/*
 *  Copyright (c) 2008 Cavium Networks 
 * 
 *  This file is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License, Version 2, as 
 *  published by the Free Software Foundation. 
 *
 *  This file is distributed in the hope that it will be useful, 
 *  but AS-IS and WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE, TITLE, or 
 *  NONINFRINGEMENT.  See the GNU General Public License for more details. 
 *
 *  You should have received a copy of the GNU General Public License 
 *  along with this file; if not, write to the Free Software 
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA or 
 *  visit http://www.gnu.org/licenses/. 
 *
 *  This file may also be available under a different license from Cavium. 
 *  Contact Cavium Networks for more information
 *
 */

#include <asm/io.h>
#include <common.h>
#include <command.h>
#include <cnw5xxx.h>

static void cnw5xxx_wfi (void)
{
	unsigned int zero = 0;
	__asm__ __volatile__("nop\n"
			     "mcr p15, 0, %0, c7, c0, 4"
			     :
			     : "r" (zero)
			     : "cc");
}

void set_cpu_clock (void)
{
	unsigned int cpu_speed;
	unsigned int cpu_sel;
	char *cpu_clock = getenv ("cpu_clock");
	u32 u32tmp;
	int i;

	if (cpu_clock) {
		cpu_speed = simple_strtoul (cpu_clock, NULL, 10);
	} else {
		/* check pin setting (SW7/SW8) */
		u32 chip_cfg = readl(CNW5XXX_MISC_BASE + MISC_CHIP_CFG_OFFSET);

		switch ((chip_cfg & 0x000001e0) >> 5) {
			case 0x0:
				cpu_speed = 200;
				break;
			case 0x1:
				cpu_speed = 233;
				break;
			case 0x2:
				cpu_speed = 266;
				break;
			case 0x3:
				cpu_speed = 300;
				break;
			case 0x4:
				cpu_speed = 333;
				break;
			case 0x5:
				cpu_speed = 366;
				break;
			case 0x6:
				cpu_speed = 400;
				break;
			case 0x7:
				cpu_speed = 433;
				break;
			case 0x8:
				cpu_speed = 466;
				break;
			case 0x9:
				cpu_speed = 500;
				break;
			case 0xa:
				cpu_speed = 533;
				break;
			case 0xb:
				cpu_speed = 566;
				break;
			case 0xc:
				cpu_speed = 600;
				break;
			default:
				cpu_speed = 466;	/* default */
		}
	}

	if (cpu_speed == get_cpu_speed()) 
	{
        if ( 1 == gMyCpuCoreNo )
        {
    		printf("CPU Speed: %d MHz \n", cpu_speed);
    	}	
		return;
	}

	switch (cpu_speed) {
		case 200:
			cpu_sel = 0;
			break;
		case 233:
			cpu_sel = 1;
			break;
		case 266:
			cpu_sel = 2;
			break;
		case 300:
			cpu_sel = 3;
			break;
		case 333:
			cpu_sel = 4;
			break;
		case 366:
			cpu_sel = 5;
			break;
		case 400:
			cpu_sel = 6;
			break;
		case 433:
			cpu_sel = 7;
			break;
		case 466:
			cpu_sel = 8;
			break;
		case 500:
			cpu_sel = 9;
			break;
		case 533:
			cpu_sel = 10;
			break;
		case 566:
			cpu_sel = 11;
			break;
		case 600:
			cpu_sel = 12;
			break;
		default:
			cpu_sel = 8;	/* default */
	}

	disable_interrupts();

	/****** interrupt distributor init ******/
	/* disable GIC */
	u32tmp = readl(CNW5XXX_TC11MP_GIC_DIST_BASE);
	u32tmp &= ~(0x1);
	writel(u32tmp, CNW5XXX_TC11MP_GIC_DIST_BASE);

	/* clear */
	for (i = 0; i < 8; i++) {
		writel(0xffffffff, CNW5XXX_TC11MP_SCU_BASE + 0x1180 + i * 4);
		writel(0xffffffff, CNW5XXX_TC11MP_SCU_BASE + 0x1280 + i * 4);
	}

	for (i = 0; i < 64; i++) {
		writel(0, CNW5XXX_TC11MP_SCU_BASE + 0x1400 + i * 4);
		writel(0, CNW5XXX_TC11MP_SCU_BASE + 0x1800 + i * 4);
	}

	for (i = 0; i < 16; i++) {
		writel(0x55555555, CNW5XXX_TC11MP_SCU_BASE + 0x1C00 + i * 4);
	}

	/* enable GIC */
	u32tmp = readl(CNW5XXX_TC11MP_GIC_DIST_BASE);
	u32tmp |= (0x1);
	writel(u32tmp, CNW5XXX_TC11MP_GIC_DIST_BASE);

	/****** interrupt interface init ******/
	/* clear */
	writel(0xffffffff, CNW5XXX_TC11MP_SCU_BASE + 0x1280);
	
	/* reset interrupt priority */
	for (i = 0; i < 8; i++) {
		writel(0, CNW5XXX_TC11MP_SCU_BASE + 0x1400 + i * 4);
	}
	
	/* set interrupt configuration (ID0~ID31): rising edge */
	writel(0xaaaaaaaa, CNW5XXX_TC11MP_SCU_BASE + 0x1C00);
	writel(0xaaaaaaaa, CNW5XXX_TC11MP_SCU_BASE + 0x1C04);

	/* disable the CPU interface */
	writel(0, CNW5XXX_TC11MP_GIC_CPU_BASE);

	/* allow interrupts with higher priority than 0xf */
	writel(0x000000f0, CNW5XXX_TC11MP_SCU_BASE + 0x0104);

	/* all priority bits are compared for pre-emption */
	writel(0x00000003, CNW5XXX_TC11MP_SCU_BASE + 0x0108);

	/* clear any pending interrupts */
	for (i = 0; i < 64; i++) {
		u32tmp = readl(CNW5XXX_TC11MP_SCU_BASE + 0x010C);
		if ((u32tmp & 0x03ff) == 1023)
			break;
		writel(u32tmp, CNW5XXX_TC11MP_SCU_BASE + 0x0110);
	}

	/* enable the CPU interface */
	writel(1, CNW5XXX_TC11MP_GIC_CPU_BASE);

	/****** register interrupt ******/
	/* register interrupt ID32 */
	u32tmp = readl(CNW5XXX_TC11MP_SCU_BASE + 0x1C08);
	u32tmp |= (1 << 1);		/* edge trigger */
	u32tmp &= ~(1 << 0);		/* N-N */
	writel(u32tmp, CNW5XXX_TC11MP_SCU_BASE + 0x1C08);

	u32tmp = readl(CNW5XXX_TC11MP_SCU_BASE + 0x1420);
	u32tmp &= ~(0x0f << 4);		/* priority level */
	writel(u32tmp, CNW5XXX_TC11MP_SCU_BASE + 0x1420);

	u32tmp = readl(CNW5XXX_TC11MP_SCU_BASE + 0x1820);
	u32tmp |= 1;			/* set CPU target */
	writel(u32tmp, CNW5XXX_TC11MP_SCU_BASE + 0x1820);

	u32tmp = readl(CNW5XXX_TC11MP_SCU_BASE + 0x1104);
	u32tmp |= 1;			/* enable ID32 */
	writel(u32tmp, CNW5XXX_TC11MP_SCU_BASE + 0x1104);

	/* */
	u32tmp = readl(CNW5XXX_PMU_BASE + PMU_SYS_CLK_CTRL_OFFSET);
	u32tmp &= ~(0x0f);
	u32tmp |= cpu_sel;		/* PLL_CPU frequency select */
	writel(u32tmp, CNW5XXX_PMU_BASE + PMU_SYS_CLK_CTRL_OFFSET);

	u32tmp = readl(CNW5XXX_PMU_BASE + PMU_SYS_CLK_CTRL_OFFSET);
	u32tmp &= ~(0x07 << 9);		/* set CPU power in DFS mode */
	writel(u32tmp, CNW5XXX_PMU_BASE + PMU_SYS_CLK_CTRL_OFFSET);

	/* disable all interrupt except ID32 (clkscale_intr) */
	writel(0xfffffffe, CNW5XXX_TC11MP_SCU_BASE + 0x1184);
	writel(0xffffffff, CNW5XXX_TC11MP_SCU_BASE + 0x1188);

	u32tmp = readl(CNW5XXX_TC11MP_SCU_BASE + 0x1104);
	u32tmp |= (1 << 0);
	writel(u32tmp, CNW5XXX_TC11MP_SCU_BASE + 0x1104);

	/* set DMC to low power hand shake */
	u32tmp = readl(CNW5XXX_PMU_BASE + PMU_HS_CFG_OFFSET);
	u32tmp |= (0x1 << 2);
	writel(u32tmp, CNW5XXX_PMU_BASE + PMU_HS_CFG_OFFSET);

	/* disable DMC */
	u32tmp = readl(CNW5XXX_PMU_BASE + PMU_CLK_GATE_OFFSET);
	u32tmp &= ~(0x1 << 2);
	writel(u32tmp, CNW5XXX_PMU_BASE + PMU_CLK_GATE_OFFSET);

	if (1 == gMyCpuCoreNo)
	{
		enable_interrupts();
		/* let CPU enter WFI mode */
		cnw5xxx_wfi();
	    printf("CPU Speed: %d MHz \n", get_cpu_speed());
	}
}

static unsigned int get_ddr2_speed (void)
{
	u32 chip_cfg = *(volatile unsigned int*)(CNW5XXX_PMU_BASE + PMU_SYS_CLK_CTRL_OFFSET);
	unsigned int ddr2_speed;

	switch ((chip_cfg & 0x00000180) >> 7) 
	{
		case 0:
			ddr2_speed = 400;
			break;
		case 1:
			ddr2_speed = 533;
			break;
		case 2:
			ddr2_speed = 667;
			break;
		case 3:
			ddr2_speed = 800;
			break;
		default:
			ddr2_speed = 800;
	}
	return ddr2_speed;
}

void set_smc_clock (void)
{
    if ( 1 == gMyCpuCoreNo )
    {
	    printf("DDR2 Speed: %d MHz \n", get_ddr2_speed());
	}	
}

