/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * (C) Copyright 2002
 * David Mueller, ELSOFT AG, <d.mueller@elsoft.ch>
 *
 * (C) Copyright 2003
 * Texas Instruments, <www.ti.com>
 * Kshitij Gupta <Kshitij@ti.com>
 *
 * (C) Copyright 2004
 * ARM Ltd.
 * Philippe Robin, <philippe.robin@arm.com>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
/*
 * The RealView Emulation BaseBoard provides timers and soft reset
 * - the cpu code does not need to provide these.
 */
#include <common.h>

#include <cnw5xxx.h>

DECLARE_GLOBAL_DATA_PTR;

static ulong timestamp;
static ulong lastdec;
static int timer_div = 75000;
static unsigned long cnw5xxx_l2_base;

#define IO_WRITE(addr, val) (*(volatile unsigned int *)(addr) = (val))
#define IO_READ(addr) (*(volatile unsigned int *)(addr))

#define READ_TIMER (*(volatile ulong *)(CFG_TIMERBASE))

static void flash__init (void);
static void ether__init (void);
static void timer_init(void);
static void l2cc_init(void);

#if defined(CONFIG_SHOW_BOOT_PROGRESS)
void show_boot_progress(int progress)
{
    printf("Boot reached stage %d\n", progress);
}
#endif

#define COMP_MODE_ENABLE ((unsigned int)0x0000EAEF)

static inline void delay (unsigned long loops)
{
	__asm__ volatile ("1:\n"
		"subs %0, %1, #1\n"
		"bne 1b":"=r" (loops):"0" (loops));
}

/*
 * Miscellaneous platform dependent initialisations
 */

int board_init (void)
{
	gd->bd->bi_arch_number = MACH_TYPE_VEGA;

	/* adress of boot parameters */
	gd->bd->bi_boot_params = PHYS_SDRAM_1 + 0x00000100;

	gd->flags = 0;
	gd->flags |= GD_FLG_RELOC;

	icache_enable();
	dcache_enable();

	//Enable L2 Cache controller
	l2cc_init();

	flash__init ();
	ether__init ();
	timer_init();
    dram_init();
	pci_init_board();
	return 0;
}


int misc_init_r (void)
{
	setenv("verify", "n");
	return (0);
}

/******************************
 Routine:
 Description:
******************************/
static void flash__init (void)
{

}
/*************************************************************
 Routine:ether__init
 Description: take the Ethernet controller out of reset and wait
	  		   for the EEPROM load to complete.
*************************************************************/
static void ether__init (void)
{

}

/*************************************************************
 Routine:checkboard
 Description: Check Board Identity
*************************************************************/
int checkboard(void)
{
#ifdef CONFIG_DISPLAY_BOARDINFO
	char *s = getenv("serial#");
	char *s2 = getenv("board");
	if (s2 != NULL )
	{
	    puts("Board: CNWXXXX PureVu ");
	    puts (s2);
	    puts (" Board" );
	}
	else
	{
		puts("Board: CNWXXXX PureVu Validation Board");
	}

	if (s != NULL) {
		puts(", serial# ");
		puts(s);
	}
	putc('\n');
#endif
	return (0);
}

/******************************
 Routine:
 Description:
******************************/
int dram_init (void)
{
	gd->bd->bi_dram[0].start = PHYS_SDRAM_1;
        gd->bd->bi_dram[0].size = PHYS_SDRAM_1_SIZE;

    return 0;
}

/*
 * Start the timer
 * U-Boot expects a 32 bit timer, running at CFG_HZ == 1000
 */
static void timer_init(void)
{
	unsigned int u32tmp;
	unsigned int cpu_clk, cpu_sel, div_sel;

	/* UART clock enable */
	u32tmp = IO_READ(CNW5XXX_PMU_BASE + PMU_CLK_GATE_OFFSET);
	u32tmp |= (0x1 << 14);
	IO_WRITE(CNW5XXX_PMU_BASE + PMU_CLK_GATE_OFFSET, u32tmp);

	/* UART software reset */
	u32tmp = IO_READ(CNW5XXX_PMU_BASE + PMU_SOFT_RST_OFFSET);
	u32tmp &= ~(0x1 << 14);
	IO_WRITE(CNW5XXX_PMU_BASE + PMU_SOFT_RST_OFFSET, u32tmp);
	u32tmp |= (0x1 << 14);
	IO_WRITE(CNW5XXX_PMU_BASE + PMU_SOFT_RST_OFFSET, u32tmp);

	/*
	 * Now setup timer1
	 */	
	*(volatile ulong *)(CFG_TIMERBASE + 0x00) = CFG_TIMER_RELOAD;
	*(volatile ulong *)(CFG_TIMERBASE + 0x04) = CFG_TIMER_RELOAD;
	*(volatile ulong *)(CFG_TIMERBASE + 0x30) |= 0x0201;	/* Enabled,
								 * down counter,
								 * no interrupt,
								 * 32-bit,
								 */

	/* scott.silicon */
	/* PCLK is timer1 clock source, pclk = (pll_cpu_clk_div / 8) */

	#define CPU_BASE	300

	u32tmp = IO_READ(CNW5XXX_PMU_BASE + PMU_SYS_CLK_CTRL_OFFSET);
	cpu_sel = (u32tmp >> 0) & 0xf;

	u32tmp = IO_READ(CNW5XXX_PMU_BASE + PMU_SYS_CLK_CTRL_OFFSET);
	div_sel = (u32tmp >> 4) & 0x3;

	cpu_clk = (CPU_BASE + ((cpu_sel / 3) * 100) + ((cpu_sel % 3) * 33)) >> div_sel;
	timer_div = (cpu_clk >> 3) * CFG_HZ;

	reset_timer_masked();
}

int interrupt_init (void){
	return 0;
}

/*
 * Write the system control status register to cause reset
 */
/* reset_cpu in cpu/arm11mpcore/start.S
void reset_cpu(ulong addr)
{

}
*/

/* delay x useconds AND perserve advance timstamp value */
/* ASSUMES timer is ticking at 1 msec			*/
void udelay (unsigned long usec)
{
	/* scott.silicon */
#ifdef CONFIG_HARD_TIMER
	delay(usec);
	return;
#endif

	ulong tmo, tmp;

	tmo = usec/100;

	/* scott.silicon */
	if (1000 <= usec) {
		tmo = usec / 1000;
		tmo *= CFG_HZ;
		tmo /= 1000;
	} else {
		tmo = usec * CFG_HZ;
		tmo /= (1000 * 1000);
	}

	tmp = get_timer (0);		/* get current timestamp */

	if( (tmo + tmp + 1) < tmp )	/* if setting this forward will roll time stamp */
		reset_timer_masked ();	/* reset "advancing" timestamp to 0, set lastdec value */
	else
		tmo += tmp;		/* else, set advancing stamp wake up time */

	while (get_timer_masked () < tmo)/* loop till event */
		/*NOP*/;
}

ulong get_timer (ulong base)
{
	return get_timer_masked () - base;
}

void reset_timer_masked (void)
{
	/* reset time */ /* scott.silicon */
	//lastdec = READ_TIMER/1000;  /* capure current decrementer value time */
	lastdec = READ_TIMER/timer_div;  /* capure current decrementer value time */
	timestamp = 0;	       	    /* start "advancing" time stamp from 0 */
}

void reset_timer (void)
{
#ifdef CONFIG_CMD_NAND
	reset_timer_masked();
#endif
}

/* ASSUMES 1MHz timer */
ulong get_timer_masked (void)
{
	/* scott.silicon */
	//ulong now = READ_TIMER/1000;	/* current tick value @ 1 tick per msec */
	ulong now = READ_TIMER/timer_div;	/* current tick value @ 1 tick per msec */

	if (lastdec >= now) {		/* normal mode (non roll) */
		/* normal mode */
		timestamp += lastdec - now; /* move stamp forward with absolute diff ticks */
	} else {			/* we have overflow of the count down timer */
		/* nts = ts + ld + (TLV - now)
		 * ts=old stamp, ld=time that passed before passing through -1
		 * (TLV-now) amount of time after passing though -1
		 * nts = new "advancing time stamp"...it could also roll and cause problems.
		 */
		timestamp += lastdec + TIMER_LOAD_VAL - now;
	}
	lastdec = now;

	return timestamp;
}

/*
 *  u32 get_board_rev() for ARM supplied development boards
 */
ARM_SUPPLIED_GET_BOARD_REV

#if defined(CONFIG_CNW5XXX) && defined(CONFIG_FLASH_CFI_LEGACY)
ulong board_flash_get_legacy (ulong base, int banknum, flash_info_t *info)
{
	if (banknum == 0) {     /* non-CFI boot flash */
		info->portwidth = FLASH_CFI_16BIT;
		info->chipwidth = FLASH_CFI_BY16;
		info->interface = FLASH_CFI_X8X16;
		return 1;
	} else
		return 0;
}
#endif

static inline void cache_wait(unsigned long reg, unsigned long mask)
{
	/* wait for the operation to complete */
	while (IO_READ(reg) & mask)
	{
		;	
	}
}

static inline void sync_writel(unsigned long val, unsigned long reg, unsigned long complete_mask)
{
	//unsigned long flags;
	IO_WRITE( cnw5xxx_l2_base + reg, val);
	/* wait for the operation to complete */
	while (IO_READ(cnw5xxx_l2_base + reg) & complete_mask)
	{
		;
	}
}

static inline void cache_sync(void)
{
	sync_writel(0, L2CC_CACHE_SYNC, 1);
}

void cnw5xxx_l2_inv_all(void)
{
	/* invalidate all ways */
	sync_writel(0xffff, L2CC_INV_WAY, 0xffff);
	cache_sync();
}

#if 0 //NOT_USED - disabled to remove compiler warning.
static void cnw5xxx_l2_inv_range(unsigned long start, unsigned long end)
{
	unsigned long addr;

	if (start & (CACHE_LINE_SIZE - 1)) 
	{
		start &= ~(CACHE_LINE_SIZE - 1);
		IO_WRITE( cnw5xxx_l2_base + L2CC_CLEAN_INV_LINE_PA, start);
		start += CACHE_LINE_SIZE;
	}

	if (end & (CACHE_LINE_SIZE - 1)) 
	{
		end &= ~(CACHE_LINE_SIZE - 1);
		IO_WRITE( cnw5xxx_l2_base + L2CC_CLEAN_INV_LINE_PA, end);
	}

	for (addr = start; addr < end; addr += CACHE_LINE_SIZE)
	{
		IO_WRITE( cnw5xxx_l2_base + L2CC_INV_LINE_PA, addr);
	}
	cache_sync();
}

static void cnw5xxx_l2_clean_range(unsigned long start, unsigned long end)
{
	unsigned long addr;

	start &= ~(CACHE_LINE_SIZE - 1);
	for (addr = start; addr < end; addr += CACHE_LINE_SIZE)
	{
		IO_WRITE( cnw5xxx_l2_base + L2CC_CLEAN_LINE_PA, addr);
	}

	cache_wait(cnw5xxx_l2_base + L2CC_CLEAN_LINE_PA, 1);
	cache_sync();
}

static void cnw5xxx_l2_flush_range(unsigned long start, unsigned long end)
{
	unsigned long addr;

	start &= ~(CACHE_LINE_SIZE - 1);
	for (addr = start; addr < end; addr += CACHE_LINE_SIZE)
	{
		IO_WRITE( cnw5xxx_l2_base + L2CC_CLEAN_INV_LINE_PA, addr );
	}
	cache_wait(cnw5xxx_l2_base + L2CC_CLEAN_INV_LINE_PA, 1);
	cache_sync();
}

#endif //NOT_USED - disabled to remove compiler warning.
static void l2cc_init(void)
{
	if ( 1 == gMyCpuCoreNo )
	{
	unsigned long aux, prefetch, tag, data;

	cnw5xxx_l2_base = CNW5XXX_L2CC_BASE_VIRT;

	/* disable L2CC */
	IO_WRITE( cnw5xxx_l2_base + L2CC_CTRL, 0);

	/*
	 * Auxiliary control register 
	 *
	 * bit[22]	- shared attribute internally ignored
	 * bit[21]	- parity enabled
	 * bit[20]	- 
	 * bit[19:17]	- 32kB way size 
	 * bit[16]	- way associative
	 * bit[12]	- exclusive cache disabled
	 *
	 */
	aux = IO_READ(cnw5xxx_l2_base + L2CC_AUX_CTRL);
	aux &= 0xfe000fff;
#ifdef CONFIG_CACHE_L2_I_PREFETCH
	aux |= 0x20000000;	/* bit[29]: Instruction prefetching enable, bit[28]: Data prefetching enable */
#endif
	aux |= 0x00520000;	/* ...001..., 16KB, 8-way, Parity Disable*/
	IO_WRITE( cnw5xxx_l2_base + L2CC_AUX_CTRL, aux);

	prefetch = IO_READ(cnw5xxx_l2_base + 0xF60);
	prefetch |= 0x00000008; /* prefetch offset, bit[4..0] */
#ifdef CONFIG_CACHE_L2_I_PREFETCH
	prefetch |= 0x20000000;
#endif
#ifdef CONFIG_CACHE_L2_D_PREFETCH
	prefetch |= 0x10000000;
#endif
	IO_WRITE( cnw5xxx_l2_base + 0xF60 , prefetch);

	/* Tag RAM Control register
	 * 
	 * bit[10:8]	- 1 cycle of write accesses latency
	 * bit[6:4]	- 1 cycle of read accesses latency
	 * bit[3:0]	- 1 cycle of setup latency
	 *
	 * 1 cycle of latency for setup, read and write accesses
	 */
	tag = IO_READ(cnw5xxx_l2_base + L2CC_TAG_RAM_LATENCY_CTRL);
	tag &= 0xfffff888;
	tag |= 0x00000000;
	IO_WRITE( cnw5xxx_l2_base + L2CC_TAG_RAM_LATENCY_CTRL, tag);

	/* Data RAM Control register
	 *
	 * bit[10:8]	- 1 cycles of write accesses latency
	 * bit[6:4]	- 1 cycles of read accesses latency
	 * bit[3:0]	- 1 cycle of setup latency
	 *
	 * 1 cycle of setup latency, 2 cycles of read and write accesses latency
	 */
	data = IO_READ(cnw5xxx_l2_base + L2CC_DATA_RAM_LATENCY_CTRL);
	data &= 0xfffff888;
	data |= 0x00000000;
	IO_WRITE( cnw5xxx_l2_base + L2CC_DATA_RAM_LATENCY_CTRL, data);

	cnw5xxx_l2_inv_all();

	/* lockdown required ways for different effective size of the L2 cache */
#ifdef CONFIG_CACHE_L2CC_16KB
        /* 16KB, lock way7..1 */
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_D, 0xfe);
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_I, 0xfe);
#elif defined(CONFIG_CACHE_L2CC_32KB)
        /* 32KB, lock way7..2 */
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_D, 0xfc);
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_I, 0xfc);
#elif defined(CONFIG_CACHE_L2CC_48KB)
        /* 48KB, lock way7..3 */
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_D, 0xf8);
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_I, 0xf8);
#elif defined(CONFIG_CACHE_L2CC_64KB)
        /* 64KB, lock way7..4 */
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_D, 0xf0);
        IO_WRITE( cnw5xxx_l2_base + L2CC_LOCKDOWN_0_WAY_I, 0xf0);
#endif

	/* enable L2CC */
	IO_WRITE( cnw5xxx_l2_base + L2CC_CTRL, 1);
	}
}

