#include <common.h>
#include <command.h>
#include <environment.h>
#include <watchdog.h>
#include <serial.h>
#include <linux/stddef.h>
#include <asm/byteorder.h>
#if defined(CONFIG_CMD_NET)
#include <net.h>
#endif

#include <malloc.h>




#define SECURED_PARAMETER_IN_FLASH  ( 0x100C0000 )
#define SECURED_PARAMETER_SIGNATURE ( 0xCA98CADE )
#define HDCP_PASSWORD   ( "cavm_hdcp" )
#define FLASH_SECTOR_SIZE (128*1024)

static int AllowEdit = 0;

SecNvRam_t gSecNvRam;

extern int do_protect (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);
extern int do_flerase (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);
extern int do_mem_cp ( cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);

//-----------------------------------------------------------------------------------------

static UInt32 Sec_GetCrc32(void)
{
  return crc32(0, (void*) & gSecNvRam.mNvRamBuffer[0] , sizeof ( gSecNvRam.mNvRamBuffer ) );
}

#if 0   //not used - disabled to remove warning.
static int Sec_VerifyCrc32(void)
{
    UInt32 crc ;
    crc = crc32(0, (void * ) & gSecNvRam.mNvRamBuffer[0] , sizeof ( gSecNvRam.mNvRamBuffer ) ) ;
    if ( gSecNvRam.mCrc32 ==  crc )
    {
        return 0;
    }
    return -1;
}
#endif

static int Sec_ProtectSector(void)
{
    int status;
	char TempCmd2_1[16];	char TempCmd2_2[16];

    UInt32 aDestinationAddressInFlash = SECURED_PARAMETER_IN_FLASH ;
    UInt32 aLength = ((sizeof (gSecNvRam) + FLASH_SECTOR_SIZE -1 ) / FLASH_SECTOR_SIZE ) * FLASH_SECTOR_SIZE;

    char* pCmd3[4] = {  "protect" , "on" ,  NULL ,  NULL  } ;

    sprintf(TempCmd2_1 , "0x%x", aDestinationAddressInFlash );		sprintf(TempCmd2_2 , "0x%x", aDestinationAddressInFlash + aLength - 1 );		//erase command.

   	pCmd3[2] =  TempCmd2_1 ;	pCmd3[3] =  TempCmd2_2 ;
    status = do_protect(NULL, 0,4,pCmd3); //un protect sector
    if ( 0 != status )
    {
        status = -1;
        printf("ERROR: fail to un-protect secure parameter sector\n");
    }
    return status;
}
static int Sec_UnProtectSector(void)
{
    int status;
	char TempCmd2_1[16];	char TempCmd2_2[16];

    UInt32 aDestinationAddressInFlash = SECURED_PARAMETER_IN_FLASH ;
    UInt32 aLength = ((sizeof (gSecNvRam) + FLASH_SECTOR_SIZE -1 ) / FLASH_SECTOR_SIZE ) * FLASH_SECTOR_SIZE;

    char* pCmd3[4] = {  "protect" , "off" ,  NULL ,  NULL  } ;

    sprintf(TempCmd2_1 , "0x%x", aDestinationAddressInFlash );		sprintf(TempCmd2_2 , "0x%x", aDestinationAddressInFlash + aLength - 1 );		//erase command.

   	pCmd3[2] =  TempCmd2_1 ;	pCmd3[3] =  TempCmd2_2 ;
    status = do_protect(NULL, 0,4,pCmd3); //un protect sector
    if ( 0 != status )
    {
        status = -1;
        printf("ERROR: fail to un-protect secure parameter sector\n");
    }
    return status;
}
static int Sec_EraseSector(void)
{
    UInt32 aDestinationAddressInFlash = SECURED_PARAMETER_IN_FLASH ;
    UInt32 aLength = ((sizeof (gSecNvRam) + FLASH_SECTOR_SIZE -1 ) / FLASH_SECTOR_SIZE ) * FLASH_SECTOR_SIZE;
    int status=-1;
	char TempCmd2_1[16];	char TempCmd2_2[16];
   	char* pCmd2[3] = {  "erase" ,   NULL ,  NULL  } ;
    sprintf(TempCmd2_1 , "0x%x", aDestinationAddressInFlash );		sprintf(TempCmd2_2 , "0x%x", aDestinationAddressInFlash + aLength - 1 );		//erase command.
	pCmd2[1] =  TempCmd2_1 ;	pCmd2[2] =  TempCmd2_2 ;
    status = do_flerase(NULL, 0,3, pCmd2) ;	
    if ( 0 != status )
    {
        printf("ERROR: fail to erase secure parameter region\n");
    }
    return status;
}
static int Sec_LoadNvramFromFlash(void)
{
    memcpy ( (void*) & gSecNvRam , (void*) SECURED_PARAMETER_IN_FLASH , sizeof ( gSecNvRam ) );
	if (gSecNvRam.mSignature != SECURED_PARAMETER_SIGNATURE )
    {
		printf("ERROR: Secure Nvram signature missing\n");
        return -1;
    }
	if ( gSecNvRam.mCrc32 != Sec_GetCrc32() )
	{
		printf("ERROR: Secure Nvram CRC corrupted\n");
		return -1;
	}
	return 0;
}
static int Sec_SaveNvramToFlash(void)
{
     UInt32 crc ;
    int status=0;
	//int fail_cnt=0;
    unsigned int aSourceAddressInMemory = (UInt32) & gSecNvRam;
    unsigned int aDestinationAddressInFlash = SECURED_PARAMETER_IN_FLASH ;
    unsigned int aLength = sizeof (gSecNvRam);
	//char BackupBuffer[512];
	//char TempCmd2_1[16];	char TempCmd2_2[16];
	char TempCmd31_1[16];	char TempCmd31_2[16];	char TempCmd31_3[16];
	//char TempCmd32_1[16];	char TempCmd32_2[16];	char TempCmd32_3[16];
    char* pCmd31[4] = {  "cp.b"  ,   NULL ,  NULL, NULL } ;

    sprintf(TempCmd31_1 , "0x%x", aSourceAddressInMemory+0 );	sprintf(TempCmd31_2 , "0x%x", aDestinationAddressInFlash + 0 );	sprintf(TempCmd31_3 , "0x%x", aLength );		
	pCmd31[1] =  TempCmd31_1 ;	pCmd31[2] =  TempCmd31_2 ;	pCmd31[3] =  TempCmd31_3 ;

    crc = Sec_GetCrc32();

    gSecNvRam.mCrc32 = crc;

    gSecNvRam.mSignature = SECURED_PARAMETER_SIGNATURE;


    Sec_UnProtectSector();

    Sec_EraseSector();

    printf("%s %s %s %s\n", pCmd31[0], pCmd31[1], pCmd31[2], pCmd31[3]);
	status = do_mem_cp(NULL, 0,4, pCmd31);


    if ( 0 != memcmp( (void*) SECURED_PARAMETER_IN_FLASH , (void*) & gSecNvRam , sizeof(gSecNvRam) ) )
    {
        status = -1;
        printf("ERROR: Failed to write Secured parameter to flash\n");
    }
    else
    {
        status = 0;
        printf("Secured paramter written to flash successfully\n");
    }

    Sec_ProtectSector();

    return status;

}
static char* Sec_FindEndPtr(void)
{
    int i;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];
    for( i=0; i < (SECURED_PARAMETER_SIZE -1)  ; i ++ )
    {
        if ( ( *(pPtr+i) == 0) &&  ( * (pPtr+i+1) == 0) )
        {
			if ( i== 0 )
			{
				i--;
			}
			pPtr = pPtr + i +1;
			return pPtr;
        }
    }
	return NULL;
}
static char* Sec_FindEndPtrFromCurrentPTr(char* apCurrentPtr)
{
    int i;
    char* pPtr = apCurrentPtr ;
    for(i=0 ; i < ( SECURED_PARAMETER_SIZE - 1 )  ; i ++ )
    {
        if ( ( *(pPtr+i) == 0) &&  ( * (pPtr+i+1) == 0) )
        {
			if ( i== 0 )
			{
				i--;
			}
             pPtr = pPtr + i+1;
            return pPtr;
        }
    }
	return NULL;
}
static int Sec_FindPositionForAGivenTag( char* apTag, char ** apValue )
{
	int status = -1;
    char TagName[128]={0};
    int Cnt;
    char* p1;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];

    while(*pPtr)
    {
        Cnt = strlen(pPtr);
        p1 = strstr( pPtr, "=" );
        if (p1 != NULL )
        {
            memset(TagName,0,sizeof(TagName) );
            memcpy(TagName , pPtr, p1 - pPtr); 
            if ( 0 == strcmp ( TagName , apTag) )
            {
                //tag is mataching.
                *apValue = pPtr ;
                return 0;
            }
            pPtr = pPtr + Cnt +1;
        }
    }
    return status;
}

void Sec_ClearAll(void)
{
	UInt32 crc;
	memset( & gSecNvRam , 0, sizeof (SecNvRam_t) );
	crc = Sec_GetCrc32();

    gSecNvRam.mCrc32 = crc;

    gSecNvRam.mSignature = SECURED_PARAMETER_SIGNATURE;

}

int Sec_GetTagCount(void)
{
    //int status = -1;
    int Cnt;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];
	//char * x=NULL;
    //char** pValue=&x;
    int TagCount=0;    

    while(*pPtr)
    {
        Cnt = strlen(pPtr);
        TagCount++;
        pPtr = pPtr + Cnt + 1 ;
    }
    return TagCount;
}
char* Sec_GetTagValAtOffset(int aOffset)
{
    //int status = -1;
    int Cnt;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];
	//char * x=NULL;
    //char** pValue=&x;
    int TagCount=0;    

    while(*pPtr)
    {
        Cnt = strlen(pPtr);
        if (TagCount == aOffset)
        {
            return pPtr;
        }
        TagCount++;
        pPtr = pPtr + Cnt + 1 ;
    }
    return NULL;
}

void Sec_Sort(void)
{
    return;
	int i;
	char** p;
	char*p1;

	//char buffer[256];    /* space to store an input string    */
	
	char *pTemp = NULL;         /* Temporary pointer                 */
	int sorted = 0;         /* Indicated when strings are sorted */


	int Count = Sec_GetTagCount();
	if (Count <2)
	{
		return; //nothing to sort
	}
	
	p = ( char** ) malloc ( sizeof (char*) * Count );
	if ( p == NULL )
	{
		return;
	}

	for(i=0;i<Count;i++)
	{
		p1 = Sec_GetTagValAtOffset(i);
		p [i] = (char*) malloc (sizeof(char) *   strlen(p1) ) ;
		strcpy( p[i] , p1);
	}

	i = 0;                  /* Loop counter                      */

	while(!sorted)
	{
		sorted = 1;
		for (i = 0 ; i < Count - 1 ; i++)
		if(strcmp(p[i], p[i + 1]) > 0)
		{
			sorted = 0;     /* We were out of order */
			pTemp= p[i];       /* Swap pointers pS[i]  */
			p[i] = p[i + 1];  /*       and            */
			p[i + 1]  = pTemp; /*     pS[i + 1]        */
		} 
	}

	Sec_ClearAll();
	int k=0;
	for(i=0;i<Count;i++)
	{
		 strcpy( gSecNvRam.mNvRamBuffer + k , p[i]);
		 k = k + strlen(p[i]);
		 k++;
	}


	//cleanup....
	for(i=0;i<Count;i++)
	{
		if ( p[i] != NULL ) 
		{
			free(p [i] );
		}
	}
	if ( p != NULL ) 
	{
		free(p);
	}
}

//-----------------------------------------------------------------------------------------
int Sec_GetNvRam(char* apTag, char** apValue)
{
    int status = -1;
    char TagName[128]={0};
    int Cnt;
    char* p1;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];

    while(*pPtr)
    {
        Cnt = strlen(pPtr);
        p1 = strstr( pPtr, "=" );
        if (p1 != NULL )
        {
            memcpy(TagName , pPtr, p1 - pPtr); 
            if ( 0 == strcmp ( TagName , apTag) )
            {
                //tag is mataching.
                *apValue = p1+1;
                return 0;
            }
            pPtr = pPtr + Cnt;
        }
    }
    return status;
}

void Sec_NvramPrint(char* apTag)
{
    int status = -1;
    //char TagName[128]={0};
    int Cnt;
    //char* p1;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];
	char * x=NULL;
    char** pValue=&x;

    if (apTag == NULL )
    {
        //print all
        while(*pPtr)
        {
            Cnt = strlen(pPtr);
            if ( pPtr !=  strstr(pPtr, "seccer_" ) )
            {
                printf("%s\n", pPtr);
            }
            pPtr = pPtr + Cnt + 1 ;
        }
    }
    else
    {
        //Print reqested
        status = Sec_GetNvRam( apTag, pValue);
        if (status == 0 )
        {
            printf("%s=%s\n" , apTag , *pValue );
        }
    }
}

int Sec_SetNvRam(char* apTag, char* apValue)
{
    int status = -1;
    //char TagName[128]={0};
    int Cnt;
    //char* p1;
    //char* pPtr = & gSecNvRam.mNvRamBuffer[0];

	char * x=NULL;
    char** pTagPosition=&x;
    char* pStoragePtr;

    char* pSrcPtr=NULL;
    char* pDestPtr=NULL;
    

     status = Sec_FindPositionForAGivenTag( apTag, pTagPosition);

     if (status == 0 )
     {
         //Tag exist - we have to move remaing blob.
         pSrcPtr = *pTagPosition + strlen( *pTagPosition) + 1;

         Cnt =  Sec_FindEndPtrFromCurrentPTr(pSrcPtr) - *pTagPosition  ;

         pDestPtr = *pTagPosition ;
         memcpy(pDestPtr, pSrcPtr, Cnt );
     }

    //Tag doesn't exist - store at tail
	 if( (NULL != apValue ) && ( 0 != strcmp(apValue, "" ) ) )
	 {
			pStoragePtr = Sec_FindEndPtr();
			if (pStoragePtr != NULL )
			{
				if( (pStoragePtr + strlen(apTag) + 1 + strlen(apValue) - & gSecNvRam.mNvRamBuffer[0] ) < sizeof (gSecNvRam.mNvRamBuffer) )
				{
					strcpy(pStoragePtr , apTag);       pStoragePtr = pStoragePtr + strlen(apTag);
					strcpy( pStoragePtr , "=" );       pStoragePtr = pStoragePtr + 1 ;
					strcpy( pStoragePtr , apValue );   
                    status = 0;
				}
				else
				{
                    status = -1;
					printf("ERROR-2: No room to store parameter\n");
				}
			}
			else
			{
                status = -1;
				printf("ERROR: No room to store parameter\n");
			}
	 }

	 Sec_Sort();
     return status;
}

int Sec_Init(void)
{
	int status = -1;
    memset( & gSecNvRam , 0, sizeof (SecNvRam_t) );
    status = Sec_LoadNvramFromFlash();
    if (0 == status )
    {
        printf("Secured parameter OK...\n");
    }
    else
    {
        printf("ERROR: Invalid Secured parameter found, OR it looks like secured parameter was never programmed on this board.\n");
        memset( & gSecNvRam , 0, sizeof (SecNvRam_t) );
        //Sec_SaveNvramToFlash();
    }
	Sec_ProtectSector();
	return status;
}


int do_printsecenv(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    int i;
    int status = 0;
    if (argc == 1 )
    {
        Sec_NvramPrint(NULL);
    }
    else
    {
        for(i=1;i<argc;i++)
        {
            Sec_NvramPrint(argv[i]);
        }
    }
    return status;
}

int do_printseccer(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{

    //int status = -1;
    //char TagName[128]={0};
    int Cnt;
    //char* p1;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];
	//char * x=NULL;
    //char** pValue=&x;

    
    //print only hdcp certificate only.
    while(*pPtr)
    {
        Cnt = strlen(pPtr);
        if ( pPtr ==  strstr(pPtr, "seccer_" ) )
        {
            printf("%s\n", pPtr);
        }
        pPtr = pPtr + Cnt + 1 ;
    }
    return 0;

}

int do_setsecenv(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    int i;
    int status = 0;
    char TempBuf[256]={0};
    if (argc == 2 )
    {
        Sec_SetNvRam(argv[1], NULL );
    }
    else
    {
        for( i = 2 ;  i < argc ; i++ )
        {
            strncat(TempBuf, argv[i] , sizeof(TempBuf) );
            strncat(TempBuf, " " , sizeof(TempBuf) );
        }
        if (TempBuf[strlen(TempBuf) -1] == ' ' )
        {
            TempBuf[strlen(TempBuf) -1] =0;
        }
        Sec_SetNvRam(argv[1], TempBuf );
    }
    return status;
}

int Sec_SaveToFlash(void)
{
    int status = 0;


    if ( 0 == AllowEdit  )
    {
        printf("ERROR: Sorry, you need password to save secured parameter\n");
        return -1;
    }

    status = Sec_SaveNvramToFlash();
    if ( 0 != status )
    {
        printf("ERROR: failed to store Secured paramter, please check hardware or contact vendor\n");
    }
    return status;
}
int do_savesecenv(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    return Sec_SaveToFlash();
}

int do_pswdsec(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    int status = -1;
    if ( 0 == strcmp(argv[1] , HDCP_PASSWORD ) )
    {
        AllowEdit = 1;
        printf("Password OK, now you can save secured parameter\n");
        return 0;
    }
    else
    {
        printf("ERROR: wrong password entered, please contact Cavium.\n");
    }
    return status;
}



/**************************************************/
U_BOOT_CMD(
	printsecenv, CFG_MAXARGS, 1,	do_printsecenv,
	"printsecenv- print secured environment variables\n",
	"\n    - print values of all secured environment variables\n"
	"printenv name ...\n"
	"    - print value of environment variable 'name'\n"
);

U_BOOT_CMD(
	printseccer, 1+0, 0,	do_printseccer,
	"printseccer- print secured certificate\n",
	NULL
);

U_BOOT_CMD(
	setsecenv, CFG_MAXARGS, 0,	do_setsecenv,
	"setsecenv  - set environment variables\n",
	"name value ...\n"
	"    - set secured environment variable 'name' to 'value ...'\n"
	"setsecenv name\n"
	"    - delete environment variable 'name'\n"
);

U_BOOT_CMD(
	savesecenv, 1+0, 0,	do_savesecenv,
	"savesecenv - save secured environment variables to storage\n",
	NULL
);

U_BOOT_CMD(
	pswdsec, 1+1, 0,	do_pswdsec,
	"pswdsec - to save secured parameter enter password\n",
	NULL
);

//--------------------------------------------------------------------------------------

#if 0

int do_restore_nvram(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    int status = -1;
    int Cnt;
    char* pPtr = & gSecNvRam.mNvRamBuffer[0];
	char * x=NULL;
    char** pValue=&x;

   
        //get all
        while(*pPtr)
        {
            Cnt = strlen(pPtr);
            if ( pPtr !=  strstr(pPtr, "seccer_" ) )
            {
                printf("%s\n", pPtr);

                char * p1 = strstr( pPtr, "=" );
                char TagBuf[256];
                char ValBuf[256];
                memset(TagBuf, 0, sizeof(TagBuf) );
                memset(ValBuf, 0, sizeof(ValBuf) );
                if ( p1 != NULL )
                {
                    memcpy( TagBuf , pPtr,  p1 - pPtr);
                    strcpy( ValBuf, p1+1 );
                    setenv(TagBuf, ValBuf);
                }
            }
            pPtr = pPtr + Cnt + 1 ;
        }
        saveenv();


    return status;
}
U_BOOT_CMD(
	restore_nvram, 1+0, 0,	do_restore_nvram,
	"restore nvram - copies secured nvram to regular nvram(except hdcp certificate)\n",
	NULL
);

typedef enum BOARD_TYPE
{
	MAUI_ENCODER=0,	//Don't change order - otherwise update mDefaultNvRamParameter table also.
	MAUI_DECODER=1,
	MOKU_ENCODER=2,
	MOKU_DECODER=3,
    CUSTOM_ENCODER=4,
    CUSTOM_DECODER=5,
	BOARD_TYPE_ENDS
}BOARD_TYPE;

static BOARD_TYPE mSec_BoardType = MAUI_DECODER;

static const char* mSec_BoardName[] = { "maui_encoder",		"maui_decoder",		"moku_encoder",		"moku_decoder" , "unknown"};


static const char* mSec_DefaultNvRamParameter[] = 
{
	//PARAMETER						MAUI_ENCODER		                            MAUI_DECODER		                            MOKU_ENCODER		                                MOKU_DECODER                                    CUSTOM_ENCODER		                                CUSTOM_DECODER
    "board",                        "maui_encoder",                                 "maui_decoder",                                 "moku_encoder",                                     "moku_decoder",                                 "custom_encoder",                                  "custom_decoder",
	"AUTORUN",						"1",				                            "1",				                            "1",					                            "1",                                            "1",					                            "1",
	"USB_UPDATES2",					"1",				                            "1",				                            "1",					                            "1",                                            "1",					                            "1",
	"WIFICH",						"AUTO",				                            "AUTO",				                            "AUTO",					                            "AUTO",                                         "AUTO",					                            "AUTO",
	"USB_MODE",						"0",				                            "0",				                            "0",					                            "0",                                            "0",					                            "0",
//	"bootcmd",						"go 10200000",		                            "go 10200000",		                            "go 10200000",			                            "go 10200000",                                  "go 10200000",			                            "go 10200000",
	"bootcmd2",						"",					                            "",                                             "",						                            "",                                             "",						                            "",
	"bootdelay2",					"0",				                            "0",				                            "0",					                            "0",                                            "0",					                            "0",
	"bootdelay",					"0",				                            "0",				                            "0",					                            "0",                                            "0",					                            "0",
	"bootfile",						"bootpImage",		                            "bootpImage",		                            "bootpImage",			                            "bootpImage",                                   "bootpImage",			                            "bootpImage",
	"mtdids",						"nor0=jffs2",		                            "nor0=jffs2",		                            "nor0=jffs2",			                            "nor0=jffs2",                                   "nor0=jffs2",			                            "nor0=jffs2",
	"mtdparts",						"mtdparts=jffs2:768K@0x140000(nethd)",	        "mtdparts=jffs2:768K@0x140000(nethd)",	        "mtdparts=jffs2:768K@0x140000(nethd)",	            "mtdparts=jffs2:768K@0x140000(nethd)",          "mtdparts=jffs2:768K@0x140000(nethd)",	            "mtdparts=jffs2:768K@0x140000(nethd)",
	"mtddevnum",					"0",				                            "0",				                            "0",					                            "0",                                            "0",					                            "0",
	"mtddevname",					"nethd",			                            "nethd",			                            "nethd",				                            "nethd",                                        "nethd",				                            "nethd",
	"netmask",						"255.255.0.0",		                            "255.255.0.0",		                            "255.255.0.0",			                            "255.255.0.0",                                  "255.255.0.0",			                            "255.255.0.0",
	"partition",					"nor0,0",			                            "nor0,0",			                            "nor0,0",				                            "nor0,0",                                       "nor0,0",				                            "nor0,0",
	"pv_ddr2_speed",				"667",				                            "0",				                            "667",					                            "0",                                            "667",					                            "0",
	"wfi2",							"0",				                            "0",				                            "0",					                            "0",                                            "0",					                            "0",
	"DHCP",							"0",				                            "0",				                            "0",					                            "0",                                            "0",					                            "0",
	"ENABLE_GUI",					"0",				                            "1",				                            "0",					                            "1",                                            "0",					                            "1",
	"DISABLE_GO2",					"0",				                            "0",				                            "0",					                            "0",                                            "0",					                            "0",
	"WIFIDIR",						"2",				                            "2",				                            "2",					                            "2",                                            "",					                                "",
    "GPIO_LED",                     "c",                                            "c",                                            "c",                                                "c",                                            "",                                                 "",
    "GPIO_USER_SW",                 "9",                                            "9",                                            "9",                                                "9",                                            "9",                                                "9",
    "ROLE",                         "Encoder",                                      "Decoder",                                      "Encoder",                                          "Decoder",                                      "Encoder",                                          "Decoder",
    "UPDATES_CONTROLLED_BY_GPIO",   "1",                                            "1",                                            "1",                                                "1",                                            "1",                                                "1",
    "ethaddr",                       "??:??:??:??:??:??",                           "??:??:??:??:??:??",                            "??:??:??:??:??:??",                                "??:??:??:??:??:??",                            "??:??:??:??:??:??",                                "??:??:??:??:??:??",
    "GROUPNUM",                     "?",                                            "?",                                            "?",                                                "?",                                            "?",                                                "?",
};


int Sec_GetNoOfDefaultParameter()
{
	int NoOfEntry = sizeof (mSec_DefaultNvRamParameter) / sizeof (char*) ;
	return NoOfEntry / (BOARD_TYPE_ENDS +1) ; //+1 for parameter name itself.
}

int Sec_GetDefaultParameterAndItsValue(int aParameterNo, BOARD_TYPE aBoardType,  char** apParameterName, char** apParameterValue)
{
	if ( aParameterNo >= Sec_GetNoOfDefaultParameter() )
	{
		* apParameterName = NULL;
		* apParameterValue = NULL;
		return -1;
	}

	* apParameterName = (char*) mSec_DefaultNvRamParameter[ (aParameterNo * (BOARD_TYPE_ENDS+1) )  ];
	* apParameterValue = (char*) mSec_DefaultNvRamParameter[ (aParameterNo * (BOARD_TYPE_ENDS+1) + aBoardType +1) ];

	return 0;
}

const char* Sec_GetBoardName( BOARD_TYPE aBoardType)
{
	if (aBoardType >= BOARD_TYPE_ENDS )
	{
		return NULL;
	}
	return mSec_BoardName[aBoardType];
}

int do_sec_nvram_set_defaults( cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	char* Tag;
	char* Val;
	int Cnt;
	int i;

	char arg1_buf[16]={0};

	int rc = 0;	

	if (argc < (1+1)  )
	{
		printf ("Usage:\n%s\n", cmdtp->usage);
		return -1;
	}	

	strncpy(arg1_buf, argv[1], sizeof(arg1_buf)-1 );
	strtolower(arg1_buf);

	if ( 0 == strcmp(arg1_buf, "maui_encoder") )
	{
		mSec_BoardType = MAUI_ENCODER;
		goto board_found;
	}
	else if ( 0 == strcmp(arg1_buf, "maui_decoder" ))
	{
		mSec_BoardType = MAUI_DECODER;
		goto board_found;
	}
	else if ( 0 == strcmp (arg1_buf , "moku_encoder" ))
	{
		mSec_BoardType = MOKU_ENCODER;
		goto board_found;
	}
	else if ( 0 == strcmp( arg1_buf, "moku_decoder" ))
	{
		mSec_BoardType = MOKU_DECODER;
		goto board_found;
	}
    else if ( 0 == strcmp( arg1_buf, "custome_decoder" ))
	{
		mSec_BoardType = CUSTOM_DECODER;
		goto board_found;
	}
    else if ( 0 == strcmp( arg1_buf, "custome_encoder" ))
	{
		mSec_BoardType = CUSTOM_ENCODER;
		goto board_found;
	}
	else
	{
		printf("ERROR: invalid parameter, try help\n");
		printf ("Usage:\n%s\n", cmdtp->usage);
		return -1;
	}

board_found:
	Cnt =  Sec_GetNoOfDefaultParameter() ;
	//printf("Total no. of parameter = %d\n", Cnt );

	for(i=0;i< Cnt; i++)
	{
		Sec_GetDefaultParameterAndItsValue(i, mSec_BoardType, &Tag, &Val );
		//printf(" TAG = %s, Value = %s\n" , Tag, Val );
        Sec_SetNvRam(Tag, Val);
	}
	Sec_SaveToFlash();

	Sec_LoadNvramFromFlash();

	return (rc);
}

/**************************************************/
U_BOOT_CMD(
	sec_nvram_set_defaults ,	1+1,	0,	do_sec_nvram_set_defaults ,
	"sec_nvram_set_defaults	- set nvram default parameter\n",
	"sec_nvram_set_defaults moku_encoder\n  nvram_set_defaults moku_decoder\n  nvram_set_defaults maui_encoder\n  nvram_set_defaults maui_decoder\n"
);



//-----------------------------------------------------------------------------------------------------------------------------------------------------
int do_board( cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{

	char* Tag;
	char* Val;

	char arg1_buf[16]={0};

	int rc = 0;	

	if (argc < (1)  )
	{
		printf ("Usage:\n%s\n", cmdtp->usage);
		return -1;
	}	

   	if (argc == 1)  
	{
		printf("BOARD_TYPE=%s\n",  Sec_GetBoardName( mSec_BoardType ) );
		return 0;
	}	

	strncpy(arg1_buf, argv[1], sizeof(arg1_buf)-1 );
	strtolower(arg1_buf);
    
	if ( 0 == strcmp(arg1_buf, "maui_encoder") )
	{
		mSec_BoardType = MAUI_ENCODER;
		goto board_found2;
	}
	else if ( 0 == strcmp(arg1_buf, "maui_decoder" ))
	{
		mSec_BoardType = MAUI_DECODER;
		goto board_found2;
	}
	else if ( 0 == strcmp (arg1_buf , "moku_encoder" ))
	{
		mSec_BoardType = MOKU_ENCODER;
		goto board_found2;
	}
	else if ( 0 == strcmp( arg1_buf, "moku_decoder" ))
	{
		mSec_BoardType = MOKU_DECODER;
		goto board_found2;
	}
    else if ( 0 == strcmp( arg1_buf, "custome_decoder" ))
	{
		mSec_BoardType = CUSTOM_DECODER;
		goto board_found2;
	}
    else if ( 0 == strcmp( arg1_buf, "custome_encoder" ))
	{
		mSec_BoardType = CUSTOM_ENCODER;
		goto board_found2;
	}
	else
	{
		printf("ERROR: invalid parameter, try help\n");
		printf ("Usage:\n%s\n", cmdtp->usage);
		return -1;
	}

board_found2:
	printf("BOARD_TYPE=%s\n",  Sec_GetBoardName( mSec_BoardType ) );
    Sec_SetNvRam("board" , Sec_GetBoardName(mSec_BoardType ) );
    Sec_SaveToFlash();
	return (rc);
}


U_BOOT_CMD(
	board ,	1+1,	0,	do_board ,
	"board	- get / set <BOARD_NAME> board type\n",
	"board moku_encoder\n board moku_decoder\n board maui_encoder\n board maui_decoder\n"
);

#endif
