/*
 * Copyright (c) 2008 Cavium Networks
 *
 * Scott Shu
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2, as
 * published by the Free Software Foundation.
 *
 * This file is distributed in the hope that it will be useful,
 * but AS-IS and WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE, TITLE, or
 * NONINFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this file; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA or
 * visit http://www.gnu.org/licenses/.
 *
 * This file may also be available under a different license from Cavium.
 * Contact Cavium Networks for more information
 */

#include <common.h>

#ifdef CONFIG_HARD_I2C

#include <asm/errno.h>
#include <asm/io.h>
#include <asm/arch/hardware.h>
#include <cnw5xxx.h>

#ifdef DEBUG
#define DPRINTF(args...)  printf(args)
#else
#define DPRINTF(args...)
#endif

#define I2C_ACTDONE_FG		(1 << 1)
#define I2C_BUSERR_FG		(1 << 0)

#define I2C_FS_MODE		8
#define I2C_HS_MODE		20

#define I2C_CMD_RO		0	/* useless */
#define I2C_CMD_WO		1	/* i2c_write(), write address bytes then data bytes */
#define I2C_CMD_WR		2	/* i2c_read(), write address bytes then read data bytes */
#define I2C_CMD_RW		3	/* useless */

#define I2C_DATA_LEN_1_BYTE	0
#define I2C_DATA_LEN_2_BYTE	1
#define I2C_DATA_LEN_3_BYTE	2
#define I2C_DATA_LEN_4_BYTE	3

static int i2c_transfer(uchar cmd_type, uchar chip, uchar wr_data[], ushort wr_data_len, uchar rd_data[], ushort rd_data_len);

static int i2c_read_byte (uchar chip, uint addr, uchar * buffer)
{
	static uchar wbuffer[4];

	wbuffer[0] = (addr >> 8) & 0xff;
	wbuffer[1] = addr & 0xff;

	return i2c_transfer(I2C_CMD_WR, chip, wbuffer, I2C_DATA_LEN_2_BYTE, buffer, I2C_DATA_LEN_1_BYTE);
}

static int i2c_write_byte (uchar chip, uint addr, uchar * buffer)
{
	static uchar wbuffer[4];

	wbuffer[0] = (addr >> 8) & 0xff;
	wbuffer[1] = addr & 0xff;
	wbuffer[2] = *buffer;

	return i2c_transfer(I2C_CMD_WO, chip, wbuffer, I2C_DATA_LEN_3_BYTE, 0, I2C_DATA_LEN_1_BYTE);
}

static int i2c_wait(void)
{
	int timeout = 1000;
	uint volatile u32status, u32cfg;
	uchar errno, opdone;

	do {
		u32status = readl(CNW5XXX_SSP_BASE + I2C_INTR_STAT_OFFSET);
		writel(u32status, CNW5XXX_SSP_BASE + I2C_INTR_STAT_OFFSET);
		errno = (u32status & I2C_BUSERR_FG) ? ((u32status >> 8) & 0xff) : 0;
		opdone = (u32status & I2C_ACTDONE_FG) ? 1 : 0;

		if (opdone || (errno == 0xff))
			break;

		udelay(10);
	} while (timeout--);

	if (errno && (errno != 0xff))
		return errno;
	else
		return 0;
}

static int i2c_transfer(uchar cmd_type, uchar chip, uchar wr_data[], ushort wr_data_len, uchar rd_data[], ushort rd_data_len)
{
	int i, result;
	uint u32tmp;

	/* clear pending interrupt status */
	u32tmp = readl(CNW5XXX_SSP_BASE + I2C_INTR_STAT_OFFSET);
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_INTR_STAT_OFFSET);
	//writel((I2C_ACTDONE_FG | I2C_BUSERR_FG), CNW5XXX_SSP_BASE + I2C_INTR_STAT_OFFSET);

	switch (cmd_type) {
	case I2C_CMD_RO:
	case I2C_CMD_WO:
	case I2C_CMD_WR:
		break;

	case I2C_CMD_RW:
		printf("i2c_transfer: not implemented\n");
		return -EPERM;
		break;

	default:
		printf("i2c_transfer: bad call\n");
		return -EPERM;
		break;
	}

	u32tmp = readl(CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);
	u32tmp &=  ~(0x3f);
	u32tmp |= ((u32)((cmd_type & 0x03) << 4) | (u32)((wr_data_len & 0x03) << 2) | (u32)(rd_data_len & 0x03) & 0x3f) | (1 << 31);
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);

	u32tmp = (u32)(wr_data[3] << 24) | (u32)(wr_data[2] << 16) | (u32)(wr_data[1] << 8) | (u32)wr_data[0];
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_WR_DATA_OFFSET);

	u32tmp = readl(CNW5XXX_SSP_BASE + I2C_SLAVE_ADDR_OFFSET);
	u32tmp &= ~(0xfe);
	u32tmp |= ((chip << 1) & 0xfe);
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_SLAVE_ADDR_OFFSET);

	u32tmp = readl(CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);
	u32tmp |= (1 << 6);
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);

	switch (cmd_type) {
	case I2C_CMD_RO:
	case I2C_CMD_WR:
		result = i2c_wait();
		if (result)
			return -1;

		u32tmp = readl(CNW5XXX_SSP_BASE + I2C_RD_DATA_OFFSET);
		for (i = 0; i <= rd_data_len; i++)
			rd_data[i] = (uchar)(u32tmp >> (i*8));
		break;

	case I2C_CMD_WO:
		result = i2c_wait();
		break;

	default:
		printf("i2c_transfer: bad call\n");
		return -EPERM;
		break;
	}

	return result;
}

void i2c_init(int speed, int unused)
{
	uint u32tmp;
	uint i2c_clkdiv;
   
	DPRINTF("%s: speed: %d\n",__FUNCTION__, speed);

	/* disable I2C, disable data swap, idle I2C bus */
	u32tmp = readl(CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);
	u32tmp &=  ~((0x1 << 31)|(0x1 << 24)|(0x1 << 6));
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);

	/* CLKDIV = 100MHz / (2 * clock frequency) */
	i2c_clkdiv = 100 * 1000 * 1000 / (2 * speed);

	/* set I2C clock, timeout enable for bus-stack check */
	u32tmp = ((i2c_clkdiv & 0x03ff) << 8) | (0x1 << 7) | (0x40);
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_TIMEOUT_OFFSET);

	/* clear interrupt status */
	writel((I2C_ACTDONE_FG | I2C_BUSERR_FG), CNW5XXX_SSP_BASE + I2C_INTR_STAT_OFFSET);

	/* enable I2C */
	u32tmp = readl(CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);
	u32tmp |= (0x1 << 31);
	writel(u32tmp, CNW5XXX_SSP_BASE + I2C_CFG_OFFSET);
}

int i2c_probe(uchar chip)
{
	printf("i2c_probe chip %d\n", (int) chip);
	return -1;
}

int i2c_read(uchar chip, uint addr, int alen, uchar *buf, int len)
{
	DPRINTF("%s chip: 0x%02x addr: 0x%04x alen: %d len: %d\n",__FUNCTION__, chip, addr, alen, len);

	while (len--) {
		if (i2c_read_byte(chip, addr++, buf++))
			return -1;
	}

	return 0;
}

int i2c_write(uchar chip, uint addr, int alen, uchar *buf, int len)
{
	DPRINTF("%s chip: 0x%02x addr: 0x%04x alen: %d len: %d\n",__FUNCTION__, chip, addr, alen, len);

	while (len--) {
		if (i2c_write_byte(chip, addr++, buf++))
			return -1;
	}

	return 0;
}

int i2c_set_bus_speed(unsigned int speed)
{
	return -1;
}

unsigned int i2c_get_bus_speed(void)
{
	return CFG_I2C_SPEED;
}

#endif /* CONFIG_HARD_I2C */
