/*
 * Copyright (c) 2008 Cavium Networks
 *
 * Scott Shu
 *
 * CNW5XXX GEC Network driver
 *
 * This file is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License, Version 2, as 
 * published by the Free Software Foundation. 
 *
 * This file is distributed in the hope that it will be useful, 
 * but AS-IS and WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE, TITLE, or 
 * NONINFRINGEMENT.  See the GNU General Public License for more details. 
 *
 * You should have received a copy of the GNU General Public License 
 * along with this file; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA or 
 * visit http://www.gnu.org/licenses/. 
 *
 * This file may also be available under a different license from Cavium. 
 * Contact Cavium Networks for more information
 */

#include <common.h>
#include <command.h>
#include <malloc.h> 
#include <net.h>

#include "cnw5xxx_gec.h"

#define mdelay(n)       udelay((n)*1000)

#define GEC_RX_HW_CHECKSUM
#define GEC_TX_HW_CHECKSUM

#define GEC_CHECK_LINK_PHY
#undef GEC_CHECK_LINK_MAC

#undef CONFIG_GEC_100M_FD
#undef CONFIG_GEC_GIGA_MODE
#undef CONFIG_GEC_MERGE_INTERRUPT
#undef CONFIG_GEC_DELAYED_INTERRUPT

#undef GEC_RX_DEBUG
#undef GEC_TX_DEBUG
#undef GEC_PHY_DEBUG

static gec_device gec_dev;
static int initialized = 0;
static int memory_allocated = 0;

typedef struct
{
	u32 vid;	/* VLAN ID: 0 ~ 4095 */
	u32 control;	/* Filter enable (1) or disable (0) */
} my_vlan_entry_t;

static my_vlan_entry_t my_vlan_id[4] =
{
	{ 1, 0},	/* My VLAN ID 0 */
	{ 2, 1},	/* My VLAN ID 1 */
	{ 3, 1},	/* My VLAN ID 2 */
	{ 4, 1}		/* My VLAN ID 3 */
};

#if defined(GEC_RX_DEBUG) || defined(GEC_TX_DEBUG)
static void print_packet(unsigned char * buf, int length)
{
	int i;
	int remainder;
	int lines;

	printf(DRV_NAME ": Packet of length %d \n", length );

	lines = length / 16;
	remainder = length % 16;

	for ( i = 0; i < lines ; i ++ ) {
		int cur;

		for ( cur = 0; cur < 8; cur ++ ) {
			unsigned char a, b;

			a = *(buf ++ );
			b = *(buf ++ );
			printf("%02x%02x ", a, b );
		}
		printf("\n");
	}
	for ( i = 0; i < remainder/2 ; i++ ) {
		unsigned char a, b;

		a = *(buf ++ );
		b = *(buf ++ );
		printf("%02x%02x ", a, b );
	}
	printf("\n");
}
#endif

int cnw5xxx_get_ethaddr(bd_t * bd)
{
	int env_size = 0;
	int env_present = 0;
	int reg = 0;
	char *s = NULL;
	char *e = NULL;
	char *v_mac, es[] = "11:22:33:44:55:66";
	char s_env_mac[64];
	uchar v_env_mac[6];

	env_size = getenv_r("ethaddr", s_env_mac, sizeof (s_env_mac));
	if (env_size != sizeof(es)) {	/* Ignore if env is bad or not set */
		printf("\n*** Warning: ethaddr is not set properly, ignoring!!\n");
	} else {
		env_present = 1;
		s = s_env_mac;

		for (reg = 0; reg < 6; ++reg) {	/* turn string into mac value */
			v_env_mac[reg] = s ? simple_strtoul (s, &e, 16) : 0;
			if (s)
				s = (*e) ? e + 1 : e;
		}
	}

	if(env_present) {
		v_mac = (char *)v_env_mac;
	}

	memcpy(bd->bi_enetaddr, v_mac, 6);	/* update global address to match env (allows env changing) */
	printf(DRV_NAME ": Using MAC Address %02X:%02X:%02X:%02X:%02X:%02X\n", v_mac[0], v_mac[1],
		v_mac[2], v_mac[3], v_mac[4], v_mac[5]);
	return 0;
}

static int cnw5xxx_write_phy(u8 phy_addr, u8 phy_reg, u16 write_data)
{
	int i;

	if (phy_addr > 31)
		return 0;

	GEC_PHY_CTRL0 = (0x1 << 15);	/* clear previous rw_ok status */
	GEC_PHY_CTRL0 = ((phy_addr & 0x1F) | ((phy_reg & 0x1F) << 8) | (0x1 << 13) |
			((write_data & 0xFFFF) << 16));

	for (i = 0; i < 10000; i++) {
		if ((GEC_PHY_CTRL0) & (0x1 << 15)) {
			GEC_PHY_CTRL0 = (0x1 << 15);
			return 0;
		}
		udelay(1000);
	}

	return -1;
}

static int cnw5xxx_read_phy(u8 phy_addr, u8 phy_reg, u16 *read_data)
{
	int i;

	if (phy_addr > 31)
		return 0;

	GEC_PHY_CTRL0 = (0x1 << 15);	/* clear previous rw_ok status */
	GEC_PHY_CTRL0 = ((phy_addr & 0x1F) | ((phy_reg & 0x1F) << 8) | (0x1 << 14));

	for (i = 0; i < 10000; i++) {
		u32 status = GEC_PHY_CTRL0;
		if (status & (0x1 << 15)) {
			GEC_PHY_CTRL0 = (0x1 << 15);
			*read_data = (u16)((status >> 16) & 0xFFFF);
			return 0;
		}
		udelay(1000);
	}

	return -1;
}

static int cnw5xxx_ksz9021rl_read_extended(u8 phy_addr, u8 page, u8 addr,
                                           u16 *read_data)
{
        int ret;

        ret = cnw5xxx_write_phy(phy_addr, 11, (0 << 15) | ((page & 1) << 8) | addr);
        if (!ret)
                ret = cnw5xxx_read_phy(phy_addr, 13, read_data);

        return ret;
}

static int cnw5xxx_ksz9021rl_write_extended(u8 phy_addr, u8 page, u8 addr, u16 write_data)
{
        int ret;

        ret = cnw5xxx_write_phy(phy_addr, 11, (1 << 15) | ((page & 1) << 8) | addr);
        if (!ret)
                cnw5xxx_write_phy(phy_addr, 12, write_data);

        return ret;
}

static inline pkt_t *fn_alloc_pkt(gec_device *dev)
{
	pkt_t *pkt = 0;

	pkt = dev->fn_free_pkt_list;
	if (pkt) {
		dev->fn_free_pkt_list = pkt->next;
		pkt->next = 0;
		dev->fn_free_pkt_count--;
	}
	return pkt;
}

static inline void fn_free_pkt(gec_device *dev, pkt_t *pkt)
{
	pkt->next = dev->fn_free_pkt_list;
	dev->fn_free_pkt_list = pkt;
	dev->fn_free_pkt_count++;
}

static int cnw5xxx_init_memory(gec_device *dev)
{
	int i;

	/* clear_all_dma_desc(1); */
	GEC_DMA_CFG |= (1 << 31);

	/* Initial TN/FN DMA Descriptor Point */
	/* Initial TN Descriptor Point of Ring 0-3 */
	GEC_TN0_DESC_PTR = 0;
	GEC_TN1_DESC_PTR = 0;
	GEC_TN2_DESC_PTR = 0;
	GEC_TN3_DESC_PTR = 0;
	/* Initial TN Descriptor Base Address of Ring 0-3 */
	GEC_TN0_BASE_ADDR = 0;
	GEC_TN1_BASE_ADDR = 0;
	GEC_TN2_BASE_ADDR = 0;
	GEC_TN3_BASE_ADDR = 0;
	/* Initial FN Descriptor Point */
	GEC_FN_DESC_PTR = 0;
	/* Initial FN Descriptor Base Address */
	GEC_FN_BASE_ADDR = 0;

	/* allocate and align memory for descriptor rings */
	dev->fn_desc_ring = (FNDesc *) memalign(32, RX_DESC_ALIGNED_SIZE * RX_DESC_SIZE + 1);
	dev->tn_desc_ring = (TNDesc *) memalign(32, TX_DESC_ALIGNED_SIZE * TX_DESC_SIZE + 1);

	memset(dev->fn_desc_ring, 0, sizeof(FNDesc) * RX_DESC_SIZE);
	memset(dev->tn_desc_ring, 0, sizeof(TNDesc) * TX_DESC_SIZE);

	/* allocate memory for FN packet buffer */
	dev->fn_pkt_pool = (pkt_t *) malloc(sizeof(pkt_t) * (NUM_PKT_BUFFER + 1));
	if (dev->fn_pkt_pool == 0) {
		goto err_out;
	}

	dev->fn_pkt_buffer_pool = (u8 *) malloc(PACKET_BUFFER_SIZE * (NUM_PKT_BUFFER + 1));
	if (dev->fn_pkt_buffer_pool == 0) {
		goto err_out;
	}

	memory_allocated = 1;

	memset(dev->fn_pkt_ring, 0, sizeof(pkt_t *) * RX_DESC_SIZE);

	dev->fn_free_pkt_list = &dev->fn_pkt_pool[0];

	/* chain the rings */
	for (i = 0; i < (NUM_PKT_BUFFER - 1); i++) {
		dev->fn_pkt_pool[i].next = &dev->fn_pkt_pool[i + 1];
		dev->fn_pkt_pool[i].pkt_buffer = dev->fn_pkt_buffer_pool + (i * PACKET_BUFFER_SIZE);
	}
	dev->fn_pkt_pool[i].next = 0;
	dev->fn_pkt_pool[i].pkt_buffer = dev->fn_pkt_buffer_pool + (i * PACKET_BUFFER_SIZE);

	/* chain the FN rings */
	for (i = 0; i < RX_DESC_SIZE; i++) {
		dev->fn_pkt_ring[i] = fn_alloc_pkt(dev); /* */
		dev->fn_desc_ring[i].sdp = (u32)dev->fn_pkt_ring[i]->pkt_buffer; /* */
		dev->fn_desc_ring[i].sdl = MAX_PACKET_SIZE;
		dev->fn_desc_ring[i].fsd = 1;
		dev->fn_desc_ring[i].lsd = 1;
		dev->fn_desc_ring[i].cown = 0;
		if (i == (RX_DESC_SIZE - 1))
			dev->fn_desc_ring[i].eor = 1;
	}
	dev->fn_desc_cur_index = 0;

	/* chain the TN rings */
	for (i = 0; i < TX_DESC_SIZE; i++) {
		dev->tn_desc_ring[i].cown = 1;
		if (i == (TX_DESC_SIZE - 1)) {
			dev->tn_desc_ring[i].eor = 1;
		}
		dev->tn_desc_ring[i].ico = 0;
		dev->tn_desc_ring[i].uco = 0;
		dev->tn_desc_ring[i].tco = 0;
	}
	dev->tn_desc_cur_index = 0;

	/* initial packet counter */
	dev->fn_pkt_count = 0;
	dev->tn_pkt_count = 0;

	/* clear_all_dma_desc(0); */
	GEC_DMA_CFG &= (~(1 << 31));

	return 0;

err_out:
	printf(DRV_NAME ": call cnw5xxx_init_memory() failed\n");
	if (dev->fn_pkt_pool)
		free(dev->fn_pkt_pool);
	if (dev->fn_pkt_buffer_pool)
		free(dev->fn_pkt_buffer_pool);
	return -1;
}

int eth_init(bd_t *bd)
{
	gec_device *dev = &gec_dev;
	unsigned long i;
	u32 mac_config0 = 0, mac_config1 = 0;
	u32 fc_config0 = 0, fc_config1 = 0;
	u32 phy_addr = 0;
	u16 phy_data;
	u16 phy_id = 0;
	u32 vlan_id = 0;
	u32 arl_config = 0;
	u32 tc_config = 0;
	u32 dma_config = 0;
	int err;

	if (initialized) {
		mac_config0 = GEC_MAC_CFG0;
		mac_config0 &= ~(0x1 << 23);	/* Enable MAC port */
		GEC_MAC_CFG0 = mac_config0;

		mdelay(10);

		return 0;
	}

	printf(DRV_NAME ": initializing\n");

	/* allocate memory for packets and descriptor rings */
	cnw5xxx_init_memory(dev);

	/* Initial TN/FN DMA Descriptor Point */
	/* configure DMA descriptors */
	GEC_FN_DESC_PTR = (u32)dev->fn_desc_ring;
	GEC_FN_BASE_ADDR = (u32)dev->fn_desc_ring;
#ifdef GEC_RX_DEBUG
	printf(DRV_NAME ": GEC_FN_DESC_PTR = %p\n", dev->fn_desc_ring);
	printf(DRV_NAME ": GEC_FN_BASE_ADDR[1]  = %p\n", &dev->fn_desc_ring[1]);
#endif

	GEC_TN0_DESC_PTR = (u32)dev->tn_desc_ring;
	GEC_TN0_BASE_ADDR = (u32)dev->tn_desc_ring;
#ifdef GEC_TX_DEBUG
	printf(DRV_NAME ": GEC_TN0_DESC_PTR = %p\n", dev->tn_desc_ring);
	printf(DRV_NAME ": GEC_TN0_BASE_ADDR[1] = %p\n", &dev->tn_desc_ring[1]);
#endif

	/* ------ MAC config ------ */
	/* MAC Configuration 0 */
	mac_config0 = GEC_MAC_CFG0;
	mac_config0 |= (0x1 << 23);	/* Disable MAC port */
	GEC_MAC_CFG0 = mac_config0;

	/* TN/FN DMA Control */
	GEC_TN0_DMA_CTRL = 0;		/* Disable TN/FN DMA */
	GEC_TN1_DMA_CTRL = 0;
	GEC_TN2_DMA_CTRL = 0;
	GEC_TN3_DMA_CTRL = 0;
	GEC_FN_DMA_CTRL = 0;

	/* Interrupt Status */
	GEC_INT_STATUS = 0x03FF003F;	/* Clear all interrupt */

	GEC_INT_MASK = 0x03FF003F;

	/* Merge all DMA Edge-Trigger Interrupt & Delay Interrupt Configuration */
#ifdef CONFIG_GEC_DELAYED_INTERRUPT
	GEC_DLY_INT_CFG = (0x1 << 31) | (0x1 << 16) | (MAX_PEND_INT_CNT << 8) | (MAX_PEND_INT_TIME << 0);
#endif

	/* Merge all DMA Edge-Trigger Interrupt */
#ifdef CONFIG_GEC_MERGE_INTERRUPT
	GEC_DLY_INT_CFG = (0x1 << 31);
#endif

	/* Extended PHY Control 1 */
	cnw5xxx_read_phy(phy_addr, 3, &phy_data);
	if ((phy_data & 0xffff) == 0xc914) {
		printf(DRV_NAME ": Realtek RTL8211D \n");

		/* reset phy */
		if (cnw5xxx_read_phy(phy_addr, 0, &phy_data))
			return -1;
		phy_data |= (1 << 15);
		if (cnw5xxx_write_phy(phy_addr, 0, phy_data))
			return -1;

		phy_id = 0xc914;

	} else if ((phy_data & 0xffff) == 0xc852) {/* RTL phy switch */

                printf(DRV_NAME ": Realtek RTL8304E \n");
                phy_id = 0xc852; /* phy0_id of Switch */

	} else if ((phy_data & 0xfff0) == 0x1610) {
		printf(DRV_NAME ": Micrel KSZ9021RL/RN\n");
		phy_id = 0x1610;
	} else if ((phy_data & 0xffff) == 0x0C54) {
		cnw5xxx_read_phy(phy_addr, 2, &phy_data);
		if ((phy_data & 0xffff) == 0x0243) {
			printf(DRV_NAME ": ICplus IP101A \n");
		}
		phy_id = 0x0243;
	} else if ((phy_data & 0x000f) == 0x0000) {
		printf(DRV_NAME ": VSC8601 Type A Chip \n");
	} else if ((phy_data & 0x000f) == 0x0001) {
		if ((phy_data & 0x03f0) == 0x0030) {
			printf(DRV_NAME ": VSC8641 Chip \n");
			cnw5xxx_read_phy(phy_addr, 23, &phy_data);
                	phy_data |= (0x1 << 8);        /* RGMII skew timing compensation, adds 2ns delay */
			cnw5xxx_write_phy(phy_addr, 23, phy_data);

			#if 0
			/* Patch code from VITESSEi: Long Link-Up Times Caused by Noise on the Twisted Pair Interface */
			cnw5xxx_write_phy(phy_addr, 31, 0x52B5);
			cnw5xxx_write_phy(phy_addr, 16, 0xA7FA);
			cnw5xxx_read_phy(phy_addr, 18, &phy_data);
			cnw5xxx_write_phy(phy_addr, 18, phy_data);
			cnw5xxx_read_phy(phy_addr, 17, &phy_data);
			phy_data |= 0x0008;
			cnw5xxx_write_phy(phy_addr, 17, phy_data);
			cnw5xxx_write_phy(phy_addr, 16, 0x87FA);
			cnw5xxx_write_phy(phy_addr, 31, 0x0000);
			#endif
		} else if ((phy_data & 0x03f0) == 0x0020) {
			printf(DRV_NAME ": VSC8601 Type B Chip \n");
			cnw5xxx_read_phy(phy_addr, 23, &phy_data);
                	phy_data |= ( 0x1 << 8);        /* RGMII skew timing compensation, adds 2ns delay */
			cnw5xxx_write_phy(phy_addr, 23, phy_data);
		} else {
			printf(DRV_NAME ": unknown (0x%x)\n", phy_data);
                        cnw5xxx_read_phy(phy_addr, 2, &phy_data);
                        printf(DRV_NAME ": OUI (0x%x) \n", phy_data);
			return -1;
		}
	} else {
		printf(DRV_NAME ": unknown (0x%x) \n", phy_data);
	        cnw5xxx_read_phy(phy_addr, 2, &phy_data);
		printf(DRV_NAME ": OUI (0x%x) \n", phy_data);
		return -1;
	}

	/* MAC Configuration 0 */
	mac_config0 = GEC_MAC_CFG0;
	mac_config0 |= (0x1 << 13);	/* Enable TX clock period check */
	//mac_config0 &= (~(0x1 << 13));	/* Disable TX clock period check */
#ifdef CONFIG_GEC_GIGA_MODE
	mac_config0 |= (0x1 << 17);
#else
	mac_config0 &= ~(0x1 << 17);
#endif
	GEC_MAC_CFG0 = mac_config0;

	/* MAC Configuration 1 */
	mac_config1 = GEC_MAC_CFG1;
//	mac_config1 |= (0x1 << 0);	/* Force MAC to do fast back-off after collision */
//	mac_config1 |= (0x1 << 1);	/* Allow MAC to re-transmit a packet after 16 consecutive collisions */
	mac_config1 |= (0x1f << 2);	/* IPG */
//	mac_config1 |= (0x1 << 20);	/* Accept oversize packets */
	mac_config1 &= ~(0x1 << 20);	/* Discard oversize packets */
//	mac_config1 |= (0x1 << 21);	/* Accept CRC error packets */
	mac_config1 |= (0x1 << 22);	/* Accept checksum error packets */
//	mac_config1 |= (0x1 << 23);	/* Strip CRC in RX packet and generate CRC for TX packet */
//	mac_config1 &= ~(0x1 << 24);	/* Strip VLAN tag */
#ifdef GEC_RX_HW_CHECKSUM
	mac_config1 |= (0x1 << 25);	/* Enable IP/TCP/UDP checksum generation offload in RX */
#else
	mac_config1 &= ~(0x1 << 25);
#endif
#ifdef GEC_TX_HW_CHECKSUM
	mac_config1 |= (0x1 << 26);	/* Enable IP/TCP/UDP checksum generation offload in TX */
#else
	mac_config1 &= ~(0x1 << 26);
#endif
	GEC_MAC_CFG1 = mac_config1;

	/* MAX Length Configuration */
	GEC_MAX_LENGTH_CFG = 1518;

	/* ------ FC config ------ */
	fc_config0 = GEC_FC_CFG0;
	fc_config0 &= ~(0x0eff);	/* Send pause off frame threshold */
	fc_config0 |= (0x10);
//	fc_config0 &= ~(0x0eff << 16);	/* Send pause on frame threshold */
//	fc_config0 |= (0x360 << 16);
	GEC_FC_CFG0 = fc_config0;

	fc_config1 = GEC_FC_CFG1;
	fc_config1 &= ~(0x1f);		/* Max backpressure collision count */
	fc_config1 |= (0x0c);
	fc_config1 &= ~(0x1 << 5);	/* Allow unlimit backpressure collisions */
	fc_config1 &= ~(0x1 << 6);	/* Jam all incoming packets until backpressure condition released */
	fc_config1 |= (0x1 << 7);	/* Enable half-duplex backpressure */
	fc_config1 &= ~(0x1 << 8);	/* Disable to treat unicast pause frame as 802.3x pause frame */
	GEC_FC_CFG1 = fc_config1;

	/* ------ PHY config ------ */
	/* PHY Control Register 0 */
	GEC_PHY_CTRL0 &= (~(0x1 << 7));	/* Enable MDC/MDIO Interface */

	/* Mode Control - Reset PHY */
	cnw5xxx_read_phy(phy_addr, 0, &phy_data);
	phy_data |= (0x1 << 15);
	cnw5xxx_write_phy(phy_addr, 0, phy_data);
	udelay(100);

	/* MAC Configuration 0 */
	mac_config0 = GEC_MAC_CFG0;
#ifdef CONFIG_GEC_100M_FD
	mac_config0 &= (~(0x1 << 7));	/* Disable Auto-Negotiation */
	mac_config0 &= (~(0x3 << 8));	/* Force speed 100 Mbps */
	mac_config0 |= (0x1 << 8);
	mac_config0 |= (0x1 << 10);	/* Force full duplex */
	mac_config0 |= (0x1 << 11);	/* Force RX flow control */
	mac_config0 |= (0x1 << 12);	/* Force TX flow control */
#else /* !CONFIG_GEC_100M_FD */
	mac_config0 |= (0x1 << 7);	/* Enable Auto-Negotiation */
	if (phy_id == 0x1610) {
		mac_config0 |= (0x1 << 11);	/* Force RX flow control */
		mac_config0 &= (~(0x1 << 12));	/* Disable TX flow control */
	} else {
		mac_config0 |= (0x1 << 11);	/* Force RX flow control */
		mac_config0 |= (0x1 << 12);	/* Force TX flow control */
	}

	mac_config0 &= (~(0x1f << 24));	/* PHY address for auto-polling */
	mac_config0 |= (phy_addr << 24);
	mac_config0 |= (0x1 << 31);	/* Enable PHY auto polling */
#endif /* CONFIG_GEC_100M_FD */
	mac_config0 &= (~(0x1 << 14));	/* Normal MII/RGMII mode */
	mac_config0 &= (~(0x3 << 15));	/* MII/RMII/RGMII mode */

	if (phy_id != 0xc852) /* RTL8304E supports MII mode as per schematic */
		mac_config0 |= (0x2 << 15);	/* RGMII mode */

	mac_config0 |= (0x1 << 22);	/* Enable MAC clock */
	mac_config0 &= ~(0x1 << 23);	/* Enable MAC port */
	GEC_MAC_CFG0 = mac_config0;

	/* 1000BASE-T Control */
	cnw5xxx_read_phy(phy_addr, 9, &phy_data);
#ifdef CONFIG_GEC_GIGA_MODE
	phy_data |= (1 << 9);	        /* PHY is 1000BASE-T FDX capable */
	phy_data |= (1 << 8);	        /* PHY is 1000BASE-T HDX capable */

	//if (phy_id ==  0x1610) {
	//	phy_data |= (3 << 11);	/* Force to MASTER mode */
	//}
#else
	phy_data &= (~(1 << 9));	/* PHY is NOT 1000BASE-T FDX capable */
	phy_data &= (~(1 << 8));	/* PHY is NOT 1000BASE-T HDX capable */
#endif
	cnw5xxx_write_phy(phy_addr, 9, phy_data);

	/* Mode Control */
	cnw5xxx_read_phy(phy_addr, 0, &phy_data);
#ifdef CONFIG_GEC_100M_FD
	phy_data |= (0x1 << 8);		/* Full-duplex */
	phy_data |= (0x1 << 13);	/* Force speed 100 Mbps */
	phy_data &= (~(0x1 << 6));
	phy_data &= (~(0x1 << 12));	/* Auto-negotiation disabled */
#else
	if (phy_id == 0x1610) {
		phy_data |= (0x1 << 9);		/* Restart auto-negotiation process */
	}
	phy_data |= (0x1 << 12);	/* Auto-negotiation enable */
#endif
	cnw5xxx_write_phy(phy_addr, 0, phy_data);

	if (phy_id == 0xc914 || phy_id == 0xc852) {
		/* nop */
        } else if (phy_id == 0x1610) {
                // RHMII Clock and Control Pad Skew.
                // Adjust TXC/RXC PAD Skew Control to 3ns, and
                // TX/RX CTL PAD Skew Control to 0ns.
                cnw5xxx_ksz9021rl_write_extended(phy_addr, 1, 4, 0xf0f0);
                cnw5xxx_ksz9021rl_read_extended(phy_addr, 1, 4, &phy_data);

		//cnw5xxx_ksz9021rl_write_extended(phy_addr, 1, 5, 0xaaaa);
		//cnw5xxx_ksz9021rl_read_extended(phy_addr, 1, 5, &phy_data);

		// Turn off auto-crossover function
		//cnw5xxx_read_phy(phy_addr, 28, &phy_data);
		//phy_data |= (3 << 6);
		//cnw5xxx_write_phy(phy_addr, 28, phy_data);
	} else {
		/* (Extended Page) RGMII Skew Control */
		cnw5xxx_write_phy(phy_addr, 31, 0x0001);	/* change to extended registers */
		cnw5xxx_read_phy(phy_addr, 28, &phy_data);
		phy_data &= ~(0x3 << 14);			/* RGMII TX skew timing compensation, 2ns */
		phy_data |= (0x3 << 14);
		phy_data &= ~(0x3 << 12);			/* RGMII RX skew timing compensation, 2ns */
		phy_data |= (0x3 << 12);
		cnw5xxx_write_phy(phy_addr, 28, phy_data);
		cnw5xxx_write_phy(phy_addr, 31, 0x0000);	/* change to normal registers */
	}

	/* ------ VLAN config ------ */
	/* My VLAN ID 0 & 1 */
	vlan_id = (my_vlan_id[0].vid & 0x0fff);
	vlan_id |= ((my_vlan_id[1].vid & 0x0fff) << 16);
	GEC_VLAN_ID_01 = vlan_id;

	/* My VLAN ID 2 & 3 */
	vlan_id = (my_vlan_id[2].vid & 0x0fff);
	vlan_id |= ((my_vlan_id[3].vid & 0x0fff) << 16);
	GEC_VLAN_ID_23 = vlan_id;

	/* My VLAN ID Control */
	GEC_VLAN_CTRL = ((my_vlan_id[0].control << 0) | (my_vlan_id[1].control << 1) |
			 (my_vlan_id[2].control << 2) | (my_vlan_id[3].control << 3) );

	/* ------ ARL config ------ */
	/* ARL Configuration */
	arl_config = GEC_ARL_CFG;
	arl_config &= ~(0x1 << 0);	/* MAC address hash algorithm: direct mode */
	arl_config &= ~(0x1 << 1);	/* Forward multicast to CPU */

	arl_config |= (0x1 << 2);	/* Disable learning SMAC of TX packet into hash table */
//	arl_config &= ~(0x1 << 2);	/* Learning SMAC of TX packet into hash table */

//	arl_config |= (0x1 << 3);	/* Only My MAC or BC packets are received */
	arl_config &= ~(0x1 << 3);	/* My MAC or BC or hash table hit packets are received */

	//arl_config |= (0x1 << 4);	/* Promiscuous mode */
	arl_config &= ~(0x1 << 4);	/* Disable promiscuous mode */
	GEC_ARL_CFG = arl_config;

	/* ------ TC config ------ */
	/* Traffic Class Configuration */
	tc_config = GEC_TC_CFG;
	tc_config &= (~(0x7 << 0));	/* Queue 0 weight, 0 ~ 4 */
	tc_config |= (0x0 << 0);
	tc_config &= (~(0x7 << 4));	/* Queue 1 weight, 0 ~ 4 */
	tc_config |= (0x0 << 4);
	tc_config &= (~(0x7 << 8));	/* Queue 2 weight, 0 ~ 4 */
	tc_config |= (0x0 << 8);
	tc_config &= (~(0x7 << 12));	/* Queue 3 weight, 0 ~ 4 */
	tc_config |= (0x0 << 12);
	tc_config &= (~(0x3 << 16));	/* Scheduling mode */
	tc_config |= (0x0 << 16);	/* 0: WRR, 1: Strict priority, 2: Mix */
	tc_config &= (~(0x3 << 30));	/* Number of traffic class */
	tc_config |= (0x0 << 30);	/* 0: 1 traffic class(TC), 1: 2TC, 2: 3TC, 3: 4TC */
	GEC_TC_CFG = tc_config;

	/* ------ DMA config ------ */
	/* DMA Configuration */
	dma_config = GEC_DMA_CFG;
	dma_config &= ~(0x1 << 0);	/* FN DMA can receive the packet */
	dma_config |= (0x1 << 1);	/* FN DMA auto polling C-bit of the FN descriptor */
	dma_config &= ~(0x3 << 2);	/* FN DMA auto polling period is 100us */
	dma_config |= (0x2 << 2);
	dma_config &= ~(0x1 << 4);	/* FN DMA payload swap disabled */

	dma_config &= ~(0x1 << 8);	/* TN DMA can receive the packet */
	dma_config |= (0x1 << 9);	/* TN DMA auto polling C-bit of the FN descriptor */
	dma_config &= ~(0x3 << 10);	/* TN DMA auto polling period is 100us */
	dma_config |= (0x2 << 10);
	dma_config &= ~(0x1 << 12);	/* TN DMA payload swap disabled */

	dma_config &= ~(0x3 << 24);	/* Descriptor queue threshold of FN ring */
	dma_config |= (0x3 << 24);
	dma_config &= ~(0x3 << 26);	/* Descriptor queue threshold of TN ring */
	dma_config |= (0x3 << 26);

	dma_config |= (0x1 << 28);	/* Align to word-aligned (4N) address offset */
//	dma_config &= ~(0x1 << 28);	/* De-align to (4N + 2) address offset */
	GEC_DMA_CFG = dma_config;

	/* MAC address */
	err = cnw5xxx_get_ethaddr(bd);
	if (err < 0) {
		memset(bd->bi_enetaddr, 0, 6);
		return -1;
	}

	GEC_MY_MAC_H = (bd->bi_enetaddr[0] << 8) | bd->bi_enetaddr[1];
	GEC_MY_MAC_L = (bd->bi_enetaddr[2] << 24) | (bd->bi_enetaddr[3] << 16) | (bd->bi_enetaddr[4] << 8) | (bd->bi_enetaddr[5]);

	/* Bring the PHY up */
	cnw5xxx_read_phy(phy_addr, 0, &phy_data);
	phy_data &= ~(0x1 << 11);	/* Power-up */
	cnw5xxx_write_phy(phy_addr, 0, phy_data);
	
#ifdef CONFIG_GEC_100M_FD
	mdelay(2000);
#else
	printf(DRV_NAME ": Auto-Negotiation .");	/* check AN complete on PHY */
	for (i = 0; i < 100; i++) {
		cnw5xxx_read_phy(phy_addr, 1, &phy_data);
		phy_data &= (0x1 << 5);
		if (phy_data >> 5) {
			printf("done\n");
			break;
		} else {
			printf(".");
			mdelay(1000);
		}
		if (i == (100 - 1)) {
			printf("failed\n");
			return -1;
		}
	}
#endif

#ifdef GEC_CHECK_LINK_PHY
	printf(DRV_NAME ": (PHY) Link up .");		/* check link status on PHY */
	for (i = 0; i < 500; i++) {
		cnw5xxx_read_phy(phy_addr, 1, &phy_data);
		phy_data &= (0x1 << 2);
		if (phy_data >> 2) {
			printf("done\n");
			break;
		} else {
			printf(".");
			mdelay(1000);		/* link down, wait a moment */
		}
	}

	if (phy_id == 0x1610) {
		cnw5xxx_read_phy(phy_addr, 31, &phy_data);
		if (phy_data & 0x1) {
			printf(DRV_NAME ": (PHY31) Link Status Check Fail...\n");
		} else {
			printf(DRV_NAME ": (PHY31) %dM %s (0x%x)\n",
				((phy_data & 0x10) == (1 << 4) ? 10 :
				 (phy_data & 0x20) == (1 << 5) ? 100 : 1000),
				 (phy_data & 0x08) == (1 << 3) ? "Full duplex" : "Half duplex", phy_data);
		}
	}
#endif

#ifdef GEC_PHY_DEBUG
	cnw5xxx_read_phy(phy_addr, 28, &phy_data);
	printf(DRV_NAME ": Auxiliary Control and Status = %4x\n", phy_data);
#endif

#ifdef GEC_CHECK_LINK_MAC
	mac_config0 = GEC_MAC_CFG0;		/* check link status on MAC */
	mac_config0 &= 0x1;
        if (mac_config0) {
                printf(DRV_NAME ": (MAC) Link up .");
        }

	for (i = 0; i < 200; i++) {
		if (~GEC_MAC_CFG0) {
			printf(".");
			mdelay(1000);		/* link down, wait a moment */
		} else {
			printf("\n");
			break;
		}
	}
#endif

	mac_config0 = GEC_MAC_CFG0;		/* Show link speed */
        if (mac_config0 & 0x1) {
                printf(DRV_NAME ": (MAC) %dM %s.\n",
                        ((mac_config0 & 0x0C) == (0 << 2) ? 10 :
                         (mac_config0 & 0x0C) == (1 << 2) ? 100 : 1000),
                         (mac_config0 & 0x10) == 0 ? "Half duplex" : "Full duplex");
        }

	GEC_FN_DMA_CTRL = 1;

	initialized = 1;

	return 0;
}

int eth_send(volatile void *packet, int length)
{
	gec_device *dev = &gec_dev;
	int i;

#ifdef GEC_TX_DEBUG
	printf(DRV_NAME ": eth_send()\n");
#endif

	/* skb_padto */
	if (length < MIN_PACKET_SIZE) {
		memset((void*) (packet + length) , 0, (MIN_PACKET_SIZE - length));
		length = MIN_PACKET_SIZE;
	}

#ifdef GEC_TX_DEBUG
	print_packet((unsigned char *)packet, length);
#endif

	dev->tn_desc_ring[dev->tn_desc_cur_index].sdp = (u32) packet;
#ifdef GEC_TX_DEBUG
	printf(DRV_NAME ": tn_desc_cur_index = %d, tn_desc_ring = %p\n",
			dev->tn_desc_cur_index, &dev->tn_desc_ring[dev->tn_desc_cur_index]);
#endif
	dev->tn_desc_ring[dev->tn_desc_cur_index].ico = 0;
	dev->tn_desc_ring[dev->tn_desc_cur_index].uco = 0;
	dev->tn_desc_ring[dev->tn_desc_cur_index].tco = 0;
	dev->tn_desc_ring[dev->tn_desc_cur_index].intr = 0;

	dev->tn_desc_ring[dev->tn_desc_cur_index].sdl = length;
	dev->tn_desc_ring[dev->tn_desc_cur_index].ctv = 0;
	dev->tn_desc_ring[dev->tn_desc_cur_index].fsd = 1;
	dev->tn_desc_ring[dev->tn_desc_cur_index].lsd = 1;
	dev->tn_desc_ring[dev->tn_desc_cur_index].cown = 0;

	if (dev->tn_desc_cur_index == (TX_DESC_SIZE - 1))
		dev->tn_desc_ring[dev->tn_desc_cur_index].eor = 1;
	else
		dev->tn_desc_ring[dev->tn_desc_cur_index].eor = 0;

	dev->tn_desc_cur_index = ((dev->tn_desc_cur_index + 1) % TX_DESC_SIZE);

	GEC_TN0_DMA_CTRL = 1;

	while (GEC_TN0_DMA_CTRL) { 
		for(i = 0; i < 0x1000; i++);	/* sleep */
	}
	GEC_TN0_DESC_PTR = (u32) &dev->tn_desc_ring[dev->tn_desc_cur_index];

#ifdef GEC_TX_DEBUG
	printf(DRV_NAME ": GEC_TN0_DESC_PTR = %p\n", GEC_TN0_DESC_PTR);
#endif
	return 0;
}

void eth_halt(void)
{
	u32 mac_config0 = (*((u32 volatile *)(0x70000004)));

#ifdef GEC_DEBUG
	printf(DRV_NAME ": eth_halt()\n");
#endif
	mac_config0 |= (0x1 << 23);     /* Disable MAC port */
	(*(volatile unsigned int*)(0x70000004) = (mac_config0));
}

int receive_packet(void) {
	gec_device *dev = &gec_dev;
	FNDesc volatile *fn_desc = 0;
	pkt_t *rcvpkt;
	pkt_t *newpkt;
	u32 rxcount = 0;

#ifdef GEC_RX_DEBUG
	printf(DRV_NAME ": eth_rx()\n");
#endif

	while (1) {
		u32 fn_head = (GEC_FN_DESC_PTR - GEC_FN_BASE_ADDR) / sizeof(FNDesc);
		int counter = 0;

		if (fn_head > dev->fn_desc_cur_index)
			counter = fn_head - dev->fn_desc_cur_index;
		else
			counter = RX_DESC_SIZE - dev->fn_desc_cur_index + fn_head;

#ifdef GEC_RX_DEBUG
		printf(DRV_NAME ": fn_desc_cur_index = %d, fn_desc_ring = %p, fn_head = %d, counter = %d.\n", 
				dev->fn_desc_cur_index, &dev->fn_desc_ring[dev->fn_desc_cur_index], fn_head, counter);
#endif

		/* current free descriptor */
		fn_desc = &dev->fn_desc_ring[dev->fn_desc_cur_index];

		if (fn_desc->cown == 0) {
			/* no packets */
#ifdef GEC_RX_DEBUG
			printf(DRV_NAME ": no packets!\n");
#endif
			break;
		}

		rcvpkt = dev->fn_pkt_ring[dev->fn_desc_cur_index];
		rcvpkt->length = fn_desc->sdl;

		newpkt = fn_alloc_pkt(dev);
		if (newpkt == 0) {
			printf("Allocate pkt failed on RX...\n");
		}
		dev->fn_pkt_ring[dev->fn_desc_cur_index] = newpkt;
		fn_desc->sdp = (u32)newpkt->pkt_buffer;
		fn_desc->sdl = MAX_PACKET_SIZE;
		fn_desc->fsd = 1;
		fn_desc->lsd = 1;
		fn_desc->cown = 0;
		if (dev->fn_desc_cur_index == (RX_DESC_SIZE - 1))
			fn_desc->eor = 1;
		else
			fn_desc->eor = 0;

#ifdef GEC_RX_DEBUG
		print_packet((unsigned char *)rcvpkt->pkt_buffer, rcvpkt->length);
#endif
		NetReceive(rcvpkt->pkt_buffer, rcvpkt->length);

		fn_free_pkt(dev, rcvpkt);

		dev->fn_desc_cur_index = ((dev->fn_desc_cur_index + 1) % RX_DESC_SIZE);

		rxcount++;
		if (rxcount == RX_DESC_SIZE)
			break;
	}

	dev->fn_pkt_count += rxcount;

	GEC_FN_DMA_CTRL = 1;

	return 0;
}

int eth_rx(void)
{
	receive_packet();
	
	return 0;
}
