/** @file launcher.c
 *
 *   The AQUOS Launcher
 *
 *     Copyright (c) 2005, 2006 Sharp Corporation
 *
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <pthread.h>
#include <unistd.h>
#include <errno.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/reboot.h>
#include <sys/time.h>
#include <sys/mount.h>

#include "../launcher.h"

#define DBGPRINT(...)                            \
  do {                                           \
    if (the_message_mode == MSGMODE_VERBOSE) {   \
      fprintf(stderr, "Launcher: " __VA_ARGS__); \
    }                                            \
  } while(0)

#define DBGPERROR(str)                           \
  do {                                           \
    if (the_message_mode == MSGMODE_VERBOSE) {   \
      perror("Launcher: " str);                  \
    }                                            \
  } while(0)

#define NELMS(array) (sizeof(array)/sizeof((array)[0]))
#define CUE_STR "cue"

typedef enum msg_mode {
  MSGMODE_QUIET,
  MSGMODE_VERBOSE,
} msg_mode_t;

typedef enum launch_mode {
  LAUNCHMODE_NORMAL,
  LAUNCHMODE_DEBUG,
} launch_mode_t;

typedef enum fifo_mode {
  FIFOMODE_AVAILABLE,
  FIFOMODE_NOTAVAILABLE,
} fifo_mode_t;

#if defined(UNHOSTED)
static msg_mode_t      the_message_mode = MSGMODE_QUIET;
#else
static msg_mode_t      the_message_mode = MSGMODE_VERBOSE;
#endif /* defined(UNHOSTED) */

static launch_mode_t   the_launch_mode = LAUNCHMODE_NORMAL;
static char*           the_argv[MAX_NUM_OF_ARGS + 1];
static pthread_t       the_watcher_thread;
static int             the_fifo_mode = FIFOMODE_AVAILABLE;

extern int insmod_main(int argc, char* argv[]);

static void  print_timestamp(const char* string);
static void  print_status(int status, pid_t pid, const char* string);
static int   open_pipe(process_table_t* table);
static int   close_pipe_internally(int* fd);
static int   close_pipe_for_read(process_table_t* table);
static int   close_pipe_for_write(process_table_t* table);
static int   setup_args(process_table_t* table);
static int   minimize_thread_priority(void);
static void  stand_by(process_table_t* table);
static void  launch(process_table_t* table);
static void  launch_again(process_table_t* table);
static void  startup(void);
static int   find_process(char* buffer);
static void  cue(int idx);
static void  halt(int idx);
static char* find_next_token(char* buffer);
static void* watcher(void* arg);
static void  start_watcher(void);
static void  maintain(void);
static void* thread_main(void* arg);
static void  restart_system();
static void  setup_mode(const char* cmdline);

static void print_timestamp(const char* string)
{
#ifdef ENABLE_DBGTIME
  char str[MAX_LENGTH_OF_COMMAND];
  snprintf(str, sizeof(str),"$DT Launcher:%s",string);
  access(str, 0);
#else  
  if (the_message_mode != MSGMODE_VERBOSE) {
    return;
  }

  struct timeval timeofday;
  gettimeofday(&timeofday, NULL);
  fprintf(stderr, "Launcher: %s\n", string);
  fprintf(stderr, "Launcher: %ld.%06ld\n",
          timeofday.tv_sec, timeofday.tv_usec);
#endif /* ENABLE_DBGTIME */
}

static void print_status(int status, pid_t pid, const char* string)
{
  if (the_message_mode != MSGMODE_VERBOSE) {
    return;
  }
    
  fprintf(stderr, "Launcher: Process #%d (%s) ", pid, string);
  
  if (WIFEXITED(status)) {
    fprintf(stderr, "exited with status %d.\n", WEXITSTATUS(status));
  }
  else if (WIFSTOPPED(status)) {
    fprintf(stderr, "is stooped by Process #%d.\n", WSTOPSIG(status));
  }
  else if (WIFSIGNALED(status)) {
    int signal = WTERMSIG(status);
    switch (signal) {
      
    case SIGSEGV:
      fprintf(stderr, "caused Segmentation Fault.\n");
      break;

    case SIGBUS:
      fprintf(stderr, "caused Bus Error.\n");
      break;

    case SIGFPE:
      fprintf(stderr, "caused Floating Pointer Exception.\n");
      break;

    default:
      fprintf(stderr, "terminated by signal #%d.\n", signal);
    }
  }
  else {
    fprintf(stderr, "terminated by Unknown Reason (status = %d).\n", status);
  }
  fflush(stderr);
}

static int open_pipe(process_table_t* table)
{
  if ((the_fifo_mode == FIFOMODE_AVAILABLE) && (table->delay != 0)) {
    if (pipe(table->pipe_fd) == -1) {
      DBGPERROR("pipe failure");

      table->pipe_fd[0] = table->pipe_fd[1] = -1;
      return -1;
    }
  }
  else {
    table->pipe_fd[0] = table->pipe_fd[1] = -1;
  }

  return 0;
}

static int close_pipe_internally(int* fd)
{
  if (*fd == -1) {
    return 0;
  }

  if (close(*fd) == -1) {
    return -1;
  }

  *fd = -1;
  return 0;
}

static int close_pipe_for_read(process_table_t* table)
{
  return close_pipe_internally(&(table->pipe_fd[0]));
}

static int close_pipe_for_write(process_table_t* table)
{
  return close_pipe_internally(&(table->pipe_fd[1]));
}

static int setup_args(process_table_t* table)
{
  static char  args_buffer[MAX_LENGTH_OF_COMMAND];
  char* p    = args_buffer;
  int   argc = 0;

  strncpy(args_buffer, table->string, sizeof(args_buffer) - 1);

  if ((the_launch_mode == LAUNCHMODE_DEBUG) && (table->dbgopts != NULL)) {
    int optstart = strlen(table->string);
    int optsize  = sizeof(args_buffer) - optstart;
    
    strncpy(args_buffer + optstart, table->dbgopts, optsize - 1);
  }
      
  args_buffer[sizeof(args_buffer) - 1] = '\0';

  while (argc < sizeof(the_argv) - 1) {
    while (isspace(*p))
      ++p;
    
    if (*p == '\0') {
      break;
    }
    
    the_argv[argc++] = p;
    
    while (!isspace(*p) && (*p != '\0'))
      ++p;
    
    if (*p == '\0') {
      break;
    }
    
    *p++ = '\0';
  }
  
  the_argv[argc] = NULL;
  return argc;
}

static int minimize_thread_priority(void)
{
#if defined(USE_THREAD)
  struct sched_param param;
  pthread_t thread = pthread_self();

  param.sched_priority = sched_get_priority_min(PROGRAMS_SCHED_POLICY);
  return pthread_setschedparam(thread, PROGRAMS_SCHED_POLICY, &param);
#else
  return 0;
#endif /* defined(USE_THREAD) */
}

static void stand_by(process_table_t* table)
{
  int waiting_time = table->delay;
  int waiting_fd   = table->pipe_fd[0];

  int             rfd;
  fd_set          rfds_impl;
  fd_set*         rfds;
  struct timeval  tv_impl;
  struct timeval* tv;

  if (0 == waiting_time) {
    return;
  }

  DBGPRINT("%s: standing by: time = %d\n", table->string, table->delay);
  
  if (0 < waiting_time) {
    tv = &tv_impl;
    tv->tv_sec  = waiting_time;
    tv->tv_usec = 0;
  }
  else {
    tv = NULL;
  }

  if (0 <= waiting_fd) {
    rfd  = waiting_fd + 1;
    rfds = &rfds_impl;
    FD_ZERO(rfds);
    FD_SET (waiting_fd, rfds);
  }
  else {
    rfd  = 0;
    rfds = NULL;
  }

  if (select(rfd, rfds, NULL, NULL, tv) == -1) {
    DBGPERROR("select failure");
  }
}

static void launch(process_table_t* table)
{
  open_pipe(table);

  table->pid = fork();

  if (table->pid == 0) {
    close_pipe_for_write(table);
    stand_by(table);
    close_pipe_for_read(table);

    DBGPRINT("%s: launched\n", table->string);

    if (table->action != Launch_Primary) {
      freopen("/dev/null", "r", stdin);
    }

    setup_args(table);
    
    minimize_thread_priority();
    
    execvp(the_argv[0], the_argv);
    DBGPERROR("execvp failure");
    exit(EXIT_FAILURE);
  }
  else if (table->pid == -1) {
    DBGPERROR("fork failure");
    exit(EXIT_FAILURE);
  }

  close_pipe_for_read(table);
}

static void launch_again(process_table_t* table)
{
  table->delay = 0;
  launch(table);
}

static void startup(void)
{
  int   i;

  for (i = 0; i < NELMS(the_proctab); ++i) {

    print_timestamp(the_proctab[i].string);

    switch (the_proctab[i].action) {

    case Insmod:
      {
        int argc = setup_args(&the_proctab[i]);
        insmod_main(argc, the_argv);
      }
      break;

    default:
      launch(&the_proctab[i]);
    }
  }
}

static int find_process(char* buffer)
{
  int i;
  
  for (i = 0; i < NELMS(the_proctab); ++i) {
    const char* s = the_proctab[i].string;
    int         l = strlen(s);
    char*       p = strchr(s, ' ');
    
    if (p != NULL) {
      l = p - s;
    }
    
    if (strncmp(s, buffer, l) == 0) {
      return i;
    }
  }
  
  return -1;
}

static void cue(int idx)
{
  process_table_t* table = &the_proctab[idx];
  int fd                 = table->pipe_fd[1];
  
  DBGPRINT("The process '%s' has gotten a cue.\n", table->string);
  
  if (0 <= fd) {
    write(fd, CUE_STR, sizeof(CUE_STR));
    close_pipe_for_write(table);
  }
}

static void halt(int idx)
{
  process_table_t* table = &the_proctab[idx];
  int pid                = table->pid;
  
  DBGPRINT("The process '%s' is requested to restart.\n", table->string);

  if (0 < pid) {
    kill(pid, SIGKILL);
  }
}

static char* find_next_token(char* buffer)
{
  char* p = buffer;
  
  for (;;) {
    if (*p == '\0') {
      return NULL;
    }
    if (isspace(*p)) {
      break;
    }
    ++p;
  }

  for (;;) {
    if (*p == '\0') {
      return NULL;
    }
    if (!isspace(*p)) {
      break;
    }
    ++p;
  }

  return p;
}

static void* watcher(void* arg)
{
  FILE* fifo;
  char  buffer[MAX_LENGTH_OF_COMMAND];
  int   idx;

  signal(SIGPIPE, SIG_IGN);

  while ((fifo = fopen(FIFOFILE, "r")) != NULL) {
    while (fgets(buffer, sizeof(buffer), fifo) != NULL) {
      char* command  = buffer;
      char* argument = find_next_token(buffer);
      
      if (argument == NULL) {
        DBGPRINT("Invalid string: '%s'.\n", buffer);
        continue;
      }

      if ((idx = find_process(argument)) < 0) {
        DBGPRINT("No such process: '%s'.\n", buffer);
      }
      else if (idx == 0) {
        DBGPRINT("Process is not running: '%s'.\n", buffer);
      }
      else {
        switch(*command) {
        case 'S':
        case 's':
          cue(idx);
          break;
        case 'R':
        case 'r':
          halt(idx);
          break;
        default:
          DBGPRINT("Unknown command '%c'.\n", *command);
        }
      }
    }
      
    fclose(fifo);
  }

  DBGPERROR("opening FIFO failure");
  return NULL;
}

static void start_watcher(void)
{
  struct sched_param param;
  pthread_attr_t     attr;
  int                result;

  if (the_fifo_mode != FIFOMODE_AVAILABLE) {
    return;
  }

  param.sched_priority = sched_get_priority_max(LAUNCHER_SCHED_POLICY);
  pthread_attr_init(&attr);
  pthread_attr_setschedpolicy (&attr, LAUNCHER_SCHED_POLICY);
  pthread_attr_setschedparam  (&attr, &param);

  result = pthread_create(&the_watcher_thread, &attr, watcher, NULL);

  if (result != 0) {
    DBGPRINT("Failed to start the watcher thread.\n");
  }
}

static void maintain(void)
{
  int   i;
  pid_t pid;
  int   status;

  while ((pid = wait(&status)) != -1) {

    for (i = 0; i < NELMS(the_proctab); ++i) {
      if (pid == the_proctab[i].pid) {
        break;
      }
    }
    
    if (i == NELMS(the_proctab)) {
      continue;
    }
    
    print_status(status, pid, the_proctab[i].string);

    switch (the_proctab[i].action) {
      
    case Launch_Primary:
    case Launch_Essential:
      if (the_proctab[i].delay < 0) {
        launch_again(&the_proctab[i]);
        break;
      }
      
      return;
      
    case Launch_OneShot:
      break;
      
    case Launch_Repetitive:
      launch_again(&the_proctab[i]);
      break;
      
    default:
      assert(0);
    }
  }
}

static void* thread_main(void* arg)
{
  startup();
  start_watcher();
  maintain();
  restart_system();
  
  return NULL;
}

static void restart_system()
{
  int i;

  DBGPRINT("Rebooting...\n");

  for (i = 0; i < NELMS(the_proctab); ++i) {
    pid_t pid = the_proctab[i].pid;

    if (pid != 0) {
      DBGPRINT("The process #%d was killed.\n", pid);

      if (kill(pid, SIGTERM) < 0) {
        DBGPERROR("kill");
      }
    }
  }
  sync();

  sleep(WATCHDOG_INTERVAL);
  kill(0, SIGKILL);
}

static void setup_mode(const char* cmdline)
{
  const char* postfix = DEBUG_PROGNAME_POSTFIX;
  const char* p       = strrchr(cmdline, postfix[0]);

  if ((p != NULL) && (strncmp(p, postfix, strlen(postfix)) == 0)) {
    the_message_mode = MSGMODE_VERBOSE;
    the_launch_mode  = LAUNCHMODE_DEBUG;
  }
}

int main(int argc, char* argv[])
{
  pid_t pid = fork();

  setup_mode(argv[0]);

  if (pid == 0) {

    /* Setup /proc filesystem */
    if (mount("proc", "/proc", "proc", 0, "defaults") == -1) {
      DBGPERROR("mount (/proc) failure");
    }

    /* Setup FIFO */
    if (mkfifo(FIFOFILE, S_IRUSR | S_IWUSR) == -1 && errno != EEXIST) {
      DBGPERROR("mkfifo failure");
      the_fifo_mode = FIFOMODE_NOTAVAILABLE;
    }

#if defined(USE_THREAD)
    {
      struct sched_param param;
      pthread_attr_t     attr;
      pthread_t          thread;
      int result;
      
      param.sched_priority = sched_get_priority_max(LAUNCHER_SCHED_POLICY);
      pthread_attr_init(&attr);
      pthread_attr_setschedpolicy (&attr, LAUNCHER_SCHED_POLICY);
      pthread_attr_setschedparam  (&attr, &param);
      
      result = pthread_create(&thread, &attr, thread_main, NULL);
      
      if (result == 0) {
        pthread_join(thread, NULL);
      }
    }
#else
    thread_main(NULL);
#endif /* defined(USE_THREAD) */

  }
  else {
    for (;;) {
      pid_t id;
      int   status;

      id = wait(&status);

      DBGPRINT("'wait' returned: retval=%d, status=%d\n", id, status);
    }
  }

  return 0;
}
