/* 
 * rt3.c: rt3 trace log related routines.
 *
 * Copyright (C) Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include <linux/string.h>
#include <linux/types.h>
#include <linux/interrupt.h> 
#include <linux/irq.h> 

#include <asm/irq.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/compiler.h>
#include <linux/sched.h>
#ifdef CONFIG_SMP
#include <linux/spinlock.h>
#endif

#ifdef CONFIG_RT3_TRACE
#include <asm/rt3.h>
#include <mach/hardware.h>

#ifndef CONFIG_RT3_TRACE_DO_ENABLE
volatile int rt3_trace_enabled = 0;

#include <linux/irq.h>
#ifdef CONFIG_SMP
volatile unsigned int rt3_trace_current_pid[NR_CPUS];
#endif
#endif

#ifdef CONFIG_RT3_TRACE_LOG_ON_MEM
u32 rt3_trace_buff[RT3_TRACE_BUFF_SIZE_IN_WORD];
u32 rtt_buff_size = sizeof(rt3_trace_buff);
volatile u32 *rt3_trace_buff_ptr = rt3_trace_buff;
volatile int rt3_trace_buff_ovrflw = 0;
#endif

#ifdef CONFIG_RT3_LOG_EXIT_TRACE_LOG
struct rt3_trace_exit_struct rt3_exit_log_table[CONFIG_RT3_EXIT_TRACE_LOG_MAX_NUM];
int rt3_exit_log_cnt;
#endif /* CONFIG_RT3_EXIT_TRACE_LOG */

#ifdef CONFIG_SMP
spinlock_t rt3_trace_lock = SPIN_LOCK_UNLOCKED;
#endif

#endif	/* CONFIG_RT3_TRACE */
#define	LOGID(a)	(((u32)(a) >> 24) & 0x000000ff)
#define	STAMP(a)	(((u32)(a) >>  8) & 0x0000ffff)
#define	VALUE(a)	(((u32)(a) >>  0) & 0x000000ff)
#define	VALUE2(a)	(((u32)(a) >>  0) & 0x00ffffff)

/* for rt3ctrl */

static char *exit_task_name_by_pid(pid_t pid){
	
	int i;
	for(i=0;i<rt3_exit_log_cnt;i++){
		if(rt3_exit_log_table[i].pid == pid){
			return rt3_exit_log_table[i].comm;
		}
	}
	
	return NULL;
}


static int
rt3_trace_get_tskname(OBS_FNAME_INFO *exinf)
{
pid_t				pid;
struct task_struct	*task;

char				buf[16+1];
int					i;
unsigned long 		flags;
	get_user(pid, &(exinf->pid));
	if (pid == 0)	return 0;

	rt3_trace_spin_lock_irqsave(flags);

	if ((task = find_task_by_vpid (pid)) != NULL) {
		memcpy(buf, task->comm, 16);
		for (i=0; i<16; i++) {
			if ((buf[i] < 0x20) || (buf[i] > 0x7e))	break;
		}
		buf[i] = '\0';
		if(copy_to_user(exinf->name, buf, 16)<1){
		}
	}else if(exit_task_name_by_pid(pid) != NULL){
	        memcpy(buf,exit_task_name_by_pid(pid),16);
	        for (i=0; i<16; i++) {
			if ((buf[i] < 0x20) || (buf[i] > 0x7e))	break;
		}
		buf[i] = '\0';
		if(copy_to_user(exinf->name, buf, 16) < 1){
		}
	}
	
	rt3_trace_spin_unlock_irqrestore(flags);
	return 0;
}

static void
rt3_trace_init_buffer(void)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	
	rt3_trace_buff_ptr		= rt3_trace_buff;
	rt3_trace_buff_ovrflw	= 0;

	rt3_trace_spin_unlock_irqrestore(flags);
	return;
}

static int
rt3_trace_get_rt3info(rt3_info *exinf)
 {
	unsigned long flags;
	ulong timer_cycle = (UNIPHIER_GLOBAL_TIMER_SRC >> RT3_TRACE_TIMESTAMP_SHIFT);
	rt3_trace_spin_lock_irqsave(flags);
#ifdef	CONFIG_SMP
 	put_user( 1,								&(exinf->linux_smp));
#else
	put_user( 0,								&(exinf->linux_smp));
#endif		
	put_user((u32)RT3_TRACE_SOFTIRQ_PID,	&(exinf->softirq_pid));
	put_user((u32)rt3_trace_buff,			&(exinf->buff));
	put_user((u32)rtt_buff_size,				&(exinf->buffsize));
	put_user((u32)rt3_trace_buff_ptr,		&(exinf->ptr));
	put_user((u32)rt3_trace_buff_ovrflw, 	&(exinf->ovrflw));
#ifndef CONFIG_RT3_TRACE_DO_ENABLE
 	put_user((u32)rt3_trace_enabled,		&(exinf->enabled));
#else
 	put_user( 1,								&(exinf->enabled));
#endif
 	put_user((u32)timer_cycle,					&(exinf->timer_cycle));

	rt3_trace_spin_unlock_irqrestore(flags);
	return 0;
}

static int
rt3_trace_get_buf(rt3_getlog *exinf)
{
	u32		*ptr, c, cprev = 0;
	u32		tmp;

	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);

	get_user(tmp, &(exinf->ptr));
	ptr = (u32 *)tmp;


	if (rt3_trace_buff_ovrflw == 0) {

	    if (ptr == rt3_trace_buff) {						// Buffer Empty
			put_user(0, &(exinf->ptr));						// End of buffer
			goto out_with_unlock;
	    }
	    if (ptr == 0) {										// Set buffer top
			ptr = (u32 *)rt3_trace_buff_ptr;
	    }
	    while (1) {
			ptr--;
			if (LOGID(c = *ptr) != 0x00) break;
			cprev = c;
			if (ptr == rt3_trace_buff) {					// Bottom ?
				put_user(0, &(exinf->ptr));					// End of buffer
				goto out_with_unlock;
			}
	    }

	} else { /* In case of rt3_trace_buff_ovrflw == 1 */

	    if (ptr == rt3_trace_buff_ptr) {					// Already Bottom ?
			put_user(0, &(exinf->ptr));						// End of buffer
			goto out_with_unlock;
	    }
	    if (ptr == 0) {										// Set buffer top
			ptr = (u32 *)rt3_trace_buff_ptr;
	    }
	    while (1) {
			ptr--;
			if (ptr < rt3_trace_buff) {					// Boundary of buffer
				ptr += RT3_TRACE_BUFF_SIZE_IN_WORD;
			}
			if (LOGID(c = *ptr) != 0x00) break;
			cprev = c;
			if (ptr == rt3_trace_buff_ptr) {
				put_user(0, &(exinf->ptr));					// End of buffer
				goto out_with_unlock;
			}
	    }
	}

	put_user((u32)ptr,           &(exinf->ptr));
	put_user((u32)LOGID(c),      &(exinf->logid));	// 1st word  8bit logid
	put_user((u32)STAMP(c),      &(exinf->ts));		// 1st word 16bit tstamp
	put_user((u32)VALUE(c),      &(exinf->val));	// 1st word  8bit value
	put_user((u32)VALUE2(cprev), &(exinf->val2));	// 2nd word 24bit value

out_with_unlock:
	rt3_trace_spin_unlock_irqrestore(flags);
	return 0;
}

int
rt3_trace_get_isrname(int isr, char *isrname)
{
struct irq_desc		*irq = &irq_desc[isr];
struct irqaction	*action;
char				buf[48+1], *dst;
unsigned long		flags;
int					i;

	spin_lock_irqsave(&irq->lock, flags);
 
	buf[0]	= '\0';
	dst		= buf;

	action = irq->action;
	while (action) {
		strncpy(dst, action->name, 16);
		for (i=0; i<16; i++) {
 			if ((dst[i] < 0x20) || (dst[i] > 0x7e))	break;
		}
		dst  = dst + i;
		*dst = '\0';
		if ((dst - buf) > 32) break;

		action = action->next;
		if (!(action))	break;

		strcat(dst, "->");
		dst += 2;
		if ((dst - buf) > 32) break;
	}

	buf[31] = '\0';
	if(copy_to_user(isrname, buf, 32) < 1){
	}

	spin_unlock_irqrestore(&irq->lock, flags);
	return 0;
}


asmlinkage u32 rt3_trace(int log_id, int info, int *exinfo)
{
#ifdef CONFIG_RT3_TRACE
	switch (log_id) {

#ifndef CONFIG_RT3_TRACE_DO_ENABLE
	case RT3_TRACE_DISABLE:
		rt3_log_disable_event(current->pid);
		break;

	case RT3_TRACE_ENABLE:
	{
		unsigned long flags;
		local_save_flags(flags);
		rt3_log_enable_event(0x07 & (flags >> 8));
		break;
	}
#endif /* !CONFIG_RT3_TRACE_DO_ENABLE */
	case RT3_TRACE_USERLOG:
		rt3_log_user_event(info);
		break;

	case RT3_TRACE_USERLOG_EXT:
		rt3_log_user8_24_event(info >> 24, info & 0xffffff);
		break;

#ifdef CONFIG_RT3_LOG_ISR
	case RT3_TRACE_ENTISR:
		rt3_log_isr_entry_event(info);
		break;

	case RT3_TRACE_LEAISR:
		rt3_log_isr_exit_event(info);
		break;
#endif /* CONFIG_RT3_LOG_ISR */

	case RT3_TRACE_ENTIRQ:
		rt3_log_irq_entry_event(info);
		break;

	case RT3_TRACE_LEAIRQ:
		rt3_log_irq_exit_event(info);
		break;
	case RT3_SYS_TRACE_RT3INFO:
	{
		rt3_trace_get_rt3info((rt3_info *)exinfo);
		break;
	}
	case RT3_SYS_TRACE_GETBUF:
	{
		rt3_trace_get_buf((rt3_getlog *)exinfo);
		break;
	}
 
	case RT3_SYS_TRACE_GETNAME:
	{
		rt3_trace_get_tskname((OBS_FNAME_INFO *)exinfo);
		break;
	}
	case RT3_SYS_TRACE_GETISR:
	{
		rt3_trace_get_isrname(info, (char *)exinfo);
		break;
	}
	case RT3_SYS_TRACE_INIT:
	{
		rt3_trace_init_buffer();
		break;
	}
	default:
		break;

	}
#endif /* CONFIG_RT3_TRACE */

	return 0;
}

EXPORT_SYMBOL(rt3_trace);

#ifdef CONFIG_RT3_TRACE_LOG_ON_MEM
void rt3_log_event(u32 data)
{
	*rt3_trace_buff_ptr = (u32)(data);
	if (rt3_trace_buff_ptr
	    < rt3_trace_buff + RT3_TRACE_BUFF_SIZE_IN_WORD - 1) {
		rt3_trace_buff_ptr++;
	}
	else {
		rt3_trace_buff_ovrflw = 1;
		rt3_trace_buff_ptr = rt3_trace_buff;
	}
}
#endif /* CONFIG_RT3_TRACE_LOG_ON_MEM */
