/*
 * linux/arch/arm/lib/lib1funcs.S: Optimized ARM division routines
 *
 * Author: Nicolas Pitre <nico@fluxnic.net>
 *   - contributed to gcc-3.4 on Sep 30, 2003
 *   - adapted for the Linux kernel on Oct 2, 2003
 */

/* Copyright 1995, 1996, 1998, 1999, 2000, 2003 Free Software Foundation, Inc.

This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file into combinations with other programs,
and to distribute those combinations without any restriction coming
from the use of this file.  (The General Public License restrictions
do apply in other respects; for example, they cover modification of
the file, and distribution when not linked into a combine
executable.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#include <linux/linkage.h>
#include <asm/assembler.h>


.macro ARM_DIV_BODY dividend, divisor, result, curbit

#if __LINUX_ARM_ARCH__ >= 5

	clz	\curbit, \divisor
	clz	\result, \dividend
	sub	\result, \curbit, \result
	mov	\curbit, #1
	mov	\divisor, \divisor, lsl \result
	mov	\curbit, \curbit, lsl \result
	mov	\result, #0
	
#else

	@ Initially shift the divisor left 3 bits if possible,
	@ set curbit accordingly.  This allows for curbit to be located
	@ at the left end of each 4 bit nibbles in the division loop
	@ to save one loop in most cases.
	tst	\divisor, #0xe0000000
	moveq	\divisor, \divisor, lsl #3
	moveq	\curbit, #8
	movne	\curbit, #1

	@ Unless the divisor is very big, shift it up in multiples of
	@ four bits, since this is the amount of unwinding in the main
	@ division loop.  Continue shifting until the divisor is 
	@ larger than the dividend.
1:	cmp	\divisor, #0x10000000
	cmplo	\divisor, \dividend
	movlo	\divisor, \divisor, lsl #4
	movlo	\curbit, \curbit, lsl #4
	blo	1b

	@ For very big divisors, we must shift it a bit at a time, or
	@ we will be in danger of overflowing.
1:	cmp	\divisor, #0x80000000
	cmplo	\divisor, \dividend
	movlo	\divisor, \divisor, lsl #1
	movlo	\curbit, \curbit, lsl #1
	blo	1b

	mov	\result, #0

#endif

	@ Division loop
1:	cmp	\dividend, \divisor
	subhs	\dividend, \dividend, \divisor
	orrhs	\result,   \result,   \curbit
	cmp	\dividend, \divisor,  lsr #1
	subhs	\dividend, \dividend, \divisor, lsr #1
	orrhs	\result,   \result,   \curbit,  lsr #1
	cmp	\dividend, \divisor,  lsr #2
	subhs	\dividend, \dividend, \divisor, lsr #2
	orrhs	\result,   \result,   \curbit,  lsr #2
	cmp	\dividend, \divisor,  lsr #3
	subhs	\dividend, \dividend, \divisor, lsr #3
	orrhs	\result,   \result,   \curbit,  lsr #3
	cmp	\dividend, #0			@ Early termination?
	movnes	\curbit,   \curbit,  lsr #4	@ No, any more bits to do?
	movne	\divisor,  \divisor, lsr #4
	bne	1b

.endm


.macro ARM_DIV2_ORDER divisor, order

#if __LINUX_ARM_ARCH__ >= 5

	clz	\order, \divisor
	rsb	\order, \order, #31

#else

	cmp	\divisor, #(1 << 16)
	movhs	\divisor, \divisor, lsr #16
	movhs	\order, #16
	movlo	\order, #0

	cmp	\divisor, #(1 << 8)
	movhs	\divisor, \divisor, lsr #8
	addhs	\order, \order, #8

	cmp	\divisor, #(1 << 4)
	movhs	\divisor, \divisor, lsr #4
	addhs	\order, \order, #4

	cmp	\divisor, #(1 << 2)
	addhi	\order, \order, #3
	addls	\order, \order, \divisor, lsr #1

#endif

.endm


.macro ARM_MOD_BODY dividend, divisor, order, spare

#if __LINUX_ARM_ARCH__ >= 5

	clz	\order, \divisor
	clz	\spare, \dividend
	sub	\order, \order, \spare
	mov	\divisor, \divisor, lsl \order

#else

	mov	\order, #0

	@ Unless the divisor is very big, shift it up in multiples of
	@ four bits, since this is the amount of unwinding in the main
	@ division loop.  Continue shifting until the divisor is 
	@ larger than the dividend.
1:	cmp	\divisor, #0x10000000
	cmplo	\divisor, \dividend
	movlo	\divisor, \divisor, lsl #4
	addlo	\order, \order, #4
	blo	1b

	@ For very big divisors, we must shift it a bit at a time, or
	@ we will be in danger of overflowing.
1:	cmp	\divisor, #0x80000000
	cmplo	\divisor, \dividend
	movlo	\divisor, \divisor, lsl #1
	addlo	\order, \order, #1
	blo	1b

#endif

	@ Perform all needed substractions to keep only the reminder.
	@ Do comparisons in batch of 4 first.
	subs	\order, \order, #3		@ yes, 3 is intended here
	blt	2f

1:	cmp	\dividend, \divisor
	subhs	\dividend, \dividend, \divisor
	cmp	\dividend, \divisor,  lsr #1
	subhs	\dividend, \dividend, \divisor, lsr #1
	cmp	\dividend, \divisor,  lsr #2
	subhs	\dividend, \dividend, \divisor, lsr #2
	cmp	\dividend, \divisor,  lsr #3
	subhs	\dividend, \dividend, \divisor, lsr #3
	cmp	\dividend, #1
	mov	\divisor, \divisor, lsr #4
	subges	\order, \order, #4
	bge	1b

	tst	\order, #3
	teqne	\dividend, #0
	beq	5f

	@ Either 1, 2 or 3 comparison/substractions are left.
2:	cmn	\order, #2
	blt	4f
	beq	3f
	cmp	\dividend, \divisor
	subhs	\dividend, \dividend, \divisor
	mov	\divisor,  \divisor,  lsr #1
3:	cmp	\dividend, \divisor
	subhs	\dividend, \dividend, \divisor
	mov	\divisor,  \divisor,  lsr #1
4:	cmp	\dividend, \divisor
	subhs	\dividend, \dividend, \divisor
5:
.endm


ENTRY(__udivsi3)
ENTRY(__aeabi_uidiv)

	subs	r2, r1, #1
	moveq	pc, lr
	bcc	Ldiv0
	cmp	r0, r1
	bls	11f
	tst	r1, r2
	beq	12f

	ARM_DIV_BODY r0, r1, r2, r3

	mov	r0, r2
	mov	pc, lr

11:	moveq	r0, #1
	movne	r0, #0
	mov	pc, lr

12:	ARM_DIV2_ORDER r1, r2

	mov	r0, r0, lsr r2
	mov	pc, lr

ENDPROC(__udivsi3)
ENDPROC(__aeabi_uidiv)

ENTRY(__umodsi3)

	subs	r2, r1, #1			@ compare divisor with 1
	bcc	Ldiv0
	cmpne	r0, r1				@ compare dividend with divisor
	moveq   r0, #0
	tsthi	r1, r2				@ see if divisor is power of 2
	andeq	r0, r0, r2
	movls	pc, lr

	ARM_MOD_BODY r0, r1, r2, r3

	mov	pc, lr

ENDPROC(__umodsi3)

ENTRY(__divsi3)
ENTRY(__aeabi_idiv)

	cmp	r1, #0
	eor	ip, r0, r1			@ save the sign of the result.
	beq	Ldiv0
	rsbmi	r1, r1, #0			@ loops below use unsigned.
	subs	r2, r1, #1			@ division by 1 or -1 ?
	beq	10f
	movs	r3, r0
	rsbmi	r3, r0, #0			@ positive dividend value
	cmp	r3, r1
	bls	11f
	tst	r1, r2				@ divisor is power of 2 ?
	beq	12f

	ARM_DIV_BODY r3, r1, r0, r2

	cmp	ip, #0
	rsbmi	r0, r0, #0
	mov	pc, lr

10:	teq	ip, r0				@ same sign ?
	rsbmi	r0, r0, #0
	mov	pc, lr

11:	movlo	r0, #0
	moveq	r0, ip, asr #31
	orreq	r0, r0, #1
	mov	pc, lr

12:	ARM_DIV2_ORDER r1, r2

	cmp	ip, #0
	mov	r0, r3, lsr r2
	rsbmi	r0, r0, #0
	mov	pc, lr

ENDPROC(__divsi3)
ENDPROC(__aeabi_idiv)

ENTRY(__modsi3)

	cmp	r1, #0
	beq	Ldiv0
	rsbmi	r1, r1, #0			@ loops below use unsigned.
	movs	ip, r0				@ preserve sign of dividend
	rsbmi	r0, r0, #0			@ if negative make positive
	subs	r2, r1, #1			@ compare divisor with 1
	cmpne	r0, r1				@ compare dividend with divisor
	moveq	r0, #0
	tsthi	r1, r2				@ see if divisor is power of 2
	andeq	r0, r0, r2
	bls	10f

	ARM_MOD_BODY r0, r1, r2, r3

10:	cmp	ip, #0
	rsbmi	r0, r0, #0
	mov	pc, lr

ENDPROC(__modsi3)

#ifdef CONFIG_AEABI

ENTRY(__aeabi_uidivmod)

	stmfd	sp!, {r0, r1, ip, lr}
	bl	__aeabi_uidiv
	ldmfd	sp!, {r1, r2, ip, lr}
	mul	r3, r0, r2
	sub	r1, r1, r3
	mov	pc, lr

ENDPROC(__aeabi_uidivmod)

ENTRY(__aeabi_idivmod)

	stmfd	sp!, {r0, r1, ip, lr}
	bl	__aeabi_idiv
	ldmfd	sp!, {r1, r2, ip, lr}
	mul	r3, r0, r2
	sub	r1, r1, r3
	mov	pc, lr

ENDPROC(__aeabi_idivmod)

ENTRY(__aeabi_ldivmod)

	cmp	r3, #0
	cmpeq	r2, #0
	bne	1f
	cmp	r1, #0
	cmpeq	r0, #0
	movlt	r1, #0x80000000
	movlt	r0, #0
	mvngt	r1, #0x80000000
	mvngt	r0, #0
	b	Ldiv0
1:
	sub	sp, sp, #8
	stmfd	sp!, {sp, lr}
	bl	__gnu_ldivmod_helper
	ldr	lr, [sp, #4]
	add	sp, sp, #8
	ldmfd	sp!, {r2, r3}
	bx	lr

ENDPROC(__aeabi_ldivmod)

ENTRY(__aeabi_uldivmod)

	cmp	r3, #0
	cmpeq	r2, #0
	bne	1f
	cmp	r1, #0
	cmpeq	r0, #0
	mvnne	r1, #0
	mvnne	r0, #0
	b	Ldiv0
1:
	sub	sp, sp, #8
	stmfd	sp!, {sp, lr}
	bl	__gnu_uldivmod_helper
	ldr	lr, [sp, #4]
	add	sp, sp, #8
	ldmfd	sp!, {r2, r3}
	bx	lr

ENDPROC(__aeabi_uldivmod)

ENTRY(__gnu_uldivmod_helper)

	stmfd	sp!, {r3, r4, r5, r6, r7, r8, r9, lr}
	mov	r4, r2
	mov	r5, r3
	mov	r6, r0
	mov	r7, r1
	bl	__udivdi3
	mul	r5, r0, r5
	umull	r8, r9, r0, r4
	mla	r4, r1, r4, r5
	mov	r2, r8
	add	r3, r4, r9
	subs	r6, r6, r2
	sbc	r7, r7, r3
	ldr	r3, [sp, #32]
	strd	r6, [r3]
	ldmfd	sp!, {r3, r4, r5, r6, r7, r8, r9, pc}

ENDPROC(__gnu_uldivmod_helper)

ENTRY(__gnu_ldivmod_helper)

	stmfd	sp!, {r3, r4, r5, r6, r7, r8, r9, lr}
	mov	r4, r2
	mov	r5, r3
	mov	r6, r0
	mov	r7, r1
	bl	__divdi3
	mul	r5, r0, r5
	umull	r8, r9, r0, r4
	mla	r4, r1, r4, r5
	mov	r2, r8
	add	r3, r4, r9
	subs	r6, r6, r2
	sbc	r7, r7, r3
	ldr	r3, [sp, #32]
	strd	r6, [r3]
	ldmfd	sp!, {r3, r4, r5, r6, r7, r8, r9, pc}

ENDPROC(__gnu_ldivmod_helper)

ENTRY(__divdi3)

	cmp	r1, #0
	stmfd	sp!, {r4, r5, r6, r7, r8, r9, sl, fp, lr}
	movge	r6, #0
	sub	sp, sp, #20
	blt	LS19
LS01:
	cmp	r3, #0
	blt	LS18
LS02:
	subs	ip, r3, #0
	mov	fp, r1
	mov	r4, r2
	mov	r5, r0
	bne	LS05
	cmp	r2, r1
	bls	LS09
	clz	r3, r2
	cmp	r3, #0
	rsbne	r2, r3, #32
	lsrne	r2, r0, r2
	lslne	r4, r4, r3
	orrne	fp, r2, r1, lsl r3
	lsr	r8, r4, #16
	lslne	r5, r0, r3
	mov	r1, r8
	mov	r0, fp
	bl	__aeabi_uidiv
	mov	r1, r8
	lsl	r7, r4, #16
	lsr	r7, r7, #16
	mov	r9, r0
	mov	r0, fp
	bl	__aeabi_uidivmod
	mul	r3, r7, r9
	lsr	fp, r5, #16
	orr	fp, fp, r1, lsl #16
	cmp	r3, fp
	bls	LS03
	adds	fp, fp, r4
	sub	r9, r9, #1
	bcs	LS03
	cmp	r3, fp
	subhi	r9, r9, #1
	addhi	fp, fp, r4
LS03:
	rsb	fp, r3, fp
	mov	r1, r8
	mov	r0, fp
	bl	__aeabi_uidiv
	mov	r1, r8
	mov	sl, r0
	mov	r0, fp
	bl	__aeabi_uidivmod
	mul	r7, sl, r7
	lsl	r0, r5, #16
	lsr	r5, r0, #16
	orr	r5, r5, r1, lsl #16
	cmp	r7, r5
	bls	LS04
	adds	r5, r5, r4
	sub	sl, sl, #1
	bcs	LS04
	cmp	r7, r5
	subhi	sl, sl, #1
LS04:
	orr	sl, sl, r9, lsl #16
	mov	r7, #0
	b	LS07
LS05:
	cmp	ip, r1
	bls	LS14
LS06:
	mov	r7, #0
	mov	sl, r7
LS07:
	cmp	r6, #0
	mov	r2, sl
	mov	r3, r7
	beq	LS08
	rsbs	r2, r2, #0
	rsc	r3, r3, #0
LS08:
	mov	r1, r3
	mov	r0, r2
	add	sp, sp, #20
	ldmfd	sp!, {r4, r5, r6, r7, r8, r9, sl, fp, pc}
LS09:
	cmp	r2, #0
	bne	LS10
	mov	r1, r2
	mov	r0, #1
	bl	__aeabi_uidiv
	mov	r4, r0
LS10:
	clz	sl, r4
	cmp	sl, #0
	bne	LS20
	lsl	r8, r4, #16
	mov	r1, #1
	rsb	sl, r4, fp
	lsr	r8, r8, #16
	lsr	r7, r4, #16
	str	r1, [sp, #12]
LS11:
	mov	r1, r7
	mov	r0, sl
	bl	__aeabi_uidiv
	mov	r1, r7
	mov	r9, r0
	mov	r0, sl
	bl	__aeabi_uidivmod
	mul	r3, r8, r9
	lsr	sl, r5, #16
	orr	fp, sl, r1, lsl #16
	cmp	r3, fp
	bls	LS12
	adds	fp, fp, r4
	sub	r9, r9, #1
	bcs	LS12
	cmp	r3, fp
	subhi	r9, r9, #1
	addhi	fp, fp, r4
LS12:
	rsb	fp, r3, fp
	mov	r1, r7
	mov	r0, fp
	bl	__aeabi_uidiv
	mov	r1, r7
	mov	sl, r0
	mov	r0, fp
	bl	__aeabi_uidivmod
	mul	r8, sl, r8
	lsl	r3, r5, #16
	lsr	r5, r3, #16
	orr	r5, r5, r1, lsl #16
	cmp	r8, r5
	bls	LS13
	adds	r5, r5, r4
	sub	sl, sl, #1
	bcs	LS13
	cmp	r8, r5
	subhi	sl, sl, #1
LS13:
	orr	sl, sl, r9, lsl #16
	ldr	r7, [sp, #12]
	b	LS07
LS14:
	clz	r7, ip
	cmp	r7, #0
	beq	LS23
	rsb	r1, r7, #32
	lsr	r8, r2, r1
	lsr	r3, r0, r1
	orr	r8, r8, ip, lsl r7
	lsr	sl, fp, r1
	orr	lr, r3, fp, lsl r7
	lsr	r9, r8, #16
	mov	r1, r9
	mov	r0, sl
	str	lr, [sp, #12]
	bl	__aeabi_uidiv
	mov	r1, r9
	mov	fp, r0
	mov	r0, sl
	bl	__aeabi_uidivmod
	lsl	r0, r8, #16
	ldr	r2, [sp, #12]
	lsr	sl, r0, #16
	mul	r0, sl, fp
	lsr	ip, r2, #16
	lsl	r2, r4, r7
	orr	r3, ip, r1, lsl #16
	cmp	r0, r3
	bls	LS15
	adds	r3, r3, r8
	sub	fp, fp, #1
	bcc	LS25
LS15:
	rsb	r4, r0, r3
	mov	r1, r9
	mov	r0, r4
	stmib	sp, {r2, r4}
	bl	__aeabi_uidiv
	ldr	r3, [sp, #8]
	mov	r1, r9
	mov	r4, r0
	mov	r0, r3
	bl	__aeabi_uidivmod
	ldr	r0, [sp, #12]
	mul	sl, r4, sl
	lsl	r2, r0, #16
	lsr	ip, r2, #16
	ldr	r2, [sp, #4]
	orr	r3, ip, r1, lsl #16
	cmp	sl, r3
	bls	LS16
	adds	r3, r3, r8
	sub	r4, r4, #1
	bcc	LS24
LS16:
	rsb	r1, sl, r3
	orr	sl, r4, fp, lsl #16
	lsr	ip, sl, #16
	lsr	lr, r2, #16
	bic	r8, sl, ip, lsl #16
	bic	r4, r2, lr, lsl #16
	mul	r0, r8, r4
	mul	r4, ip, r4
	mul	r8, lr, r8
	mul	lr, ip, lr
	adds	r8, r4, r8
	addcs	lr, lr, #0x10000
	adds	r0, r0, r8, lsl #16
	adc	lr, lr, r8, lsr #16
	cmp	r1, lr
	bcc	LS17
	movne	ip, #0
	moveq	ip, #1
	cmp	r0, r5, lsl r7
	movls	r7, #0
	andhi	r7, ip, #1
	cmp	r7, #0
	beq	LS07
LS17:
	sub	sl, sl, #1
	mov	r7, #0
	b	LS07
LS18:
	rsbs	r2, r2, #0
	rsc	r3, r3, #0
	mvn	r6, r6
	b	LS02
LS19:
	rsbs	r0, r0, #0
	rsc	r1, r1, #0
	mvn	r6, #0
	b	LS01
LS20:
	rsb	r7, sl, #32
	lsr	r3, r5, r7
	lsl	r4, r4, sl
	lsr	r8, fp, r7
	orr	lr, r3, fp, lsl sl
	lsr	r7, r4, #16
	mov	r1, r7
	mov	r0, r8
	str	lr, [sp, #12]
	bl	__aeabi_uidiv
	mov	r1, r7
	mov	fp, r0
	mov	r0, r8
	bl	__aeabi_uidivmod
	lsl	r2, r4, #16
	lsr	r8, r2, #16
	ldr	ip, [sp, #12]
	mul	r2, r8, fp
	lsr	r9, ip, #16
	orr	r3, r9, r1, lsl #16
	cmp	r2, r3
	bls	LS21
	adds	r3, r3, r4
	sub	fp, fp, #1
	bcc	LS26
LS21:
	rsb	r9, r2, r3
	mov	r1, r7
	mov	r0, r9
	str	r9, [sp, #8]
	bl	__aeabi_uidiv
	ldr	r1, [sp, #8]
	mov	r9, r0
	mov	r0, r1
	mov	r1, r7
	bl	__aeabi_uidivmod
	ldr	r2, [sp, #12]
	lsl	ip, r2, #16
	mul	r2, r8, r9
	lsr	r0, ip, #16
	orr	r3, r0, r1, lsl #16
	cmp	r2, r3
	bls	LS22
	adds	r3, r3, r4
	sub	r9, r9, #1
	bcs	LS22
	cmp	r2, r3
	subhi	r9, r9, #1
	addhi	r3, r3, r4
LS22:
	orr	lr, r9, fp, lsl #16
	lsl	r5, r5, sl
	str	lr, [sp, #12]
	rsb	sl, r2, r3
	b	LS11
LS23:
	cmp	ip, r1
	cmpcs	r2, r0
	movls	sl, #1
	bls	LS07
	b	LS06
LS24:
	cmp	sl, r3
	addhi	r3, r3, r8
	subhi	r4, r4, #1
	b	LS16
LS25:
	cmp	r0, r3
	subhi	fp, fp, #1
	addhi	r3, r3, r8
	b	LS15
LS26:
	cmp	r2, r3
	subhi	fp, fp, #1
	addhi	r3, r3, r4
	b	LS21

ENDPROC(__divdi3)

ENTRY(__udivdi3)

	cmp	r3, #0
	stmfd	sp!, {r4, r5, r6, r7, r8, r9, sl, fp, lr}
	mov	r4, r1
	sub	sp, sp, #12
	mov	r8, r0
	mov	r6, r2
	mov	r9, r1
	mov	r7, r3
	mov	r5, r2
	mov	sl, r1
	bne	LU03
	cmp	r2, r1
	bls	LU07
	clz	r3, r2
	cmp	r3, #0
	rsbne	r2, r3, #32
	lsrne	r2, r0, r2
	lslne	r5, r6, r3
	orrne	sl, r2, r1, lsl r3
	lsr	r6, r5, #16
	lslne	r8, r0, r3
	mov	r1, r6
	mov	r0, sl
	bl	__aeabi_uidiv
	mov	r1, r6
	lsl	r4, r5, #16
	lsr	r4, r4, #16
	mov	r9, r0
	mov	r0, sl
	bl	__aeabi_uidivmod
	mul	r3, r4, r9
	lsr	sl, r8, #16
	orr	sl, sl, r1, lsl #16
	cmp	r3, sl
	bls	LU01
	adds	sl, sl, r5
	sub	r9, r9, #1
	bcs	LU01
	cmp	r3, sl
	subhi	r9, r9, #1
	addhi	sl, sl, r5
LU01:
	rsb	sl, r3, sl
	mov	r1, r6
	mov	r0, sl
	bl	__aeabi_uidiv
	mov	r1, r6
	mov	r7, r0
	mov	r0, sl
	bl	__aeabi_uidivmod
	mul	r4, r7, r4
	lsl	r2, r8, #16
	lsr	r0, r2, #16
	orr	r8, r0, r1, lsl #16
	cmp	r4, r8
	bls	LU02
	adds	r8, r8, r5
	sub	r7, r7, #1
	bcs	LU02
	cmp	r4, r8
	subhi	r7, r7, #1
LU02:
	orr	r9, r7, r9, lsl #16
	mov	r1, #0
	b	LU14
LU03:
	cmp	r3, r1
	bhi	LU13
	clz	r6, r3
	cmp	r6, #0
	beq	LU12
	rsb	r1, r6, #32
	lsr	sl, r2, r1
	lsr	lr, r0, r1
	orr	r7, sl, r3, lsl r6
	lsr	fp, r4, r1
	orr	r9, lr, r4, lsl r6
	lsr	sl, r7, #16
	mov	r1, sl
	mov	r0, fp
	str	r9, [sp, #4]
	bl	__aeabi_uidiv
	mov	r1, sl
	mov	r9, r0
	mov	r0, fp
	bl	__aeabi_uidivmod
	lsl	r0, r7, #16
	ldr	r3, [sp, #4]
	lsr	r4, r0, #16
	mul	r2, r4, r9
	lsr	fp, r3, #16
	lsl	r3, r5, r6
	orr	fp, fp, r1, lsl #16
	cmp	r2, fp
	bls	LU04
	adds	fp, fp, r7
	sub	r9, r9, #1
	bcc	LU19
LU04:
	rsb	fp, r2, fp
	mov	r1, sl
	mov	r0, fp
	str	r3, [sp]
	bl	__aeabi_uidiv
	mov	r1, sl
	mov	r5, r0
	mov	r0, fp
	bl	__aeabi_uidivmod
	ldr	ip, [sp, #4]
	mul	r4, r5, r4
	lsl	r3, ip, #16
	lsr	r2, r3, #16
	ldr	r3, [sp]
	orr	r2, r2, r1, lsl #16
	cmp	r4, r2
	bls	LU05
	adds	r2, r2, r7
	sub	r5, r5, #1
	bcc	LU18
LU05:
	rsb	r4, r4, r2
	orr	r9, r5, r9, lsl #16
	lsr	r0, r9, #16
	lsr	lr, r3, #16
	bic	ip, r9, r0, lsl #16
	bic	r5, r3, lr, lsl #16
	mul	r2, ip, r5
	mul	r5, r0, r5
	mul	ip, lr, ip
	mul	lr, r0, lr
	adds	ip, r5, ip
	addcs	lr, lr, #0x10000
	adds	r2, r2, ip, lsl #16
	adc	lr, lr, ip, lsr #16
	cmp	r4, lr
	bcc	LU06
	movne	r0, #0
	moveq	r0, #1
	cmp	r2, r8, lsl r6
	movls	r1, #0
	andhi	r1, r0, #1
	cmp	r1, #0
	beq	LU14
LU06:
	sub	r9, r9, #1
	mov	r1, #0
	b	LU14
LU07:
	cmp	r2, #0
	bne	LU08
	mov	r1, r2
	mov	r0, #1
	bl	__aeabi_uidiv
	mov	r5, r0
LU08:
	clz	r9, r5
	cmp	r9, #0
	bne	LU15
	lsl	r1, r5, #16
	rsb	r4, r5, r4
	lsr	sl, r1, #16
	lsr	r7, r5, #16
	mov	r6, #1
LU09:
	mov	r1, r7
	mov	r0, r4
	bl	__aeabi_uidiv
	mov	r1, r7
	mov	r9, r0
	mov	r0, r4
	bl	__aeabi_uidivmod
	mul	r3, sl, r9
	lsr	r2, r8, #16
	orr	fp, r2, r1, lsl #16
	cmp	r3, fp
	bls	LU10
	adds	fp, fp, r5
	sub	r9, r9, #1
	bcs	LU10
	cmp	r3, fp
	subhi	r9, r9, #1
	addhi	fp, fp, r5
LU10:
	rsb	fp, r3, fp
	mov	r1, r7
	mov	r0, fp
	bl	__aeabi_uidiv
	mov	r1, r7
	mov	r4, r0
	mov	r0, fp
	bl	__aeabi_uidivmod
	mul	sl, r4, sl
	lsl	ip, r8, #16
	lsr	r8, ip, #16
	orr	r8, r8, r1, lsl #16
	cmp	sl, r8
	bls	LU11
	adds	r8, r8, r5
	sub	r4, r4, #1
	bcs	LU11
	cmp	sl, r8
	subhi	r4, r4, #1
LU11:
	orr	r9, r4, r9, lsl #16
	mov	r1, r6
	b	LU14
LU12:
	cmp	r3, r1
	cmpcs	r2, r0
	movls	r1, r6
	movls	r9, #1
	bls	LU14
LU13:
	mov	r1, #0
	mov	r9, r1
LU14:
	mov	r0, r9
	add	sp, sp, #12
	ldmfd	sp!, {r4, r5, r6, r7, r8, r9, sl, fp, pc}
LU15:
	rsb	r7, r9, #32
	lsr	lr, r8, r7
	lsl	r5, r5, r9
	lsr	r6, r4, r7
	orr	fp, lr, r4, lsl r9
	lsr	r7, r5, #16
	mov	r1, r7
	mov	r0, r6
	str	fp, [sp, #4]
	bl	__aeabi_uidiv
	mov	r1, r7
	lsl	sl, r5, #16
	lsr	sl, sl, #16
	mov	fp, r0
	mov	r0, r6
	bl	__aeabi_uidivmod
	ldr	ip, [sp, #4]
	mul	r3, sl, fp
	lsr	r6, ip, #16
	orr	r4, r6, r1, lsl #16
	cmp	r3, r4
	bls	LU16
	adds	r4, r4, r5
	sub	fp, fp, #1
	bcc	LU20
LU16:
	rsb	r4, r3, r4
	mov	r1, r7
	mov	r0, r4
	bl	__aeabi_uidiv
	mov	r1, r7
	mov	r6, r0
	mov	r0, r4
	bl	__aeabi_uidivmod
	ldr	r0, [sp, #4]
	mul	r2, sl, r6
	lsl	r4, r0, #16
	lsr	r3, r4, #16
	orr	r4, r3, r1, lsl #16
	cmp	r2, r4
	bls	LU17
	adds	r4, r4, r5
	sub	r6, r6, #1
	bcs	LU17
	cmp	r2, r4
	subhi	r6, r6, #1
	addhi	r4, r4, r5
LU17:
	lsl	r8, r8, r9
	rsb	r4, r2, r4
	orr	r6, r6, fp, lsl #16
	b	LU09
LU18:
	cmp	r4, r2
	addhi	r2, r2, r7
	subhi	r5, r5, #1
	b	LU05
LU19:
	cmp	r2, fp
	subhi	r9, r9, #1
	addhi	fp, fp, r7
	b	LU04
LU20:
	cmp	r3, r4
	subhi	fp, fp, #1
	addhi	r4, r4, r5
	b	LU16

ENDPROC(__udivdi3)

#endif

Ldiv0:

	str	lr, [sp, #-8]!
	bl	__div0
	mov	r0, #0			@ About as wrong as it could be.
	ldr	pc, [sp], #8


