/* clockevents
 *
 * Copyright (C) 2010 Red Hat, Inc. All Rights Reserved.
 * Written by Mark Salter (msalter@redhat.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public Licence
 * as published by the Free Software Foundation; either version
 * 2 of the Licence, or (at your option) any later version.
 */
#include <linux/clockchips.h>
#include <linux/interrupt.h>
#include <linux/percpu.h>
#include <linux/smp.h>
#include <linux/irq.h>
#include <asm/timex.h>

void clockevent_set_clock(struct clock_event_device *cd, unsigned int clock);

static int next_event(unsigned long delta,
		      struct clock_event_device *evt)
{
	stop_jiffies_counter();
	reload_jiffies_counter(delta - 1);
	return 0;
}

static void set_clock_mode(enum clock_event_mode mode,
				struct clock_event_device *evt)
{
	/* Nothing to do ...  */
}

struct clock_event_device uniphier_clockevent_device;
struct irqaction timer_irq;

static irqreturn_t timer_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *cd;

	stop_jiffies_counter();

	cd = &uniphier_clockevent_device;
	cd->event_handler(cd);

	return IRQ_HANDLED;
}

static void event_handler(struct clock_event_device *dev)
{
}

int __init init_clockevents(void)
{
	struct clock_event_device *cd;
	struct irqaction *iact;
	unsigned long flags;

	cd = &uniphier_clockevent_device;

	stop_jiffies_counter();
	cd->irq	= TMJCIRQ;

	cd->name		= "Timestamp";
	cd->features		= CLOCK_EVT_FEAT_ONESHOT;

	/* Calculate the min / max delta */
	clockevent_set_clock(cd, MN_JCCLK);

	cd->max_delta_ns	= clockevent_delta2ns(TMJCBR_MAX, cd);
	cd->min_delta_ns	= clockevent_delta2ns(100, cd);

	cd->rating		= 200;
	cd->cpumask		= cpumask_of(smp_processor_id());
	cd->set_mode		= set_clock_mode;
	cd->event_handler	= event_handler;
	cd->set_next_event	= next_event;

	iact = &timer_irq;
	iact->flags = IRQF_DISABLED | IRQF_SHARED | IRQF_TIMER;
	iact->handler = timer_interrupt;

	clockevents_register_device(cd);

#if defined(CONFIG_SMP) && !defined(CONFIG_GENERIC_CLOCKEVENTS_BROADCAST)
	/* setup timer irq affinity so it only runs on this cpu */
	{
		struct irq_desc *desc;
		unsigned int cpu = smp_processor_id();
		desc = irq_to_desc(cd->irq);
		cpumask_copy(desc->affinity, cpumask_of(cpu));
		iact->flags |= IRQF_NOBALANCING;
	}
#endif

	reload_jiffies_counter(MN_JC_PER_HZ - 1);
	iact->name = "CPU0 Timer";

	setup_irq(cd->irq, iact);

	local_irq_save(flags);
	get_irq_chip(cd->irq)->unmask(cd->irq);
	local_irq_restore(flags);

	return 0;
}
