/*
 * linux/arch/arm/mach-uniphier/include/mach/l2ca.c
 *
 * Copyright (C) 2011 Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/io.h>

#include <asm/cacheflush.h>
#include <asm/mach/map.h>
#include <mach/l2ca-regs.h>

#define L2_CACHE_ADRS_MASK	(~(L2_CACHE_BYTES - 1))

#define	L2PCMD_INV		L2COPA_CM_INV
#define	L2PCMD_WB		L2COPA_CM_WB
#define	L2PCMD_WBINV		L2COPA_CM_WBINV
#define	L2PCMD_L2CPREFETCH	L2COPA_CM_PF_L2C
#define	L2PCMD_DPREFETCH	L2COPA_CM_PF_PFBUF
#define	L2PCMD_IPREFETCH	(L2COPA_CM_PF_PFBUF | L2COPA_TID)
#define	L2PCMD_TOUCH		L2COPA_CM_TOUCH
#define	L2PCMD_TOUCH_ZERO	L2COPA_CM_TOUCH_ZERO
#define	L2PCMD_TOUCH_DIRTY	L2COPA_CM_TOUCH_DIRTY
#define	L2PCMD_SYNC		L2COPE_CM_SYNC
#define	L2PCMD_PBFLUSH		L2COPE_CM_FLSH_PFBUF

#define	L2QCMD_INV		L2COQM_CM_INV
#define	L2QCMD_WB		L2COQM_CM_WB
#define	L2QCMD_WBINV		L2COQM_CM_WBINV
#define	L2QCMD_L2CPREFETCH	L2COQM_CM_PF_L2C
#define	L2QCMD_TOUCH		L2COQM_CM_TOUCH
#define	L2QCMD_TOUCH_ZERO	L2COQM_CM_TOUCH_ZERO
#define	L2QCMD_TOUCH_DIRTY	L2COQM_CM_TOUCH_DIRTY

#define	MAXSZPURGE_L2C		0x00100000

static inline void
l2ca_writel(unsigned long val, unsigned long offset)
{
	outl(val, SSC_OFFSET_BASE + offset);
}

static inline unsigned long
l2ca_readl(unsigned long offset)
{
	return inl(SSC_OFFSET_BASE + offset);
}

#define	sync_opt_ex_bus()	\
		do { u32 tmp; tmp = *(volatile u32 *)UNIPHIER_KERNEL_UNCACHE_BASE; } while (0)

static void l2ca_sync(void);

#if  MAXSZPURGE_L2C >= L2COQSZ_SIZE
#error MAXSZPURGE_L2C must be smaller than L2COQSZ_SIZE
#endif

void _set_l2cache_qcmd(unsigned long start, unsigned long size, int cmd)
{
unsigned long	flags, l2coqm_val;
unsigned long	l2co_start, l2co_end, l2co_size, l2co_mask;

    l2coqm_val = L2COQM_CE | L2COQM_CW_FLAG | (cmd & (L2COQM_CM | L2COQM_S));

    l2co_start = start & L2_CACHE_ADRS_MASK;
    l2co_end   = (start + size + (L2_CACHE_BYTES - 1)) & L2_CACHE_ADRS_MASK;
    l2co_size  = l2co_end - l2co_start;
    l2co_mask  = ~((1 << (fls(l2co_size) - 1)) - 1);

    if (likely((cmd & L2COQM_S) == L2COQM_S_ADDRESS)) {
	if (!(l2co_start & ~l2co_mask) && !(l2co_end & ~l2co_mask)) {
	    l2co_size = 0; /* mask operation */
	}
	else {
	    if (l2co_size > MAXSZPURGE_L2C) {
		unsigned long l2co_tmp_size = ~l2co_mask + 1;
		if (!(l2co_start & ~l2co_mask)) {
		    _set_l2cache_qcmd(l2co_start, l2co_tmp_size, cmd);
		    l2co_start += l2co_tmp_size;
		    l2co_size -= l2co_tmp_size;
		    _set_l2cache_qcmd(l2co_start, l2co_size, cmd);
		}
		else if (!(l2co_end & ~l2co_mask)) {
		    _set_l2cache_qcmd(l2co_end - l2co_tmp_size, l2co_tmp_size, cmd);
		    l2co_size -= l2co_tmp_size;
		    _set_l2cache_qcmd(l2co_start, l2co_size, cmd);
		}
		else {
		    unsigned long l2co_tmp_end = l2co_end & l2co_mask;
		    _set_l2cache_qcmd(l2co_start , l2co_tmp_end - l2co_start, cmd);
		    _set_l2cache_qcmd(l2co_tmp_end , l2co_end - l2co_tmp_end, cmd);
		}
		return;
	    }
	}
    }

    preempt_disable();
#ifdef CONFIG_HYBRID
	hybrid_preempt_disable();
#endif /* CONFIG_HYBRID */
    sync_opt_ex_bus();
    __asm__ __volatile__("dsb":::"memory");
    local_save_flags(flags);

    do {
	local_irq_restore(flags);

	// Enter an local atomic section
#ifdef CONFIG_HYBRID
	hybrid_local_irq_disable();
#else /* !CONFIG_HYBRID */
	local_irq_disable();
#endif /* CONFIG_HYBRID */

	// Get a right to use L2-cache operation
	l2ca_writel(l2coqm_val, L2COQM_OFFSET);

	if (likely((cmd & L2COQM_S) == L2COQM_S_ADDRESS)) {
	    // Set address
	    l2ca_writel(l2co_start, L2COQAD_OFFSET);
	    if (l2co_size) {
		// Set size
		l2ca_writel(l2co_size, L2COQSZ_OFFSET);
	    }
	    else {
		// Set mask
		l2ca_writel(l2co_mask, L2COQMASK_OFFSET);
	    }
	}
    } while ((l2ca_readl(L2COPPQSEF_OFFSET) & (L2COPPQSEF_FE | L2COPPQSEF_OE)) != 0x00);

    local_irq_restore(flags);

    /* wait till all queue commands are completed */
    while ((l2ca_readl(L2COLPQS_OFFSET) & L2COLPQS_QST) == L2COLPQS_QST);
    l2ca_writel(L2COLPQS_EF, L2COLPQS_OFFSET);

    l2ca_sync();
    sync_opt_ex_bus();
#ifdef CONFIG_HYBRID
	hybrid_preempt_enable();
#endif /* CONFIG_HYBRID */
    preempt_enable();
}

static void
_set_l2cache_pcmd(unsigned long start, int cmd)
{
unsigned long	dummy, flags;
unsigned long	l2cope;

	l2cope = (unsigned long)__uniphier_io(SSC_OFFSET_BASE + L2COPE_OFFSET);

    __asm__ __volatile__("dsb":::"memory");
    if ((cmd == L2PCMD_SYNC)) {
	local_irq_save(flags);
	__asm__ __volatile__(
	"dmb\n"
	"str	%1, [%2]\n"
	"ldr	%0, [%2]\n"
	"dmb\n"
	: "=&r"(dummy)
	: "r"(cmd & L2COPE_CM), "r"(l2cope)
	: "memory"
	);
	local_irq_restore(flags);
    } else if (cmd == L2PCMD_PBFLUSH) {
	/* Set L2Cache primitive command */
	l2ca_writel((cmd & L2COPE_CM), L2COPE_OFFSET);
	/* Dummy read for synchronization */
	dummy	= l2ca_readl(L2COPE_OFFSET);

    } else {
	/* Set target address and L2Cache primitive command */
	l2ca_writel((start & L2COPA_ADDRESS) | (cmd & L2COPA_TID_CM), L2COPA_OFFSET);
	/* Dummy read for synchronization */
	dummy	= l2ca_readl(L2COPA_OFFSET);
    }
}

static unsigned long *ucwg_dummy_p;
unsigned long ucwg_dummy[L2_CACHE_BYTES/sizeof(unsigned long)] __attribute__((__aligned__(L2_CACHE_BYTES)));

static void
l2ca_sync(void) {
	unsigned long dummy;
	_set_l2cache_pcmd(0, L2PCMD_PBFLUSH);
	_set_l2cache_pcmd(0, L2PCMD_SYNC);
	__asm__ __volatile__(
	"	str	%2, [%1]	\n"
	"	ldr	%0, [%1]	\n"
	"	dsb			\n"
	: "=&r"(dummy)
	: "r"(ucwg_dummy_p), "r"(0)
	: "memory"
	);
}

static void
l2ca_clean_all(void)
{
#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
	 _set_l2cache_qcmd(0, 0, L2QCMD_WB | L2COQM_S_ALL);
#else /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
	l2ca_sync();
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
}

static void
l2ca_flush_all(void)
{
#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
	 _set_l2cache_qcmd(0, 0, L2QCMD_WBINV | L2COQM_S_ALL);
#else /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
	l2ca_sync();
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
}

static void
l2ca_inv_all(void)
{
#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
	_set_l2cache_qcmd(0, 0, L2QCMD_INV | L2COQM_S_ALL);
#else /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
	l2ca_sync();
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
}

static void
l2ca_clean_range(unsigned long start, unsigned long end)
{
	if (end > start) {
#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
		_set_l2cache_qcmd(start, end - start, L2QCMD_WB | L2COQM_S_ADDRESS);
#else /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
		l2ca_sync();
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
	}
}

static void
l2ca_flush_range(unsigned long start, unsigned long end)
{
	if (end > start) {
#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
		_set_l2cache_qcmd(start, end - start, L2QCMD_WBINV | L2COQM_S_ADDRESS);
#else /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
		l2ca_sync();
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
	}
}

static void
l2ca_inv_range(unsigned long start, unsigned long end)
{
	if (end > start) {
#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
		if (start & ~L2_CACHE_ADRS_MASK) {
			start = start & L2_CACHE_ADRS_MASK;
			_set_l2cache_qcmd(start, L2_CACHE_BYTES,
						L2PCMD_WBINV | L2COQM_S_ADDRESS);
			start += L2_CACHE_BYTES;
			if (end <= start) {
				l2ca_sync();
				return;
			}
		}
		if (end & ~L2_CACHE_ADRS_MASK) {
			end = end & L2_CACHE_ADRS_MASK;
			_set_l2cache_qcmd(end, L2_CACHE_BYTES,
						L2PCMD_WBINV | L2COQM_S_ADDRESS);
			if (end <= start) {
				l2ca_sync();
				return;
			}
		}
		_set_l2cache_qcmd(start, end - start, L2QCMD_INV | L2COQM_S_ADDRESS);
#else /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
		l2ca_sync();
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
	}
}

int __init l2ca_init(void)
{
#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
	unsigned long tmp;

	if(l2ca_readl(L2CC_OFFSET) & L2CC_ON) {
		goto __l2ca_enabled;
	}

	l2ca_inv_all();

	tmp = l2ca_readl(L2CC_OFFSET);
	tmp &= L2CC_MASK;;
	tmp |= L2CC_SETTING;
	l2ca_writel(tmp, L2CC_OFFSET);
	tmp = l2ca_readl(L2CC_OFFSET);	/* dummy read */

__l2ca_enabled:
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */
	ucwg_dummy_p
		= (unsigned long *)(UNIPHIER_KERNEL_UCWG_BASE +
		  (((unsigned long)&ucwg_dummy) & (UNIPHIER_KERNEL_UCWG_SIZE-1)));
	outer_cache.inv_all = l2ca_inv_all;
	outer_cache.clean_all = l2ca_clean_all;
	outer_cache.flush_all = l2ca_flush_all;
	outer_cache.sync = l2ca_sync;
	outer_cache.inv_range = l2ca_inv_range;
	outer_cache.clean_range = l2ca_clean_range;
	outer_cache.flush_range = l2ca_flush_range;

#ifdef CONFIG_UNIPHIER_L2CACHE_ENABLE
	printk(KERN_INFO "Level2 cache controller enabled\n");
#endif /* CONFIG_UNIPHIER_L2CACHE_ENABLE */

	return 0;
}
#ifdef CONFIG_UNIPHIER_HAS_L2CA
early_initcall(l2ca_init);
#endif /* CONFIG_UNIPHIER_HAS_L2CA */
