/*** CONFIDENTIAL ***/
/* Copyright (C) 2011, Panasonic Corporation */


#ifndef __FUSION__ENTRIES_H__
#define __FUSION__ENTRIES_H__

#include <linux/mutex.h>

#include "types.h"
#include "list.h"

typedef struct __FD_FusionEntry FusionEntry;

typedef const struct {
     int object_size;

     int  (*Init)   ( FusionEntry *entry, void *ctx, void *create_ctx );
     void (*Destroy)( FusionEntry *entry, void *ctx );
     int  (*Print)  ( FusionEntry *entry, void *ctx, char *buf );
} FusionEntryClass;

typedef struct {
     FusionEntryClass  *class;
     void              *ctx;

     FusionLink        *list;
     int                ids;
     struct semaphore   lock;
} FusionEntries;

struct __FD_FusionEntry {
     FusionLink         link;

     FusionEntries     *entries;

     int                id;
     pid_t              pid;

     pid_t              lock_pid;

     struct semaphore   lock;
     wait_queue_head_t  wait;
     int                waiters;

     struct timeval     last_lock;

     char               name[FUSION_ENTRY_INFO_NAME_LENGTH];
};

void fusion_entries_init  ( FusionEntries    *entries,
                            FusionEntryClass *class,
                            void             *ctx );

void fusion_entries_deinit( FusionEntries    *entries );

int  fusion_entries_read_proc( char *buf, char **start, off_t offset,
                               int len, int *eof, void *private );

int  fusion_entry_create        ( FusionEntries  *entries,
                                  int            *ret_id,
                                  void           *create_ctx );

int  fusion_entry_destroy       ( FusionEntries  *entries,
                                  int             id );

void fusion_entry_destroy_locked( FusionEntries  *entries,
                                  FusionEntry    *entry );

int  fusion_entry_set_info( FusionEntries          *entries,
                            const FusionEntryInfo  *info );

int  fusion_entry_get_info( FusionEntries          *entries,
                            FusionEntryInfo        *info );

int  fusion_entry_lock    ( FusionEntries    *entries,
                            int               id,
                            bool              keep_entries_lock,
                            FusionEntry     **ret_entry );

void fusion_entry_unlock  ( FusionEntry      *entry );

int  fusion_entry_wait    ( FusionEntry      *entry,
                            long             *timeout );

void fusion_entry_notify  ( FusionEntry      *entry,
                            bool              all );

#define FUSION_ENTRY_CLASS( Type, name, init_func, destroy_func, print_func )   \
                                                                                \
     static FusionEntryClass name##_class = {                                   \
          .object_size = sizeof(Type),                                          \
          .Init        = init_func,                                             \
          .Destroy     = destroy_func,                                          \
          .Print       = print_func                                             \
     };                                                                         \
                                                                                \
     static inline int fusion_##name##_lock( FusionEntries  *entries,           \
                                             int             id,                \
                                             bool            keep,              \
                                             Type          **ret_##name )       \
     {                                                                          \
          int          ret;                                                     \
          FusionEntry *entry;                                                   \
                                                                                \
          ret = fusion_entry_lock( entries, id, keep, &entry );                 \
                                                                                \
          if (!ret)                                                             \
               *ret_##name = (Type *) entry;                                    \
                                                                                \
          return ret;                                                           \
     }                                                                          \
                                                                                \
     static inline void fusion_##name##_unlock( Type *name )                    \
     {                                                                          \
          fusion_entry_unlock( (FusionEntry*) name );                           \
     }                                                                          \
                                                                                \
     static inline int fusion_##name##_wait( Type *name, long *timeout )        \
     {                                                                          \
          return fusion_entry_wait( (FusionEntry*) name, timeout );             \
     }                                                                          \
                                                                                \
     static inline void fusion_##name##_notify( Type *name, bool all )          \
     {                                                                          \
          fusion_entry_notify( (FusionEntry*) name, all );                      \
     }

#endif
