/*
 *  Pseudo keyboard driver
 *
 *  Copyright (c) 2011 SHARP Corporation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/input.h>

static unsigned short pseudo_kbd_keycode[] = {
	KEY_RESERVED,
	KEY_1,
	KEY_2,
	KEY_3,
	KEY_4,
	KEY_5,
	KEY_6,
	KEY_7,
	KEY_8,
	KEY_9,
	KEY_0,

	KEY_UP,
	KEY_DOWN,
	KEY_LEFT,
	KEY_RIGHT,

	KEY_ENTER,
	KEY_BACK,
	KEY_HOME,
	KEY_END,

	KEY_PAGEUP,
	KEY_PAGEDOWN,

	KEY_MUTE,
	KEY_VOLUMEUP,
	KEY_VOLUMEDOWN,

	KEY_CHANNELUP,
	KEY_CHANNELDOWN,

	KEY_RED,
	KEY_GREEN,
	KEY_YELLOW,
	KEY_BLUE,

	KEY_PLAY,
	KEY_RECORD,
	KEY_STOP,
	KEY_PLAYPAUSE,
	KEY_FASTFORWARD,
	KEY_REWIND,

	KEY_FORWARD,
	KEY_NEXT,
	KEY_PREVIOUS,

	KEY_LAST,
	KEY_MENU,
	KEY_INFO,
	KEY_SOUND,
	KEY_EXIT,
	KEY_AUDIO,
	KEY_SUBTITLE,
	KEY_SETUP,
	KEY_HELP,

	KEY_PAUSE,

	KEY_LEFTALT,
	KEY_RIGHTALT,
	KEY_LEFTSHIFT,
	KEY_RIGHTSHIFT,
	KEY_CAPSLOCK,
	KEY_LEFTCTRL,
	KEY_RIGHTCTRL,
	KEY_NUMLOCK,
	KEY_SCROLLLOCK,
	KEY_ZENKAKUHANKAKU,
	KEY_KATAKANAHIRAGANA,
	KEY_INSERT,
	KEY_DELETE,
	KEY_BACKSPACE,
	KEY_ESC,
	KEY_SPACE,
	KEY_TAB,
	KEY_POWER,

	KEY_F1,
	KEY_F2,
	KEY_F3,
	KEY_F4,
	KEY_F5,
	KEY_F6,
	KEY_F7,
	KEY_F8,
	KEY_F9,
	KEY_F10,
	KEY_F11,
	KEY_F12,
	KEY_F13,
	KEY_F14,
	KEY_F15,
	KEY_F16,
	KEY_F17,
	KEY_F18,
	KEY_F19,
	KEY_F20,
	KEY_F21,
	KEY_F22,
	KEY_F23,
	KEY_F24,

	KEY_PROG1,
	KEY_PROG2,
	KEY_PROG3,
	KEY_PROG4,

	KEY_A,
	KEY_B,
	KEY_C,
	KEY_D,
	KEY_E,
	KEY_F,
	KEY_G,
	KEY_H,
	KEY_I,
	KEY_J,
	KEY_K,
	KEY_L,
	KEY_M,
	KEY_N,
	KEY_O,
	KEY_P,
	KEY_Q,
	KEY_R,
	KEY_S,
	KEY_T,
	KEY_U,
	KEY_V,
	KEY_W,
	KEY_X,
	KEY_Y,
	KEY_Z,

	KEY_LEFTBRACE,
	KEY_RIGHTBRACE,
	KEY_SEMICOLON,
	KEY_APOSTROPHE,
	KEY_GRAVE,
	KEY_BACKSLASH,
	KEY_COMMA,
	KEY_DOT,
	KEY_SLASH,
	KEY_MINUS,
	KEY_EQUAL,
	KEY_SEMICOLON,
};

static struct input_dev *pseudo_dev = NULL;

static int __init pseudo_kbd_init(void)
{
	int i, error;

	pseudo_dev = input_allocate_device();

	if (!pseudo_dev) {
		error = -ENOMEM;
		return error;
	}

    pseudo_dev->name = "Pseudo keyboard";
    pseudo_dev->phys = "pseudo-kbd/input0";
    pseudo_dev->id.bustype = BUS_HOST;
    pseudo_dev->id.vendor = 0x0001;
    pseudo_dev->id.product = 0x0001;
    pseudo_dev->id.version = 0x0100;

    pseudo_dev->evbit[0] = BIT_MASK(EV_KEY) | BIT_MASK(EV_LED);
    pseudo_dev->ledbit[0] = BIT_MASK(LED_NUML) | BIT_MASK(LED_CAPSL) | BIT_MASK(LED_SCROLLL) | BIT_MASK(LED_COMPOSE) | BIT_MASK(LED_KANA);

	for (i = 0; i < sizeof(pseudo_kbd_keycode)/sizeof(pseudo_kbd_keycode[0]); i++) {
		set_bit(pseudo_kbd_keycode[i], pseudo_dev->keybit);
	}

#ifdef CONFIG_KEYBOARD_PSEUDO_WITH_MOUSE
	set_bit(EV_REL, pseudo_dev->evbit);
	set_bit(BTN_LEFT, pseudo_dev->keybit);
	set_bit(BTN_RIGHT, pseudo_dev->keybit);
	set_bit(BTN_MIDDLE, pseudo_dev->keybit);
	set_bit(BTN_SIDE, pseudo_dev->keybit);
	set_bit(BTN_EXTRA, pseudo_dev->keybit);
	set_bit(REL_X, pseudo_dev->relbit);
	set_bit(REL_Y, pseudo_dev->relbit);
	set_bit(REL_WHEEL, pseudo_dev->relbit);
#endif

	clear_bit(0, pseudo_dev->keybit);

	error = input_register_device(pseudo_dev);

    if (error) {
        pr_err("pseudo-kbd: Unable to register input device, "
			   "error: %d\n", error);
		input_free_device(pseudo_dev);
		return error;
    }

	return 0;
}

module_init(pseudo_kbd_init);

static void __exit pseudo_kbd_exit(void)
{
	input_unregister_device(pseudo_dev);
	pseudo_dev = NULL;
}
module_exit(pseudo_kbd_exit);

MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Tomohiko Ozeki");
MODULE_DESCRIPTION("Pseudo keyboard driver");
