/* ===================================================================== *
 * Copyright    : @(#)Copyright(C) 2009
 *              :        Panasonic Corporation.
 *              :                                    All Right reserved.
 * FileName     : eeprom.c
 * System       : EEPROM to NAND Access Driver 
 * Date         : 2009/12/09
 * Comment      : EEPROM ؿ
 * ===================================================================== */

/* -------------------------------------------------------------------------- *
 *  ͥ
 * -------------------------------------------------------------------------- */
#ifndef __KERNEL__
#define	__KERNEL__
#endif

/* ===================================================================== *
 *  include
 * ===================================================================== */
#include <linux/module.h>
#include <linux/init.h>

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/pagemap.h>
#include <linux/poll.h>

#include <asm/system.h>
#include <asm/hardirq.h>
#include <asm/delay.h>
#include <asm/mmu_context.h>
#include <asm/pgtable.h>
#include <asm/eeprom.h>

/* -------------------------------------------------------------------------- *
 *  ǥХֹ
 * -------------------------------------------------------------------------- */
static unsigned int eeprom_major = EEPROM_MAJOR;
static unsigned int eeprom_nr_devs = EEPROM_NR_DEVS;

module_param (eeprom_major, int, S_IRUGO);
module_param (eeprom_nr_devs, int, S_IRUGO);

/* ========================================================================== *
 *  ޥ
 * ========================================================================== */
#if defined( CONFIG_MEI_DTVREC )
#define EEPROM_MALLOC( count )	vmalloc( count )
#define EEPROM_FREE( addr )		vfree( addr )
#else
#define EEPROM_MALLOC( count )	kmalloc( count, GFP_KERNEL )
#define EEPROM_FREE( addr )		kfree( addr )
#endif

/* ========================================================================== *
 *  ץȥ
 * ========================================================================== */
/* -------------------------------------------------------------------------- *
 *  եؿ
 * -------------------------------------------------------------------------- */
static signed int __init eeprom_init(void);
static void       __exit eeprom_cleanup(void);
static signed int eeprom_open(struct inode *inode, struct file *filp);
static signed int eeprom_release(struct inode *inode, struct file *filp);
#ifdef CONFIG_DTV_USE_UNLOCKED_IOCTL
static signed long eeprom_ioctl (struct file *filp, unsigned int cmd, unsigned long arg);
#else
static signed int eeprom_ioctl (struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg);
#endif
static loff_t     eeprom_lseek(struct file *filp, loff_t off, int whence);
static ssize_t    eeprom_read(struct file *filp, char *buf, size_t count, loff_t *f_pos);
static ssize_t    eeprom_write(struct file *filp, const char *buf, size_t count, loff_t *f_pos);

/* ========================================================================== *
 *  Хѿ
 * ========================================================================== */
/* -------------------------------------------------------------------------- *
 *  եؿ
 * -------------------------------------------------------------------------- */
struct file_operations eeprom_fops =
{
	llseek:		eeprom_lseek,
#ifdef CONFIG_DTV_USE_UNLOCKED_IOCTL
	unlocked_ioctl:	eeprom_ioctl,
#else
	ioctl:		eeprom_ioctl,
#endif
	read:		eeprom_read,
	write:		eeprom_write,
	open:		eeprom_open,
	release:	eeprom_release,
	owner:		THIS_MODULE,
};

/* -------------------------------------------------------------------------- *
 *  ޥե
 * -------------------------------------------------------------------------- */
static eeprom_dev_t *eeprom_devices;

/* ===================================================================== *
 *  ؿ
 * ===================================================================== */
/* ========================================================================== *
 * ModuleName   : eeprom_init
 * Function     : eepromɥ饤Фν
 * Args         : ʤ
 * Return       : 0                          : ｪλ
 *              : -ENOMEM                    : Out of memory
 *              : ¾                     : ɥ饤Ͽ
 * Comment      : - ʲεǽ¸
 *              :   1) ͥ˥饯ǥХȤƥ⥸塼Ͽ
 *              :   2) ޥեΰγݤԤ
 *              :   3) ޥեνԤ
 * ========================================================================== */
static signed int __init
eeprom_init(void)
{
	signed int ret;
	unsigned int i;
	
	/* ɥ饤Ͽ */
	ret = register_chrdev(eeprom_major, "eeprom", &eeprom_fops);
	if(ret < 0){
		printk (KERN_WARNING "eeprom: can't register major %d\n", eeprom_major);
		return ret;
	}
	
	/* ޥեΰγ */
	eeprom_devices = EEPROM_MALLOC(eeprom_nr_devs * sizeof(eeprom_dev_t));
	if(!eeprom_devices){
		ret = -ENOMEM;
		unregister_chrdev(eeprom_major, "eeprom");
		return ret;
	}
	
	memset(eeprom_devices, 0, eeprom_nr_devs * sizeof(eeprom_dev_t));
	
	/* ޥեν */
	for(i = 0; i < eeprom_nr_devs; i++){
		sema_init(&eeprom_devices[i].sem, 1);
	}
	
	return 0;
}

/* ========================================================================== *
 * ModuleName   : eeprom_open
 * Function     : eepromɥ饤ФΥץ
 * Args         : struct inode *inode       : inodeݥ[]
 *              : struct file  *filp        : file¤Υݥ[]
 * Return       : 0                          : ｪλ
 *              : -ENOMEM                    : Out of memory
 *              : ¾                     : MTDɥ饤ФΥ顼
 * Comment      : - ʲεǽ¸
 *              :   1) ǥХ
 *              :   2) mtdblock11򥪡ץ󤷡fd
 *              :   3) file¤Τ˥ǥХ򥳥ԡ
 * ========================================================================== */
static signed int
eeprom_open(struct inode *inode, struct file *filp)
{
	eep_mydev_t *mydev;
	signed int minor = MINOR (inode->i_rdev);
	signed int ret;
	char mtdpath[EEPROM_PATH_LEN];
	
	/* ǥХ¤Τν */
	mydev = EEPROM_MALLOC(sizeof(eep_mydev_t));
	if(!mydev)
		return -ENOMEM;
	
	memset(mydev, 0, sizeof (*mydev));
	
	/* ǥХμ */
	mydev->dev = &eeprom_devices[minor];
	mydev->dev->minor = minor;
	
	/* ǽϾmtdblock11open */
#if defined(CONFIG_MTD_NAND_MAP_V1)
	sprintf(mtdpath, "%s%d", EEPROM_BLOCK_PATH, EEPROM_SELECT_MTD11);
#elif defined(CONFIG_MTD_NAND_MAP_V2)
	sprintf(mtdpath, "%s%d", EEPROM_BLOCK_PATH, EEPROM_SELECT_DEFAULT_MTD);
#else
# error Unknown NAND MAP version !!
#endif
	
	/* mtdopenƿեǥץ */
	down(&mydev->dev->sem);
	mydev->myfilp = filp_open(mtdpath, O_RDWR, 0);
	if(IS_ERR(mydev->myfilp)){
		ret = PTR_ERR(mydev->myfilp);
		EEPROM_FREE(mydev);
		up(&mydev->dev->sem);
		return ret;
	}
	up(&mydev->dev->sem);
	
	/* ǥХfile¤Τ˥ԡ */
	filp->private_data = mydev;
	
	return 0;
}

/* ========================================================================== *
 * ModuleName   : eeprom_release
 * Function     : eepromɥ饤ФΥ
 * Args         : struct inode *inode       : inodeݥ[]
 *              : struct file  *filp        : file¤Υݥ[]
 * Return       : 0                          : ｪλ
 *              : ¾                     : MTDɥ饤ФΥ顼
 * Comment      : - ʲεǽ¸
 *              :   1) file¤Τեǥץ
 *              :   2) եǥץ򥯥
 * ========================================================================== */
static signed int
eeprom_release(struct inode *inode, struct file *filp)
{
	eep_mydev_t *mydev;
	signed int ret;
	
	/* file¤Τեǥץ */
	mydev = filp->private_data;
	
	/* եǥץclose */
	down(&mydev->dev->sem);
	ret = filp_close(mydev->myfilp, 0);
	if(ret < 0){
		printk("filp_close failed.\n");
		up(&mydev->dev->sem);
		return ret;
	}
	up(&mydev->dev->sem);
	
	/* ǥХ */
	EEPROM_FREE(mydev);
	
	return 0;
}

/* ========================================================================== *
 * ModuleName   : eeprom_ioctl
 * Function     : MTD Blockڤؤ
 * Args         : struct inode *inode       : inodeݥ[]
 *              : struct file  *filp        : file¤Υݥ[]
 *              : unsigned int cmd          : ޥ[]
 *              : unsigned long arg         : []
 * Return       : 0                          : ｪλ
 *              : -ENOMEM                    : Out of memory
 *              : -EFAULT                    : Bad Address
 *              : ¾                     : MTDɥ饤ФΥ顼
 * Comment      : - ʲεǽ¸
 *              : [EEPROM_CMD_SELECT]
 *              :   1) 桼ΰ褫
 *              :   2) ߥץ󤷤ƤǥХclose
 *              :   3) ꤵ줿MTDǥХ򥪡ץ󤷡եǥץ
 * ========================================================================== */
#ifdef CONFIG_DTV_USE_UNLOCKED_IOCTL
static signed long
eeprom_ioctl (struct file *filp, unsigned int cmd, unsigned long arg)
#else
static signed int
eeprom_ioctl (struct inode *inode, struct file *filp, 
			unsigned int cmd, unsigned long arg)
#endif
{
	eeprom_dev_t *dev;
	eep_mydev_t *mydev;
	signed int ret;
	
	mydev = filp->private_data;
	dev = mydev->dev;
	
	switch (cmd){
		case EEPROM_CMD_SELECT: /* Ѥmtdѹ */
			{
				eep_select sel;
				unsigned char mtdpath[EEPROM_PATH_LEN];
				
				/* 桼ΰ褫饫ͥΰ˰򥳥ԡ */
				memset(&sel, 0, sizeof(eep_select));
				if( copy_from_user((void*)&sel, (void *)arg, sizeof(eep_select)) != 0 ){
					printk("copy_from_user failed.\n");
					return -EFAULT;
				}
				
				down(&dev->sem);
				/* openƤեǥץclose */
				ret = filp_close(mydev->myfilp, 0);
				if(ret < 0){
					printk("filp_close failed.\n");
					up(&dev->sem);
					return ret;
				}
				
				/* select_mtdmtdpath */
				sprintf(mtdpath, "%s%d", EEPROM_BLOCK_PATH, sel.select_mtd);
				
				/* ꤵ줿mtdopenƥեǥץ */
				mydev->myfilp = filp_open(mtdpath, O_RDWR, 0);
				if(IS_ERR(mydev->myfilp)){
					ret = PTR_ERR(mydev->myfilp);
					up(&dev->sem);
					return ret;
				}
				up(&dev->sem);
			}
			break;
		default :
			printk("ioctl cmd error.\n");
			return -EINVAL;
	}
	
	return 0;
}

/* ========================================================================== *
 * ModuleName   : eeprom_lseek
 * Function     : ɤ߽񤭥եåȤΰ֤ѹ
 * Args         : struct file *filp       : file¤ΤΥݥ[]
 *              : loff_t off              : եå[]
 *              : int whence              : ޥ[]
 * Return       : 0                          : ｪλ
 *              : ¾                     : MTDɥ饤ФΥ顼
 * Comment      : - ʲεǽ¸
 *              :   1) file¤Τեǥץ
 *              :   2) read/write륪եåȤΰ֤ѹ
 *              :   3) file¤Τѹ줿եåȰ֤Ǽ
 * ========================================================================== */
static loff_t
eeprom_lseek(struct file *filp, loff_t off, int whence)
{
	loff_t newpos;
	eep_mydev_t *mydev;
	
	/* file¤Τեǥץ */
	mydev = filp->private_data;
	
	/* read/write륪եåȤΰ֤ѹ */
	down(&mydev->dev->sem);
	newpos = mydev->myfilp->f_op->llseek( mydev->myfilp, off, whence );
	filp->f_pos = newpos;
	up(&mydev->dev->sem);
	
	return newpos;
}

/* ========================================================================== *
 * ModuleName   : eeprom_read
 * Function     : ǡɤ߹
 * Args         : struct file *filp       : file¤ΤΥݥ[]
 *              : const char *buf         : ɤ߹ǡ[]
 *              : size_t count            : []
 *              : loff_t *f_pos           : եå[]
 * Return       : 0                          : ｪλ
 *              : ¾                     : MTDɥ饤ФΥ顼
 * Comment      : - ʲεǽ¸
 *              :   1) file¤Τեǥץ
 *              :   2) readޥɤǥǡɤ߹
 * ========================================================================== */
static ssize_t
eeprom_read(struct file *filp, char *buf, size_t count, loff_t *f_pos)
{
	ssize_t byte;
	eep_mydev_t *mydev;
	mm_segment_t oldfs;
	
	/* file¤Τեǥץ */
	mydev = filp->private_data;
	
	/* Ͽ줿readڥ졼ǥǡɤ߹ */
	down(&mydev->dev->sem);
	oldfs = get_fs();
	set_fs(KERNEL_DS);
	mydev->myfilp->f_pos = *f_pos;
	byte = mydev->myfilp->f_op->read(mydev->myfilp, buf, count, &mydev->myfilp->f_pos);
	if(byte < 0){
		printk("read error.\n");
	}else{
		*f_pos += byte;
	}
	set_fs(oldfs);
	up(&mydev->dev->sem);
	
	return byte;
}

/* ========================================================================== *
 * ModuleName   : eeprom_write
 * Function     : ǡν񤭹
 * Args         : struct file *filp       : file¤ΤΥݥ[]
 *              : const char *buf         : 񤭹ǡ[]
 *              : size_t count            : []
 *              : loff_t *f_pos           : եå[]
 * Return       : 0                          : ｪλ
 *              : ¾                     : MTDɥ饤ФΥ顼
 * Comment      : - ʲεǽ¸
 *              :   1) file¤Τեǥץ
 *              :   2) writeޥɤǥǡ񤭹
 * ========================================================================== */
static ssize_t
eeprom_write(struct file *filp, const char *buf, size_t count, loff_t *f_pos)
{
	ssize_t byte;
	eep_mydev_t *mydev;
	mm_segment_t oldfs;
	
	/* file¤Τեǥץ */
	mydev = filp->private_data;
	
	/* Ͽ줿writeڥ졼ǥǡ񤭹 */
	down(&mydev->dev->sem);
	mydev->myfilp->f_pos = *f_pos;
	oldfs = get_fs();
	set_fs(KERNEL_DS);
	byte = mydev->myfilp->f_op->write(mydev->myfilp, buf, count, &mydev->myfilp->f_pos);
	if(byte < 0){
		printk("write error.\n");
	}else{
		*f_pos += byte;
	}
	set_fs(oldfs);
	up(&mydev->dev->sem);
	
	return byte;
}

/* ========================================================================== *
 * ModuleName   : eeprom_cleanup
 * Function     : eepromɥ饤Фβ
 * Args         : ʤ
 * Return       : ʤ
 * Comment      : - ʲεǽ¸
 *              :   1) ⥸塼Ͽ
 * ========================================================================== */
static void __exit 
eeprom_cleanup(void)
{
	EEPROM_FREE(eeprom_devices);
	/* ɥ饤в */
	unregister_chrdev(eeprom_major, "eeprom");
}

module_init(eeprom_init);
module_exit(eeprom_cleanup);

/* ===================================================================== *
 * Unpublished Copyright(C) 2009 by Panasonic Corporation.
 * ===================================================================== */
