/******************************************************************************
 * $Id: logdev.c,v 1.4 2012/02/18 06:16:23 txbsd Exp $
 *
 * FileName     : log_dev.c
 *
 * Description  : Message logging driver for debug and timestamp
 *
 * Copyright    : Panasonic Corporation
 *
 *****************************************************************************/
#undef LOGDEV_VERBOSE

#define __NO_VERSION__ /* Must be first in all but one driver source file! */

#include <linux/autoconf.h>
#include <linux/module.h>  /* Definitions needed for kernel modules */
#include <linux/kernel.h>  /* We run in the kernel so we need this */
#include <linux/time.h>
//#include <linux/init.h>
#include <linux/fs.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <asm/uaccess.h>

#include <linux/iosc/iosc-devices.h>

#define LOG_BUF_SIZE (1*1024*1024)

#define ERR(fmt, args...) printk(KERN_ERR "[logdev] " fmt "\n", ##args)
#ifdef LOGDEV_VERBOSE
# define MSG(fmt, args...) printk(KERN_INFO "{logdev} " fmt "\n", ##args)
#else
# define MSG(...) /*empty*/
#endif

static struct semaphore logdev_sem = __SEMAPHORE_INITIALIZER(logdev_sem, 1);
static char *log_buf_head, *log_buf_tail = NULL;
static char *write_ptr = NULL;

/* local function */
static int logdev_init( void );
static void logdev_exit( void );
static ssize_t logdev_write(struct file *file, const char __user *buf, size_t count, loff_t *offset);
static ssize_t logdev_read(struct file *file, char __user *buf, size_t count, loff_t *offset);

struct logdev_priv_struct {
  char *last_read_ptr;
  struct timeval prev;
};

static int logdev_init_buf(void)
{
  if (!log_buf_head) {          /* first access, do initialize */
    log_buf_head = vmalloc(LOG_BUF_SIZE);
    if (!log_buf_head) {
      ERR("cannot allocate memory for logging buffer");
      return -ENOMEM;
    }
    log_buf_tail = log_buf_head + LOG_BUF_SIZE - 1;
    write_ptr = log_buf_head;
  }

  return 0;
}


/*****************************************************************
 * Method functions for Linux-side device
 *****************************************************************/

/*
 * [Method] open
 * RETURN: length if success to log
 *         under 0 if error occured
 */
static int
logdev_open(struct inode *inode, struct file *file)
{
  int ret;
  struct logdev_priv_struct *priv;

  down(&logdev_sem);

  ret = logdev_init_buf();
  if (ret)
    return ret;

  priv = kmalloc(sizeof(struct logdev_priv_struct), GFP_KERNEL);
  if (!priv) {
      ERR("cannot allocate memory for private data");
      ret = -ENOMEM;
      goto EXIT;
  }
  priv->last_read_ptr = log_buf_head;
  priv->prev.tv_sec = priv->prev.tv_usec = 0;
  file->private_data = (void *)priv;
  ret = 0;
 EXIT:
  up(&logdev_sem);
  return ret;

}

/*
 * [Method] close
 * RETURN: length if success to log
 *         under 0 if error occured
 */
static int
logdev_release(struct inode *inode, struct file *file)
{
  struct logdev_priv_struct *priv = (struct logdev_priv_struct *)(file->private_data);

  if (priv) {
    kfree(priv);
  }

  return 0;
}


/*
 * [Method] write
 * RETURN: length if success to log
 *         under 0 if error occured
 */
static ssize_t
logdev_write(struct file *file, const char __user *buf, size_t count, loff_t *offset)
{
  int ret;
  struct timeval tv;
  size_t *count_ptr;

  down(&logdev_sem);

  if (write_ptr + count + sizeof(tv) + sizeof(size_t) + 1 >= log_buf_tail) {
    ret = -EOVERFLOW;
    goto EXIT;
  }

  do_gettimeofday(&tv);
  *(time_t *)write_ptr = tv.tv_sec;
  write_ptr += sizeof(time_t);
  *(suseconds_t *)write_ptr = tv.tv_usec;
  write_ptr += sizeof(suseconds_t);
  count_ptr = (size_t *)write_ptr;
  *count_ptr = count;
  write_ptr += sizeof(size_t);

  if (offset==NULL) {             /* called from kernel space */
    memcpy(write_ptr, buf, count);
  } else {
    if (copy_from_user(write_ptr, buf, count)) {
      ret = -EFAULT;
      goto EXIT;
    }
  }
  write_ptr += count;
  if (*(write_ptr-1) != '\n') {
    *write_ptr++ = '\n';
    (*count_ptr)++;
  }

  while (((unsigned int)write_ptr & 0x3)!=0) {
    *write_ptr++ = '\n';        /* padding for int-size alignment */
  }

  ret = count;
 EXIT:
  up(&logdev_sem);
  return ret;
}

/*
 * [Method] read
 * RETURN: length from one entry of log
 *         under 0 if error occured
 */

static ssize_t
logdev_read(struct file *file, char __user *buf, size_t count, loff_t *offset)
{
  int ret;
  size_t log_len, ts_len;
  ssize_t total_len;
  struct timeval tv;
  int diff_sec, diff_usec;
  char clock_buf[100];
  struct logdev_priv_struct *priv = (struct logdev_priv_struct *)(file->private_data);
  char *read_ptr = priv->last_read_ptr;

  if (read_ptr >= write_ptr) {
    ret = 0;
    goto EXIT;
  }

  tv.tv_sec = *(time_t *)read_ptr;
  read_ptr += sizeof(time_t);
  tv.tv_usec = *(suseconds_t *)read_ptr;
  read_ptr += sizeof(suseconds_t);
  log_len = *(size_t *)read_ptr;
  read_ptr += sizeof(size_t);

  if (priv->prev.tv_sec == 0 && priv->prev.tv_usec== 0) {
    diff_sec = diff_usec = 0;
  } else {
    diff_sec = tv.tv_sec - priv->prev.tv_sec;
    diff_usec = tv.tv_usec - priv->prev.tv_usec;
    if (diff_usec < 0) {
      diff_sec--;
      diff_usec += 1000*1000;
    }
  }
  priv->prev.tv_sec = tv.tv_sec;
  priv->prev.tv_usec = tv.tv_usec;
  ts_len = snprintf(clock_buf, sizeof(clock_buf)-1, "%d.%06d\t%d.%06d\t",
                    (int)tv.tv_sec, (int)tv.tv_usec, diff_sec, diff_usec);

  total_len = ts_len + log_len;
  if (total_len >= count) {
    ERR("too long message: given is %d, but message length is %d",
        count, total_len);
    ret = -EINVAL;
    goto EXIT;
  }

  if (copy_to_user(buf, clock_buf, ts_len)) {
    ERR("failed to copy time-stamp part");
    ret = -EFAULT;
    goto EXIT;
  }
  if (copy_to_user(buf+ts_len, read_ptr, log_len)) {
    ERR("failed to copy log-message part");
    ret = -EFAULT;
    goto EXIT;
  }
  buf[ts_len+log_len] = 0;

  read_ptr += log_len;
  while (((unsigned int)read_ptr & 0x3)!=0) {
    read_ptr++;
  }

  priv->last_read_ptr = read_ptr;

  ret = total_len;

 EXIT:
  return ret;
}

/* file operation structure */
static struct file_operations logdev_fops = {
  open:    logdev_open,
  release: logdev_release,
  write:   logdev_write,
  read:    logdev_read,
};

/*
 * static int logdev_init( void )
 *-----------------------------------------------------------------------------
 * function: initialize module
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
static int logdev_init( void )
{
  int result;

  result = register_chrdev(LOG_DEV_MAJOR, "logdev", &logdev_fops);
  if (result < 0) {
    ERR("failed to register device\n");
    goto ERROR_REGDEV;
  }

  return 0;

 ERROR_REGDEV:
  return -1;
}

/*
 * static void logdev_exit( void )
 *-----------------------------------------------------------------------------
 * function: exit module
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
static void logdev_exit( void )
{
  unregister_chrdev(LOG_DEV_MAJOR, "logdev");
}

module_init(logdev_init);
module_exit(logdev_exit);

#include <stdarg.h>

/*
 * global library
 *-----------------------------------------------------------------------------
 * function: 
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
void logdev(const char *fmt, ...)
{
  char buf[256];
  int size;
  va_list args;

  if (logdev_init_buf())
    return;

  va_start(args, fmt);
  size = vsnprintf(buf, sizeof(buf)-1, fmt, args);
  logdev_write(NULL, buf, size, NULL);

  va_end(args);
}

EXPORT_SYMBOL(logdev);
