#include <linux/autoconf.h>

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/pagemap.h>
#include <linux/mutex.h>
#include <asm/io.h>

#include <asm/vbiout.h>

#ifndef __KERN__
#define __KERN__
#endif  /* __KERN__ */

#ifndef VBIOUT_MAJOR
#define VBIOUT_MAJOR 210
#endif

/*
#define VBI_DEBUG
*/

#ifdef VBI_DEBUG
#define vbi_debug( fmt, args... )      printk( fmt, ##args );
#else
#define vbi_debug( ... )
#endif

typedef struct _vbioutstat_t {
    unsigned int txtbuf;
    unsigned char txtcount;
    struct file *filp;
} VBIDDSTAT_t;

VBIDDSTAT_t  vbioutstat;

/***********************************************************************/
int vbiout_major = VBIOUT_MAJOR;

static unsigned int
VBIOUT_read_reg (unsigned int addr)
{
    return inl(addr);
}

static int
VBIOUT_write_reg (unsigned int addr, unsigned int value)
{
    outl(value, addr);
    return 0;
}

static int
vbiout_set_txtenb (unsigned int value)
{
    vbi_debug(KERN_INFO "[vbiout] TXTENB %x\n",value);
    VBIOUT_write_reg(VBIREG_TXTENB, value&0x3FFFF);
    return 0;
}

static int
vbiout_set_ttsdramrp (unsigned int value)
{
    vbi_debug(KERN_INFO "[vbiout] TTSDRAMRP %x\n",value);
    VBIOUT_write_reg(VBIREG_TTSDRAMRP, value&0xFFFFFFF);
    return 0;
}

static int
vbiout_set_cc21data (unsigned char *value)
{
    vbi_debug(KERN_INFO "[vbiout] CC21 %x %x\n",value[0],value[1]);
    VBIOUT_write_reg(VBIREG_CC21DATA, ((value[1]<<8)|value[0])&0xFFFF);
    return 0;
}

static int
vbiout_set_cc284data (unsigned char *value)
{
    vbi_debug(KERN_INFO "[vbiout] CC284 %x %x\n",value[0],value[1]);
    VBIOUT_write_reg(VBIREG_CC284DATA, ((value[1]<<8)|value[0])&0xFFFF);
    return 0;
}

static char
vbiout_get_fieldflag ( void )
{
    char val;
    val = (char)VBIOUT_read_reg(VBIREG_FIELDFLAG) & 0x1;
    vbi_debug(KERN_INFO "[vbiout] GETFIELD %x\n",val);
    return val;
}

static int
vbiout_check_text (unsigned char *value)
{
    char field;

    vbi_debug(KERN_INFO "[vbiout] CHECK_TEXT %2x%2x %x  %x\n",value[0],value[1],value[2],vbioutstat.txtbuf+vbioutstat.txtcount*VBITXT_BUFSIZE);

    field = vbiout_get_fieldflag();
    if ( value[2] == field ) {
        vbiout_set_txtenb( value[0]<<8 | value[1] );
        vbiout_set_ttsdramrp( vbioutstat.txtbuf+vbioutstat.txtcount*VBITXT_BUFSIZE );
        return 0;
    } else {
        vbiout_set_txtenb(0x00);
        return -1;
    }
}

static int
vbiout_check_cc (unsigned char *value)
{
    unsigned char field, data[2];
    int err;
    vbi_debug(KERN_INFO "[vbiout] CHECK_CC %x %x %x\n",value[0],value[1],value[2]);
    field = vbiout_get_fieldflag();
    if ( field == value[2] ) {
        if ( field == 0 )
            err = vbiout_set_cc21data(value);
        else
            err = vbiout_set_cc284data(value);
        return 0;
    } else {
        data[0] = 0x80;
        data[1] = 0x80;
        if ( field == 0 )
            err = vbiout_set_cc21data(data);
        else
            err = vbiout_set_cc284data(data);
        return -1;
    }
}

/*******************************
 OPEN Method
*******************************/
static int
vbiout_open (struct inode *inode, struct file *filp)
{
#ifdef MODULE
    MOD_INC_USE_COUNT;
#endif
    return 0;
}

/*******************************
 CLOSE Method
*******************************/
static int
vbiout_release (struct inode * inode, struct file * filp)
{
#ifdef MODULE
    MOD_DEC_USE_COUNT;
#endif
    return 0;
}

/*******************************
 IOCTL Method
*******************************/
static int
vbiout_ioctl (struct inode *inode, struct file *filp,
                unsigned int cmd, unsigned long a)
{
    struct vbiout_ioctl *arg = (struct vbiout_ioctl *)a;
    int err = 0;

    if (_IOC_TYPE (cmd) != VBIOUT_IOC_MAGIC)
        return -EINVAL;

    switch (cmd) {
    case VBIOUT_SET_TXTENB:
        err = vbiout_set_txtenb(arg->ioctl.txtenb);
        break;
    case VBIOUT_SET_CC21DATA:
        err = vbiout_set_cc21data(arg->ioctl.ccdata);
        break;
    case VBIOUT_SET_CC284DATA:
        err = vbiout_set_cc284data(arg->ioctl.ccdata);
        break;
    case VBIOUT_CHECK_TEXT:
        err = vbiout_check_text(arg->ioctl.text);
        break;
    case VBIOUT_CHECK_CCDATA:
        err = vbiout_check_cc(arg->ioctl.ccdata);
        break;
    default:
        return -EINVAL;
        break;
    }

    return err;
}

static DEFINE_MUTEX(vbiout_write_mutex);

static ssize_t vbiout_write (struct file *file, const char __user *buf,
                size_t count, loff_t *offset)
{
    unsigned char *start;

    mutex_lock(&vbiout_write_mutex);

    if (count > VBITXT_DATASIZE)
        count = VBITXT_DATASIZE;

    start = (unsigned char *)(vbioutstat.txtbuf+vbioutstat.txtcount*VBITXT_BUFSIZE);
    if (copy_from_user(start, buf, count)) {
        mutex_unlock(&vbiout_write_mutex);
        return -EFAULT;
    }
    if( vbioutstat.txtcount == VBITXT_BUFNUM-1)
        vbioutstat.txtcount = 0;
    else
        vbioutstat.txtcount++;

    vbi_debug(KERN_INFO "[vbiout] write %x %x %x %x %x %x %x %x\n",start[0],start[1],start[2],start[3],start[40],start[41],start[42],start[43]);

    mutex_unlock(&vbiout_write_mutex);

    return count;
}

/*
 * The different file operations
 */
struct file_operations vbiout_fops =
{
    ioctl:vbiout_ioctl,
    write:vbiout_write,
    open:vbiout_open,
    release:vbiout_release,
};

/*******************************
 INIT Method
*******************************/
int __init
vbiout_init (void)
{
    int result;

    /*
     * Register your major, and accept a dynamic number
     */
    result = register_chrdev (vbiout_major, "vbiout", &vbiout_fops);
    if (result < 0) {
        printk (KERN_WARNING "[vbiout] can't get major %d\n", vbiout_major);
        return result;
    }
    if (vbiout_major == 0)
        vbiout_major = result;        /* dynamic */
    printk (KERN_INFO  "[vbiout] register_chrdev complete major = %d\n", vbiout_major);

    vbioutstat.txtbuf = (unsigned int)kmalloc(VBITXT_BUFSIZE*VBITXT_BUFNUM, GFP_KERNEL);
    if ( vbioutstat.txtbuf == 0 )
        return -1;
    else if ( (vbioutstat.txtbuf & 0xF0000000) == 0x90000000 )
        vbioutstat.txtbuf = (vbioutstat.txtbuf & 0x0FFFFFFF) | 0xB0000000;
    else if ( (vbioutstat.txtbuf & 0xF0000000) == 0xA0000000 )
        vbioutstat.txtbuf = (vbioutstat.txtbuf & 0x0FFFFFFF) | 0xC0000000;

    memset((unsigned char *)vbioutstat.txtbuf, 0, VBITXT_BUFSIZE*VBITXT_BUFNUM);

    vbioutstat.txtcount = 0;
    vbiout_set_txtenb ( 0x0 );
  
    return 0;            /* succeed */
}



/*******************************
 EXIT Method
*******************************/
void
vbiout_exit (void)
{
    unregister_chrdev (vbiout_major, "vbiout");

    if (vbioutstat.txtbuf) {
        kfree((unsigned char *)vbioutstat.txtbuf);
    }
}

module_init(vbiout_init);
module_exit(vbiout_exit);

