/*
 * Copyright (C) 2010, Panasonic Corporation.
 *                       All Rights Reserved.
 *
 */

#define INTERMEM_CONFIG_NUM	ARRAY_SIZE( intermem_configuration )

typedef struct _INTERMEM_GROUP{
	const INTERMEM_CONFIG* nodeConfig;
	u8*		topAddress;
	u8*		topAddress_dma;
	u8*		bottomAddress;
	int		waitingNum;
	int*	waitingStack;
	u8*		workingCheckTable;
} INTERMEM_GROUP;

typedef struct _INTERMEM_TOP{
	INTERMEM_GROUP	groups[INTERMEM_CONFIG_NUM];
	u8*				topAddress;
	spinlock_t		lock;
} INTERMEM_TOP;

#define INTERMEM_LOCK( EHCI )										\
	{	unsigned long lockflags;									\
		spin_lock_irqsave( &(EHCI)->intermem->lock, lockflags );

#define INTERMEM_UNLOCK( EHCI )											\
		spin_unlock_irqrestore( &(EHCI)->intermem->lock, lockflags );	\
	}

static inline void freeGroups( INTERMEM_GROUP* gr )
{
	int i;
	
	if( !gr ) return;
	
	for( i=0 ; i < INTERMEM_CONFIG_NUM ; i++ ){
		if( gr[i].waitingStack )		kfree( gr[i].waitingStack );
		if( gr[i].workingCheckTable )	kfree( gr[i].workingCheckTable );
	}
}

static void usbh_sharemem_finish( struct ehci_hcd *ehci )
{
	INTERMEM_TOP* top;
	
	top = ehci->intermem;
	if( !top ) return;
	ehci->intermem = NULL;
	
#if defined(CONFIG_USB_PANASONIC_SLD2H) || defined(CONFIG_USB_PANASONIC_SLD3)
	iounmap( top->topAddress );
#endif
	freeGroups( top->groups );
	kfree( top );
}

static inline int allocateGroups( INTERMEM_GROUP* gr, u8* address, u8* dma )
{
	size_t usedBufferSize;
	int i, j, num;
	
	usedBufferSize = 0;
	
	for( i=0 ; i < INTERMEM_CONFIG_NUM ; i++, gr++ ){
		gr->nodeConfig = &intermem_configuration[i];
		num = gr->nodeConfig->num;
		gr->topAddress =		address	+ usedBufferSize;
		gr->topAddress_dma =	dma		+ usedBufferSize;
		usedBufferSize +=		num * gr->nodeConfig->size;
		gr->bottomAddress =		address	+ usedBufferSize -1;
		if( INTERMEM_SIZE < usedBufferSize )
			return -1;
		
		gr->waitingStack = kzalloc( sizeof(int) * num, GFP_ATOMIC );
		if( !gr->waitingStack ) return -1;
		
		gr->workingCheckTable = kzalloc( sizeof(u8) * num, GFP_ATOMIC );
		if( !gr->workingCheckTable ) return -1;

		gr->waitingNum = num;
		for( j=0 ; j < num ; j++ ) gr->waitingStack[j] = num -j -1;
	}
	return 0;
}

static int usbh_sharemem_init( struct ehci_hcd *ehci )
{
	u8* address;
	u8* dma;
	
	assert( ehci->intermem == NULL );
	ehci->intermem = kzalloc( sizeof(INTERMEM_TOP), GFP_ATOMIC );
	if( !ehci->intermem ) return -1;
	spin_lock_init( &ehci->intermem->lock );
	dma = address = (u8*)( (u32)ehci_to_hcd(ehci)->rsrc_start ) + INTERMEM_OFFSET;
#if defined(CONFIG_USB_PANASONIC_SLD2H) || defined(CONFIG_USB_PANASONIC_SLD3)
	address = ioremap_nocache( (resource_size_t)address, INTERMEM_SIZE );
	assert( address != NULL );
#endif
	memset( address, 0x00, INTERMEM_SIZE );
	ehci->intermem->topAddress = address;
	
	if( allocateGroups( ehci->intermem->groups, address, dma ) ){
		printk( "%s: allocateGroups fail.\n", __FUNCTION__ );
		usbh_sharemem_finish( ehci );
		return -1;
	}
	
	return 0;
}

static inline INTERMEM_GROUP* getSuitableGroupFromSize( INTERMEM_TOP* top, size_t size )
{
	int i;
	INTERMEM_GROUP* gr;
	
	if( top == NULL ) return NULL;
	
	for( i = INTERMEM_CONFIG_NUM-1 ; 0 <= i ; i-- ){
		gr = &top->groups[i];
		if( gr->waitingNum != 0 && size <= gr->nodeConfig->size ) return gr;
	}
	return NULL;
}

static void* usbh_malloc_share( struct ehci_hcd *ehci, size_t size, dma_addr_t* dma )
{
	int index, offset;
	INTERMEM_GROUP* gr;
	void* ret;
	
	ret = NULL;
	
	INTERMEM_LOCK( ehci );
	gr = getSuitableGroupFromSize( ehci->intermem, size );
	if( gr ){
		index = gr->waitingStack[ gr->waitingNum -1 ];
		gr->waitingNum--;
		assert( gr->workingCheckTable[ index ] == 0 );
		gr->workingCheckTable[index] = 1;
		offset = gr->nodeConfig->size * index;
		ret = (void*) ( gr->topAddress + offset );
		if( dma ) *dma = (dma_addr_t) ( gr->topAddress_dma + offset );
	}
	INTERMEM_UNLOCK( ehci );

	return ret;
}

static inline INTERMEM_GROUP* getGroupFromAddress( INTERMEM_TOP* top, void* ptr )
{
	int i;
	INTERMEM_GROUP* gr;
	
	if( top == NULL ) return NULL;
	
	for( i = INTERMEM_CONFIG_NUM-1 ; 0 <= i ; i-- ){
		gr = &top->groups[i];
		if( gr->topAddress <= (u8*)ptr && (u8*)ptr <= gr->bottomAddress ) return gr;
	}
	return NULL;
}

static void usbh_free_share( struct ehci_hcd *ehci, void* ptr )
{
	int index, offset;
	INTERMEM_GROUP* gr;
	
	INTERMEM_LOCK( ehci );
	gr = getGroupFromAddress( ehci->intermem, ptr );
	if( gr ){
		offset = ( (u8*)ptr - gr->topAddress );
		index  = offset / gr->nodeConfig->size;
		assert( gr->workingCheckTable[index] );
		gr->workingCheckTable[ index ] = 0;
		gr->waitingStack[ gr->waitingNum ] = index;
		gr->waitingNum++;
	}
	INTERMEM_UNLOCK( ehci );
}

