/* Shared library add-on to ip6tables to add NFMARK matching support. */
#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>

#include <ip6tables.h>
#include <linux/netfilter_ipv6/ip6t_mark.h>

/* Function which prints out usage message. */
static void
help(void)
{
	printf(
"MARK match v%s options:\n"
"[!] --mark value[/mask]         Match nfmark value with optional mask\n"
"\n",
NETFILTER_VERSION);
}

static struct option opts[] = {
	{ "mark", 1, 0, '1' },
	{0}
};

/* Initialize the match. */
static void
init(struct ip6t_entry_match *m, unsigned int *nfcache)
{
	/* Can't cache this. */
	*nfcache |= NFC_UNKNOWN;
}

/* Function which parses command options; returns true if it
   ate an option */
static int
parse(int c, char **argv, int invert, unsigned int *flags,
      const struct ip6t_entry *entry,
      unsigned int *nfcache,
      struct ip6t_entry_match **match)
{
	struct ip6t_mark_info *markinfo = (struct ip6t_mark_info *)(*match)->data;

	switch (c) {
		char *end;
	case '1':
		if (check_inverse(optarg, &invert))
			optind++;
		markinfo->mark = strtoul(optarg, &end, 0);
		if (*end == '/') {
			markinfo->mask = strtoul(end+1, &end, 0);
		} else
			markinfo->mask = 0xffffffff;
		if (*end != '\0' || end == optarg)
			exit_error(PARAMETER_PROBLEM, "Bad MARK value `%s'", optarg);
		if (invert)
			markinfo->invert = 1;
		*flags = 1;
		break;

	default:
		return 0;
	}
	return 1;
}

static void
print_mark(unsigned long mark, unsigned long mask, int invert, int numeric)
{
	if (invert)
		fputc('!', stdout);

	if(mask != 0xffffffff)
		printf("0x%lx/0x%lx ", mark, mask);
	else
		printf("0x%lx ", mark);
}

/* Final check; must have specified --mark. */
static void
final_check(unsigned int flags)
{
	if (!flags)
		exit_error(PARAMETER_PROBLEM,
			   "MARK match: You must specify `--mark'");
}

/* Prints out the matchinfo. */
static void
print(const struct ip6t_ip6 *ip,
      const struct ip6t_entry_match *match,
      int numeric)
{
	printf("MARK match ");
	print_mark(((struct ip6t_mark_info *)match->data)->mark,
		  ((struct ip6t_mark_info *)match->data)->mask,
		  ((struct ip6t_mark_info *)match->data)->invert, numeric);
}

/* Saves the union ip6t_matchinfo in parsable form to stdout. */
static void
save(const struct ip6t_ip6 *ip, const struct ip6t_entry_match *match)
{
	printf("--mark ");
	print_mark(((struct ip6t_mark_info *)match->data)->mark,
		  ((struct ip6t_mark_info *)match->data)->mask,
		  ((struct ip6t_mark_info *)match->data)->invert, 0);
}

static
struct ip6tables_match mark
= { NULL,
    "mark",
    NETFILTER_VERSION,
    IP6T_ALIGN(sizeof(struct ip6t_mark_info)),
    IP6T_ALIGN(sizeof(struct ip6t_mark_info)),
    &help,
    &init,
    &parse,
    &final_check,
    &print,
    &save,
    opts
};

void _init(void)
{
	register_match6(&mark);
}
