/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/****************************************************************************/
#include "fmb_api.h"
#include "fmb_hw_usb.h"
#include "fmb_sys_abs.h"

#ifndef VERSION
#define VERSION "?.?.?.?"
#endif

PREPARE_DEBUG_FOR_FUNC();
//static DECLARE_WAIT_QUEUE_HEAD(sys_cmd_ack_wait);
//static DECLARE_WAIT_QUEUE_HEAD(sys_msg_wait);
//static DECLARE_WAIT_QUEUE_HEAD(eject_event_wait);

/********************************************************/
/*      GROBAL VALUES               */
/********************************************************/
#if 0 /* During a function add */
extern const struct fmb_param_head_table g_Fmb_param_table[];
#endif

/********************************************************/
/*                          */
/*  MB86H57 core register processing function  */
/*  (Application part)              */
/*                          */
/********************************************************/
#if 0 /* During a function add */
/**
*  @brief       Get the parameter information for MB86H57
*  @param[in]   hw_priv_p           private data pointer
*  @param[in]   id                  parameter id
*  @return      fmb_param_table*    parameter table data
*  @note        None
*  @attention   None
*/
static struct fmb_param_table*  fmb_api_get_param_table_data(struct fmb_hard_private*   hw_priv_p,
                    unsigned long id )
{
    int minor;
    int i;
    struct fmb_param_table* second_table_addr;

    minor = hw_priv_p->minor;
    i = 0;
    second_table_addr = NULL;
    MSG(INTERNAL_LOG, minor, "START");

    /* serch first table */
    while (g_Fmb_param_table[i].second_table_addr != NULL) {

        if (((id >> 16) & 0xffff) == g_Fmb_param_table[i].id_up_16bit) {
            /* id(upper 16bit) found! */
            second_table_addr = (void *)g_Fmb_param_table[i].second_table_addr;
            break;
        }
        else {
            i++;    /* next */
        }
    }

    if (second_table_addr == NULL) {
        MSG(INTERNAL_ERR_LVL, minor, "second table not found.....");
        return NULL;
    }

    /* check array size(lower 16bit) */
    if (((id & 0xffff) + 1) > g_Fmb_param_table[i].second_table_size) {
        MSG(INTERNAL_ERR_LVL, minor, "second table is invalid.....");
        return NULL;
    }

    if (second_table_addr[(id & 0xffff)].valid == 1) {
        MSG(INTERNAL_LOG, minor, "END");
        return &second_table_addr[(id & 0xffff)];
    }
    else {
        MSG(INTERNAL_ERR_LVL, minor, "id is not supported.....");
        return NULL;
    }
}

#endif


void
local_mask_create32( unsigned long *mask, unsigned short pos, unsigned short len, unsigned short reg_bit_size )
{
    *mask = ~(0);
    *mask <<= reg_bit_size - (pos + len);
    *mask >>= reg_bit_size - (pos + len);
    *mask >>= pos;
    *mask <<= pos;
    
    return;
}

void
local_mask_create( unsigned short *mask, unsigned short pos, unsigned short len, unsigned short reg_bit_size )
{
    *mask = ~(0);
    *mask <<= reg_bit_size - (pos + len);
    *mask >>= reg_bit_size - (pos + len);
    *mask >>= pos;
    *mask <<= pos;
    
    return;
}

/**
*  @brief       ioctl() FMB_API_GET_PARAM Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer(struct fmb_param_info *)
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_get_param( struct fmb_hard_private * hw_priv_p, unsigned long argp )
{
    int minor;
    struct fmb_param_info   param_info;
    int    register_bit_size;
    int mask_flg;
    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    PRIV_P_CHECK( (void *)argp,      -EINVAL );
    
    minor = hw_priv_p->minor;
    
    register_bit_size = sizeof( param_info.value ) * 8;
    
    
    if (copy_from_user(&param_info, (void*)argp, sizeof(struct fmb_param_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_param_info)");
        return -EFAULT;
    }

    mask_flg = param_info.mask;
    
    if ( mask_flg == 0 )
    {
        if ( param_info.bit_pos >= register_bit_size )  return -EINVAL;
        if ( param_info.bit_pos + param_info.bit_len > register_bit_size )  return -EINVAL;
        if ( param_info.bit_len == 0 ) return -EINVAL;

        local_mask_create( &(param_info.mask), param_info.bit_pos, param_info.bit_len, register_bit_size );
    }
    
    param_info.value = Fmb_hw_reg_read_lsi(hw_priv_p, param_info.offset, &result_internal );
    
    param_info.value &= param_info.mask;
    
    if ( mask_flg == 0 )
    {
        param_info.value >>= param_info.bit_pos;
        param_info.mask    = 0;
    }

    if ( result_internal < 0 ) return result_internal;

    if (copy_to_user((void*)argp, &param_info, sizeof(struct fmb_param_info))) {
//        up(&priv_p->sys_cmd_sync_send_mutex);
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_param_info)\n");
        return -EFAULT;
    }
    
    
    return 0;
}
/**
*  @brief       ioctl() FMB_API_GET_PARAM32 Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer(struct fmb_param_info *)
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_get_param32( struct fmb_hard_private * hw_priv_p, unsigned long argp )
{
    int minor;
    struct fmb_param_info32   param_info32;
    int    register_bit_size;
    int mask_flg;
    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    PRIV_P_CHECK( (void *)argp,      -EINVAL );
    
    minor = hw_priv_p->minor;
    
    register_bit_size = sizeof( param_info32.value ) * 8;
    
    
    if (copy_from_user(&param_info32, (void*)argp, sizeof(struct fmb_param_info32))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_param_info32)");
        return -EFAULT;
    }

    mask_flg = param_info32.mask;
    
    if ( mask_flg == 0 )
    {
        if ( param_info32.bit_pos >= register_bit_size )  return -EINVAL;
        if ( param_info32.bit_pos + param_info32.bit_len > register_bit_size )  return -EINVAL;
        if ( param_info32.bit_len == 0 ) return -EINVAL;

        local_mask_create32( &(param_info32.mask), param_info32.bit_pos, param_info32.bit_len, register_bit_size );
    }
    
    param_info32.value = Fmb_hw_reg_read_lsi32(hw_priv_p, param_info32.offset, &result_internal );
    if ( result_internal < 0 ) return result_internal;

    param_info32.value &= param_info32.mask;
    
    if ( mask_flg == 0 )
    {
        param_info32.value >>= param_info32.bit_pos;
        param_info32.mask    = 0;
    }

    if (copy_to_user((void*)argp, &param_info32, sizeof(struct fmb_param_info32))) {
//        up(&priv_p->sys_cmd_sync_send_mutex);
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_param_info32)\n");
        return -EFAULT;
    }
    
    
    return 0;
}

/**
*  @brief       ioctl() FMB_API_SET_PARAM Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer(struct fmb_param_info *)
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_set_param( struct fmb_hard_private * hw_priv_p, unsigned long argp )
{
    int minor;
    struct fmb_param_info   param_info;
    int    register_bit_size;
    int mask_flg;
    unsigned long   w_mask;
    unsigned long   w_value;
    unsigned long   w_offset_acheck;
    unsigned long   work_offset;
    
    int ret;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    PRIV_P_CHECK( (void *)argp,      -EINVAL );
    
    minor = hw_priv_p->minor;
    
    register_bit_size = sizeof( param_info.value ) * 8;
    
    
    if (copy_from_user(&param_info, (void*)argp, sizeof(struct fmb_param_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_param_info)");
        return -EFAULT;
    }
    /* 2byte align check */
    w_offset_acheck =   param_info.offset;
    w_offset_acheck >>= 1;
    w_offset_acheck <<= 1;
    if ( w_offset_acheck != param_info.offset )
    {
        return -EINVAL;
    }
    /* 4byte align check */
    w_offset_acheck =   param_info.offset;
    w_offset_acheck >>= 2;
    w_offset_acheck <<= 2;
    if ( w_offset_acheck != param_info.offset )
    {
        work_offset = param_info.offset - 2;
    }
    else
    {
        work_offset = param_info.offset;
    }
    
    mask_flg = param_info.mask;
    
    if ( mask_flg == 0 )
    {
        if ( param_info.bit_pos >= register_bit_size )  return -EINVAL;
        if ( param_info.bit_pos + param_info.bit_len > register_bit_size )  return -EINVAL;
        if ( param_info.bit_len == 0 ) return -EINVAL;

        local_mask_create( &(param_info.mask), param_info.bit_pos, param_info.bit_len, register_bit_size );
        param_info.value <<= param_info.bit_pos;
    }
    if ( work_offset == param_info.offset )
    {
#ifdef MULTIENDIAN_FUNCTION
        w_value = cpu_to_be32(param_info.value << ( 2 * 8 ) );
        w_mask  = cpu_to_be32(param_info.mask  << ( 2 * 8 ) );
#else
        w_value = (unsigned long)param_info.value;
        w_mask  = (unsigned long)param_info.mask;
#endif
    }
    else
    {
#ifdef MULTIENDIAN_FUNCTION
        w_value = cpu_to_be32(param_info.value);
        w_mask  = cpu_to_be32(param_info.mask);
#else
        w_value = (unsigned long)param_info.value << ( 2 * 8 );
        w_mask  = (unsigned long)param_info.mask  << ( 2 * 8 );
#endif
    }
    
    ret = Fmb_hw_reg_rmw_lsi32(hw_priv_p, work_offset, w_mask, w_value );
    
    if ( ret < 0 ) return ret;
    
    return 0;
}
/**
*  @brief       ioctl() FMB_API_SET_PARAM32 Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer(struct fmb_param_info *)
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_set_param32( struct fmb_hard_private * hw_priv_p, unsigned long argp )
{
    int minor;
    struct fmb_param_info32   param_info32;
    int    register_bit_size;
    int mask_flg;
    unsigned long  w_offset_acheck;
    int ret;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    PRIV_P_CHECK( (void *)argp,      -EINVAL );
    
    minor = hw_priv_p->minor;
    
    register_bit_size = sizeof( param_info32.value ) * 8;
    
    
    if (copy_from_user(&param_info32, (void*)argp, sizeof(struct fmb_param_info32))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_param_info32)");
        return -EFAULT;
    }
    
    /* 4byte align check */
    w_offset_acheck =   param_info32.offset;
    w_offset_acheck >>= 2;
    w_offset_acheck <<= 2;
    if ( w_offset_acheck != param_info32.offset )
    {
        return -EINVAL;
    }
    mask_flg = param_info32.mask;
    
    if ( mask_flg == 0 )
    {
        if ( param_info32.bit_pos >= register_bit_size )  return -EINVAL;
        if ( param_info32.bit_pos + param_info32.bit_len > register_bit_size )  return -EINVAL;
        if ( param_info32.bit_len == 0 ) return -EINVAL;

        local_mask_create32( &(param_info32.mask), param_info32.bit_pos, param_info32.bit_len, register_bit_size );
        param_info32.value <<= param_info32.bit_pos;
    }
    
#ifdef MULTIENDIAN_FUNCTION
    param_info32.mask  = cpu_to_be32(param_info32.mask);
    param_info32.value = cpu_to_be32(param_info32.value);
#endif

    ret = Fmb_hw_reg_rmw_lsi32(hw_priv_p, param_info32.offset, param_info32.mask, param_info32.value );
    if ( mask_flg == 0 )
    {
        param_info32.mask = 0;
    }
    
    if ( ret < 0 ) return ret;
    
    return 0;
}

struct local_event_area_info
{
    int             area_num;
    unsigned long   area1_addr;
    size_t          area1_num;
    unsigned long   area2_addr;
    size_t          area2_num;
};


int
local_get_event_area_info( struct local_event_area_info *reg_info, unsigned long en_mask )
{
    int ret = 0;
    
    
    switch( en_mask )
    {
    case FMB_FACT_STATE:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_STATE_CHAGE_ADDR;
        reg_info->area1_num  = FMB_EVENT_STATE_CHAGE_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_PCR_IN:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_PCR_ADDR;
        reg_info->area1_num  = FMB_EVENT_PCR_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_STREAM_BUFF:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_STREAM_BUFF_ADDR;
        reg_info->area1_num  = FMB_EVENT_STREAM_BUFF_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_PID_IN:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_PID_ADDR;
        reg_info->area1_num  = FMB_EVENT_PID_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_I2C:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_I2C_ADDR;
        reg_info->area1_num  = FMB_EVENT_I2C_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_BCAS_STATE:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_BCAS_STATE_CHANGE_ADDR;
        reg_info->area1_num  = FMB_EVENT_BCAS_STATE_CHANGE_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_BCAS_DATA:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_BCAS_DATA_RET_ADDR;
        reg_info->area1_num  = FMB_EVENT_BCAS_DATA_RET_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_SECURE:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_SECURE_ADDR;
        reg_info->area1_num  = FMB_EVENT_SECURE_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_VFRAME_ERR:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_VFRAME_ERR_ADDR;
        reg_info->area1_num  = FMB_EVENT_VFRAME_ERR_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_VFRAME:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_VFRAME_ADDR;
        reg_info->area1_num  = FMB_EVENT_VFRAME_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_AFRAME:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_AFRAME_ADDR;
        reg_info->area1_num  = FMB_EVENT_AFRAME_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_STREAM_SIZE:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_STREAM_SIZE_ADDR;
        reg_info->area1_num  = FMB_EVENT_STREAM_SIZE_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_STREAM_SIZE2:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_STREAM_SIZE2_ADDR;
        reg_info->area1_num  = FMB_EVENT_STREAM_SIZE2_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_GOP_INFO:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_GOP_INFO_ADDR;
        reg_info->area1_num  = FMB_EVENT_GOP_INFO_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_GOP_INFO2:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_GOP_INFO2_ADDR;
        reg_info->area1_num  = FMB_EVENT_GOP_INFO2_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    case FMB_FACT_XERROR:
        reg_info->area_num   = 1;
        reg_info->area1_addr = FMB_EVENT_XERROR_ADDR;
        reg_info->area1_num  = FMB_EVENT_XERROR_NUM;
        reg_info->area2_addr = 0x000000;
        reg_info->area2_num  = 0;
        break;
    default:
        ret = -1;
        break;
    }

    return ret;

}
int
local_get_event_area_data( struct fmb_hard_private* hw_priv_p, unsigned char *out_data, unsigned long addr, int size_num )
{
    int cnt_num = 0;
    unsigned short *out_work;
    int result_internal = 0;
    
    out_work = (unsigned short *)out_data; /** todo: check endian */
    
    
    for(;;)
    {
        if ( cnt_num >= size_num ) break;
        
        *(out_work + cnt_num) = Fmb_hw_reg_read_lsi( hw_priv_p, addr + (cnt_num * 2), &result_internal );
        
        if ( result_internal < 0 ) break;
        
        cnt_num++;
    }
    
    if ( result_internal < 0 ) return result_internal;
    
    return 0;
}

/**
*  @brief       ioctl() FMB_API_GET_ASYNC_INFO_SIZE Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[io]   argp        command parameter pointer (struct fmb_async_info *)
*                           [in ](fmb_async_info).en_master       expect irq mask data(bit 0:none 1:wait irq)
*                           [out](fmb_async_info).data_num        number of data area
*                           [out](fmb_async_info).data_data1_size size of data1 area
*                           [out](fmb_async_info).data_data2_size size of data2 area
*                           [---](fmb_async_info).(other member)  none
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_get_async_info_size( struct fmb_hard_private* hw_priv_p, unsigned long argp )
{
    int minor;
    int result = 0;
    int ret    = 0;
    struct fmb_async_info async_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    struct local_event_area_info reg_info;
    
    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&async_info, (void*)argp, sizeof(struct fmb_async_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_async_info)");
        return -EFAULT;
    }
    async_info.data_num   = 0;
    async_info.data1_size = 0;
    async_info.data2_size = 0;
    
    ret = local_get_event_area_info( &reg_info, async_info.en_mask );
    if ( ret )  return -EINVAL;
    
    async_info.data_num   = reg_info.area_num;
    async_info.data1_size = reg_info.area1_num * 2;
    async_info.data2_size = reg_info.area2_num * 2;
    
    if (copy_to_user( (void*)argp, &async_info, sizeof(struct fmb_async_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_async_info)");
        return -EFAULT;
    }

    return result;
    
}


/**
*  @brief       ioctl() FMB_API_GET_ASYNC_INFO Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[io]   argp        command parameter pointer (struct fmb_async_info *)
*                           [in ](fmb_async_info).en_master      expect irq mask data(bit 0:none 1:wait irq)
*                           event data output:
*                           [io ](fmb_async_info).data1_addr     user memory address for data1 output
*                           [in ](fmb_async_info).data1_size     user memory size    for data1 output
*                           [io ](fmb_async_info).data2_addr     user memory address for data2 output(only data_num=2(FMB_API_GET_ASYNC_INFO_SIZE))
*                           [in ](fmb_async_info).data2_size     user memory size    for data2 output(only data_num=2(FMB_API_GET_ASYNC_INFO_SIZE))
*                           [---](fmb_async_info).(other member) none
*                     local:[in ] mode                           0:get_async_info mode 1:async_info mode
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_get_async_info( struct fmb_hard_private* hw_priv_p, unsigned long argp, int mode )
{
    int minor;
    int result = 0;
    int ret    = 0;
    struct fmb_async_info async_info;
    struct fmb_fact_info  fact_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    struct local_event_area_info reg_info;
    
    unsigned char *work_data1 = NULL;
    unsigned char *work_data2 = NULL;

    int result_internal = 0;
    
    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if ( mode != 0 && mode != 1 )
    {
        MSG(KERN_ERR_LVL, minor, "Failed in a mode error.(mode=%d)", mode );
            return -EFAULT;
    }
    
    
    if ( mode == 0 )
    {
        if (copy_from_user(&async_info, (void*)argp, sizeof(struct fmb_async_info))) {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_async_info)");
            return -EFAULT;
        }
    }
    else /* mode == 1 */
    {
        if (copy_from_user(&fact_info, (void*)argp, sizeof(struct fmb_fact_info))) {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_async_info)");
            return -EFAULT;
        }
        async_info.en_mask    = fact_info.en_mask;
        async_info.data_num   = 1;
        async_info.data1_addr = fact_info.data;
        async_info.data1_size = fact_info.size;
        async_info.data2_addr = 0;
        async_info.data2_size = 0;
    }
    
    if (down_interruptible(&priv_p->async_info_mutex)) {
        /** Signal(ctrl+c etc) interrupt */
        MSG(KERN_ERR_LVL, minor, "Signal interrupt in wait mutex.(async_info))");
        return -ERESTARTSYS;
    }
    
    
    ret = local_get_event_area_info( &reg_info, async_info.en_mask );
    if ( ret != 0 )
    {
        up(&priv_p->async_info_mutex);
        return -EINVAL;
    }
    
    if ( mode == 0 )
    {
        if ( reg_info.area_num >= 1 && ( reg_info.area1_num * 2 > async_info.data1_size ) )
        {
                up(&priv_p->async_info_mutex);
                return -EINVAL;
        }
        if ( reg_info.area_num >= 2 && ( reg_info.area2_num * 2 > async_info.data2_size ) )
        {
                up(&priv_p->async_info_mutex);
                return -EINVAL;
        }
    }
    else /* mode == 1 */
    {
        if ( reg_info.area_num >= 1 && ( reg_info.area1_num * 2 > async_info.data1_size ) )
        {
                up(&priv_p->async_info_mutex);
                return -EINVAL;
        }
        
        if ( reg_info.area_num >= 2 && ( reg_info.area1_num * 2 + reg_info.area2_num * 2 ) > (async_info.data1_size + async_info.data2_size) )
        {
                up(&priv_p->async_info_mutex);
                return -EINVAL;
        }
    }
    if ( reg_info.area_num >= 1 )
    {
        if ( reg_info.area1_addr != 0 )
        {
            work_data1 = (unsigned char *)kmalloc( reg_info.area1_num * 2, GFP_KERNEL );
            if ( work_data1 == NULL )
            {
                up(&priv_p->async_info_mutex);
                return -ENOMEM; /* not enough memory */
            }
        }
    }

    if ( reg_info.area_num >= 2 )
    {
        if ( reg_info.area2_addr != 0 )
        {
            work_data2 = (unsigned char *)kmalloc( reg_info.area2_num * 2, GFP_KERNEL );
            if ( work_data2 == NULL )
            {
                kfree( work_data1 );
                up(&priv_p->async_info_mutex);
                return -ENOMEM; /* not enough memory */
            }
        }
    }
    
    if ( mode == 0 )
    {
    
        if ( work_data1 != NULL )
        {
            result_internal = local_get_event_area_data( hw_priv_p, work_data1, reg_info.area1_addr, reg_info.area1_num );
            if ( result_internal < 0 )
            {
                kfree( work_data1 );
                if ( work_data2 != NULL ) kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return result_internal;
            }
            if (copy_to_user( async_info.data1_addr, work_data1, reg_info.area1_num * 2)) {
                MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_async_info.data1_addr)");
                kfree( work_data1 );
                if ( work_data2 != NULL ) kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return -EFAULT;
            }
            kfree( work_data1 );
        }
        if ( work_data2 != NULL )
        {
            result_internal = local_get_event_area_data( hw_priv_p, work_data2, reg_info.area2_addr, reg_info.area2_num );
            if ( result_internal < 0 )
            {
                kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return result_internal;
            }
            if (copy_to_user( async_info.data2_addr, work_data2, reg_info.area2_num * 2)) {
                MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_async_info.data2_addr)");
                kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return -EFAULT;
            }
            kfree( work_data2 );
        }
    }
    else
    {
        if ( work_data1 != NULL )
        {
            result_internal = local_get_event_area_data( hw_priv_p, work_data1, reg_info.area1_addr, reg_info.area1_num );
            if ( result_internal < 0 )
            {
                kfree( work_data1 );
                if ( work_data2 != NULL ) kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return result_internal;
            }
            if (copy_to_user( async_info.data1_addr, work_data1, (reg_info.area1_num + reg_info.area2_num) * 2)) {
                MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_async_info.data1_addr)");
                kfree( work_data1 );
                if ( work_data2 != NULL ) kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return -EFAULT;
            }
            kfree( work_data1 );
        }
        if ( work_data2 != NULL )
        {
            result_internal = local_get_event_area_data( hw_priv_p, work_data2, reg_info.area2_addr, reg_info.area2_num );
            if ( result_internal < 0 )
            {
                kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return result_internal;
            }
            if (copy_to_user( async_info.data1_addr + reg_info.area1_num * 2, work_data2, reg_info.area2_num * 2)) {
                MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_async_info.data2_addr)");
                kfree( work_data2 );
                up(&priv_p->async_info_mutex);
                return -EFAULT;
            }
            kfree( work_data2 );
        }
    }

    
    up(&priv_p->async_info_mutex);

    if ( mode == 0 )
    {
        if (copy_to_user( (void*)argp, &async_info, sizeof(struct fmb_async_info))) {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_async_info)");
            return -EFAULT;
        }
    }
    else
    {
        if (copy_to_user( (void*)argp, &fact_info, sizeof(struct fmb_fact_info))) {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_fact_info)");
            return -EFAULT;
        }
    }

    return result;
    
}
/**
*  @brief       ioctl() FMB_API_ASYNC_INFO Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[io]   argp        command parameter pointer (struct fmb_fact_info *)
*                           [in ](fmb_fact_info).en_master      expect irq mask data(bit 0:none 1:wait irq)
*                           event data output:
*                           [io ](fmb_fact_info).data           (i)user memory address for data output(data1 area(+data2 area)) (o)event data
*                           [in ](fmb_fact_info).size           user memory size    for data output(need data1 size(+data2 size))
*                           [---](fmb_fact_info).(other member) none
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_async_info( struct fmb_hard_private* hw_priv_p, unsigned long argp )
{
    int ret;
    ret = fmb_api_get_async_info( hw_priv_p, argp, 1 );
    return ret;
}
/**
*  @brief       ioctl() FMB_API_WAIT_FACT Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[io]   argp        command parameter pointer (struct fmb_fact_info *)
*                           [in ](fmb_fact_info).en_mask        expect irq mask data(bit 0:none 1:wait irq)
*                           [out](fmb_fact_info).status         irq data( only return=0 )
*                           [out](fmb_fact_info).cret           reason of wait cancel
*                           [---](fmb_fact_info).(other member) none
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_wait_fact( struct fmb_hard_private* hw_priv_p, unsigned long argp )
{
    int minor;
    int result = 0;
    struct fmb_fact_info fact_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    
    unsigned long val_status;
    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if ( !(fmb_hw_is_exist_intr( hw_priv_p )) )
    {
        return -EPERM;
    }

    if (copy_from_user(&fact_info, (void*)argp, sizeof(struct fmb_fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_fact_info)");
        return -EFAULT;
    }
    if (down_interruptible(&priv_p->wait_fact_mutex)) {
        /** Signal(ctrl+c etc) interrupt */
            MSG(KERN_ERR_LVL, minor, "Signal interrupt in wait mutex.(wait_fact))");
            return -ERESTARTSYS;
    }

    fact_info.cret   = 0;
    fact_info.status = 0;
    priv_p->fact_irq_act_flag = 1;
    
    for(;;)
    {

        result_internal = Fmb_core_lsi_get_irq_status( hw_priv_p, &val_status ); if (  result_internal < 0 ) break;
        /* check irq status */
        if ( (val_status & fact_info.en_mask) != 0 )
        {
			/* Factor having */
            fact_info.cret   = FMB_FACT_IRQ_EVENT;
            fact_info.status = val_status;
			result           = 0;
            break;
        }
        
        if ( priv_p->fact_irq_type == FMB_FACT_IRQ_REMOVE_CANCEL )
        {
            result          = -ECANCELED;
            fact_info.cret  = FMB_FACT_IRQ_REMOVE_CANCEL;
            break;
        }
        if ( priv_p->fact_irq_stop_flag == FMB_FACT_IRQ_REMOVE_CANCEL )
        {
            result          = -ECANCELED;
            fact_info.cret  = FMB_FACT_IRQ_REMOVE_CANCEL;
            break;
        }
        if ( priv_p->fact_irq_stop_flag == FMB_FACT_IRQ_IOCTL_CANCEL )
        {
            result          = -ECANCELED;
            fact_info.cret  = FMB_FACT_IRQ_IOCTL_CANCEL;
            priv_p->fact_irq_stop_flag = FMB_FACT_IRQ_NO_CANCEL;
            break;
        }

#if 1
        result = FMB_SYS_ABS_wait_event(priv_p->fact_irq_wait,
                              priv_p->fact_irq_flag == FMB_FACT_IRQ_ON
                              );
        
#else
    /* for debug */
        result = FMB_SYS_ABS_wait_event_timeout(priv_p->fact_irq_wait,
                          priv_p->fact_irq_flag == FMB_FACT_IRQ_ON,
                          (15 * HZ));

#endif
        priv_p->fact_irq_flag = FMB_FACT_IRQ_OFF;

        switch( priv_p->fact_irq_type )
        {
        case FMB_FACT_IRQ_IOCTL_CANCEL:
                result = -ECANCELED;
                fact_info.cret        = FMB_FACT_IRQ_IOCTL_CANCEL;
                priv_p->fact_irq_type = FMB_FACT_IRQ_NO_CANCEL;
                priv_p->fact_irq_stop_flag = FMB_FACT_IRQ_NO_CANCEL;
                break;
        case FMB_FACT_IRQ_REMOVE_CANCEL:
                result = -ECANCELED;
                fact_info.cret        = FMB_FACT_IRQ_REMOVE_CANCEL;
                break;
        default:
                break;
        }
        if (  result != 0 ) break;
        
    }

    priv_p->fact_irq_act_flag = 0;

    up(&priv_p->wait_fact_mutex);
    if ( result_internal < 0 ) return result_internal;
    
    if (copy_to_user( (void*)argp, &fact_info, sizeof(struct fmb_fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_fact_info)");
        return -EFAULT;
    }

    return result;
    
}
/**
*  @brief       ioctl() FMB_API_CANCEL_WAIT_FACT Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -ERESTARTSYS    internal error
*  @note        None
*  @attention   None
*/
static int fmb_api_cancel_wait_fact(struct fmb_hard_private*  hw_priv_p)
{
    int minor;
    struct fmb_core_private*    priv_p;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    priv_p->fact_irq_type = FMB_FACT_IRQ_IOCTL_CANCEL;
    priv_p->fact_irq_stop_flag = FMB_FACT_IRQ_IOCTL_CANCEL;
    priv_p->fact_irq_flag    = FMB_FACT_IRQ_ON;
    FMB_SYS_ABS_wake_up( &priv_p->fact_irq_wait );

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}


/**
*  @brief       ioctl() FMB_API_SEND_SYS_CMD Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @return      -EAGAIN     try again
*  @note        None
*  @attention   None
*/
static int fmb_api_send_sys_cmd(struct fmb_hard_private*    hw_priv_p,
                 unsigned long          argp)
{
    int minor;
    int result;
    struct fmb_cmd_sys_info cmd_sys_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb_setapicmd_param cmd_param;
#ifndef MULTIENDIAN_FUNCTION
    unsigned char   data_work[2];
#endif
    struct fmb_hw_usb* usb_priv;
    int work_ccnt;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if ( !(fmb_hw_is_exist_intr( hw_priv_p )) )
    {
        return -EPERM;
    }

    if (copy_from_user(&cmd_sys_info, (void*)argp, sizeof(struct fmb_cmd_sys_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_cmd_sys_info)");
        return -EFAULT;
    }

    if (cmd_sys_info.sync != FMB_MODE_SYNC)
    {
        info("%03d:FMB_API_SEND_SYS_CMD Warning: sync mode is FMB_MODE_SYNC only.Sync mode is changed to FMB_MODE_SYNC.", minor );
        cmd_sys_info.sync = FMB_MODE_SYNC;
    }

    /* check SYNC ASYNC mode */
    if (cmd_sys_info.sync == FMB_MODE_SYNC) {
        /* sync mode */
        if (down_interruptible(&priv_p->sys_cmd_sync_send_mutex)) {
            /** Signal(ctrl+c etc) interrupt */
            MSG(KERN_ERR_LVL, minor, "Signal interrupt in wait mutex.");
            return -ERESTARTSYS;
        }
    }

    mutex_lock(&(usb_priv->io_mutex));

    /* ToDo: start SET_API_CMD fmb_hw_usb_ep_xxxx func call */
    /* new for usb */
    memset( &cmd_param, 0, sizeof(cmd_param) );
    cmd_param.continuity_counter = priv_p->sys_cmd_continuity_counter;

    cmd_param.cmd_ID     = cmd_sys_info.id;
    cmd_param.sub_cmd_ID = cmd_sys_info.sub_id;

#ifdef MULTIENDIAN_FUNCTION
    cmd_param.body00     = cpu_to_be16( cmd_sys_info.body );
#else
    *(unsigned short *)data_work = cmd_sys_info.body;
    cmd_sys_info.body = ( (unsigned short)data_work[0] << 8 ) + data_work[1];

    cmd_param.body00     = cmd_sys_info.body;
#endif

    result = fmb_hw_usb_ep_set_api_cmd( hw_priv_p->dev_priv, &cmd_param, 10*HZ );
    mutex_unlock(&(usb_priv->io_mutex));

    priv_p->sys_cmd_continuity_counter++;
    priv_p->sys_cmd_continuity_counter &= FMB_SYS_CMD_CONTINUITY_COUNTER_MASK;

    if (cmd_sys_info.sync == FMB_MODE_SYNC) {
        /* wait for ACK receive */
#ifndef FMB_DEBUG_GDB
        result = FMB_SYS_ABS_wait_event_interruptible_timeout(priv_p->sys_cmd_ack_wait_sync,
                              priv_p->sys_cmd_ack_rcv_flag == FMB_SYS_CMD_ACK_RECEIVE,
                              (FMB_CODEC_CACK_SYS_TIME * HZ) / 1000);
#else
        result = FMB_SYS_ABS_wait_event_timeout(priv_p->sys_cmd_ack_wait_sync,
                              priv_p->sys_cmd_ack_rcv_flag == FMB_SYS_CMD_ACK_RECEIVE,
                              (FMB_CODEC_CACK_SYS_TIME * HZ) / 1000);
#endif
        priv_p->sys_cmd_ack_rcv_flag = FMB_SYS_CMD_ACK_NONE;

        if (result == 0) {
            /** system command ack receive timeout */
            up(&priv_p->sys_cmd_sync_send_mutex);
            MSG(KERN_ERR_LVL, minor, "FMB Codec LSI system command ack receive timeout.");
            return -ETIMEDOUT;
        }
        else if (result == -ERESTARTSYS) {
            /** Signal(ctrl+c etc) interrupt */
            up(&priv_p->sys_cmd_sync_send_mutex);
            MSG(KERN_ERR_LVL, minor, "Signal interrupt with system command ack receive wait.");
            return -ERESTARTSYS;
        }

        cmd_sys_info.id = priv_p->sys_cmd_ack_info.id;
        cmd_sys_info.sub_id = priv_p->sys_cmd_ack_info.sub_id;

        if (copy_to_user((void*)argp, &cmd_sys_info, sizeof(struct fmb_cmd_sys_info))) {
            up(&priv_p->sys_cmd_sync_send_mutex);
            MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_cmd_sys_info)\n");
            return -EFAULT;
        }

        up(&priv_p->sys_cmd_sync_send_mutex);
    }
    
    
    if ( priv_p->sys_cmd_continuity_counter == 0 ) work_ccnt = FMB_SYS_CMD_CONTINUITY_COUNTER_MASK;
    else work_ccnt = priv_p->sys_cmd_continuity_counter -1;
    
    if ( work_ccnt != priv_p->sys_cmd_ack_info_ccnt )
    {
        MSG( DEBUG_LVL, minor, "ACK continue conuter error before:%d, after:%d\n", work_ccnt, priv_p->sys_cmd_ack_info_ccnt );
    }

#ifdef MB86E61_FUNCTION
    if((priv_p->sys_cmd_ack_info_ccnt & GET_CMD_ACK_RESET_MASK) != 0x00){
        /* reset continuity counter */
        priv_p->sys_cmd_continuity_counter = 0x00;
    }else if((priv_p->sys_cmd_ack_info_ccnt & GET_CMD_ACK_NOT_INCREMENT_MASK) != 0x00){
        /* continuity counter not increment */
        priv_p->sys_cmd_continuity_counter = work_ccnt;
    }
    if((cmd_param.cmd_ID == 0x03) && (cmd_param.sub_cmd_ID == 0x00)){
        priv_p->sys_cmd_continuity_counter = 0;
    }
#endif /* MB86E61_FUNCTION */
    
    if ( cmd_sys_info.sub_id != 0x00 ) return -EAGAIN; /* ack error */
    
    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}

#ifdef MB86E61_FUNCTION
/**
*  @brief       ioctl() FMB_API_SEND_CTL_CMD Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @return      -ETIMEDOUT  command ack timeout
*  @return      return value of fmb_hw_usb_ep_control_cmd()
*  @note        None
*  @attention   None
*/
static int fmb_api_send_ctl_cmd(struct fmb_hard_private*    hw_priv_p,
                 unsigned long          argp)
{
    int minor;
    struct fmb_cmd_ctl_info *cmd_ctl_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    int result;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");
    cmd_ctl_info = usb_priv->cmd_ctl_info_apl;

    if ( !(fmb_hw_is_exist_intr( hw_priv_p )) )
    {
        return -EPERM;
    }

    if (copy_from_user(cmd_ctl_info, (void*)argp, sizeof(struct fmb_cmd_ctl_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_cmd_ctl_info)");
        return -EFAULT;
    }

    mutex_lock(&(usb_priv->io_mutex));

    result = fmb_hw_usb_ep_control_cmd( hw_priv_p->dev_priv, cmd_ctl_info, 10000 );

    mutex_unlock(&(usb_priv->io_mutex));

    if(result < 0){
        MSG(KERN_ERR_LVL, minor, "Failed in fmb_hw_usb_ep_control_cmd()\n");
        return result;
    }

    if (copy_to_user((void*)argp, cmd_ctl_info, sizeof(struct fmb_cmd_ctl_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_cmd_ctl_info)\n");
        return -EFAULT;
    }

    
    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}
#endif /* MB86E61_FUNCTION */

/**
*  @brief           ioctl() FMB_API_GET_LSI_REG Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   argp        command parameter pointer
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_get_lsi_reg(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    unsigned short  read_data;
    unsigned short* prm_data;
    unsigned long   counter;
    unsigned long   offset_addr;
    unsigned long   access_size;
    struct fmb_reg_info reg_info;
    struct fmb_core_private*    priv_p;

    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&reg_info, (void*)argp, sizeof(struct fmb_reg_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_reg_info)");
        return -EFAULT;
    }

    prm_data = reg_info.prm_data;
    offset_addr = reg_info.offset;
    access_size = reg_info.reg_num * sizeof(unsigned short);

    if (Fmb_hw_check_addr_lsi(hw_priv_p, offset_addr, access_size)){
        MSG(INTERNAL_ERR_LVL, minor, "Specified parameter is abnormal.");
        return -EINVAL;
    }

    for (counter = 0 ; counter < reg_info.reg_num ; counter++) {

        read_data = Fmb_hw_reg_read_lsi(hw_priv_p, offset_addr, &result_internal );
        if ( result_internal < 0 ) break;

        if (copy_to_user(prm_data, &read_data, sizeof(unsigned short))) {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_reg_info)");
            return -EFAULT;
        }
        prm_data++;
        offset_addr += 2;
    }

    if ( result_internal < 0 ) return result_internal;

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}

/**
*  @brief           ioctl() FMB_API_SET_LSI_REG Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   argp        command parameter pointer
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_set_lsi_reg(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    unsigned short  write_data;
    unsigned short* prm_data;
    unsigned long   counter;
    unsigned long   offset_addr;
    unsigned long   access_size;
    struct fmb_reg_info reg_info;
    struct fmb_core_private*    priv_p;

    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&reg_info, (void*)argp, sizeof(struct fmb_reg_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_reg_info)");
        return -EFAULT;
    }

    prm_data = reg_info.prm_data;

    offset_addr = reg_info.offset;
    access_size = reg_info.reg_num * sizeof(unsigned short);

    if (Fmb_hw_check_addr_lsi(hw_priv_p, offset_addr, access_size)){
        MSG(INTERNAL_ERR_LVL, minor, "Specified parameter is abnormal.");
        return -EINVAL;
    }

    for (counter = 0 ; counter < reg_info.reg_num ; counter++) {
        if (copy_from_user(&write_data, prm_data, sizeof(unsigned short))) {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_reg_info)");
            return -EFAULT;
        }

        Fmb_hw_reg_write_lsi(hw_priv_p, offset_addr, write_data, &result_internal );
        if ( result_internal < 0 ) break;

        prm_data++;
        offset_addr += 2;
    }

    if ( result_internal < 0 ) return result_internal;

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}

/**
*  @brief           ioctl() FMB_API_READ_I2C_REG Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   argp        command parameter pointer
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_read_i2c_reg(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result;
    struct fmb_read_i2c_info read_i2c_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&read_i2c_info, (void*)argp, sizeof(read_i2c_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_read_i2c_info)");
        return -EFAULT;
    }

	result = fmb_hw_usb_ep_control_i2c( hw_priv_p->dev_priv, 
										(read_i2c_info.acs_mode == FMB_I2C_R_REPEATED_START) ? USB_IMODE_REPEATED_START:USB_IMODE_NORMAL, 
										USB_CMDTYPE_IN, 
										read_i2c_info.adrs, 
										read_i2c_info.buf, 
										read_i2c_info.length, 
										10*HZ 
									  );

    if (copy_to_user((void *)argp, &read_i2c_info, sizeof(read_i2c_info)))
    {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_read_i2c_info)");
        result = -EFAULT;
    }

    MSG(INTERNAL_LOG, minor, "END");
    return result;
}

/**
*  @brief           ioctl() FMB_API_WRITE_I2C_REG Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   argp        command parameter pointer
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_write_i2c_reg(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result;
    struct fmb_write_i2c_info write_i2c_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&write_i2c_info, (void*)argp, sizeof(write_i2c_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_write_i2c_info)");
        return -EFAULT;
    }

	result = fmb_hw_usb_ep_control_i2c( hw_priv_p->dev_priv, 
										(write_i2c_info.acs_mode == FMB_I2C_W_WITHOUT_STOP) ? USB_IMODE_WITHOUT_STOP:USB_IMODE_NORMAL, 
										USB_CMDTYPE_OUT, 
										write_i2c_info.adrs, 
										write_i2c_info.buf, 
										write_i2c_info.length, 
										10*HZ 
									  );

    MSG(INTERNAL_LOG, minor, "END");
    return result;
}

/**
*  @brief           ioctl() FMB_API_GET_I2C_STATUS Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   argp        command parameter pointer
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_get_i2c_status(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result;
    struct fmb_get_i2c_status_info get_i2c_status_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&get_i2c_status_info, (void*)argp, sizeof(get_i2c_status_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_get_i2c_status_info)");
        return -EFAULT;
    }

	result = fmb_hw_usb_ep_control_i2c( hw_priv_p->dev_priv, 
										USB_IMODE_STATUS, 
										USB_CMDTYPE_IN, 
										get_i2c_status_info.adrs, 
										get_i2c_status_info.buf, 
										FMB_I2C_STATUS_SIZE, 
										10*HZ 
									  );

    if (copy_to_user((void *)argp, &get_i2c_status_info, sizeof(get_i2c_status_info)))
    {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_get_i2c_status_info)");
        result = -EFAULT;
    }

    MSG(INTERNAL_LOG, minor, "END");
    return result;
}

/**
*  @brief           ioctl() FMB_API_STOP_I2C Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   argp        command parameter pointer
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_stop_i2c(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result;
    unsigned char buf;
    struct fmb_stop_i2c_info stop_i2c_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&stop_i2c_info, (void*)argp, sizeof(stop_i2c_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_stop_i2c_info)");
        return -EFAULT;
    }

	result = fmb_hw_usb_ep_control_i2c( hw_priv_p->dev_priv, 
										USB_IMODE_STOP, 
										USB_CMDTYPE_OUT, 
										stop_i2c_info.adrs, 
										&buf, 
										FMB_I2C_STOP_SIZE, 
										10*HZ 
									  );

    MSG(INTERNAL_LOG, minor, "END");
    return result;
}

/**
*  @brief           ioctl() FMB_API_GET_FACTOR Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[out]      argp        command parameter pointer(struct fmb_fact_info *)
*                   [out](fmb_fact_info).status         interrupt bit data(bit 0:none 1:irq exists)
*                   [---](fmb_fact_info).(other member) none
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_get_factor(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result = 0;
    struct fmb_fact_info fact_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    
    unsigned short val_st1;
    unsigned short val_st2;
    unsigned short val_xerror;
    unsigned short val_st1_mask;
    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&fact_info, (void*)argp, sizeof(fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_factor_info)");
        return -EFAULT;
    }

    val_st1      = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_API_IRQST,  &result_internal ); if ( result_internal < 0 ) return result_internal;
    val_st2      = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_API_IRQST2, &result_internal ); if ( result_internal < 0 ) return result_internal;
    if ( ( val_st1 & FMB_FACT_STATE ) != 0 )
    {
        val_xerror = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_STATE, &result_internal );    if ( result_internal < 0 ) return result_internal;
        if ( val_xerror == 0xffff ) val_xerror = 1;
        else    val_xerror = 0;
    }
    else    val_xerror = 0;
        
    if ( val_xerror == 1 )
    {
        /* xerror : disable FMB_FACT_STATE */
        val_st1_mask = FMB_FACT_STATE;
        val_st1      &= ~val_st1_mask;
    }
    
    
    fact_info.status  = (unsigned long)val_st2 << ( 8 * 2 );
    fact_info.status += (unsigned long)val_st1;
    fact_info.status &= (unsigned long)(~FMB_FACT_XERROR);
    if ( val_xerror == 1 )
    {
        fact_info.status |= FMB_FACT_XERROR;
    }
    
    
    if (copy_to_user( (void*)argp, &fact_info, sizeof(fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_fact_info)");
        return -EFAULT;
    }
    
    
    MSG(INTERNAL_LOG, minor, "END");
    return result;
}
/**
*  @brief           ioctl() FMB_API_RESET_FACTOR Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in]       argp        command parameter pointer(struct fmb_fact_info *)
*                   [in ](fmb_fact_info).status         reset bit data(bit 0:none 1:reset)
*                   [---](fmb_fact_info).(other member) none
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_reset_factor(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result = 0;
    struct fmb_fact_info fact_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    
    unsigned short val_st1;
    unsigned short val_st2;
    
    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&fact_info, (void*)argp, sizeof(fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_factor_info)");
        return -EFAULT;
    }

    val_st1  = fact_info.status & 0xffff;
    val_st2  = fact_info.status >> ( 8 * 2 );
    val_st2 &= 0x7FFF;
    
    if ( (FMB_FACT_XERROR & fact_info.status) != 0 )
    {
        val_st1 |= FMB_FACT_STATE;
    }
    
    
    Fmb_hw_reg_write_lsi( hw_priv_p, FMB_REG_API_IRQST,  val_st1,    &result_internal ); if ( result_internal < 0 ) return result_internal;
    Fmb_hw_reg_write_lsi( hw_priv_p, FMB_REG_API_IRQST2, val_st2,    &result_internal ); if ( result_internal < 0 ) return result_internal;
    
    MSG(INTERNAL_LOG, minor, "END");
    return result;
}

/**
*  @brief           ioctl() FMB_API_GET_MASK_FACTOR Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[out]      argp        command parameter pointer(struct fmb_fact_info *)
*                   [out](fmb_fact_info).en_mask        interrupt bit mask data(bit 0:irq not mask 1:irq mask)
*                   [---](fmb_fact_info).(other member) none
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_get_mask_factor(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result = 0;
    struct fmb_fact_info fact_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    
    unsigned short val_msk1;
    unsigned short val_msk2;

    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&fact_info, (void*)argp, sizeof(fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_factor_info)");
        return -EFAULT;
    }

    val_msk1         = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_API_IRQEN,  &result_internal ); if ( result_internal < 0 ) return result_internal;
    val_msk2         = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_API_IRQEN2, &result_internal ); if ( result_internal < 0 ) return result_internal;
    
    
    fact_info.en_mask  = (unsigned long)val_msk2 << ( 8 * 2 );
    fact_info.en_mask += (unsigned long)val_msk1;
    fact_info.en_mask |= (unsigned long)(FMB_FACT_XERROR); /* forec xerror bit 1:enable */
    
    
    fact_info.en_mask = ~fact_info.en_mask; /* Mask Data <--- EnableDisable Data */
    
    if (copy_to_user( (void*)argp, &fact_info, sizeof(fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_fact_info)");
        return -EFAULT;
    }
    
    
    MSG(INTERNAL_LOG, minor, "END");
    return result;
}
/**
*  @brief           ioctl() FMB_API_SET_MASK_FACTOR Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in]       argp        command parameter pointer(struct fmb_fact_info *)
*                   [in ](fmb_fact_info).en_mask        interrupt bit mask data(bit 0:irq not mask 1:irq mask)
*                   [---](fmb_fact_info).(other member) none
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_set_mask_factor(struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result = 0;
    struct fmb_fact_info fact_info;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb* usb_priv;
    
    unsigned short val_msk1;
    unsigned short val_msk2;
    unsigned short work_val_msk2;
    
    
    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&fact_info, (void*)argp, sizeof(fact_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_factor_info)");
        return -EFAULT;
    }
    
    fact_info.en_mask = ~fact_info.en_mask; /* EnableDisable Data <--- Mask data */

    /* read IRQEN2_GPIO_BIT */
    work_val_msk2  = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_API_IRQEN2, &result_internal ); if ( result_internal < 0 ) return result_internal;
    work_val_msk2 &= 0x8000;

    val_msk1  = fact_info.en_mask & 0xffff;
    val_msk2  = fact_info.en_mask >> ( 8 * 2 );
    val_msk2 &= 0x7FFF;        /* Disregard xerror */
    val_msk2 |= work_val_msk2; /* IRQEN2_GPIO_BIT no change */
    
    
    Fmb_hw_reg_write_lsi( hw_priv_p, FMB_REG_API_IRQEN,  val_msk1,       &result_internal ); if ( result_internal < 0 ) return result_internal;
    Fmb_hw_reg_write_lsi( hw_priv_p, FMB_REG_API_IRQEN2, val_msk2,       &result_internal ); if ( result_internal < 0 ) return result_internal;
    
    MSG(INTERNAL_LOG, minor, "END");
    return result;
}
#if 0 /* During a function add */
/**
*  @brief           ioctl() FMB_API_GET_INTERNAL_DATA Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   type        data type
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
static int fmb_api_get_internal_data(struct fmb_hard_private*   hw_priv_p,
                unsigned long           argp)
{
    int minor;
    unsigned long   flags = 0;
    struct fmb_internal_info    internal_info;
    struct fmb_core_private*    priv_p;

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&internal_info, (void*)argp, sizeof(struct fmb_internal_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_internal_info)");
        return -EFAULT;
    }

    internal_info.data = -1;

    spin_lock_irqsave(&priv_p->lock_core_priv, flags);
    switch (internal_info.type) {
        case FMB_DATA_TYPE_ENCODE:
            internal_info.data = priv_p->encode_mode;
            break;
        case FMB_DATA_TYPE_NOTIFY:
            internal_info.data = priv_p->notify_mode;
            break;
        default:
            spin_unlock_irqrestore(&priv_p->lock_core_priv, flags);
            MSG(INTERNAL_ERR_LVL, minor, "Specified parameter is abnormal.");
            return -EINVAL;
    }
    spin_unlock_irqrestore(&priv_p->lock_core_priv, flags);

    if (copy_to_user((void*)argp, &internal_info, sizeof(struct fmb_internal_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed (fmb_internal_info)\n");
        return -EFAULT;
    }

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}

/**
*  @brief       ioctl() FMB_API_SET_INTERNAL_DATA Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   type        data type
*  @param[in]   data        setting data
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -EINVAL     invalid content of parameter
*  @note        None
*  @attention   None
*/
static int fmb_api_set_internal_data(struct fmb_hard_private*   hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int ret;
    unsigned long   flags = 0;
    struct fmb_internal_info    internal_info;
    struct fmb_core_private*    priv_p;

    ret = 0;
    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    MSG(INTERNAL_LOG, minor, "START");

    if (copy_from_user(&internal_info, (void*)argp, sizeof(struct fmb_internal_info))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_internal_info)");
        return -EFAULT;
    }

    spin_lock_irqsave(&priv_p->lock_core_priv, flags);
    switch (internal_info.type) {
        case FMB_DATA_TYPE_ENCODE:
            if( (internal_info.data < FMB_DATA_ENCODE_TOP) || (internal_info.data > FMB_DATA_ENCODE_BOTTOM) ) {
                ret = -EINVAL;
                break;
            }
            priv_p->encode_mode = internal_info.data;
            break;
        case FMB_DATA_TYPE_NOTIFY:
            if( (internal_info.data < FMB_DATA_NOTIFY_NONE) || (internal_info.data > FMB_DATA_NOTIFY_ALL) ) {
                ret = -EINVAL;
                break;
            }
            priv_p->notify_mode = internal_info.data;
            break;
        default:
            ret = -EINVAL;
            break;
    }
    spin_unlock_irqrestore(&priv_p->lock_core_priv, flags);

    MSG(INTERNAL_LOG, minor, "END");
    return ret;
}
#endif

/**
*   @brief      ioctl() FMB_API_GET_VERSION Command Function
*
*   @param[in]  priv_p      private data pointer
*   @param[in]  argp        command parameter pointer
*   @return     0           normal end
*   @return     -EFAULT     user buffer(argp) access error
*   @note       None
*   @attention  None
*/

static int fmb_api_get_version(struct fmb_hard_private* hw_priv_p,
                unsigned long           argp)
{
    int             minor;
    int             cnt;
    int             ret;
    unsigned short  read_firm_ver;
    unsigned char*  drv_ver_p;
    struct fmb_version_info version_info;
    struct fmb_core_private*    priv_p;
    size_t          ver_size;

    int result_internal = 0;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    ret = 0;

    MSG(DRIVER_FUNC_LVL, minor, "START");

    if( argp == 0 ){
        MSG(KERN_ERR_LVL, minor, "argument pointer is NULL.");
        return -EFAULT;
    }

    memset(&version_info, 0, sizeof(version_info) );
    
    ver_size = sizeof(VERSION);
    if ( (FMB_DRIVER_VERSION_LENGTH - 1) < ver_size )    ver_size = FMB_DRIVER_VERSION_LENGTH - 1;
    
    drv_ver_p = VERSION;
    memcpy( &(version_info.driver_ver[0]), drv_ver_p, ver_size );

    for (cnt = 0; cnt < (FMB_VERSION_REG_SIZE / sizeof(unsigned short)); cnt++) {
        read_firm_ver = Fmb_hw_reg_read_lsi(hw_priv_p, FMB_VERSION_BASE + (cnt * sizeof(unsigned short)), &result_internal);
        if ( result_internal < 0 ) break;
        memcpy(&(version_info.firm_ver[cnt]), &read_firm_ver, sizeof(unsigned short));
    }

    memcpy(version_info.spirom_ver, hw_priv_p->spirom_ver, FMB_SPIROM_VERSION_LENGTH );
    
    if (copy_to_user((void *)argp, &version_info, sizeof(version_info)))
    {
        ret = -EFAULT;
    }
    
    if ( result_internal < 0 ) return result_internal;
    
    MSG(DRIVER_FUNC_LVL, minor, "END");
    return ret;
}

/**
*  @brief       ioctl() FMB_API_GET_MSG Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -ETIMEDOUT  command ack timeout
*  @return      -ERESTARTSYS    internal error
*  @note        None
*  @attention   None
*/
static int fmb_api_get_msg(struct fmb_hard_private* hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int result;
    struct fmb_core_private*    priv_p;
    struct fmb_get_msg_info get_msg_info;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    if ( !(fmb_hw_is_exist_intr( hw_priv_p )) )
    {
        return -EPERM;
    }

    result = copy_from_user(&get_msg_info, (void*)argp, sizeof(struct fmb_get_msg_info));
    if(result != 0){
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_version_info)");
        return -EFAULT;
    }

    if ( get_msg_info.timeout == 0 )    return -EINVAL;

    priv_p->get_msg_act_flag = 1;

    /* wait for msg receive */
#ifndef FMB_DEBUG_GDB
    result = FMB_SYS_ABS_wait_event_interruptible_timeout(priv_p->sys_msg_wait,
                          priv_p->rcv_get_msg_cnt > 0,
                          (get_msg_info.timeout * HZ));
#else
    result = FMB_SYS_ABS_wait_event_timeout(priv_p->sys_msg_wait,
                          priv_p->rcv_get_msg_cnt > 0,
                          (get_msg_info.timeout * HZ));
#endif

    priv_p->sys_cmd_msg_rcv_flag = FMB_SYS_CMD_MSG_NONE;

    priv_p->get_msg_act_flag = 0;

    if (result == 0) {
        /** system command ack receive timeout */
        MSG(KERN_ERR_LVL, minor, "FMB Codec LSI system command message receive timeout.");
        return -ETIMEDOUT;
    }
    else if (result == -ERESTARTSYS) {
        /** Signal(ctrl+c etc) interrupt */
        MSG(KERN_ERR_LVL, minor, "Signal interrupt with system command message receive wait.");
        return -ERESTARTSYS;
    }

    priv_p->rcv_get_msg_cnt--;

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}

/**
*  @brief       ioctl() FMB_API_GET_EJECT_EVENT Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -ERESTARTSYS    internal error
*  @note        None
*  @attention   None
*/
static int fmb_api_get_eject_event(struct fmb_hard_private* hw_priv_p)
{
    int minor;
    int result;
    int ret;
    struct fmb_core_private*    priv_p;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    /* wait for msg receive */
#if 1
    result = FMB_SYS_ABS_wait_event(priv_p->eject_event_wait,
                          priv_p->eject_event_flag == FMB_EJECT_EVENT_ON
                          );
#else
    /* for debug */
    result = FMB_SYS_ABS_wait_event_timeout(priv_p->eject_event_wait,
                          priv_p->eject_event_flag == FMB_EJECT_EVENT_ON,
                          (15 * HZ));

#endif
    priv_p->eject_event_flag = FMB_EJECT_EVENT_OFF;

    if (result == -ERESTARTSYS) {
        priv_p->eject_event_type = FMB_EJECT_EVENT_NO_CANCEL;
        /** Signal(ctrl+c etc) interrupt */
        MSG(DEBUG_LVL, minor, "Signal interrupt with system command message receive wait.");
        return -ERESTARTSYS;
    }

    switch (priv_p->eject_event_type)
    {
        case FMB_EJECT_EVENT_REMOVE_CANCEL:
            ret = FMB_EJECT_EVENT_REMOVE_CANCEL;
            break;
        case FMB_EJECT_EVENT_IOCTL_CANCEL:
            ret = FMB_EJECT_EVENT_IOCTL_CANCEL;
            break;
        case FMB_EJECT_EVENT_NO_CANCEL:
            ret = FMB_EJECT_EVENT_NO_CANCEL;
            break;
        default:
            ret = -1;
            break;
    }
    priv_p->eject_event_type = FMB_EJECT_EVENT_NO_CANCEL;

    MSG(INTERNAL_LOG, minor, "END");
    return ret;
}

/**
*  @brief       ioctl() FMB_API_CANCEL_EJECT_EVENT Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -ERESTARTSYS    internal error
*  @note        None
*  @attention   None
*/
static int fmb_api_cancel_eject_event(struct fmb_hard_private*  hw_priv_p)
{
    int minor;
    struct fmb_core_private*    priv_p;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    priv_p->eject_event_type = FMB_EJECT_EVENT_IOCTL_CANCEL;
    priv_p->eject_event_flag = FMB_EJECT_EVENT_ON;
    FMB_SYS_ABS_wake_up( &priv_p->eject_event_wait );

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}

#ifdef MB86E61_FUNCTION
/**
*  @brief       ioctl() FMB_API_GET_NOTIFY_FAILURE_MSG Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -ERESTARTSYS    internal error
*  @note        None
*  @attention   None
*/
static int fmb_api_get_notify_failure_msg(struct fmb_hard_private* hw_priv_p)
{
    int minor;
    int result;
    int ret;
    struct fmb_core_private*    priv_p;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    /* wait for msg receive */
#if 1
    result = FMB_SYS_ABS_wait_event(priv_p->boot_event_wait,
                          priv_p->boot_event_flag == FMB_BOOT_EVENT_ON
                          );
#else
    /* for debug */
    result = FMB_SYS_ABS_wait_event_timeout(priv_p->boot_event_wait,
                          priv_p->boot_event_flag == FMB_BOOT_EVENT_ON,
                          (15 * HZ));

#endif
    priv_p->boot_event_flag = FMB_BOOT_EVENT_OFF;

    if (result == -ERESTARTSYS) {
        priv_p->boot_event_type = FMB_BOOT_EVENT_NO_CANCEL;
        /** Signal(ctrl+c etc) interrupt */
        MSG(KERN_ERR_LVL, minor, "Signal interrupt with system command message receive wait.");
        return -ERESTARTSYS;
    }

    switch (priv_p->boot_event_type)
    {
        case FMB_BOOT_EVENT_REMOVE_CANCEL:
            ret = FMB_BOOT_EVENT_REMOVE_CANCEL;
            break;
        case FMB_BOOT_EVENT_IOCTL_CANCEL:
            ret = FMB_BOOT_EVENT_IOCTL_CANCEL;
            break;
        case FMB_BOOT_EVENT_NO_CANCEL:
            ret = FMB_BOOT_EVENT_NO_CANCEL;
            break;
        default:
            ret = -1;
            break;
    }
    priv_p->boot_event_type = FMB_BOOT_EVENT_NO_CANCEL;

    MSG(INTERNAL_LOG, minor, "END");
    return ret;
}

/**
*  @brief       ioctl() FMB_API_CANCEL_NOTIFY_FAILURE_MSG Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      -ERESTARTSYS    internal error
*  @note        None
*  @attention   None
*/
static int fmb_api_cancel_notify_failure_msg(struct fmb_hard_private* hw_priv_p)
{
    int minor;
    struct fmb_core_private*    priv_p;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    priv_p->boot_event_type = FMB_BOOT_EVENT_IOCTL_CANCEL;
    priv_p->boot_event_flag = FMB_BOOT_EVENT_ON;
    FMB_SYS_ABS_wake_up( &priv_p->boot_event_wait );

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}
#endif /* MB86E61_FUNCTION */

/**
*  @brief           ioctl() FMB_API_READ_THUMB Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in,out]   argp        command parameter pointer(struct fmb_read_thmbnail *)
*                   [out](fmb_read_thmbnail).buffer            read buffer address
*                   [in ](fmb_read_thmbnail).count             read buffer size
*                   [out](fmb_read_thmbnail).actual_size       actual read data size
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @note            None
*  @attention       None
*/
static int fmb_api_read_thumb( struct fmb_hard_private*     hw_priv_p,
                unsigned long           argp)
{
    int minor;
    int ret;
    int result;
    struct fmb_read_thumbnail read_thumbnail;
    
    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    minor = hw_priv_p->minor;
    
    ret = copy_from_user(&read_thumbnail, (void*)argp, sizeof(struct fmb_read_thumbnail));
    if(ret != 0){
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_read_thumbnail)");
        return -EFAULT;
    }
    
    result = fmb_hw_usb_thumbnail_read( hw_priv_p, &read_thumbnail );
    
    ret = copy_to_user( (void*)argp, &read_thumbnail, sizeof(struct fmb_read_thumbnail));
    if(ret != 0){
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_read_thumbnail)");
        return -EFAULT;
    }

    if ( result < 0  && read_thumbnail.actual_size <= 0 ) ret = result;

    return ret;
}


#ifdef MB86E61_FUNCTION
int
fmb_hw_usb_cvt_epaddr(int minor, struct fmb_hw_usb *dev, int io_mode, int endpoint, unsigned int *pipe )
{
    
    switch( io_mode )
    {
    case USB_CMDTYPE_OUT:
        *pipe = usb_sndctrlpipe( dev->udev, endpoint );
        break;
    case USB_CMDTYPE_IN:
        *pipe = usb_rcvctrlpipe( dev->udev, endpoint );
        break;
    default:
        MSG(KERN_ERR_LVL, minor, "io mode error (%d)\n", io_mode );
        return -1;
    }
    
    return 0;
}

/**
*  @brief       ioctl() FMB_API_FIFO_CLEAR Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @param[in]   argp        command parameter pointer
*  @return      0           normal end
*  @return      -EFAULT     user buffer(argp) access error
*  @return      return value of fmb_hw_usb_cvt_epaddr()
*  @return      return value of usb_control_msg()
*  @return      return value of usb_clear_halt()
*  @note        None
*  @attention   None
*/
static int fmb_api_fifo_clear_cmd(struct fmb_hard_private*    hw_priv_p,
                 unsigned long          argp)
{
    int minor;
    int ret;
    struct fmb_core_private*    priv_p;
    struct fmb_hw_usb *usb_priv;
    struct fmb_fifoclear_info fifoclear_info;
    unsigned char   bulk_type;
    unsigned char   endpoint;
    int pipe;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );

    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );

    MSG(INTERNAL_LOG, minor, "START");

    ret = copy_from_user(&fifoclear_info, (void*)argp, sizeof(struct fmb_fifoclear_info));
    if(ret != 0){
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_fifoclear_info)");
        return -EFAULT;
    }

    bulk_type = fifoclear_info.bulk_type;
    endpoint = fifoclear_info.endpoint;

    if(bulk_type == FMB_BULK_TYPE_IN){
        endpoint = endpoint | 0x80;
        ret = fmb_hw_usb_cvt_epaddr( minor, usb_priv, USB_CMDTYPE_OUT, usb_priv->control_out_endpointAddr, &pipe );
        if(ret == 0){
            ret = usb_control_msg( usb_priv->udev, pipe, 0x03, 0x02, 0x00, 0x86, NULL, 0x00, 1000 );
            if(ret != 0){
                MSG(KERN_ERR_LVL, minor, "usb control message ret=%d\n", ret );
                return ret;
            }
        }else
        {
            MSG(KERN_ERR_LVL, minor, " usb control message : create pipe error ret=%d\n", ret );
            return ret;
        }

        ret = fmb_hw_usb_cvt_epaddr( minor, usb_priv, USB_CMDTYPE_IN, endpoint, &pipe );
        if(ret == 0){
            ret = usb_clear_halt( usb_priv->udev, pipe );
            if(ret != 0){
                MSG(KERN_ERR_LVL, minor, "usb control clear halt ep:%x io_mode:0x%x ret=%d\n", endpoint, USB_CMDTYPE_IN, ret );
                return ret;
            }
        }
        else
        {
            MSG(KERN_ERR_LVL, minor, "usb control clear halt : create pipe error ep:%x io_mode:0x%x ret=%d\n", endpoint, USB_CMDTYPE_IN, ret );
            return ret;
        }
    }else if(bulk_type == FMB_BULK_TYPE_OUT){
        ret = fmb_hw_usb_cvt_epaddr( minor, usb_priv, USB_CMDTYPE_OUT, usb_priv->control_out_endpointAddr, &pipe );
        if(ret == 0){
            ret = usb_control_msg( usb_priv->udev, pipe, 0x03, 0x02, 0x00, 0x07, NULL, 0x00, 1000 );
            if(ret != 0){
                MSG(KERN_ERR_LVL, minor, "usb control message ret=%d\n", ret );
                return ret;
            }
        }else
        {
            MSG(KERN_ERR_LVL, minor, "usb control message : create pipe error ret=%d\n", ret );
            return ret;
        }

        ret = fmb_hw_usb_cvt_epaddr( minor, usb_priv, USB_CMDTYPE_OUT, endpoint, &pipe );
        if(ret == 0){
            ret = usb_clear_halt( usb_priv->udev, pipe );
            if(ret != 0){
                MSG(KERN_ERR_LVL, minor, "usb control clear halt ep:%x io_mode:0x%x ret=%d\n", endpoint, USB_CMDTYPE_OUT, ret );
                return ret;
            }
        }
        else
        {
            MSG(KERN_ERR_LVL, minor, "usb control clear halt : create pipe error ep:%x io_mode:0x%x ret=%d\n", endpoint, USB_CMDTYPE_OUT, ret );
            return ret;
        }
    }

    MSG(INTERNAL_LOG, minor, "END");
    return ret;
}
#endif /* MB86E61_FUNCTION */

/********************************************************/
/*  systemcall basic API Function           */
/********************************************************/

/**
*  @brief           ioctl() Command Function
*  @param[in]       fmb_hard_private_data_p     hard private data pointer
*  @param[in]       cmd         Command number from application
*  @param[in,out]   arg         Argument from application
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @return          -ETIMEDOUT  timeout
*  @return          -ERESTARTSYS    internal error
*  @return          -EAGAIN     try again
*  @note            None
*  @attention       None
*/
int Fmb_api_ioctl(struct  fmb_hard_private* fmb_hard_private_data_p,
                unsigned int cmd, unsigned long arg)
{
    int minor;
    int rc;

    rc = 0;

    PRIV_P_CHECK( fmb_hard_private_data_p, -ENODEV );

    minor = fmb_hard_private_data_p->minor;
    MSG(INTERNAL_FUNC, minor, "start");

    switch (cmd) {
        case FMB_API_SEND_SYS_CMD:
            rc = fmb_api_send_sys_cmd(fmb_hard_private_data_p, arg);
            break;
#ifdef MB86E61_FUNCTION
        case FMB_API_SEND_CTL_CMD:
            rc = fmb_api_send_ctl_cmd(fmb_hard_private_data_p, arg);
            break;
#endif /* MB86E61_FUNCTION */
        case FMB_API_GET_LSI_REG:
            rc = fmb_api_get_lsi_reg(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_SET_LSI_REG:
            rc = fmb_api_set_lsi_reg(fmb_hard_private_data_p, arg);
            break;
#if 0 /* During a function add */
        case FMB_API_GET_INTERNAL_DATA:
            rc = fmb_api_get_internal_data(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_SET_INTERNAL_DATA:
            rc = fmb_api_set_internal_data(fmb_hard_private_data_p, arg);
            break;
#endif
        case FMB_API_GET_VERSION:
            rc = fmb_api_get_version(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_GET_MSG:
            rc = fmb_api_get_msg(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_GET_EJECT_EVENT:
            rc = fmb_api_get_eject_event(fmb_hard_private_data_p);
            break;
        case FMB_API_CANCEL_EJECT_EVENT:
            rc = fmb_api_cancel_eject_event(fmb_hard_private_data_p);
            break;
#ifdef MB86E61_FUNCTION
        case FMB_API_GET_NOTIFY_FAILURE_MSG:
            rc = fmb_api_get_notify_failure_msg(fmb_hard_private_data_p);
            break;
        case FMB_API_CANCEL_NOTIFY_FAILURE_MSG:
            rc = fmb_api_cancel_notify_failure_msg(fmb_hard_private_data_p);
            break;
#endif /* MB86E61_FUNCTION */
        case FMB_API_GET_PARAM:
            rc = fmb_api_get_param(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_GET_PARAM32:
            rc = fmb_api_get_param32(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_SET_PARAM:
            rc = fmb_api_set_param(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_SET_PARAM32:
            rc = fmb_api_set_param32(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_READ_I2C_REG:
            rc = fmb_api_read_i2c_reg(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_WRITE_I2C_REG:
            rc = fmb_api_write_i2c_reg(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_STOP_I2C:
            rc = fmb_api_stop_i2c(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_GET_I2C_STATUS:
            rc = fmb_api_get_i2c_status(fmb_hard_private_data_p, arg);
            break;
        case FMB_API_GET_FACTOR:
            rc = fmb_api_get_factor( fmb_hard_private_data_p, arg );
            break;
        case FMB_API_RESET_FACTOR:
            rc = fmb_api_reset_factor( fmb_hard_private_data_p, arg );
            break;
        case FMB_API_GET_MASK_FACTOR:
            rc = fmb_api_get_mask_factor( fmb_hard_private_data_p, arg );
            break;
        case FMB_API_SET_MASK_FACTOR:
            rc = fmb_api_set_mask_factor( fmb_hard_private_data_p, arg );
            break;
        case FMB_API_WAIT_FACT:
            rc = fmb_api_wait_fact( fmb_hard_private_data_p, arg );
            break;
        case FMB_API_CANCEL_WAIT_FACT:
            rc = fmb_api_cancel_wait_fact( fmb_hard_private_data_p );
            break;
        case FMB_API_GET_ASYNC_INFO:
            rc = fmb_api_get_async_info( fmb_hard_private_data_p, arg, 0 );
            break;
        case FMB_API_GET_ASYNC_INFO_SIZE:
            rc = fmb_api_get_async_info_size( fmb_hard_private_data_p, arg );
            break;
        case FMB_API_ASYNC_INFO:
            rc = fmb_api_async_info( fmb_hard_private_data_p, arg );
            break;
        case FMB_API_READ_THUMB:
            rc = fmb_api_read_thumb( fmb_hard_private_data_p, arg );
            break;
#ifdef MB86E61_FUNCTION
        case FMB_API_FIFO_CLEAR:
            rc = fmb_api_fifo_clear_cmd(fmb_hard_private_data_p, arg);
            break;
#endif /* MB86E61_FUNCTION */
        default:
            rc = -EINVAL;
            MSG(INTERNAL_ERR_LVL, minor, "ioctl():Command is not supported. ");
            break;
    }
    MSG(INTERNAL_FUNC, minor, "end");
    return rc;
}
