/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/****************************************************************************/
#ifndef __FMB_CORE_H__
#define __FMB_CORE_H__

#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/usb.h>
#include <asm/io.h>
#include "fmb_drv.h"

/********************************************************/
/*  Constant definition for ioctl()                     */
/********************************************************/
/** 
 * Async system command send
 */
#define FMB_SYS_CMD_SEND_NONE                       0
#define FMB_SYS_CMD_SEND_SYNC                       1
#define FMB_SYS_CMD_SEND_ASYNC                      2

/** 
 * system command ACK receive (for sync mode)
 */
#define FMB_SYS_CMD_ACK_NONE                        0
#define FMB_SYS_CMD_ACK_RECEIVE                     1

/** 
 * system command message receive (for sync mode)
 */
#define FMB_SYS_CMD_MSG_NONE                        0
#define FMB_SYS_CMD_MSG_RECEIVE                     1

#define FMB_EJECT_EVENT_OFF                         0
#define FMB_EJECT_EVENT_ON                          1
#define FMB_FACT_IRQ_OFF                            0
#define FMB_FACT_IRQ_ON                             1
#define FMB_ASYNC_IRQ_OFF                           0
#define FMB_ASYNC_IRQ_ON                            1
#define FMB_WAIT_BIT_OFF                            0
#define FMB_WAIT_BIT_ON                             1
#ifdef MB86E61_FUNCTION
#define FMB_BOOT_EVENT_OFF                          0
#define FMB_BOOT_EVENT_ON                           1
#endif /* MB86E61_FUNCTION */

/********************************************************/
/*  Constant definition for MB86H57 LSI core            */
/********************************************************/
/**
 * Version Info
 */
#define FMB_VERSION_BASE                            0x10
#define FMB_VERSION_FJ_3                            (FMB_VERSION_BASE + 0x00)
#define FMB_VERSION_FJ_2                            (FMB_VERSION_BASE + 0x02)
#define FMB_VERSION_FJ_1                            (FMB_VERSION_BASE + 0x04)
#define FMB_VERSION_FJ_0                            (FMB_VERSION_BASE + 0x06)
#define FMB_VERSION_3                               (FMB_VERSION_BASE + 0x08)
#define FMB_VERSION_2                               (FMB_VERSION_BASE + 0x0A)
#define FMB_VERSION_1                               (FMB_VERSION_BASE + 0x0C)
#define FMB_VERSION_0                               (FMB_VERSION_BASE + 0x0E)

/** 
 * System Command
 */
#define FMB_SYS_CMD_BASE                            0x80000
#define FMB_SYS_CMD_DESTINATION_ID                  (FMB_SYS_CMD_BASE + 0x0)
#define FMB_SYS_CMD_CONTINUITY_COUNTER              (FMB_SYS_CMD_BASE + 0x2)
#define FMB_SYS_CMD_CMD_ID                          (FMB_SYS_CMD_BASE + 0x4)
#define FMB_SYS_CMD_SUB_CMD_ID                      (FMB_SYS_CMD_BASE + 0x4)
#define FMB_SYS_CMD_BODY_00                         (FMB_SYS_CMD_BASE + 0x6)

/**
 * System Command ACK
 */
#define FMB_SYS_CMD_ACK_BASE                        0x80020
#define FMB_SYS_CMD_ACK_CONTINUITY_COUNTER          (FMB_SYS_CMD_ACK_BASE + 0x2)
#define FMB_SYS_CMD_ACK_CACK_ID                     (FMB_SYS_CMD_ACK_BASE + 0x4)
#define FMB_SYS_CMD_ACK_SUB_CACK_ID                 (FMB_SYS_CMD_ACK_BASE + 0x4)

/**
 * mask & shift
 */
#define FMB_SYS_CMD_DESTINATION_ID_MASK             0x000F
#define FMB_SYS_CMD_DESTINATION_ID_SHIFT            12
#define FMB_SYS_CMD_CONTINUITY_COUNTER_MASK         0x003F
#define FMB_SYS_CMD_CONTINUITY_COUNTER_SHIFT        0
#define FMB_SYS_CMD_CMD_ID_MASK                     0x00FF
#define FMB_SYS_CMD_CMD_ID_SHIFT                    8
#define FMB_SYS_CMD_SUB_CMD_ID_MASK                 0x00FF
#define FMB_SYS_CMD_SUB_CMD_ID_SHIFT                0
#define FMB_SYS_CMD_BODY_00_MASK                    0xFFFF
#define FMB_SYS_CMD_BODY_00_SHIFT                   0

/********************************************************/
/*  Macro definition (Register Field R/W)               */
/********************************************************/
#define SWAPW(x)    (((x>>8)&0x00ff) | ((x<<8)&0xff00))


/********************************************************/
/* Structure definition                                 */
/********************************************************/
/**
 *  @brief      The MB86H57-LSI-dependent part of private data<br>
 *  @note       None
 *  @attention  None
 */
struct fmb_core_private
{
    spinlock_t                      lock_core_priv;     /**< spin lock for this private data */

    unsigned char                   sys_cmd_continuity_counter; /**< continuity counter of system command */

#if 0 /* During a function add */
    unsigned int                    encode_mode;
    unsigned int                    notify_mode;
#endif

    int                             sys_cmd_ack_rcv_flag;
    int                             sys_cmd_send_flag;
    struct fmb_cmd_ack_sys_info     sys_cmd_ack_info;           /**< system command ack info */
    unsigned char                   sys_cmd_ack_info_ccnt;      /**< ack continuity_counter */

    /* waitqueue */
    wait_queue_head_t               sys_cmd_ack_wait_sync;  /**< wait queue for system command ACK(SYNC mode) */
    wait_queue_head_t               sys_msg_wait;           /**< wait queue for system message */
    wait_queue_head_t               eject_event_wait;       /**< wait queue for eject event */
    wait_queue_head_t               fact_irq_wait;          /**< wait queue for fact irq */
    wait_queue_head_t               wait_bit_wait;          /**< wait queue for fact irq */
#ifdef MB86E61_FUNCTION
    wait_queue_head_t               boot_event_wait;        /**< wait queue for boot event */
#endif /* MB86E61_FUNCTION */

    struct semaphore                sys_cmd_sync_send_mutex; /**< mutex for system command */
    struct semaphore                wait_fact_mutex;         /**< mutex for wait_fact command */
    struct semaphore                async_info_mutex;        /**< mutex for async_info command */

    int                             minor;          /**< minor number */
    int                             minor_num;      /**< private core data number */
    struct fmb_hard_private*        hw_priv_p;      /**< pointer to the HW private data */

    int                             get_msg_act_flag;       /**< act flag is specified whether function of fmb_api_get_msg  is called or not. */
    int                             sys_cmd_msg_rcv_flag;   /**< message receive flag */
    int                             rcv_get_msg_cnt;        /**< message count */

    int                             eject_event_flag;       /**< eject event flag */
    int                             eject_event_type;       /**< kind of eject event */
    int                             fact_irq_act_flag;      /**< act flag is specified whether function of fmb_api_wait_fact  is called or not. */
    int                             fact_irq_type;          /**< fact irq type */
    int                             fact_irq_stop_flag;     /**< fact irq stop flag */
    int                             fact_irq_flag;          /**< fact irq flag */
    int                             wait_bit_act_flag;      /**< act flag is specified whether function of fmb_spec_wait_bit  is called or not. */
    int                             wait_bit_type;          /**< wait bit type */
    int                             wait_bit_flag;          /**< wait bit flag */
#ifdef MB86E61_FUNCTION
    int                             boot_event_flag;        /**< boot event flag */
    int                             boot_event_type;        /**< kind of boot event */
#endif /* MB86E61_FUNCTION */

};


/********************************************************/
/*  External Function                                   */
/********************************************************/
/**
 *  @brief      The MB86H57-LSI part of the probe function.
 *  @param[in]  hw_priv_p   the pointer to the HW private data
 *  @return     0   success
 *  @return     -1  fail
 *  @note       None
 *  @attention  None
 */
int Fmb_core_probe( struct fmb_hard_private*    hw_priv_p );


/**
 *  @brief      Kills tasklets and releases the core private data.
 *  @param[in]  priv_p  the pointer to the core private data
 *  @return     None
 *  @note       None
 *  @attention  None
 */
void Fmb_core_remove( struct fmb_core_private* priv_p );


/**
 *  @brief      The MB86H57-LSI part of the init_module function.
 *  @param      None
 *  @return     None
 *  @note       None
 *  @attention  None
 */
void Fmb_core_init_module( void );


/**
*   @brief      MB86H57 Codec TACK interrupt TaskLet
*   @param[in]  data           the pointer to the HW private data
*   @param[in]  cack_id        the value of command ack id
*   @param[in]  sub_cack_id    the value of sub command ack id
*   @return     None
*   @note       None
*   @attention  None
*/
void fmb_core_tack_func( unsigned long data, unsigned short cack_id, unsigned short sub_cack_id, unsigned char continuity_counter );

/**
 *  @brief      The MB86H57-LSI part of the irq status Check function.
 *  @param      None
 *  @return     None
 *  @note       None
 *  @attention  None
 */
int Fmb_core_lsi_get_irq_status( struct fmb_hard_private* hw_priv_p, unsigned long *val_status );

#endif /* ifndef __FMB_CORE_H__ */
