/*
 *  drivers/mtd/brcmnand/brcmnand_base.c
 *
    Copyright (c) 2005-2007 Broadcom Corporation                 
    
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License version 2 as
 published by the Free Software Foundation.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    File: brcmnand_base.c

    Description: 
    NAND driver for Samsung K9F1G08U0A chip with Broadcom NAND controller.
    The main difference between the Broadcom controller and OneNAND is that the Broadcom
    NAND controller has only a 512B cache (bufferram) regardless of the flash chip, 
    whereas OneNAND has multiple bufferram's to match the page size.
    This complicates this driver quite a bit, because, for large page flash (2K page)
    we have to read in all 4 slides before we know for sure whether a page is bad.

when	who what
-----	---	----
051011	tht	codings derived from onenand_base implementation.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/jiffies.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <linux/byteorder/generic.h>

#include <asm/io.h>
#include <asm/bug.h>

//#include "bbm.h"

#include "brcmnand.h"

#define PRINTK(...)

#define my_be32_to_cpu(x) be32_to_cpu(x)

int gdebug=0;

// Whether we should clear the BBT to fix a previous error.
/* This will eventually be on the command line, to allow a user to 
 * clean the flash
 */
extern int gClearBBT;


#define DRIVER_NAME	"brcmnand"

#define HW_AUTOOOB_LAYOUT_SIZE		32 /* should be enough */

#define BRCMNAND_CORRECTABLE_ECC_ERROR		(1)
#define BRCMNAND_UNCORRECTABLE_ECC_ERROR	(-1)

/*
 * MTD structure for Broadcom NAND
 */
//static struct mtd_info *brcmnand_mtd = NULL;

#if 0
#ifdef CONFIG_MTD_PARTITIONS
static const char *part_probes[] = { "cmdlinepart", NULL,  };
#endif
#endif

typedef struct brcmnand_chip_Id {
    	uint8 mafId, chipId;
	char* chipIdStr;
	uint32 options;
	uint32 timing1, timing2;
} brcmnand_chip_Id;

/*
 * List of supported chip
 */
static brcmnand_chip_Id brcmnand_chips[] = {
	{	/* 0 */
		.chipId = SAMSUNG_K9F1G08U0A,
		.mafId = FLASHTYPE_SAMSUNG,
		.chipIdStr = "Samsung K9F1G08U0A",
		.options = NAND_USE_FLASH_BBT, 		/* Use BBT on flash */
				//| NAND_COMPLEX_OOB_WRITE	/* Write data together with OOB for write_oob */
		.timing1 = 00070000,
		.timing2 = 0,
	},
	{	/* 1 */
		.chipId = ST_NAND01GW3B,
		.mafId = FLASHTYPE_ST,
		.chipIdStr = "ST NAND01GW3B2B",
		.options = NAND_USE_FLASH_BBT,
		.timing1 = 0, .timing2 = 0,
	},
	{	/* 2 */
		.chipId = ST_NAND512W3A,
		.mafId = FLASHTYPE_ST,
		.chipIdStr = "ST ST_NAND512W3A",
		.options = NAND_USE_FLASH_BBT,
		.timing1 = 0, //0x6474555f, 
		.timing2 = 0, //0x00000fc7,
	},
	{	/* 3 */
		.chipId = ST_NAND256W3A,
		.mafId = FLASHTYPE_ST,
		.chipIdStr = "ST ST_NAND256W3A",
		.options = NAND_USE_FLASH_BBT,
		.timing1 = 0, //0x6474555f, 
		.timing2 = 0, //0x00000fc7,
	},
#if 0 // EOL
	{	/* 4 */
		.chipId = HYNIX_HY27UF081G2M,
		.mafId = FLASHTYPE_HYNIX,
		.chipIdStr = "HYNIX HY27UF081G2M",
		.options = NAND_USE_FLASH_BBT 
			,
	},
#endif
	/* This is the new version of HYNIX_HY27UF081G2M which is EOL.
	 * Both use the same DevID
	 */
	{	/* 4 */
		.chipId = HYNIX_HY27UF081G2A,
		.mafId = FLASHTYPE_HYNIX,
		.chipIdStr = "ST HY27UF081G2A",
		.options = NAND_USE_FLASH_BBT,
		.timing1 = 0, .timing2 = 0,
	},

	{	/* 5 */
		.chipId = MICRON_MT29F2G08AAB,
		.mafId = FLASHTYPE_MICRON,
		.chipIdStr = "MICRON_MT29F2G08AAB",
		.options = NAND_USE_FLASH_BBT,
		.timing1 = 0, .timing2 = 0,
	},
/* This is just the 16 bit version of the above?
	{
		.chipId = MICRON_MT29F2G16AAB,
		.mafId = FLASHTYPE_MICRON,
		.chipIdStr = "MICRON_MT29F2G16AAB",
		.options = NAND_USE_FLASH_BBT 
			,
	}
*/
	{	/* LAST DUMMY ENTRY */
		.chipId = 0,
		.mafId = 0,
		.chipIdStr = "UNSUPPORTED NAND CHIP",
		.options = NAND_USE_FLASH_BBT,
		.timing1 = 0, .timing2 = 0,
	}
};

// Max chip account for the last dummy entry
#define BRCMNAND_MAX_CHIPS (ARRAY_SIZE(brcmnand_chips) - 1)

/**
 * brcmnand_oob oob info for 2K page
 */
static struct nand_oobinfo brcmnand_oob_64 = {
	.useecc		= MTD_NANDECC_AUTOPLACE,
	.eccbytes	= 12,
	.eccpos		= {
		6,7,8,
		22,23,24,
		38,39,40,
		54,55,56
		},
	.oobfree	= { /* 0-1 used for BBT and/or manufacturer bad block marker, 
	                    * first slice loses 2 bytes for BBT */
				{2, 4}, {9,13}, 		/* First slice {9,7} 2nd slice {16,6}are combined */ 
									/* ST uses 6th byte (offset=5) as Bad Block Indicator, 
									  * in addition to the 1st byte, and will be adjusted at run time */
				{25, 13},				/* 2nd slice  */
				{41, 13},				/* 3rd slice */
				{57, 7},				/* 4th slice */
	                   // {0, 0}				/* There are only 8 elements */
			}
};


/**
 * brcmnand_oob oob info for 512 page
 */
static struct nand_oobinfo brcmnand_oob_16 = {
	.useecc		= MTD_NANDECC_AUTOPLACE,
	.eccbytes	= 3,
	.eccpos		= {
		6,7,8
		},
	.oobfree	= { {0, 3}, {9,7}, {0, 0} 
			   }
							    /* THT Bytes 4&5 are used by BBT.  Actually only byte 5 is used, but in order to accomodate
                                                  * for 16 bit bus width, byte 4 is also not used.  If we only use byte-width chip, then we can also 
                                                  * use byte 4 as free bytes. */
                                              

};

static const unsigned char ffchars[] = {
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,	/* 16 */
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,	/* 32 */
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,	/* 48 */
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,	/* 64 */
};

static unsigned char eccmask[64]; // Will be initialized during probe

static uint32_t brcmnand_ctrl_read(uint32_t nandCtrlReg) 
{
	volatile unsigned long* pReg = (volatile unsigned long*) (BRCMNAND_CTRL_REGS 
		+ nandCtrlReg - BCHP_NAND_REVISION);

	if (nandCtrlReg < BCHP_NAND_REVISION || nandCtrlReg > BCHP_NAND_BLK_WR_PROTECT ||
		(nandCtrlReg & 0x3) != 0) {
		printk(KERN_ERR "brcmnand_ctrl_read: Invalid register value %08x\n", nandCtrlReg);
	}
if (gdebug > 2) printk("%s[@%p]=%08x\n", __FUNCTION__, pReg, *pReg);
	return (uint32_t) /*be32_to_cpu*/(*pReg);
}


static void brcmnand_ctrl_write(uint32_t nandCtrlReg, uint32_t val) 
{
	volatile unsigned long* pReg = (volatile unsigned long*) (BRCMNAND_CTRL_REGS 
		+ nandCtrlReg - BCHP_NAND_REVISION);

if (gdebug > 2) printk("%s: cmd (%08x) = %08x\n", __FUNCTION__, (uint32_t) pReg, val);
	if (nandCtrlReg < BCHP_NAND_REVISION || nandCtrlReg > BCHP_NAND_BLK_WR_PROTECT ||
		(nandCtrlReg & 0x3) != 0) {
		printk(KERN_ERR "brcmnand_ctrl_read: Invalid register value %08x\n", nandCtrlReg);
	}
	*pReg = (volatile unsigned long)/* cpu_to_be32*/(val);
}

static void brcmnand_ctrl_writeAddr(struct brcmnand_chip* this, uint32_t offset) 
{
	uint32_t pAddr = offset + this->pbase;

//if (((offset & 0xffff0000) >= 0x02800000) && ((offset & 0xffff0000) < 0x02e60000)) {gdebug=4; dump_stack();}
//else if (gdebug==4) gdebug=0; // if we set it, reset it to 0.

if (gdebug > 2) printk("%s(addr=%08x\n", __FUNCTION__, pAddr);
	this->ctrl_write(BCHP_NAND_CMD_ADDRESS, pAddr);
}


static void print_config_regs(void)
{
	unsigned long nand_acc_control = brcmnand_ctrl_read(BCHP_NAND_ACC_CONTROL);
	unsigned long nand_config = brcmnand_ctrl_read(BCHP_NAND_CONFIG);
	unsigned long flash_id = brcmnand_ctrl_read(BCHP_NAND_FLASH_DEVICE_ID);
	unsigned long nand_timing1 = brcmnand_ctrl_read(BCHP_NAND_TIMING_1);
	unsigned long nand_timing2 = brcmnand_ctrl_read(BCHP_NAND_TIMING_2);
	
	
	printk("\nFound NAND: ACC=%08x, cfg=%08x, flashId=%08x, tim1=%08x, tim2=%08x\n", 
		nand_acc_control, nand_config, flash_id, nand_timing1, nand_timing2);	
}


void print_oobbuf(const unsigned char* buf, int len)
{
int i;


if (!buf) {printk("NULL"); return;}
 for (i=0; i<len; i++) {
  if (i % 16 == 0) printk("\n");
  else if (i % 4 == 0) printk(" ");
  printk("%02x", buf[i]);
 }
 printk("\n");
}

void print_databuf(const unsigned char* buf, int len)
{
int i;


 for (i=0; i<len; i++) {
  if (i % 32 == 0) printk("\n%04x: ", i);
  else if (i % 4 == 0) printk(" ");
  printk("%02x", buf[i]);
 }
 printk("\n");
}



/*
 * BRCMNAND controller always copies the data in 4 byte chunk, and in Big Endian mode
 * from and to the flash.
 * This routine assumes that dest and src are 4 byte aligned, and that len is a multiple of 4
 (Restriction removed)

 * TBD: 4/28/06: Remove restriction on count=512B, but do restrict the read from within a 512B section.
 * Change brcmnand_memcpy32 to be 2 functions, one to-flash, and one from-flash,
 * enforcing reading from/writing to flash on a 4B boundary, but relaxing on the buffer being on 4 byte boundary.
 */
 typedef union {
 	uint32_t u;
 	char c[4];
} u32_t;


static int brcmnand_from_flash_memcpy32(void* dest, const void* flash, int len)
{
	unsigned char* pucDest = (unsigned char*) dest; 
	unsigned char* pucFlash = (unsigned char*) flash; 
	uint32_t* pSrc;
	uint32_t* pDest;
	u32_t u32;
	int odd, i;
	
/*	
	if (unlikely(dest & 0x3)) {
		printk(KERN_ERR "brcmnand_memcpy32 dest=%p not DW aligned\n", dest);
		return -EINVAL;
	}
	if (unlikely(src & 0x3)) {
		printk(KERN_ERR "brcmnand_memcpy32 src=%p not DW aligned\n", src);
		return -EINVAL;
	}
	if (unlikely(len & 0x3)) {
		printk(KERN_ERR "brcmnand_memcpy32 len=%d not DW aligned\n", len);
		return -EINVAL;
	}
*/
	/* Take care of the leading odd bytes */
	odd = (((unsigned long) pucFlash) & 0x3);
	if (odd) {
printk("****** WARNING: leading odd bytes ************\n");
		/* Guaranteed to be valid, since cache is on 512B boundary */
		pSrc = (uint32_t*) (pucFlash - odd);

		/* pSrc is now aligned on a DW boundary, 
		 * no need to call cpu_to_be32 since we write and read from the same endian
		 */
		u32.u = /*cpu_to_be32 */(*pSrc);

		for (i=0; i<odd; i++) {
			pucDest[i] = u32.c[odd+i];
		}
		pucFlash += 4 - odd; 
		len += 4 - odd;
		pucDest += 4 - odd;
	}

	/* Copy the aligned DWs */
	pSrc = (uint32_t*) pucFlash;
	pDest = (uint32_t*) pucDest;
	for (i=0; i< (len>>2); i++) {
		pDest[i] = /* THT 8/29/06  cpu_to_be32 */(pSrc[i]);
	}

	/* Take care of the trailing odd bytes */
	odd = (len & 0x3);
	if (odd) {
		pucDest = (unsigned char*) pDest;

printk("****** WARNING: trailing odd bytes ************\n");	
		/* pSrc is now aligned on a DW boundary */
		u32.u = /*cpu_to_be32 */ (*pSrc);

		for (i=0; i<odd; i++) {
			pucDest[i] = u32.c[odd+i];
		}
	}
	return 0;
}

static int brcmnand_to_flash_memcpy32(void* flash, const void* src, int len)
{
	int i;
	uint32_t* pDest = (uint32_t*) flash;
	uint32_t* pSrc = (uint32_t*) src;
	
	if (unlikely((unsigned long) pDest & 0x3)) {
		printk(KERN_ERR "brcmnand_memcpy32 dest=%p not DW aligned\n", flash);
		return -EINVAL;
	}
	if (unlikely((unsigned long) src & 0x3)) {
		printk(KERN_ERR "brcmnand_memcpy32 src=%p not DW aligned\n", src);
		return -EINVAL;
	}
	if (unlikely(len & 0x3)) {
		printk(KERN_ERR "brcmnand_memcpy32 len=%d not DW aligned\n", len);
		return -EINVAL;
	}
if (gdebug) printk("%s: \n", __FUNCTION__);
	for (i=0; i< (len>>2); i++) {
if (gdebug) {

if (i % 8 == 0) printk("\n%04x ", i);
printk("%08x ", cpu_to_be32(pSrc[i])); // This is just for printout.  
}
		pDest[i] = /* THT: 8/29/06 cpu_to_be32  */ (pSrc[i]);
	}
	return 0;
}


/*
 * Returns	 0: No errors
 *			 1: Correctable error
 *			-1: Uncorrectable error
 */
static int brcmnand_verify_ecc(struct brcmnand_chip* this, int state, u_char* buf)
{
	int err = 0;
	uint32_t addr;

	/* Only make sense on read */
	if (state != FL_READING) 
		return 0;

	addr = this->ctrl_read(BCHP_NAND_ECC_CORR_ADDR);
	if (addr) {

		// Clear it
		this->ctrl_write(BCHP_NAND_ECC_CORR_ADDR, 0);
		printk(KERN_WARNING "BRCMNAND: Correctable ECC error at %08x\n", addr);
		err = BRCMNAND_CORRECTABLE_ECC_ERROR;
	}

	addr = this->ctrl_read(BCHP_NAND_ECC_UNC_ADDR);
	if (addr) {
		this->ctrl_write(BCHP_NAND_ECC_UNC_ADDR, 0);

		/*
		 * If the block was just erased, and have not yet been written to, this will be flagged,
		 * so this could be a false alarm
		 */

		printk(KERN_DEBUG "BRCMNAND: Possible uncorrectable ECC error at %08x\n", addr);
				
		err = BRCMNAND_UNCORRECTABLE_ECC_ERROR;
	}
	return err;
}

/**
 * brcmnand_wait - [DEFAULT] wait until the command is done
 * @param mtd		MTD device structure
 * @param state		state to select the max. timeout value
 *
 * Wait for command done. This applies to all BrcmNAND command
 * Read can take up to 53, erase up to ?s and program up to 30 clk cycle ()
 * according to general BrcmNAND specs
 */
static int brcmnand_wait(struct mtd_info *mtd, int state, uint32_t* pStatus)
{
	struct brcmnand_chip * this = mtd->priv;
	unsigned long timeout;
	uint32_t ready;

	/* The 20 msec is enough */
	timeout = jiffies + msecs_to_jiffies(3000); // THT: 3secs, for now
	while (time_before(jiffies, timeout)) {
		ready = this->ctrl_read(BCHP_NAND_INTFC_STATUS);

		if (ready & BCHP_NAND_INTFC_STATUS_CTLR_READY_MASK) {
			*pStatus = ready;
			return 0;
		}

		if (state != FL_READING)
			cond_resched();
		//touch_softlockup_watchdog();
	}

	/*
	 * Get here on timeout
	 */
	return -ETIMEDOUT;
}

static int brcmnand_ctrl_ready(struct mtd_info *mtd, int state)
{
	uint32_t status;
	int err;

	err = brcmnand_wait(mtd, state, &status);
	if (!err) {
		return (status & BCHP_NAND_INTFC_STATUS_CTLR_READY_MASK);
	}
	return 0;
}

/* 
 * Returns 	 1: Success, no errors
 * 			 0: Timeout
 *			-1: Errors
 */
static int brcmnand_cache_is_valid(struct mtd_info* mtd,  int state, int offset, u_char* buf, int raw) 
{
	struct brcmnand_chip * this = mtd->priv;
	unsigned long timeout;
	uint32_t ready;
	
	/* The 20 msec is enough */
	timeout = jiffies + msecs_to_jiffies(3000); // 3 sec timeout for now
	while (time_before(jiffies, timeout)) {
		ready = this->ctrl_read(BCHP_NAND_INTFC_STATUS);

//printk("ctrl_read(BCHP_NAND_INTFC_STATUS) returns %08x\n", ready);


		if (ready & (BCHP_NAND_INTFC_STATUS_CTLR_READY_MASK | 0x1)) {
			int ecc;
			
			if (ready & 0x1) {
				printk(KERN_ERR "%s: Flash chip report error %08x\n", __FUNCTION__, ready);
				return -1;
			}

			if (!raw) {
				ecc = brcmnand_verify_ecc(this, state, buf);
				if (ecc < 0) {
printk(KERN_DEBUG "%s: Possible Uncorrectable ECC error at offset %08x\n", __FUNCTION__, (unsigned long) offset);
					return BRCMNAND_UNCORRECTABLE_ECC_ERROR;
				}
			}
			return 1;
		}
		if (state != FL_READING)
			cond_resched();

	}

printk("%s: TIMEDOUT ready=%08x\n", __FUNCTION__, ready);
	return 0; // TimeOut
}

/* 
 * Returns 	 1: Success, no errors
 * 			 0: Timeout
 *			-1: Errors
 */
static int brcmnand_spare_is_valid(struct mtd_info* mtd,  int state, int offset, int raw) 
{
	struct brcmnand_chip * this = mtd->priv;
	unsigned long timeout;
	uint32_t ready;

	/* The 20 msec is enough */
	timeout = jiffies + msecs_to_jiffies(3000);  // 3 sec timeout for now
	while (time_before(jiffies, timeout)) {
		ready = this->ctrl_read(BCHP_NAND_INTFC_STATUS);

		if (ready & BCHP_NAND_INTFC_STATUS_CTLR_READY_MASK) {
			int ecc;

			if (!raw) {
				ecc = brcmnand_verify_ecc(this, state, NULL);
				if (ecc < 0) {
					printk("%s: Uncorrectable ECC error at offset %08x\n", __FUNCTION__, (unsigned long) offset);
					return -1;
				}
			}
			return 1;
		}
		if (state != FL_READING)
			cond_resched();
	}

printk("%s: TIMEDOUT: ready=%08x\n", __FUNCTION__, ready);
	return 0; // Timed out
}

/*
 * Returns 0 = Complete with no error
 * 1 = Write complete, need BBT
 * err < 0: Other errors
 */
static int brcmnand_wait_for_write(struct mtd_info *mtd)
{
	uint32_t status;
	uint32_t flashStatus = 0;
	int err;

	err = brcmnand_wait(mtd, FL_WRITING, &status);
	if (!err) {
		if (status & BCHP_NAND_INTFC_STATUS_CTLR_READY_MASK) {
			flashStatus = status & 0x01;
			return flashStatus; // 0 = write completes with no errors
		}
		else {
			err = -ETIMEDOUT;
		}
	}
	return 1;
}

static int brcmnand_write_is_complete(struct mtd_info *mtd, int* outp_needBBT)
{
	int err;
	uint32_t status;
	uint32_t flashStatus = 0;

	*outp_needBBT = 1;
	err = brcmnand_wait(mtd, FL_WRITING, &status);
	if (!err) {
		if (status & BCHP_NAND_INTFC_STATUS_CTLR_READY_MASK) {
			flashStatus = status & 0x01;
			*outp_needBBT = flashStatus; // 0 = write completes with no errors
			return 1;
		}
		else {
			return 0;
		}
	}
	return 0;
}




#if 0

/* Compare the OOB areas, skipping over the ECC, and OOB bytes */
static int oobcmp(const unsigned char* buf1, const unsigned char* buf2, int len, struct nand_oobinfo* oobsel)
{
	int i, ret, from, num;

	/* Walk through the autoplace chunks */
	for (i = 0; oobsel->oobfree[i][1] && i < 8; i++) {
		from = oobsel->oobfree[i][0];
		num = oobsel->oobfree[i][1];

		if (from >=len) break;
		
		ret = memcmp(&buf1[from], &buf2[from], num);
		if (ret) {
printk("%s: oobfree differ around offset %d, len=%d\n", __FUNCTION__, from, num);
printk("left\n"); print_oobbuf(buf1, len);
printk("right\n"); print_oobbuf(buf2, len);
			return ret;
		}
	}
	/* Compare the OOB bytes.  For now, just compare the first 2 bytes */
	ret = memcmp(buf1, buf2, 2);
	if (ret) {
printk("%s: OOB bytes differ, left=%02x%02x right=%02x%02x\n", 
__FUNCTION__, buf1[0], buf1[1], buf2[0], buf2[1]);
	}
	return ret;
}
#endif


/* Compare the OOB areas, skipping over the ECC, and BI bytes */

static int ecccmp(struct mtd_info* mtd, int offset, const unsigned char* left, const unsigned char* right, int len)
{
	struct brcmnand_chip *this = mtd->priv;
	int i, ret=0;
	int sliceOffset = (offset & ~ (mtd->eccsize - 1));  // Beginning of slice
	int pageOffset = (offset & ~ (mtd->oobblock - 1)); // Beginning of page
	int sliceNum = (sliceOffset - pageOffset) / mtd->eccsize; // Slice number 0-3
	int oobOffset = sliceNum * mtd->eccsize;

	/* Walk through the autoplace chunks */
	for (i = 0; i < len; i++) {
		if ((left[i] & eccmask[i+oobOffset]) != (right[i] & eccmask[i+oobOffset])) {
			ret = 1;
			break;
		}
	}
	
	if (ret) {
printk("%s: OOB bytes differ, offset=%08x, oobblock=%d, eccsize=%d, \naround sliceoffset=%d, bufOff=%d, len=%d left=%02x, right=%02x mask=%02x\ncmpmask=\n", 
__FUNCTION__, offset, mtd->oobblock, mtd->eccsize, oobOffset, i, len, left[i], right[i], eccmask[i+oobOffset]);
print_oobbuf(eccmask, mtd->oobsize);
printk("\nleft=\n");
print_oobbuf(left, len);
printk("\nright=\n");
print_oobbuf(right, len);
	}
	return ret;
}


/**
 * brcmnand_convert_oobfree_to_fsbuf - [GENERIC] Read 1 page's oob buffer into fs buffer
 * @param mtd		MTD device structure
 * @oob_buf			IN: The raw OOB buffer read from the NAND flash
 * @param fsbuf		OUT: the converted file system oob buffer
 * @oob_buf			raw oob buffer read from flash
 * returns the number of bytes converted.
 */
static int brcmnand_convert_oobfree_to_fsbuf(struct mtd_info* mtd, 
	u_char* raw_oobbuf, u_char* out_fsbuf, int fslen, struct nand_oobinfo* oobsel, int autoplace) 
{
	int i;
	int len = 0;

	/* without autoplace. Legacy mode used by YAFFS1 */
	switch(oobsel->useecc) {
	case MTD_NANDECC_AUTOPLACE:
	case MTD_NANDECC_AUTOPL_USR:
if (gdebug > 2) printk("%s: MTD_NANDECC_AUTOPLACE|USR\n", __FUNCTION__);
if (gdebug > 2) print_oobbuf(raw_oobbuf, mtd->oobsize);
		/* Walk through the autoplace chunks */
		for (i = 0; oobsel->oobfree[i][1] && i < ARRAY_SIZE(oobsel->oobfree); i++) {
			int from = oobsel->oobfree[i][0];
			int num = oobsel->oobfree[i][1];
			int clen;

			clen = min_t(int, num, fslen-len);
if (gdebug > 2) {
printk("%s: i=%d, from=%d, num=%d, len=%d, clen=%d\n", __FUNCTION__, i, from, num, len, clen);
print_oobbuf(&raw_oobbuf[from], clen);
}			
			memcpy(&out_fsbuf[len], &raw_oobbuf[from], clen);
			len += clen;
		}
		break;
	case MTD_NANDECC_PLACE:
if (gdebug > 2) printk("%s: MTD_NANDECC_PLACE\n", __FUNCTION__);
		len = mtd->oobsize;
		memcpy(out_fsbuf, raw_oobbuf, len);
		break;
		
	default:
		printk(KERN_ERR "%s: Invalid useecc %d not supported\n",
			__FUNCTION__, oobsel->useecc);
		len = 0;
	}
	return len;

}




/**
 * brcmnand_prepare_oobbuf - [GENERIC] Prepare the out of band buffer
 * @mtd:	MTD device structure
 * @fsbuf:	buffer given by fs driver (input oob_buf)
 * @fslen:	size of buffer
 * @oobsel:	out of band selection structure
 * @oob_buf:	Output OOB buffer
 * @autoplace:	1 = place given buffer into the free oob bytes
 * @numpages:	number of pages to prepare. The Brcm version only handles 1 page at a time.
 * @retlen:	returned len, i.e. number of bytes copied into flash.
 *
 * Return:
 * 1. Filesystem buffer available and autoplacement is off,
 *    return filesystem buffer
 * 2. No filesystem buffer or autoplace is off, return internal
 *    buffer
 * 3. Filesystem buffer is given and autoplace selected
 *    put data from fs buffer into internal buffer and
 *    retrun internal buffer
 *
 * Note: The internal buffer is filled with 0xff. This must
 * be done only once, when no autoplacement happens
 * Autoplacement sets the buffer dirty flag, which
 * forces the 0xff fill before using the buffer again.
 *
 * 
 *
*/
static u_char * 
brcmnand_prepare_oobbuf (
		struct mtd_info *mtd, u_char *fsbuf, int fslen, u_char* oob_buf, struct nand_oobinfo *oobsel,
		int autoplace, int* retlen)
{
	struct brcmnand_chip *this = mtd->priv;
	int i, len, ofs;

	*retlen = 0;

if (gdebug > 2) printk("--> %s.  autoplace=%d, fslen=%d, fsbuf=@%p\n", __FUNCTION__, autoplace, fslen, fsbuf);
if (gdebug > 2 && fsbuf) print_oobbuf(fsbuf, fslen);

	/* Zero copy fs supplied buffer */
	if (fsbuf && !autoplace) {
		*retlen = fslen;
		return fsbuf;
	}

	/* Check, if the buffer must be filled with ff again */
	//if (this->oobdirty) 
	{
		memset (oob_buf, 0xff, mtd->oobsize );
		this->oobdirty = 0;
	}

	/* If we have no autoplacement or no fs buffer use the internal one */
	if (!autoplace || !fsbuf) {
		*retlen = 0;
		return oob_buf;
	}

	/* Walk through the pages and place the data */
	this->oobdirty = 1;
	ofs = 0;
	//while (numpages--) 
	{
		for (i = 0, len = 0; len < mtd->oobavail && len < fslen && i < ARRAY_SIZE(oobsel->oobfree); i++) {
			int to = ofs + oobsel->oobfree[i][0];
			int num = oobsel->oobfree[i][1];

#if 1
			if ((len + num) > fslen) {

				memcpy(&oob_buf[to], &fsbuf[*retlen], fslen-len);
				*retlen += fslen-len;
if (gdebug > 2) {printk("%s: partial copy i=%d, %d oob len=%d free %d, fslen %d, retlen=%d\n", __FUNCTION__, 
	i, fslen-len, len, num, fslen, *retlen);
print_oobbuf(&fsbuf[len], fslen-len);
}
				break;
			}
#endif			
if (gdebug > 2) {
printk("%s: i=%d, to=%d, num=%d, len=%d retlen=%d\n", __FUNCTION__, i, to, num, len, *retlen);
print_oobbuf(&fsbuf[len], num);
}
			memcpy (&oob_buf[to], &fsbuf[*retlen], num);
			len += num;
			*retlen += num;
		}
		ofs += mtd->oobavail;
	}
if (fslen != *retlen) printk("%s: fslen=%d, retlen=%d\n", __FUNCTION__, fslen, *retlen);
if (gdebug > 2) { printk("<-- %s: outbuf=\n", __FUNCTION__);
print_oobbuf(oob_buf, 64); }
	return oob_buf;
}



/**
 * brcmnand_posted_read_cache - [BrcmNAND Interface] Read the 512B cache area
 * Assuming brcmnand_get_device() has been called to obtain exclusive lock
 * @param mtd		MTD data structure
 * @param oobarea	Spare area, pass NULL if not interested
 * @param buffer	the databuffer to put/get data, pass NULL if only spare area is wanted.
 * @param offset	offset to read from or write to, must be 512B aligned.
 * @param raw: Ignore BBT bytes when raw = 1
 *
 * Caller is responsible to pass a buffer that is
 * (1) large enough for 512B for data and optionally an oobarea large enough for 16B.
 * (2) 4-byte aligned.
 *
 * Read the cache area into buffer.  The size of the cache is mtd-->eccsize and is always 512B.
 */
static int brcmnand_posted_read_cache(struct mtd_info* mtd, 
		void* buffer, u_char* oobarea, int offset, int raw)
{
	struct brcmnand_chip* this = mtd->priv;
	int sliceOffset = (offset & ~ (mtd->eccsize - 1));
	int i, ret;
	int retries = 2, done = 0;
	uint32_t* p32 = (uint32_t*) oobarea;
static int count;


//if (offset == 0x002a0000) gdebug=3;
//if ((((unsigned int) offset & 0xffff0000) >= 0x02800000) && (((unsigned int) offset & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x\n", __FUNCTION__, (unsigned int) offset);

while (retries > 0 && !done) {
	if (unlikely(sliceOffset != offset)) {
		printk(KERN_ERR "%s: offset %08x is not cache aligned, sliceOffset=%08x, CacheSize=%d\n", 
			__FUNCTION__, (unsigned int) offset, sliceOffset, mtd->eccsize);
		return -EINVAL;
	}
if (gdebug) {
printk("%s: sliceoffset=%08x, offset=%08x\n", __FUNCTION__, sliceOffset, offset);
}

//printk("Calling ctrl_writeAddr, sliceOffset=%08x\n", sliceOffset);
	this->ctrl_writeAddr(this, sliceOffset);
	this->ctrl_write(BCHP_NAND_CMD_START, OP_PAGE_READ);

//printk("Calling ctrl_writeAddr .... done\n");

	// Wait until cache is filled up
	switch (brcmnand_cache_is_valid(mtd, FL_READING, sliceOffset, this->vbase+offset, raw)) {
	case 1: /* Success, no errors */
//printk("PR: copy data from %08x, len=%d, vbase=%08x, offset=%d\n", this->vbase+offset, len, this->vbase, offset);
if (gdebug) {
printk("%s: after read offset=%08x\n", __FUNCTION__, offset);
print_databuf(this->vbase+offset, mtd->eccsize);
}
		if (buffer) {
			brcmnand_from_flash_memcpy32(buffer, this->vbase+offset, mtd->eccsize);
if (gdebug) {
printk("%s: after copy from flash offset=%08x\n", __FUNCTION__, offset);
print_databuf(buffer, mtd->eccsize);
}
		}

		if (oobarea) {

			for (i = 0; i < 4; i++) {
				p32[i] = /* THT 11-30-06 */ be32_to_cpu (this->ctrl_read(BCHP_NAND_SPARE_AREA_READ_OFS_0 + i*4));
			}
if (gdebug && count++ < 20) /* Conversion is just for display */
printk("PR: OOB[%d]=%08x, %08x, %08x, %08x\n", i, 
my_be32_to_cpu(p32[0]), my_be32_to_cpu(p32[1]), my_be32_to_cpu(p32[2]), my_be32_to_cpu(p32[3]));
			
		}

		ret = 0; // Success
		done = 1;
		break;
		
	case BRCMNAND_UNCORRECTABLE_ECC_ERROR:
		{
			/* Flash chip returns errors 

			|| There is a bug in the controller, where if one reads from an erased block that has NOT been written to,
			|| this error is raised.  
			|| (Writing to OOB area does not have any effect on this bug)
			|| The workaround is to also look into the OOB area, to see if they are all 0xFF
			
			*/
			u_char oobbuf[16];
			int all00, allFF;

			if (!oobarea) 
				oobarea = &oobbuf[0];
			p32 = (uint32_t*) oobarea;

			for (i = 0; i < 4; i++) {
				p32[i] = /* THT 11-30-06 */ be32_to_cpu (this->ctrl_read(BCHP_NAND_SPARE_AREA_READ_OFS_0 + i*4));
			}

			all00 = (oobarea[6] == 0xff && oobarea[7] == 0xff && oobarea[8] == 0xff);
			allFF = (oobarea[6] == 0x00 && oobarea[7] == 0x00 && oobarea[8] == 0x00);
			if ( all00 || allFF) {
				/* 
				 * For the first case, the slice is an erased block, and the ECC bytes are all 0xFF,
				 * for the 2nd, all bytes are 0xFF, so the Hamming Codes for it are all zeroes.
				 * The current version of the BrcmNAND controller treats these as un-correctable errors.
				 * For either case, fill data buffer with 0x00 or 0xff and return success.  The error has already
				 * been cleared inside brcmnand_verify_ecc.
				 * Both case will be handled correctly by the BrcmNand controller in later releases.
				 */
				p32 = (uint32_t*) buffer;
				for (i=0; i < this->eccsize/4; i++) {
					p32[i] = 0xFFFFFFFF;
				}
				ret = 0; // Success

			}
			else {
				/* Real error: Disturb read returns uncorrectable errors */
printk("%s: Uncorrectable ECC errors at %08x, oob=\n", __FUNCTION__, (unsigned int) offset);
print_oobbuf(oobarea, ARRAY_SIZE(oobbuf));
				ret = -EBADMSG; 
			}
			done = 1;
			
			break;
		}
		
	case 0:
		//Read has timed out 
		printk(KERN_ERR "brcmnand_cache_is_valid returns 0\n");
		ret = -ETIMEDOUT;
		done = 1;
		break;

	default:
		BUG_ON(1);
		/* Should never gets here */
		ret = -EINVAL;
		done = 1;
		break; 
	}
}

//gdebug = 0;
	return ret;
}

/**
 * brcmnand_posted_read_oob - [BrcmNAND Interface] Read the spare area
 * @param mtd		MTD data structure
 * @param oobarea	Spare area, pass NULL if not interested
 * @param offset	offset to read from or write to
 *
 * This is a little bit faster than brcmnand_posted_read, making this command useful for improving
 * the performance of BBT management.
 * The 512B flash cache is invalidated.
 *
 * Read the cache area into buffer.  The size of the cache is mtd->oobblock and is always 512B,
 * for this version of the BrcmNAND controller.
 */
static int brcmnand_posted_read_oob(struct mtd_info* mtd, 
		unsigned char* oobarea, int offset, int raw)
{
	struct brcmnand_chip* this = mtd->priv;
	int sliceOffset = (offset & ~ (mtd->eccsize - 1));
	int i, ret, done = 0;
	int retries = 5;

//if ((((unsigned int) offset & 0xffff0000) >= 0x02800000) && (((unsigned int) offset & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x\n", __FUNCTION__, (unsigned int) offset);

while (retries > 0 && !done) {
	if (unlikely(sliceOffset != offset)) {
		printk(KERN_ERR "%s: offset %08x is not cache aligned\n", 
			__FUNCTION__, (unsigned int) offset);
		return -EINVAL;
	}

	this->ctrl_writeAddr(this, sliceOffset);
	this->ctrl_write(BCHP_NAND_CMD_START, OP_SPARE_AREA_READ);

	// Wait until spare area is filled up
	switch (brcmnand_spare_is_valid(mtd, FL_READING, sliceOffset, raw)) {
	case 1:
		if (oobarea) {
//static int count;
			uint32_t* p32 = (uint32_t*) oobarea;
			
			for (i = 0; i < 4; i++) {
				p32[i] = /* THT 11-30-06 */ be32_to_cpu /**/(this->ctrl_read(BCHP_NAND_SPARE_AREA_READ_OFS_0 + (i<<2)));
			}
#if 0
//if (p32[0] != 0xffffffff || p32[1] != 0xffffffff || p32[2] != 0xffffffff || p32[3] != 0xffffffff)
if (count++ < 20)
printk("OOB-RD: %08x %08x %08x %08x\n", 
my_be32_to_cpu(p32[0]), my_be32_to_cpu(p32[1]), my_be32_to_cpu(p32[2]), my_be32_to_cpu(p32[3]));
#endif
		}
		
		ret = 0;
		done = 1;
		break;

	case -1:
		ret = -EBADMSG;
		/* brcmnand_spare_is_valid also clears the error bit, so just retry it */
		retries--;
		break;
		
	case 0:
		//Read has timed out or read found bad block. TBD: Find out which is which
		ret = -ETIMEDOUT;
		done = 1;
		break;
		
	default:
		BUG_ON(1);
		/* NOTREACHED */
		ret = -EINVAL;
		done = 1;
		break; /* Should never gets here */
	}

}	

	return ret;
}





/**
 * brcmnand_posted_write - [BrcmNAND Interface] Write a buffer to the flash cache
 * Assuming brcmnand_get_device() has been called to obtain exclusive lock
 *
 * @param mtd		MTD data structure
 * @param buffer	the databuffer to put/get data
 * @param oobarea	Spare area, pass NULL if not interested
 * @param offset	offset to write to, and must be 512B aligned
 *
 * Write to the cache area TBD 4/26/06
 */
static int brcmnand_posted_write_cache(struct mtd_info *mtd,
		const void* buffer, const unsigned char* oobarea, int offset)
{
	struct brcmnand_chip* this = mtd->priv;
	int sliceOffset = (offset & ~ (mtd->eccsize - 1));
	uint32_t* p32;
	int i, needBBT=0;
	int ret;

//if ((((unsigned int) offset & 0xffff0000) >= 0x02800000) && (((unsigned int) offset & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x\n", __FUNCTION__, (unsigned int) offset);

#if 0 // Will rely on the flash status for error detection //def CONFIG_MTD_BRCMNAND_VERIFY_WRITE
// Debugging
char* inbuf; //[512];
char* outbuf; //[512];
char inoob[16];
char outoob[16];

if (gdebug){ printk("--> %s\n", __FUNCTION__);}
inbuf = kmalloc(512, GFP_KERNEL);
outbuf = kmalloc(512, GFP_KERNEL);
if (buffer) memcpy(inbuf, buffer, 512);
if (oobarea) memcpy(inoob, oobarea, 16);

#endif

if (gdebug) printk("brcmnand_posted_write_cache buf=%p, offset=%08x, oob=%p\n", buffer, offset, oobarea);

	if (unlikely(sliceOffset != offset)) {
		printk(KERN_ERR "%s: offset %08x is not cache aligned\n", 
			__FUNCTION__, (unsigned int) offset);
gdebug = 0;
		ret =  -EINVAL;
		goto out;
	}
//gdebug = 1;
	this->ctrl_writeAddr(this, sliceOffset);

	if (buffer) {
//gdebug=1;
		brcmnand_to_flash_memcpy32(this->vbase+offset, buffer, mtd->eccsize);
gdebug=0;
//printk("\n%s: after toflash_memcopy:\n", __FUNCTION__);
//print_databuf(inbuf, mtd->eccsize);
	}
	/* Must write data when NAND_COMPLEX_OOB_WRITE */
	else if (this->options & NAND_COMPLEX_OOB_WRITE) {
		brcmnand_to_flash_memcpy32(this->vbase+offset, ffchars, mtd->eccsize);
	}

	if (oobarea) {
		p32 = (uint32_t*) oobarea;

#if  0
if (gdebug) {printk("PW: oob=");
for (i = 0; i < 4; i++) {
printk("%04x ", my_be32_to_cpu(p32[i]));
}
printk("\n");
}
#endif

	}
	else {
		// Fill with 0xFF if don't want to change OOB
		p32 = (uint32_t*) &ffchars[0];
	}

	for (i = 0; i < 4; i++) {
		this->ctrl_write(BCHP_NAND_SPARE_AREA_WRITE_OFS_0 + i*4, /* THT 11-30-06 */ cpu_to_be32 /* */(p32[i]));
	}

	this->ctrl_write(BCHP_NAND_CMD_START, OP_PROGRAM_PAGE);
//gdebug = 0;

	// Wait until flash is ready
	if (brcmnand_write_is_complete(mtd, &needBBT)) {
		if (!needBBT) {

#if 0// Cannot compare OOB with just 1 slice.  Must compare on whole page or compare everyting except  ECC
unsigned long writeAddr, readAddr;
// Debug, reading it back.
//writeAddr = this->ctrl_read(BCHP_NAND_PROGRAM_PAGE_ADDR);
//gdebug=1;
			brcmnand_posted_read_cache(mtd, outbuf, outoob, offset, 0);
//readAddr = this->ctrl_read(BCHP_NAND_FLASH_READ_ADDR);
//gdebug=0;
//if (gdebug) printk("writeAddr=%08x, readAddr=%08x\n", writeAddr, readAddr);

#ifdef CONFIG_MTD_BRCMNAND_VERIFY_WRITE
			if (oobarea && ecccmp(mtd, offset, inoob, outoob, 16)) {
printk("%s: input buffer:\n", __FUNCTION__);
print_oobbuf(inbuf, 16);
printk("%s: output buffer:\n", __FUNCTION__);
print_oobbuf(outbuf, 16);
			}
			if (inbuf && memcmp(inbuf, outbuf, 512)) {
printk("%s: in/out data buffers differ, in =\n", __FUNCTION__);
print_databuf(inbuf, 512);
printk("%s: output buffer:\n", __FUNCTION__);
print_databuf(outbuf, 512);
			}

#endif
#endif // #if 0

if (gdebug) printk("<-- %s\n", __FUNCTION__);		
//gdebug = 0;
			ret = 0;
			goto out;
		}
	
		else { // Need BBT
			printk(KERN_WARNING "%s: Marking bad block @%08x\n", __FUNCTION__, (unsigned int) offset);
			ret = this->block_markbad(mtd, offset);
//gdebug = 0;
			ret = -EINVAL;
			goto out;
		}
	}
	//Write has timed out or read found bad block. TBD: Find out which is which
	printk(/*KERN_INFO*/ "%s: Timeout\n", __FUNCTION__);
//gdebug = 0;
	ret = -ETIMEDOUT;

out:
#if 0 //def CONFIG_MTD_BRCMNAND_VERIFY_WRITE
	kfree(inbuf);
	kfree(outbuf);
#endif
	return ret;
}

/**
 * brcmnand_posted_write_oob - [BrcmNAND Interface] Write the spare area
 * @param mtd		MTD data structure
 * @param oobarea	Spare area, pass NULL if not interested.  Must be able to 
 *					hold mtd->oobsize (16) bytes.
 * @param offset	offset to write to, and must be 512B aligned
 *
 */
static int brcmnand_posted_write_oob(struct mtd_info *mtd,
		const unsigned char* oobarea, int offset)
{
	struct brcmnand_chip* this = mtd->priv;
	int sliceOffset = (offset & ~ (mtd->eccsize - 1));
	uint32_t* p32;
	int i, needBBT=0;
static int count;

//if ((((unsigned int) offset & 0xffff0000) >= 0x02800000) && (((unsigned int) offset & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x\n", __FUNCTION__, (unsigned int) offset);

	if (unlikely(sliceOffset != offset)) {
		printk(KERN_ERR "%s: offset %08x is not cache aligned\n", 
			__FUNCTION__, (unsigned int) offset);
	}
	
	this->ctrl_writeAddr(this, sliceOffset);

	/* Must write data when NAND_COMPLEX_OOB_WRITE option is set */
	if (this->options & NAND_COMPLEX_OOB_WRITE) {
		brcmnand_to_flash_memcpy32(this->vbase+offset, ffchars, mtd->eccsize);
	}

	// assert oobarea here
	BUG_ON(!oobarea);	
	p32 = (uint32_t*) oobarea;
		
	for (i = 0; i < 4; i++) {
		this->ctrl_write(BCHP_NAND_SPARE_AREA_WRITE_OFS_0 + i*4, /* THT 11-30-06 */ cpu_to_be32 (p32[i]));
	}

if (gdebug && count++ < 20) 
printk("OOB-WR: %08x %08x %08x %08x\n", 
my_be32_to_cpu(p32[0]), my_be32_to_cpu(p32[1]), my_be32_to_cpu(p32[2]), my_be32_to_cpu(p32[3]));
	

	if (this->options & NAND_COMPLEX_OOB_WRITE) {
printk("****** Workaround, using OP_PROGRAM_PAGE instead of OP_PROGRAM_SPARE_AREA\n");
		this->ctrl_write(BCHP_NAND_CMD_START, OP_PROGRAM_PAGE);
	}
	else {
		this->ctrl_write(BCHP_NAND_CMD_START, OP_PROGRAM_SPARE_AREA);
	}

	// Wait until flash is ready
	if (brcmnand_write_is_complete(mtd, &needBBT)) {
//gdebug = 0;
		return 0;
	}
	if (needBBT){
		int ret;
		
		printk(KERN_WARNING "%s: Marking bad block @%08x\n", __FUNCTION__, (unsigned int) offset);
		ret = this->block_markbad(mtd, offset);
		return -EINVAL;
	}
//gdebug = 0;	
	return -ETIMEDOUT;
	
}



/**
 * brcmnand_get_device - [GENERIC] Get chip for selected access
 * @param mtd		MTD device structure
 * @param new_state	the state which is requested
 *
 * Get the device and lock it for exclusive access
 */
static int brcmnand_get_device(struct mtd_info *mtd, int new_state)
{
	struct brcmnand_chip *this = mtd->priv;
	DECLARE_WAITQUEUE(wait, current);

	/*
	 * Grab the lock and see if the device is available
	 */
	while (1) {
		spin_lock(&this->chip_lock);
#if 0
		if (atomic_dec(this->semCount)) == 0) {
			// Write 1 to Semaphore register TBD
			this->ctrl_write(BCHP_NAND_SEMAPHORE, 1);
		}
		else { /* In Use by another thread */
			atomic_inc(this->semCount);
			continue;
		}
#endif
		if (this->state == FL_READY) {
			this->state = new_state;
			spin_unlock(&this->chip_lock);
			break;
		}
		if (new_state == FL_PM_SUSPENDED) {
			spin_unlock(&this->chip_lock);
			return (this->state == FL_PM_SUSPENDED) ? 0 : -EAGAIN;
		}
		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&this->wq, &wait);
		spin_unlock(&this->chip_lock);
		schedule();
		remove_wait_queue(&this->wq, &wait);
	}

	return 0;
}

/**
 * brcmnand_release_device - [GENERIC] release chip
 * @param mtd		MTD device structure
 *
 * Deselect, release chip lock and wake up anyone waiting on the device
 */
static void brcmnand_release_device(struct mtd_info *mtd)
{
	struct brcmnand_chip *this = mtd->priv;

	/* Release the chip */
	spin_lock(&this->chip_lock);
	this->state = FL_READY;
	wake_up(&this->wq);
	spin_unlock(&this->chip_lock);
}



/**
 * brcmnand_read_page - [GENERIC] Read 1 page data and the page OOB into internal buffer.
 * @param mtd		MTD device structure
 * @param offset		offset from start of flash
 * @param data_buf	Data buffer to be read to, must be at least mtd->oobblock
 * @param len		Length of data buffer, must be equal or less than page size
 * @param oob_buf	OOB buffer, must be at least mtd->oobsize
 * Assume that device lock has been held
 * returns0 on success
 */
static int brcmnand_read_page(struct mtd_info* mtd, 
		unsigned int offset, u_char* data_buf, int len, u_char* oob_buf, 
		struct nand_oobinfo* oobsel, int* retlen, int* retooblen)
{
	struct brcmnand_chip *this = mtd->priv;
	int dataRead = 0, oobRead = 0;
	int winslice;
	int oob = 0;
	int ret;
	int autoplace; 

if (gdebug) printk("--> %s(offset=%08x, len=%d, oob_buf=%p\n", __FUNCTION__, offset, len, oob_buf);
	*retlen = 0;
	*retooblen = 0;

	switch(oobsel->useecc) {
		case MTD_NANDECC_AUTOPLACE:
		case MTD_NANDECC_AUTOPL_USR:
			autoplace = 1;
			break;
		default:
			autoplace = 0;
			break;
		}

if (gdebug) printk("%s: autoplace=%d, offset=%08x\n", __FUNCTION__, autoplace, offset);
			
	if (len > mtd->oobblock || len < 0) {
		printk(KERN_ERR "%s: Invalid length %d\n", __FUNCTION__, len);
		return -EINVAL;
	}
	for (winslice = 0; winslice < this->eccsteps && dataRead < len; winslice++) {
		int thislen = min_t(int, len-dataRead, mtd->eccsize);
		int i;
		unsigned char* oobbuf = (oob_buf ? &oob_buf[oobRead] : NULL);

if (gdebug) printk("%s: winslice=%d, offset=%08x\n", __FUNCTION__, winslice, offset+dataRead);

		// If partial read, must use internal buffer
		if (thislen != mtd->eccsize) {
			ret = brcmnand_posted_read_cache(mtd, &this->data_buf[winslice*this->eccsteps], oobbuf, offset+dataRead, 0);
			memcpy(&data_buf[dataRead], &this->data_buf[winslice*this->eccsteps], thislen);
		} 
		else {
			ret = brcmnand_posted_read_cache(mtd, &data_buf[dataRead], oobbuf, offset+dataRead, 0);
		}
		if (ret) {
			printk(KERN_ERR "%s: posted read cache failed at offset=%08x, ret=%d\n", 
				__FUNCTION__, offset + dataRead, ret);
			return ret;
		}

		dataRead += thislen;

		/* Skip oob conversion if there is nothing in the first place */
		if (!oob_buf ) 
			continue;
		
		// Convert the free bytes into fsbuffer
		if (autoplace) {
			oob = brcmnand_convert_oobfree_to_fsbuf(mtd, this->oob_buf, oob_buf, mtd->oobsize, oobsel, autoplace);
			oobRead += oob;
		}
		else {
			oobRead += this->eccOobSize;
		}
		

		
	}
	*retlen = dataRead;
	*retooblen = oobRead;
if (gdebug) printk("<-- %s\n", __FUNCTION__);
	return 0;
}



/**
 * brcmnand_read_pageoob - [GENERIC] write to one page's OOB area
 * @mtd:			MTD device structure
 * @offset: 		offset  from start of the chip, must be called with (offset & this->pagemask)
 * @len: 			size of data buffer, len must be <= mtd->oobblock, and can indicate partial page.
 * @data_buf: 	data buffer.
 * @oob_buf:		out of band data buffer for 1 page (size = oobsize), must be prepared beforehand
 * @retlen: 		Returned length
 * @convert2fs: 	Convert oob_buf to fsbuf
 *
 * Nand_page_program function is used for write and writev !
 * This function will always program a full page of data
 * If you call it with a non page aligned buffer, you're lost :)
 *
 * Cached programming is not supported yet.
 */
static int brcmnand_read_pageoob (struct mtd_info *mtd, int offset,
	u_char* oob_buf, int* retooblen, struct nand_oobinfo *oobsel, int autoplace, int raw)
{
	struct brcmnand_chip *this = mtd->priv;
	int oobRead = 0;
	int winslice;
	int ret;
	u_char this_oobbuf[64];
	u_char* pOobbuf;


if (gdebug > 2) printk("-->%s: offset=%08x, autoplace=%d\n", __FUNCTION__, offset, autoplace);

	/* If autoplace, use the scratch buffer */
	if (autoplace) {
		pOobbuf = this_oobbuf;
	}
	else {
		pOobbuf = oob_buf;
	}

	/*
	 * The 4 slice OOB areas may not have the same format, (e.g. for the 2K page, the first slice
	 * lose 2 bytes.)
	 * We have to prepare the OOB buffer for the entire page, then write it out one slice at a time.
	 */
	for (winslice = 0; winslice < this->eccsteps && oobRead < mtd->oobsize; winslice++) {
		int sliceOffset = offset+this->eccsize*winslice;
		int sliceOobOffset = this->eccOobSize*winslice;

if (gdebug > 2) printk("winslice=%d, sliceoff=%08x, sliceOOBoffs=%08x\n", winslice, sliceOffset, sliceOobOffset);
		ret = brcmnand_posted_read_oob(mtd, &pOobbuf[sliceOobOffset], sliceOffset, raw);
		if (ret) {
			printk(KERN_ERR "%s: posted read cache failed at offset=%08x, ret=%d\n", 
				__FUNCTION__, sliceOffset, ret);
			return ret;
		}
		oobRead += this->eccOobSize;
	}
	
	*retooblen = oobRead;

if (gdebug > 2) {printk("%s: read raw oob B4 conv, autoplace=%d rawbuf=\n", __FUNCTION__, autoplace);
print_oobbuf(pOobbuf, mtd->oobsize);}

	// Convert the free bytes into fsbuffer
	if (autoplace) {
		*retooblen = brcmnand_convert_oobfree_to_fsbuf(mtd, pOobbuf, oob_buf, mtd->oobsize, oobsel, autoplace);
if (gdebug > 2) { printk("after conversion retlen=%d, fsbuf=\n", *retooblen);
print_oobbuf(oob_buf,*retooblen); }
	}
	

	return 0;
}


/**
 * brcmnand_read_ecc - [MTD Interface] Read data with ECC
 * @param mtd		MTD device structure
 * @param from		offset to read from
 * @param len		number of bytes to read
 * @param retlen	pointer to variable to store the number of read bytes
 * @param buf		the databuffer to put data
 * @param oob_buf	filesystem supplied oob data buffer
 * @param oobsel	oob selection structure
 *
 * BrcmNAND read with ECC
 * returns 0 on success, -errno on error.
 */
static int brcmnand_read_ecc(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u_char *buf,
	u_char *oob_buf, struct nand_oobinfo *oobsel)
{
	struct brcmnand_chip *this = mtd->priv;
	int dataRead = 0, oobRead = 0;
	int i, thislen;
	unsigned long pageAddr;
	int page;
	int ret = 0;
	u_char* raw_oob = this->oob_buf;
	int retooblen, retdatalen;
	loff_t startPageOffset;
	u_char data_buf[2048];

//if ((((unsigned int) from & 0xffff0000) >= 0x02800000) && (((unsigned int) from & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x\n", __FUNCTION__, (unsigned int) from);

	DEBUG(MTD_DEBUG_LEVEL3, "brcmnand_read_ecc: from = 0x%08x, len = %i, oob_buf=%p, autoplace=%d\n", 
	(unsigned int) from, (int) len, oob_buf, brcmnand_autoplace(mtd, oobsel));
	*retlen = 0;
	//oobarea = oob_buf;

	/* Do not allow reads past end of device */
	if (unlikely(from + len) > mtd->size) {
		DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_read_ecc: Attempt read beyond end of device\n");
		*retlen = 0;
		return -EINVAL;
	}

	/* Grab the lock and see if the device is available */
	/* THT: TBD: Move it inside while loop: It would be less efficient, but
	 * would also give less response time to multiple threads
	 */
PRINTK("brcmnand_read_ecc: get device\n");
	brcmnand_get_device(mtd, FL_READING);

	/* Setup variables and oob buffer */
	
	page = (int) (from >> this->page_shift);
	startPageOffset = (loff_t) (page << this->page_shift);
	


	while (dataRead < len) {
		u_char* oobbuf;
		
		thislen = min_t(int, mtd->oobblock, len - dataRead);
		if (oob_buf) {
			oobbuf = &oob_buf[oobRead];
		}
		else {
			oobbuf = NULL;
		}

		/* page marks the start of the  block encompassing [from, thislen] */
		pageAddr = ((unsigned long) (from+dataRead) & ~ (mtd->oobblock - 1));

		/* If not at start of page, read up to the next start of page */
		if (pageAddr != (unsigned long) (from+dataRead)) {

			thislen = min_t(int, len - dataRead, mtd->oobblock - ((from + dataRead) - pageAddr));
if(gdebug) printk("%s: partial read: thislen=%d, from=%08x, dataRead=%d, page=%08x, oobblock=%d\n", __FUNCTION__,
	thislen, (unsigned int) from,  dataRead, pageAddr, mtd->oobblock);
			retooblen = retdatalen = 0;
			/* Read the entire page into the default buffer */
			ret = brcmnand_read_page(mtd, pageAddr, data_buf, mtd->oobblock, oobbuf, oobsel, &retdatalen, &retooblen);
if(gdebug) printk("%s: partial read: read_page(pgAddr=%08x) returns %d\n", __FUNCTION__, pageAddr, ret);
			if (ret) {
				DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_read_ecc: read failed = %d\n", ret);
				// Do BTT management here
				goto out;
			}
			/* Copy from default buffer into actual outbuf */
			memcpy(&buf[dataRead], &data_buf[from+dataRead-pageAddr], thislen);
if(gdebug) { printk("Partial read: dataRead=%d, from=%08x, pageAddr=%08x, thislen=%d, buf=\n", dataRead, (int)(from+dataRead), pageAddr, thislen);
if (len > 512) print_databuf(&buf[dataRead], thislen);}

		}
		else {
			/* Check thislen against the page size, if not equal, we will have to use the default buffer, again */
			if (thislen == mtd->oobblock) {
			/* Read straight into buffer */
			ret = brcmnand_read_page(mtd, pageAddr, &buf[dataRead], thislen, oobbuf,  oobsel, &retdatalen, &retooblen);
	if(gdebug) printk("%s: boundary read: read_page(pgAddr=%08x, dataRead=%d, thislen=%d) returns %d\n", 
		__FUNCTION__, pageAddr, dataRead, thislen, ret);
				if (ret) {
					DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_read_ecc: read failed = %d\n", ret);
					// Do BTT management here
					goto out;
				}
			}
			else {
				/* thislen < mtd->oobblock
				 * This is a partial read: Read into default buffer  
				 * then copy over to avoid writing beyond the end of the input buffer
				 */
				// Here we run the risk of overwriting the OOB input buffer, but the interface does not allow
				// us to know the size of the receiving OOB.
				ret = brcmnand_read_page(mtd, pageAddr, data_buf, thislen, oobbuf, oobsel, &retdatalen, &retooblen);
if(gdebug) {printk("%s: partial boundary read at end: read_page(pgAddr=%08x, dataRead=%d, thislen=%d) returns %d\n", 
__FUNCTION__, pageAddr, dataRead, thislen, ret);
}
			if (ret) {
				DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_read_ecc: read failed = %d\n", ret);
				// Do BTT management here
				goto out;
			}
				memcpy(&buf[dataRead], data_buf, thislen);
			}
if (gdebug > 3 && len > 512) print_databuf(&buf[dataRead], thislen);
		}
		dataRead += thislen;
		oobRead += retooblen;


//printk("%s: dataRead=%d, len=%d\n", __FUNCTION__, dataRead, len);
		if (dataRead >= len)
			break;

		if (ret) {
			DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_read_ecc: read failed = %d\n", ret);
			goto out;
		}
	}

out:
	/* Deselect and wake up anyone waiting on the device */
	brcmnand_release_device(mtd);

	/*
	 * Return success, if no ECC failures, else -EBADMSG
	 * fs driver will take care of that, because
	 * retlen == desired len and result == -EBADMSG
	 */
	*retlen = dataRead;

if (gdebug > 3) {gdebug = 0;
printk("<-- %s(offset=%08x): retlen = %d\n", __FUNCTION__, (unsigned int) from, dataRead);
if (len > 512) {printk("len=%d, retlen=%d, read buffer=\n", len, *retlen); print_databuf(buf, *retlen);}
}
	return ret;
}

/**
 * brcmnand_read - [MTD Interface] MTD compability function for brcmnand_read_ecc
 * @param mtd		MTD device structure
 * @param from		offset to read from
 * @param len		number of bytes to read
 * @param retlen	pointer to variable to store the number of read bytes
 * @param buf		the databuffer to put data
 *
 * This function simply calls brcmnand_read_ecc with oob buffer and oobsel = NULL
*/
static int brcmnand_read(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u_char *buf)
{
	DEBUG(MTD_DEBUG_LEVEL3, "-->%s from=%08x, len=%d\n", __FUNCTION__,
 		(unsigned long) from, (int) len);
	return brcmnand_read_ecc(mtd, from, len, retlen, buf, NULL, NULL);
}



/**
 * brcmnand_read_oob - [MTD Interface] BrcmNAND read out-of-band
 * @param mtd		MTD device structure
 * @param from		offset to read from
 * @param len		number of bytes to read
 * @param retlen	pointer to variable to store the number of read bytes
 * @param buf		the databuffer to put data
 *
 * BrcmNAND read out-of-band data from the spare area
 */
static int brcmnand_read_oob(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u_char *buf)
{
	struct brcmnand_chip *this = mtd->priv;
	int oobread = 0, thislen;
	unsigned int page;
	int ret = 0;
	int raw;

//if ((((unsigned int) from & 0xffff0000) >= 0x02800000) && (((unsigned int) from & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x, len=%d\n", __FUNCTION__, (unsigned int) from, len);


 	DEBUG(MTD_DEBUG_LEVEL3, "%s: from = 0x%08x, len = %i\n", __FUNCTION__,
 		(unsigned long) from, (int) len);

	/* Initialize return length value */
	*retlen = 0;

	/* Do not allow reads past end of device */
	if (unlikely((from + len) > mtd->size)) {
		DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_read_oob: Attempt read beyond end of device\n");
		return -EINVAL;
	}

	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_READING);

	/* page marks the start of the page block encompassing [from, thislen] */
	page = ((unsigned long) from & ~ (mtd->oobblock - 1)) >> this->page_shift;

	/* Make sure we don't go over bound */
	while (oobread < len) {
		
		ret = brcmnand_read_pageoob(mtd, page << this->page_shift, this->oob_buf, &thislen, this->autooob, 0, 1);
//if (from==0) {
//printk("read_oob: brcmnand_read_pageoob returns thislen=%d, oobsize=%d\n", thislen, mtd->oobsize);
//print_oobbuf(this->oob_buf, mtd->oobsize);
//}
		if (ret) {
			printk(KERN_ERR "brcmnand_read_pageoob returns %d, thislen = %d\n", ret, thislen);
		}
		
		// Just copy up to thislen
		memcpy(&buf[oobread], &this->oob_buf, thislen);
		oobread += thislen;
		if (oobread >= len) {
			break;
		}

		if (ret) {
			DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_read_oob: read failed = %d\n", ret);
			goto out;
		}

		buf += thislen;		
		page++;
		
	}

out:
	/* Deselect and wake up anyone waiting on the device */
	brcmnand_release_device(mtd);

	*retlen = oobread;

if (4 == gdebug) gdebug = 0;
	return ret;
}

/*
 * Determine whether we should use autoplace
 */
static int brcmnand_autoplace(struct mtd_info *mtd, struct nand_oobinfo* oobsel)
{
	struct brcmnand_chip *this = mtd->priv;
	int autoplace;

	/* if oobsel is NULL, use chip defaults */
	if (oobsel == NULL)
		oobsel = &mtd->oobinfo;

	switch(oobsel->useecc) {
		case MTD_NANDECC_AUTOPLACE:
		case MTD_NANDECC_AUTOPL_USR:
			autoplace = 1;
			break;
		default:
			autoplace = 0;
			break;
		}

	return autoplace;
}


/**
 * brcmnand_read_oobfree - [MTD Interface] NAND read free out-of-band
 * @mtd:	MTD device structure
 * @from:	offset to read from
 * @len:	number of bytes to read
 * @retlen:	pointer to variable to store the number of read bytes
 * @buf:	the databuffer to put data
 *
 * NAND read free out-of-band data from the spare area
 */
static int brcmnand_read_oobfree (struct mtd_info *mtd, 
			loff_t from, size_t len, size_t * retlen, u_char * out_fsbuf)
{

	struct brcmnand_chip *this = mtd->priv;
	int oobread = 0, thislen;
	unsigned int page;
	int ret = 0;
	int autoplace;
static int once;

//if ((((unsigned int) from & 0xffff0000) >= 0x02800000) && (((unsigned int) from & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x, len=%d\n", __FUNCTION__, (unsigned int) from, len);

	DEBUG(MTD_DEBUG_LEVEL3, "-->%s from=%08x, len=%d\n", __FUNCTION__,
 		(unsigned long) from, (int) len);
//if (once++ < 20) {
//printk("-->%s, from=%08x, len=%d\n", __FUNCTION__, (unsigned int) from, len);
//}

	*retlen=0;

	autoplace = 1;

	/* Do not allow reads past end of device */
	if (unlikely((from + len) > mtd->size)) {
		DEBUG(MTD_DEBUG_LEVEL0, "%s: Attempt read beyond end of device\n", __FUNCTION__);
		return -EINVAL;
	}

	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_READING);

	/* page marks the start of the page block encompassing [from, thislen] */
	page = ((unsigned long) from & ~ (mtd->oobblock - 1)) >> this->page_shift;

	/* Make sure we don't go over bound */
	while (oobread < len) {
		
		ret = brcmnand_read_pageoob(mtd, page << this->page_shift,  &out_fsbuf[*retlen], &thislen, this->autooob, autoplace, 1);
	
		if (ret) {
			printk(KERN_ERR "%s: read_oob returns %d\n", __FUNCTION__, ret);
			goto out;
		}

		// Convert to fsbuf, returning # bytes converted (== mtd->oobavail)
		// already done in readpageoob
		// ret = brcmnand_convert_oobfree_to_fsbuf(mtd, this->oob_buf, &fsbuf[oobread], thislen, this->autooob, autoplace);

		// Advance to next page
		page++;
		oobread += mtd->oobsize;
		*retlen += thislen;
		from += mtd->oobblock;
	}

if (gdebug && once++ < 20) {
printk("<--%s, retlen=%d, fsbuf=\n", __FUNCTION__,*retlen);
print_oobbuf(out_fsbuf, *retlen);
}

//gdebug = 0;
	ret = 0;

out:
	brcmnand_release_device(mtd);

if (4 == gdebug) gdebug = 0;
	return ret;
}




#ifdef CONFIG_MTD_BRCMNAND_VERIFY_WRITE
/*
 * Returns 0 on success, 
 */
static int brcmnand_verify_pageoob_priv(struct mtd_info *mtd, loff_t addr, 
	u_char* fsbuf, int fslen, u_char* oob_buf, int ooblen, struct nand_oobinfo* oobsel, 
	int autoplace, int raw)
{
	struct brcmnand_chip *this = mtd->priv;
	int ret = 0;
	//int ooblen = mtd->oobsize;
	//char tmpfsbuf[64]; // Max oob size we support.
	int complen;

	
//printk("%s: fsbuf=@%p\n", __FUNCTION__, fsbuf);
//print_oobbuf(fsbuf, fslen);
//printk("%s: oob_buf=\n", __FUNCTION__);
//print_oobbuf(oob_buf, 64);
//gdebug=1;
#if 0
	ooblen = brcmnand_convert_oobfree_to_fsbuf(mtd, oob_buf, tmpfsbuf, oobsel, autoplace);
gdebug=0;
#endif
	if (autoplace) {

		complen = min_t(int, ooblen, fslen);

		/* We may have read more from the OOB area, so just compare the min of the 2 */
		if (complen == fslen) {
			ret = memcmp(fsbuf, oob_buf, complen);
			if (ret) {
				int i;
				
for (i=0; i<complen; i++) {
if (fsbuf[i] != oob_buf[i]) {printk("Comp failed at i=%d, complen=%d, fsbuf=%02x, oob_buf=%02x\n", 
i, complen, fsbuf[i], oob_buf[i]); break;}
}
				goto comparison_failed;
			}
		}
		else {
	printk("%s: OOB comparison failed, ooblen=%d is less than fslen=%d\n", 
		__FUNCTION__, ooblen, fslen);
			return  -EBADMSG;
		}
	}
	else { // No autoplace.  Skip over non-freebytes
		int i, len; 
		
		for (i = 0; oobsel->oobfree[i][1] && i < ARRAY_SIZE(oobsel->oobfree); i++) {
			int from = oobsel->oobfree[i][0];
			int num = oobsel->oobfree[i][1];

			ret = memcmp(&fsbuf[from], &oob_buf[from], num);
			if (ret) {
printk("No-Autoplace Comp failed at from=%d, len=%d, num=%d, fsbuf=%p, oob_buf=%p\n", 
from, len, num, &fsbuf[len], &oob_buf[from]); 

				goto comparison_failed;
			}
			len += num;
		}
	}
	return ret;

comparison_failed:
	{
		unsigned long nand_acc_control = brcmnand_ctrl_read(BCHP_NAND_ACC_CONTROL);
		unsigned long nand_config = brcmnand_ctrl_read(BCHP_NAND_CONFIG);
		unsigned long flash_id = brcmnand_ctrl_read(BCHP_NAND_FLASH_DEVICE_ID);
		unsigned long nand_timing1 = brcmnand_ctrl_read(BCHP_NAND_TIMING_1);
		unsigned long nand_timing2 = brcmnand_ctrl_read(BCHP_NAND_TIMING_2);
		u_char raw_oob[64];
		int retlen, i;
		struct nand_oobinfo noauto_oobsel;
		
printk("\n%s: OOB comp failed, autoplace=%d, offset=%08x, len=%d, fslen=%d, ACC=%08x, cfg=%08x, flashId=%08x, tim1=%08x, tim2=%08x\n", 
	__FUNCTION__, autoplace,
	(unsigned long) addr, ooblen, fslen, nand_acc_control, nand_config, flash_id, nand_timing1, nand_timing2);	

printk("\nfsbuf:\n"); print_oobbuf(fsbuf, fslen);
printk("\ntmpfsbuf:\n"); print_oobbuf(oob_buf, ooblen);

		noauto_oobsel = *oobsel;
		noauto_oobsel.useecc = MTD_NANDECC_PLACEONLY;
		brcmnand_read_pageoob(mtd, addr, raw_oob, &retlen, &noauto_oobsel, 0, raw);
printk("raw oobbuf:\n"); print_oobbuf(raw_oob, mtd->oobsize);

dump_stack();
		return -EBADMSG;
	}
}



/**
 * brcmnand_verify_page - [GENERIC] verify the chip contents after a write
 * @param mtd		MTD device structure
 * @param dbuf		the databuffer to verify
 * @param dlen		the length of the data buffer, and should be less than mtd->oobblock
 * @param fsbuf		the length of the file system OOB data and should be exactly
 *                             mtd-->oobavail bytes to verify.
 *
 * Assumes that lock on.  Munges the internal data and OOB buffers.
 */
//#define MYDEBUG
static int brcmnand_verify_page(struct mtd_info *mtd, loff_t addr, u_char *dbuf, int dlen, 
		u_char* fsbuf, int fslen, struct nand_oobinfo* oobsel, int autoplace
#ifdef MYDEBUG
		, int*retlen
#endif
		)
{
	struct brcmnand_chip *this = mtd->priv;
	u_char data_buf[2048];
	u_char oob_buf [64];
	int ret = 0;
	int ooblen=0, datalen=0;
	int complen;
	u_char* oobbuf = (fsbuf && fslen > 0) ? oob_buf : NULL;

if (gdebug > 2) { printk("%s: fslen=%d, fsbuf=@%p, autoplace=%d\n", __FUNCTION__, fslen, fsbuf, autoplace);
if (fsbuf) print_oobbuf(fsbuf, fslen); }

#ifdef MYDEBUG
	printk("%s: 10 retlen=%p\n", __FUNCTION__, retlen);
#endif

	// Must read entire page
	ret = brcmnand_read_page(mtd, addr, data_buf, mtd->oobblock, oobbuf, oobsel, &datalen, &ooblen);
	if (ret) {
		printk(KERN_ERR "%s: brcmnand_read_page at %08x failed ret=%d\n", 
			__FUNCTION__, (unsigned int) addr, ret);
		return ret;
	}

#ifdef MYDEBUG
	printk("%s: 20 retlen=%p\n", __FUNCTION__, retlen);
#endif

	/* 
	 * If there are no Input ECC bytes, there is nothing to verify, reading the page and checking the ECC status
	 * flag should be enough
	 */
	if (!fsbuf || fslen <= 0)
		return 0;
	
	// We only verify the ECC bytes.
	ret = brcmnand_verify_pageoob_priv(mtd, addr, fsbuf, fslen, oob_buf, ooblen, oobsel, autoplace, 0);
	if (ret) {
		return ret;
	}

#ifdef MYDEBUG
	printk("%s: 30 retlen=%p\n", __FUNCTION__, retlen);
#endif

	// but only verify dlen bytes	
	ret = memcmp(data_buf, dbuf, dlen);
#ifdef MYDEBUG
	printk("%s: 80 retlen=%p\n", __FUNCTION__, retlen);
#endif
	if (ret) {
printk("%s: Data comparison failed, len=%d, addr=%08x\n", __FUNCTION__, dlen, addr);
printk("----------- from buf ---------------\n"); print_databuf(dbuf, dlen);
printk("----------- from flash ---------------\n"); print_databuf(data_buf, dlen);
		return -EBADMSG;
	}

	return ret;
}

/**
 * brcmnand_verify_pageoob - [GENERIC] verify the chip contents after a write
 * @param mtd		MTD device structure
 * @param dbuf		the databuffer to verify
 * @param dlen		the length of the data buffer, and should be less than mtd->oobblock
 * @param fsbuf		the file system OOB data 
 * @param fslen		the length of the file system buffer
 * @param oobsel		Specify how to write the OOB data
 * @param autoplace	Specify how to write the OOB data
 * @param raw		Ignore the Bad Block Indicator when true
 *
 * Assumes that lock on.  Munges the OOB internal buffer.
 */
static int brcmnand_verify_pageoob(struct mtd_info *mtd, loff_t addr, u_char* fsbuf, int fslen,
		struct nand_oobinfo *oobsel, int autoplace, int raw)
{
	struct brcmnand_chip *this = mtd->priv;
	u_char* data_buf = this->data_buf;
	u_char oob_buf[64]; // = this->oob_buf;
	int ret = 0;
	//int complen;
	//char tmpfsbuf[64]; // Max oob size we support.
	int ooblen = 0;

if (gdebug > 2) printk("brcmnand_verify_pageoob: addr=%08x\n", (unsigned long) addr);

	// Must read entire page
	ret = brcmnand_read_pageoob(mtd, addr, oob_buf, &ooblen, oobsel, autoplace, raw);

if (gdebug > 2) {printk("%s: after read, ooblen=%d\n", __FUNCTION__, ooblen);
print_oobbuf(oob_buf, ooblen);
}
	if (ret) {
		printk(KERN_ERR "%s: brcmnand_read_page at %08x failed ret=%d\n", 
			__FUNCTION__, (unsigned int) addr, ret);
		return ret;
	}
//else printk("%s: read_pageoob(addr=%08x) returns retlen=%d\n", __FUNCTION__, addr, ooblen);

	ret = brcmnand_verify_pageoob_priv(mtd, addr, fsbuf, fslen, oob_buf, ooblen, oobsel, autoplace, raw);

	return ret;
}

#if 0
/**
 * brcmnand_verify_oob - [GENERIC] verify the oob contents after a write
 * @param mtd		MTD device structure
 * @param buf		the databuffer to verify
 * @param to			offset to read from
 * @param len		number of bytes to read and compare
 *
 */
static int brcmnand_verify_oob(struct mtd_info *mtd, const u_char *fsbuf, loff_t to, int len)
{
	struct brcmnand_chip *this = mtd->priv;
	char *oob_buf = this->oob_buf;
	int i, ret, ooblen;
	char tmpfsbuf[64]; // Max oob size we support.

	
}
#endif

#else
#define brcmnand_verify_page(...)	(0)
#define brcmnand_verify_pageoob(...)		(0)
//#define brcmnand_verify_oob(...)		(0)
#endif

#define NOTALIGNED(x)	((x & (mtd->oobblock - 1)) != 0)



/**
 * brcmnand_read_raw - [GENERIC] Read raw data including oob into buffer
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @from:	offset to read from
 * @len:	number of bytes to read
 * @ooblen:	number of oob data bytes to read
 *
 * Read raw data including oob into buffer.  Raw Reading ignores any BBT flag.
 */
int brcmnand_read_raw (struct mtd_info *mtd, uint8_t *buf, loff_t from, size_t len, size_t ooblen)
{
	struct brcmnand_chip *this = mtd->priv;
	int slice, ret, page;
	int ignoreBBT= 1;

PRINTK("%s(from=%08x, len=%08x, ooblen=%08x\n", __FUNCTION__, (unsigned long) from, len, ooblen);
	if (from & (mtd->oobblock - 1)) {
		printk(KERN_ERR "brcmnand_read_raw: from %08ul is not OOB sliceOffset aligned\n", (unsigned long) from);
		return -EINVAL;
	}
	/* len must be multiple of oobblock */
	if (((len / mtd->oobblock) * mtd->oobblock) != len) {
		printk(KERN_ERR "%s: len %08x is not multiple of oobblock %d\n", __FUNCTION__, len, mtd->oobblock);
		return -EINVAL;
	}
	/* ooblen must be multiple of oobsize */
	if (((ooblen / mtd->oobsize) * mtd->oobsize) != ooblen) {
		printk(KERN_ERR "%s: ooblen %08x is not multiple of oobblock %d\n", __FUNCTION__, ooblen, mtd->oobsize);
		return -EINVAL;
	}
	for (page=0; (page * mtd->oobblock) < len; page++) {
		/* For each 2K page, the first 2K are for data (4X512B), followed by 4X64B OOB data */
		int startPage = page*(mtd->oobblock+mtd->oobsize); // Address in buffer corresponding to start of this page
		for (slice = 0; slice < this->eccsteps; slice++) {
			int offset = from + slice*mtd->eccsize; // offset of data w.r.t. flash
			int dataOffset = startPage + slice*mtd->eccsize;
			int oobOffset = startPage + mtd->oobblock + slice*this->eccOobSize; 
			
PRINTK("%s Calling PRread at offset = %08x, bufOffset=%08x, oobOffs=%08x\n", 
__FUNCTION__, offset, dataOffset, oobOffset);		

			ret = brcmnand_posted_read_cache(mtd, &buf[dataOffset], 
				&buf[oobOffset], offset, ignoreBBT); /* Ignore BBT */
			if (ret) {
				printk(KERN_ERR "%s failed at offset = %08x, dataOffset=%08x, oobOffset=%d\n", 
					__FUNCTION__, offset, dataOffset, oobOffset);
				return ret;
			}
		}
		// if partial page, must pad with FF??
	}
	return ret;
	
}


/**
 * brcmnand_write_page - [GENERIC] write one page
 * @mtd:	MTD device structure
 * @offset: 	offset  from start of the chip, must be called with (offset & this->pagemask)
 * @len: size of data buffer, len must be <= mtd->oobblock, and can indicate partial page.
 * @data_buf: data buffer.
 * @oob_buf:	out of band data buffer for 1 page (size = oobsize), must be prepared beforehand
 *
 * Nand_page_program function is used for write and writev !
 * This function will pad the buffer to a Cache boundary before writing it out.
 *
 * Cached programming is not supported yet.
 */
static int brcmnand_write_page (struct mtd_info *mtd, int offset, int len,
	u_char* data_buf, u_char* oob_buf)
{
	struct brcmnand_chip *this = mtd->priv;
	int written = 0, oobWritten = 0;
	int winslice;
	int done = 0;
	int ret;
	unsigned long pageOffset, columnOffset;
	int column;
	int padlen = 0, trailer;

	/* Use the passed in buffer */
	this->data_poi = data_buf;
	/* page marks the start of the page encompassing [from, thislen] */
	pageOffset = ((unsigned long) offset & ~ (mtd->oobblock - 1));
	/* column marks the start of the 512B cache slice encompassing [from, thislen] */
	columnOffset = ((unsigned long) offset & ~ (mtd->eccsize - 1));
	column = (columnOffset - pageOffset) % mtd->eccsize;
	/* Pad to the front if not starting on cache slice boundary */
	if (columnOffset != (unsigned long) offset) {
		padlen = offset - columnOffset;
		trailer = min_t(int, len, mtd->oobblock - padlen);

PRINTK("%s: Padding %d bytes in front, start=%08x, trailer=%d\n", __FUNCTION__, padlen, offset, trailer);
		this->data_poi = this->data_buf;
		memcpy(this->data_buf, ffchars, padlen);
		// Make it simple stupid, just copy the remaining part of a page
		memcpy(&this->data_buf[padlen], data_buf, trailer);
		// Adjust offset accordingly
		// written = - padlen;
	}

	for (winslice = column; winslice < this->eccsteps && written < len && !done; winslice++) {
		int thislen;
		int i;


		thislen = min_t(int, len-written, mtd->eccsize - padlen);

		// Pad at the end ?
		if (thislen != mtd->eccsize - padlen) {
			int fillupLen = ((padlen+len - written + mtd->eccsize) & ~(mtd->eccsize - 1)) - (padlen+len);
			done = 1; // Writing last slice
			//Pad with 0xff before writing
PRINTK("%s: Padding %d bytes at end, start=%08x, thislen=%d\n", __FUNCTION__, fillupLen, offset+written, thislen);
			memcpy(&this->data_poi[padlen+len], &ffchars[padlen+len], fillupLen);
		}
		if (oob_buf) {
			ret = brcmnand_posted_write_cache(mtd, &this->data_poi[written], &oob_buf[oobWritten], offset+written-padlen);
		}
		else {
			ret = brcmnand_posted_write_cache(mtd, &this->data_poi[written], NULL, offset+written-padlen);
		}
		if (ret) {
			printk(KERN_ERR "%s: posted write cache failed at offset=%08x, ret=%d\n", 
				__FUNCTION__, offset + written, ret);
			return ret;
		}
		written += thislen;
		if (oob_buf)
			oobWritten += this->eccOobSize;
		padlen = 0;
	}

	return 0;
}


/**
 * brcmnand_write_pageoob - [GENERIC] write to one page's OOB area
 * @mtd:	MTD device structure
 * @offset: 	offset  from start of the chip, must be called with (offset & this->pagemask)
 * @len: size of data buffer, len must be <= mtd->oobblock, and can indicate partial page.
 * @data_buf: data buffer.
 * @oob_buf:	out of band data buffer for 1 page (size = oobsize), must be prepared beforehand
 *
 * Nand_page_program function is used for write and writev !
 * This function will always program a full page of data
 * If you call it with a non page aligned buffer, you're lost :)
 *
 * Cached programming is not supported yet.
 */
static int brcmnand_write_pageoob (struct mtd_info *mtd, int offset, int len,
	u_char* oob_buf)
{
	struct brcmnand_chip *this = mtd->priv;
	int oobWritten = 0;
	int winslice;
	int done = 0;
	int ret;


if (gdebug) printk("%s: offset=%08x, len=%d\n", __FUNCTION__, (unsigned long) offset, len);

	if (len > mtd->oobsize) {
		printk(KERN_ERR "%s: len of %d is greater than OOB size %d\n", __FUNCTION__, len, mtd->oobavail);
		return -EINVAL;
	}
	/*
	 * The 4 slice OOB areas may not have the same format, (e.g. for the 2K page, the first slice
	 * lose 2 bytes.)
	 * We have to prepare the OOB buffer for the entire page, then write it out one slice at a time.
	 */
	for (winslice = 0; winslice < this->eccsteps && oobWritten < len && !done; winslice++) {
		int sliceOffset = offset+mtd->eccsize*winslice;
		
		ret = brcmnand_posted_write_oob(mtd, &oob_buf[oobWritten], sliceOffset);
		if (ret) {
			printk(KERN_ERR "%s: posted write cache failed at offset=%08x, ret=%d\n", 
				__FUNCTION__, sliceOffset, ret);
			return ret;
		}
		oobWritten += this->eccOobSize;
	}

	return 0;
}



/**
 * brcmnand_write_ecc - [MTD Interface] BrcmNAND write with ECC
 * @param mtd		MTD device structure
 * @param to		offset to write to
 * @param len		number of bytes to write
 * @param retlen	pointer to variable to store the number of written bytes
 * @param buf		the data to write
 * @param eccbuf	filesystem supplied oob data buffer
 * @param oobsel	oob selection structure
 *
 * BrcmNAND write with ECC
 */
static int brcmnand_write_ecc(struct mtd_info *mtd, loff_t to, size_t len,
	size_t *retlen, const u_char *buf,
	u_char *eccbuf, struct nand_oobinfo *oobsel)
{
	struct brcmnand_chip *this = mtd->priv;
	int page, ret = -EIO, oob = 0, written = 0, eccWritten = 0;
	int autoplace = 0, numpages, totalpages;
	u_char *oobbuf;
	int	ppblock = (1 << (this->phys_erase_shift - this->page_shift));
	loff_t startPageOffset; 
	u_char oob_buf[64];
	int eccRetLen = 0;

static int once;
volatile unsigned int Debug = 0;


//if ((((unsigned int) to & 0xffff0000) >= 0x02800000) && (((unsigned int) to & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x, len=%d\n", __FUNCTION__, (unsigned int) to, len);



	DEBUG(MTD_DEBUG_LEVEL3, "brcmnand_write_ecc: to = 0x%08x, len = %i\n", (unsigned int) to, (int) len);
if (gdebug > 2) printk("-->%s: to=%08x, len=%d, retlen=%p, buf=%p, eccbuf=%p, oobsel=%p\n", 
	__FUNCTION__, (unsigned long) to, len, retlen, buf, eccbuf, oobsel);

	autoplace = brcmnand_autoplace(mtd, oobsel);

if (gdebug) {
printk("--> %s: autoplace=%d, oob=%p\n", __FUNCTION__, autoplace, eccbuf);
if (eccbuf) print_oobbuf(eccbuf, mtd->oobsize);
}

	/* Initialize retlen, in case of early exit */
	*retlen = 0;

//Debug=10;

	/* Do not allow writes past end of device */
	if (unlikely((to + len) > mtd->size)) {
		DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_write_ecc: Attempt write to past end of device\n");
		return -EINVAL;
	}

	/* Reject writes, which are not page aligned */
        if (unlikely(NOTALIGNED(to)) || unlikely(NOTALIGNED(len))) {
                DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_write_ecc: Attempt to write not page aligned data\n");
                return -EINVAL;
        }

//Debug = 20;
	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_WRITING);


	/* Setup variables and oob buffer */
	totalpages = len >> this->page_shift;
	page = (int) (to >> this->page_shift);
	startPageOffset = (loff_t) (page << this->page_shift);
	
	/* Invalidate the page cache, if we write to the cached page */
	if (page <= this->pagebuf && this->pagebuf < (page + totalpages))
		this->pagebuf = -1;
	eccWritten = 0;

	/*
	 * Write the part before page boundary.
	 */
	if (startPageOffset != to) {
		// How many bytes will need to be written for partial page write.
		int preface = (int)(to - startPageOffset);
		int thislen = min_t(int, len - written, mtd->oobblock - preface);
		u_char pagebuf[2048];
		
		/* Make a page, fill it up with 0xFF up to where the actual buffer starts */
		memcpy(pagebuf, ffchars, preface);
		memcpy(&pagebuf[preface], buf, thislen);
		this->data_poi = (u_char*) &pagebuf[0];

		/* We dont write ECC to a partial page */
		if (eccbuf) {
			printk("%s: OOB buffer ignored in partial page write at offset %08x\n", __FUNCTION__, (unsigned int) to);
		}
		oobbuf = NULL;
		
		/* Write one page. If this is the last page to write
		 * or the last page in this block, then use the
		 * real pageprogram command, else select cached programming
		 * if supported by the chip.
		 */
//Debug = 50;
if (gdebug) printk("050 retlen=%p\n", retlen);
		ret = brcmnand_write_page (mtd, startPageOffset, mtd->oobblock, this->data_poi, oobbuf);
		if (ret) {
			DEBUG (MTD_DEBUG_LEVEL0, "nand_write_ecc: write_page failed %d\n", ret);
			goto out;
		}

		/* We can't verify a partial page write.
		 * ret = brcmnand_verify_page (mtd, (loff_t) (page << this->page_shift), 
				&buf[written], thislen, oobbuf, eccRetLen, oobsel, autoplace
#ifdef MYDEBUG
		 *		, retlen
#endif
		 *		);
		 */

		
		/* Update written bytes count */
		written = thislen;
		*retlen = written;
	}

//Debug = 30;

	/* Loop until all data are written */
	while (written < len) {
		int thislen = min_t(int, len - written, mtd->oobblock);

		this->data_poi = (u_char*) &buf[written];
//if (!autoplace) gdebug = 1;

//Debug = 40;
//printk("40 retlen=%p\n", retlen);
		if (eccbuf) {
			oobbuf = brcmnand_prepare_oobbuf (mtd, &eccbuf[eccWritten], mtd->oobsize, oob_buf, oobsel, autoplace, &eccRetLen);
if (gdebug) printk("45 retlen=%p, oobbuf=%p\n", retlen, oobbuf);
//gdebug=0;
			oob = 0;
			eccWritten += eccRetLen;
		}
		else {
			oobbuf = NULL;
		}
		/* Write one page. If this is the last page to write
		 * or the last page in this block, then use the
		 * real pageprogram command, else select cached programming
		 * if supported by the chip.
		 */
//Debug = 50;
if (gdebug) printk("50 retlen=%p\n", retlen);
		ret = brcmnand_write_page (mtd, page << this->page_shift, thislen, this->data_poi, oobbuf);
		if (ret) {
			DEBUG (MTD_DEBUG_LEVEL0, "nand_write_ecc: write_page failed %d\n", ret);
			goto out;
		}
//if (!autoplace) gdebug = 1;
//Debug = 60; // gdebug = 3;
if (gdebug) printk("60 retlen=%p, written=%d, thislen=%d, eccRetLen=%d, oobbuf=%p\n", retlen, written, thislen, eccRetLen, oobbuf);
		ret = brcmnand_verify_page (mtd, (loff_t) (page << this->page_shift), 
				&buf[written], thislen, oobbuf, eccRetLen, oobsel, autoplace
#ifdef MYDEBUG
				, retlen
#endif
				);
//printk("65 retlen=%p\n", retlen);
//gdebug=0;
		if (ret) {
			DEBUG (MTD_DEBUG_LEVEL0, "%s: verify_pages failed %d @%08x, written=%d\n", __FUNCTION__, ret, (unsigned long) to, written);
			goto out;
		}
		/* Next oob page */
		oob += mtd->oobsize;
		/* Update written bytes count */
		written += mtd->oobblock;

//Debug = 70;
//printk("70 retlen=%p\n", retlen);
		*retlen = written;

		/* Increment page address */
		page++;
	}
	/* Verify the remaining pages */

out:
	/* Deselect and wake up anyone waiting on the device */
//Debug = 90;
	brcmnand_release_device(mtd);

// Leave debugging on at end of FS
//if ((unsigned int) to == 0x0029f800) ; else gdebug = 0;
//Debug = 99;

if (4 == gdebug) gdebug = 0;
	return ret;

}

/**
 * brcmnand_write - [MTD Interface] compability function for brcmnand_write_ecc
 * @param mtd		MTD device structure
 * @param to		offset to write to
 * @param len		number of bytes to write
 * @param retlen	pointer to variable to store the number of written bytes
 * @param buf		the data to write
 *
 * This function simply calls brcmnand_write_ecc
 * with oob buffer and oobsel = NULL
 */
static int brcmnand_write(struct mtd_info *mtd, loff_t to, size_t len,
	size_t *retlen, const u_char *buf)
{
	DEBUG(MTD_DEBUG_LEVEL3, "-->%s to=%08x, len=%d\n", __FUNCTION__,
 		(unsigned long) to, (int) len);
	return brcmnand_write_ecc(mtd, to, len, retlen, buf, NULL, NULL);
}


/*
 * Allow nested calls inside this module in order to avoid having to release-and-reacquire lock.  
 * Take same arguments as the public interface.
 * Used mostly by BBT routines.
 */
static int brcmnand_write_oob_int(struct mtd_info *mtd, loff_t to, size_t ooblen, size_t* retlen,
	const u_char *oob_buf)
{
	struct brcmnand_chip *this = mtd->priv;
	int column, page, status, eccsteps; 
	int startpage, ret = -EIO, oob = 0, written = 0, eccWritten = 0;
	int autoplace = 0, numpages, totalpages;
	//u_char *oobbuf, *bufstart;
	struct nand_oobinfo* oobsel = this->autooob;

	//DEBUG(MTD_DEBUG_LEVEL3, "%s: to = 0x%08x, len = %i\n", __FUNCTION__, (unsigned int) to, (int) ooblen);

	/* Shift to get page */
	page = (int) (to >> this->page_shift);
	//chipnr = (int) (to >> this->chip_shift);

	/* Mask to get column */
	column = to & (mtd->oobsize - 1);

	/* Initialize return length value */
	*retlen = 0;

	if ((ooblen % mtd->oobsize) != 0) {
		printk(KERN_WARNING "%s called at offset %08x with len(%d) not multiple of oobsize(%d)\n", 
			__FUNCTION__, to, ooblen, mtd->oobsize);
	}

	/* Do not allow writes past end of device */
	if (unlikely((to + ooblen) > (mtd->size+mtd->oobsize))) {
		DEBUG(MTD_DEBUG_LEVEL0, "%s: Attempt write to past end of device\n", __FUNCTION__);
		return -EINVAL;
	}

	/* Grab the lock and see if the device is available */
	// brcmnand_get_device(mtd, FL_WRITING);

	/* Invalidate the page cache, if we write to the cached page */
	if (page == this->pagebuf)
		this->pagebuf = -1;

	eccWritten = 0; // Dest offset
	oob = 0; // SOurce offset
	
	/* Loop until all data are written */
	while (eccWritten < ooblen) {
		int thislen = min_t(int, ooblen - eccWritten, mtd->oobsize);

if (gdebug) printk("%s: thislen=%d, eccWritten=%d, offset=%08x, buf=%p\n",
	__FUNCTION__, thislen, eccWritten, (unsigned int) to,  &oob_buf[oob]);
		
		/* Write one page. If this is the last page to write
		 * or the last page in this block, then use the
		 * real pageprogram command, else select cached programming
		 * if supported by the chip.
		 */
		// No autoplace when this routine is called.
		ret = brcmnand_write_pageoob (mtd, page << this->page_shift, thislen, &oob_buf[oob]);
		if (ret) {
			DEBUG (MTD_DEBUG_LEVEL0, "%s: write_pageoob failed %d\n", __FUNCTION__, ret);
			goto out;
		}

		ret = brcmnand_verify_pageoob (mtd, page << this->page_shift, &oob_buf[oob], thislen, oobsel, autoplace, 0);
		if (ret) {
			DEBUG (MTD_DEBUG_LEVEL0, "%s: verify_pages failed %d\n", __FUNCTION__, ret);
			goto out;
		}
		/* Next oob page */
		oob += mtd->oobsize;
		/* Update written bytes count */
		eccWritten += mtd->oobsize;

		*retlen += mtd->oobsize;

		/* Increment page address */
		page++;
	}
	/* Verify the remaining pages */

out:
	/* Deselect and wake up anyone waiting on the device */
	brcmnand_release_device(mtd);

	return ret;
}


/**
 * brcmnand_write_oob - [MTD Interface] BrcmNAND write out-of-band
 * @param mtd		MTD device structure
 * @param to			offset to write to (of the page, not the oob).
 * @param ooblen		number of OOB bytes to write
 * @param oob_buf	the data to write must be formatted beforehand
 *
 * BrcmNAND write out-of-band
 */
static int brcmnand_write_oob(struct mtd_info *mtd, loff_t to, size_t ooblen, size_t* retlen,
	const u_char *oob_buf)
{
	int ret;

//if ((((unsigned int) to & 0xffff0000) >= 0x02800000) && (((unsigned int) to & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x, len=%d\n", __FUNCTION__, (unsigned int) to, ooblen);

	DEBUG(MTD_DEBUG_LEVEL3, "%s: to = 0x%08x, len = %i\n", __FUNCTION__, (unsigned int) to, (int) ooblen);

	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_WRITING);

	ret = brcmnand_write_oob_int(mtd, to, ooblen, retlen, oob_buf);
	
	/* Deselect and wake up anyone waiting on the device */
	brcmnand_release_device(mtd);

if (4 == gdebug) gdebug = 0;
	return ret;
}

/**
 * brcmnand_write_oobfree - [MTD Interface] NAND write free out-of-band
 * @mtd:	MTD device structure
 * @to:		offset to write to
 * @len:	number of bytes to write
 * @retlen:	pointer to variable to store the number of written bytes
 * @buf:	the data to write
 *
 * NAND write free out-of-band area
 */
static int brcmnand_write_oobfree (struct mtd_info *mtd, 
		loff_t to, size_t len, size_t * retlen, const u_char * oob_buf)
{
	struct brcmnand_chip *this = mtd->priv;
	int column, page, status, eccsteps; 
	int startpage, ret = -EIO, oob = 0, written = 0, eccWritten = 0;
	int autoplace = 1, numpages, totalpages;
	u_char *oobbuf, *bufstart;
	struct nand_oobinfo* oobsel = this->autooob;

//if ((((unsigned int) to & 0xffff0000) >= 0x02800000) && (((unsigned int) to & 0xffff0000) < 0x02e60000))
//	printk("%s: offset=%08x, len=%d\n", __FUNCTION__, (unsigned int) to, len);


	DEBUG(MTD_DEBUG_LEVEL3, "\n%s: to = 0x%08x, len = %i\n", __FUNCTION__, (unsigned int) to, (int) len);
if (gdebug) printk("\n%s: to = 0x%08x, len = %i\n", __FUNCTION__, (unsigned int) to, (int) len);
	/* Shift to get page */
	page = (int) (to >> this->page_shift);
	//chipnr = (int) (to >> this->chip_shift);

	/* Mask to get column */
	column = to & (mtd->oobsize - 1);

	/* Initialize return length value */
	*retlen = 0;

	/* Do not allow writes past end of device */
	if (unlikely((to + len) > (mtd->size+mtd->oobsize))) {
		DEBUG(MTD_DEBUG_LEVEL0, "\n%s: Attempt write to past end of device, to=%08x, len=%d, size=%08x\n", 
			__FUNCTION__, (unsigned int) to, (unsigned int) len, mtd->size);
		return -EINVAL;
	}

	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_WRITING);

	/* Invalidate the page cache, if we write to the cached page */
	if (page == this->pagebuf)
		this->pagebuf = -1;

	eccWritten = 0;
	
	/* Loop until all data are written */
	while (eccWritten < len) {
		int eccRetLen = 0;
		int thislen = min_t(int, len - eccWritten, mtd->oobavail);
		u_char tmp_oob[64];

		//this->oob_buf = (u_char*) &oob_buf[eccWritten];
//gdebug=1;
		oobbuf = brcmnand_prepare_oobbuf (mtd, &oob_buf[eccWritten], thislen, tmp_oob, oobsel, 1, &eccRetLen);
		oob = 0;
gdebug=0;

if (gdebug) {printk("\nwrite_oobfree: oob_buf after prepare thislen=%d, eccRetLen=%d\n", thislen, eccRetLen); 
print_oobbuf(oobbuf, mtd->oobsize);}

		/* Write one page. If this is the last page to write
		 * or the last page in this block, then use the
		 * real pageprogram command, else select cached programming
		 * if supported by the chip.
		 */
//gdebug=1;
		/* We write out the entire OOB buffer, regardless of thislen */
		ret = brcmnand_write_pageoob (mtd, page << this->page_shift, mtd->oobsize, &oobbuf[oob]);
gdebug=0;
		if (ret) {
			DEBUG (MTD_DEBUG_LEVEL0, "%s: write_pageoob failed %d\n", __FUNCTION__, ret);
			goto out;
		}

//gdebug=1;
		ret = brcmnand_verify_pageoob (mtd, page << this->page_shift, &oob_buf[eccWritten], eccRetLen, 
			oobsel, autoplace, 0);
//gdebug=0;
		if (ret) {
			DEBUG (MTD_DEBUG_LEVEL0, "%s: verify_pages failed %d\n", __FUNCTION__, ret);
			goto out;
		}
		eccWritten += eccRetLen;
		/* Next oob page */
		oob += mtd->oobavail;
		/* Update written bytes count */
		eccWritten += mtd->oobavail;
		
		*retlen = written;

		/* Increment page address */
		page++;
	}
	/* Verify the remaining pages TBD */
	ret = 0;

out:
	/* Deselect and wake up anyone waiting on the device */
	brcmnand_release_device(mtd);

if (4 == gdebug) gdebug = 0;
	return ret;
}

/**
 * brcmnand_writev_ecc - [MTD Interface] write with iovec with ecc
 * @param mtd		MTD device structure
 * @param vecs		the iovectors to write
 * @param count		number of vectors
 * @param to		offset to write to
 * @param retlen	pointer to variable to store the number of written bytes
 * @param eccbuf	filesystem supplied oob data buffer 
 * @param oobsel	oob selection structure  
 *
 * BrcmNAND write with iovec with ecc
 */
static int brcmnand_writev_ecc(struct mtd_info *mtd, const struct kvec *vecs,
	unsigned long count, loff_t to, size_t *retlen,
	u_char *eccbuf, struct nand_oobinfo *oobsel)
{
	int i, len, total_len, ret = -EIO, written = 0, chipnr, oobretlen;
	unsigned long pageAddr, startAddr;
	int numpages, autoplace = 0, oobWritten = 0;
	struct brcmnand_chip *this = mtd->priv;
	int	ppblock = (1 << (this->phys_erase_shift - this->page_shift));
	u_char *oobbuf, *bufstart;
	u_char tmp_oob[64];

//if ((((unsigned int) to & 0xffff0000) >= 0x028c0000) && (((unsigned int) to & 0xffff0000) < 0x02900000)) {
//	printk("%s: offset=%08x, count=%d\n", __FUNCTION__, (unsigned int) to, count);
//	gdebug = 4;
//}


	/* Preset written len for early exit */
	*retlen = 0;

	/* Calculate total length of data */
	total_len = 0;
	for (i = 0; i < count; i++)
		total_len += vecs[i].iov_len;

DEBUG(MTD_DEBUG_LEVEL3, "brcmnand_writev_ecc: to = 0x%08x, len = %i, count = %ld, eccbuf=%p, total_len=%d\n", 
	(unsigned int) to, (unsigned int) total_len, count, eccbuf, total_len);

	/* Do not allow write past end of the device */
	if (unlikely((to + total_len) > mtd->size)) {
		DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_writev_ecc: Attempted write past end of device\n");
		return -EINVAL;
	}

	/* Reject writes, which are not page aligned */
        if (unlikely(NOTALIGNED(to)) || unlikely(NOTALIGNED(total_len))) {
                DEBUG(MTD_DEBUG_LEVEL0, "brcmnand_writev_ecc: Attempt to write not page aligned data\n");
                return -EINVAL;
        }

	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_WRITING);

	autoplace = brcmnand_autoplace(mtd, oobsel);

	/* Setup start page, we know that to is aligned on page boundary */
	startAddr = pageAddr = to;

	/* Loop until all keve's data has been written */
	len = 0;
	written = 0;
	oobWritten = 0;
	while (count) {
		/* If the given tuple is >= pagesize then
		 * write it out from the iov
		 */
		
		if ((vecs->iov_len - len) >= mtd->oobblock) {
			/* Calc number of pages we can write
			 * out of this iov in one go */
			numpages = (vecs->iov_len - len) >> this->page_shift;

			bufstart = (u_char *)vecs->iov_base;

			//oob = 0;
			for (i = 0; i < numpages; i++) {
				/* Write one page. If this is the last page to write
				 * then use the real pageprogram command, else select
				 * cached programming if supported by the chip.
				 */
				oobbuf = brcmnand_prepare_oobbuf (mtd, &eccbuf[oobWritten], mtd->oobavail, tmp_oob, oobsel, autoplace, &oobretlen);
if (gdebug > 3) printk("%s: write_page(pageAddr=%08x, oobbuf=%p oob=%d, oobretlen=%d\n", __FUNCTION__, pageAddr, oobbuf, oobWritten, oobretlen);
				ret = brcmnand_write_page (mtd, pageAddr, mtd->oobblock, &bufstart[len], oobbuf);
				if (ret) {
					printk("%s: brcmnand_write_page failed, ret=%d\n", __FUNCTION__, ret);
					goto out;
				}
				len += mtd->oobblock;
				//oob += oobretlen;
				pageAddr += mtd->oobblock;
				written += mtd->oobblock;
				oobWritten += oobretlen;
			}
			/* Check, if we have to switch to the next tuple */
			if (len >= (int) vecs->iov_len) {
				vecs++;
				len = 0;
				count--;
			}
		} else {
			/* We must use the internal buffer, read data out of each
			 * tuple until we have a full page to write
			 */
			int cnt = 0;
			len = 0;
			while (cnt < mtd->oobblock) {
				if (vecs->iov_base != NULL && vecs->iov_len)
					this->data_buf[cnt++] = ((u_char *) vecs->iov_base)[len++];
				/* Check, if we have to switch to the next tuple */
				if (len >= (int) vecs->iov_len) {
					vecs++;
					len = 0;
					count--;
				}
			}
			bufstart = &this->data_buf[0];
			numpages = 1;
			oobbuf = brcmnand_prepare_oobbuf (mtd, eccbuf, mtd->oobavail, tmp_oob, oobsel, autoplace, &oobretlen);
if (gdebug > 3) printk("%s: write_page(pageAddr=%08x, oobbuf=%p oob=%d, oobretlen=%d\n", __FUNCTION__, pageAddr, oobbuf, oobWritten, oobretlen);
			ret = brcmnand_write_page (mtd, pageAddr, mtd->oobblock, bufstart, oobbuf);
			if (ret) {
				printk("%s: brcmnand_write_page failed, ret=%d\n", __FUNCTION__, ret);
				goto out;
			}
			pageAddr += mtd->oobblock;
			written += mtd->oobblock;
			oobWritten += oobretlen;
		}

		/* All done ? */
		if (!count)
			break;

		startAddr = pageAddr;

#if 0
		/* Check, if we cross a chip boundary */
		if (!startpage) {
			chipnr++;
			this->select_chip(mtd, -1);
			this->select_chip(mtd, chipnr);
		}
#endif
	}
	ret = 0;
out:
	/* Deselect and wake up anyone waiting on the device */
	brcmnand_release_device(mtd);

	*retlen = written;

if (gdebug > 3) gdebug = 0;
	return ret;
}

/**
 * brcmnand_writev - [MTD Interface] compabilty function for brcmnand_writev_ecc
 * @param mtd		MTD device structure
 * @param vecs		the iovectors to write
 * @param count		number of vectors
 * @param to		offset to write to
 * @param retlen	pointer to variable to store the number of written bytes
 *
 * BrcmNAND write with kvec. This just calls the ecc function
 */
static int brcmnand_writev(struct mtd_info *mtd, const struct kvec *vecs,
	unsigned long count, loff_t to, size_t *retlen)
{
	DEBUG(MTD_DEBUG_LEVEL3, "-->%s to=%08x, count=%d\n", __FUNCTION__,
 		(unsigned long) to, (int) count);
	return brcmnand_writev_ecc(mtd, vecs, count, to, retlen, NULL, NULL);
}

/**
 * brcmnand_block_checkbad - [GENERIC] Check if a block is marked bad
 * @param mtd		MTD device structure
 * @param ofs		offset from device start
 * @param getchip	0, if the chip is already selected
 * @param allowbbt	1, if its allowed to access the bbt area
 *
 * Check, if the block is bad. Either by reading the bad block table or
 * calling of the scan function.
 */
static int brcmnand_block_checkbad(struct mtd_info *mtd, loff_t ofs, int getchip, int allowbbt)
{
	struct brcmnand_chip *this = mtd->priv;
	//struct bbm_info *bbm = this->bbm;
	
	/* Return info from the table */
	return this->isbad_bbt(mtd, ofs, allowbbt);
}

/**
 * brcmnand_erase_bbt - [Private] erase block(s)
 * @param mtd		MTD device structure
 * @param instr		erase instruction
 * @allowBBT			allow erase of BBT
 *
 * Erase one ore more blocks
 */
static int brcmnand_erase_bbt(struct mtd_info *mtd, struct erase_info *instr, int allowbbt)
{
	struct brcmnand_chip *this = mtd->priv;
	unsigned int block_size;
	loff_t addr;
	int len;
	int ret = 0;
	int needBBT;
	int autoplace = 0;

	DEBUG(MTD_DEBUG_LEVEL3, "%s: start = 0x%08x, len = %08x\n", __FUNCTION__, (unsigned int) instr->addr, (unsigned int) instr->len);

if (allowbbt)
printk("--> %s: AllowBBT addr=%08x, len=%d\n", __FUNCTION__, instr->addr, instr->len);

	block_size = (1 << this->erase_shift);


	/* Start address must align on block boundary */
	if (unlikely(instr->addr & (block_size - 1))) {
		DEBUG(MTD_DEBUG_LEVEL0, "%s: Unaligned address\n", __FUNCTION__);
		return -EINVAL;
	}

	/* Length must align on block boundary */
	if (unlikely(instr->len & (block_size - 1))) {
		DEBUG(MTD_DEBUG_LEVEL0, 
"%s: Length not block aligned, len=%08x, blocksize=%08x, chip->blkSize=%08x, chip->erase=%d\n",
__FUNCTION__, instr->len, block_size, this->blockSize, this->erase_shift);
		return -EINVAL;
	}

	/* Do not allow erase past end of device */
	if (unlikely((instr->len + instr->addr) > mtd->size)) {
		DEBUG(MTD_DEBUG_LEVEL0, "%s: Erase past end of device\n", __FUNCTION__);
		return -EINVAL;
	}

	instr->fail_addr = 0xffffffff;

	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_ERASING);

	/*
	 * Clear ECC registers 
	 */
	this->ctrl_write(BCHP_NAND_ECC_CORR_ADDR, 0);
	this->ctrl_write(BCHP_NAND_ECC_UNC_ADDR, 0);

	/* Loop throught the pages */
	len = instr->len;
	addr = instr->addr;

	instr->state = MTD_ERASING;

	while (len) {

#if 1
/* THT: One time deal, allow erase of BBT */
if (gClearBBT && !allowbbt)
#endif
	
		/* Check if we have a bad block, we do not erase bad blocks */
		if (brcmnand_block_checkbad(mtd, addr, 0, allowbbt)) {
			printk (KERN_ERR "%s: attempt to erase a bad block at addr 0x%08x\n", __FUNCTION__, (unsigned int) addr);
			instr->state = MTD_ERASE_FAILED;
			goto erase_one_block;
		}

		//this->command(mtd, ONENAND_CMD_ERASE, addr, block_size);

		this->ctrl_writeAddr(this, addr);
	
		this->ctrl_write(BCHP_NAND_CMD_START, OP_BLOCK_ERASE);

		// Wait until flash is ready
		ret = brcmnand_write_is_complete(mtd, &needBBT);
		
		/* Check, if it is write protected: TBD */
		if (needBBT ) {
			if ( !allowbbt) {
				printk(KERN_ERR "brcmnand_erase: Failed erase, block %d, flash status=%08x\n", 
					(unsigned) (addr >> this->erase_shift), needBBT);
				instr->state = MTD_ERASE_FAILED;
				instr->fail_addr = addr;

				printk(KERN_WARNING "%s: Marking bad block @%08x\n", __FUNCTION__, (unsigned int) addr);
				(void) this->block_markbad(mtd, addr);
				goto erase_one_block;
			}
		}
erase_one_block:

		len -= block_size;
		addr += block_size;
	}

	instr->state = MTD_ERASE_DONE;

erase_exit:

	ret = instr->state == MTD_ERASE_DONE ? 0 : -EIO;
	/* Do call back function */
	if (!ret) {
if (gdebug) printk("%s: Calling mtd_erase_callback\n", __FUNCTION__);
		mtd_erase_callback(instr);
	}


#if 0  //CONFIG_MTD_BRCMNAND_VERIFY_WRITE
{
	int retlen;

	/* Loop throught the pages */
	len = instr->len;
	addr = instr->addr;

	while (len > 0) {
		

#if 1 // Verify the OOB only, disable it, testing 1 2 3 12/16/06
		//Verify OOB erase on each page
//gdebug=1;
//printk("\n%s: verify erase OOB at %08x\n", __FUNCTION__, (unsigned long) (addr));
		ret = brcmnand_verify_pageoob(mtd, addr, ffchars, mtd->oobavail, this->autooob, 
			autoplace, allowbbt);
#else
		ret = brcmnand_verify_page(mtd, addr, ffchars, mtd->oobblock, ffchars, mtd->oobavail, this->autooob, 
			autoplace);
#endif
		if (ret) {
			printk(KERN_ERR "%s: erase failed at addr %08x\n", __FUNCTION__, addr);
			// TBD: Put it into BBT
			goto verify_exit;
		}
gdebug=0;
		len -= mtd->oobblock;
		addr += mtd->oobblock;
	}

}

verify_exit:
#endif


	/* Deselect and wake up anyone waiting on the device */
	brcmnand_release_device(mtd);

	return ret;
}


/**
 * brcmnand_erase - [MTD Interface] erase block(s)
 * @param mtd		MTD device structure
 * @param instr		erase instruction
 *
 * Erase one ore more blocks
 */
static int brcmnand_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	DEBUG(MTD_DEBUG_LEVEL3, "-->%s addr=%08x, len=%d\n", __FUNCTION__,
 		(unsigned long) instr->addr, instr->len);
	return brcmnand_erase_bbt(mtd, instr, 0); // Do not allow erase of BBT.
}

/**
 * brcmnand_sync - [MTD Interface] sync
 * @param mtd		MTD device structure
 *
 * Sync is actually a wait for chip ready function
 */
static void brcmnand_sync(struct mtd_info *mtd)
{
	DEBUG(MTD_DEBUG_LEVEL3, "brcmnand_sync: called\n");

	/* Grab the lock and see if the device is available */
	brcmnand_get_device(mtd, FL_SYNCING);

	/* Release it and go back */
	brcmnand_release_device(mtd);
}


/**
 * brcmnand_block_isbad - [MTD Interface] Check whether the block at the given offset is bad
 * @param mtd		MTD device structure
 * @param ofs		offset relative to mtd start
 *
 * Check whether the block is bad
 */
static int brcmnand_block_isbad(struct mtd_info *mtd, loff_t ofs)
{

	DEBUG(MTD_DEBUG_LEVEL3, "-->%s ofs=%08x\n", __FUNCTION__, (unsigned int) ofs);
	/* Check for invalid offset */
	if (ofs > mtd->size)
		return -EINVAL;

	return brcmnand_block_checkbad(mtd, ofs, 1, 0);
}

/**
 * brcmnand_default_block_markbad - [DEFAULT] mark a block bad
 * @param mtd		MTD device structure
 * @param ofs		offset from device start
 *
 * This is the default implementation, which can be overridden by
 * a hardware specific driver.
 */
static int brcmnand_default_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
	struct brcmnand_chip *this = mtd->priv;
	//struct bbm_info *bbm = this->bbm;
	u_char bbmarker[2] = {0, 0};
	u_char buf[64];
	size_t retlen;
	int block;
	unsigned long ulofs;

	ulofs =( (unsigned long) ofs) & ~ this->page_mask;
PRINTK("-->%s ofs=%08x, ulofs=%08x\n", __FUNCTION__, (unsigned int) ofs, ulofs);

	

	/* Get block number */
	block = ((int) ulofs) >> this->erase_shift;
      if (this->bbt)
                this->bbt[block >> 2] |= 0x01 << ((block & 0x03) << 1); 

#if 0
	/* This scheme assumes that the odd page are for OOB, BrcmNand does not use this */
       /* We write two bytes, so we dont have to mess with 16 bit access */
       ofs += mtd->oobsize + (this->badblockpos & ~0x01);
       return mtd->write_oob(mtd, ofs , 2, &retlen, bbmarker);
#else
	memcpy(buf, ffchars, sizeof(buf));
	memcpy(&buf[this->badblockpos], bbmarker, sizeof(bbmarker));

	// Lock already held by caller, so cant call mtd->write_oob directly
       return brcmnand_write_oob_int(mtd, ulofs, sizeof(buf), &retlen, buf);
#endif
}

/**
 * brcmnand_block_markbad - [MTD Interface] Mark the block at the given offset as bad
 * @param mtd		MTD device structure
 * @param ofs		offset relative to mtd start
 *
 * Mark the block as bad
 */
static int brcmnand_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
	struct brcmnand_chip *this = mtd->priv;
	int ret;

	DEBUG(MTD_DEBUG_LEVEL3, "-->%s ofs=%08x\n", __FUNCTION__, (unsigned int) ofs);
	
	ret = brcmnand_block_isbad(mtd, ofs);
	if (ret) {
		/* If it was bad already, return success and do nothing */
		if (ret > 0)
			return 0;
		return ret;
	}

	return this->block_markbad(mtd, ofs);
}

/**
 * brcmnand_unlock - [MTD Interface] Unlock block(s)
 * @param mtd		MTD device structure
 * @param ofs		offset relative to mtd start
 * @param len		number of bytes to unlock
 *
 * Unlock one or more blocks
 */
static int brcmnand_unlock(struct mtd_info *mtd, loff_t llofs, size_t len)
{
	struct brcmnand_chip *this = mtd->priv;
	int status;
	unsigned long blkAddr, ofs = (unsigned long) llofs;

	DEBUG(MTD_DEBUG_LEVEL3, "-->%s llofs=%08x, len=%d\n", __FUNCTION__,
 		(unsigned long) llofs, (int) len);

#if 0
	start = ofs >> this->erase_shift;
	end = len >> this->erase_shift;

	/* Continuous lock scheme */
	//if (this->options & ONENAND_CONT_LOCK) 
	{
		/* Set start block address */
		this->ctrl_writeAddr(this, start);
		/* Set end block address */
		this->ctrl_write(BCHP_NAND_CMD_END_ADDRESS, end - 1);
		/* Write unlock command */
		this->ctrl_write(BCHP_NAND_CMD_START, OP_BLOCKS_UNLOCK);

		/* There's no return value */
		this->wait(mtd, FL_UNLOCKING);

		/* Sanity check */
#if 0
		while (this->read_word(this->base + ONENAND_REG_CTRL_STATUS)
		    & ONENAND_CTRL_ONGO)
			continue;

		/* Check lock status */
		status = this->read_word(this->base + ONENAND_REG_WP_STATUS);
		if (!(status & ONENAND_WP_US))
			printk(KERN_ERR "wp status = 0x%x\n", status);
#endif
		return 0;
	}

#endif /* Continous lock */

	/* Block lock scheme */
	for (blkAddr = ofs; blkAddr < ofs + len; blkAddr += this->blockSize) {
		/* Set block address */
		//value = brcmnand_block_address(this, block); // THT: We don't have DDP support
//printk("%s: blockAddr=%08x\n", __FUNCTION__, blkAddr);
		this->ctrl_writeAddr(this, blkAddr); 
		/* Set end block address */
		this->ctrl_write(BCHP_NAND_CMD_END_ADDRESS, this->pbase+ blkAddr + this->blockSize - 1);
		/* Write unlock command */
		this->ctrl_write(BCHP_NAND_CMD_START, OP_BLOCKS_UNLOCK);


		/* There's no return value */
		this->wait(mtd, FL_UNLOCKING, &status);

		if (status & 0xff)  
			printk(KERN_ERR "block = %ul, wp status = 0x%x\n", blkAddr, status);

		/* Sanity check */
#if 0
		while (this->read_word(this->base + ONENAND_REG_CTRL_STATUS)
		    & ONENAND_CTRL_ONGO)
			continue;
#endif

		/* Check lock status */
		this->ctrl_writeAddr(this, blkAddr); 
		this->ctrl_write(BCHP_NAND_CMD_START, OP_READ_BLOCKS_LOCK_STATUS);
		status = this->ctrl_read(BCHP_NAND_BLOCK_LOCK_STATUS);
		//status = this->read_word(this->base + ONENAND_REG_WP_STATUS);

	}

	return 0;
}


/**
 * brcmnand_print_device_info - Print device ID
 * @param device        device ID
 *
 * Print device ID
 */
static void brcmnand_print_device_info(brcmnand_chip_Id* chipId, unsigned long flashSize)
{
	printk(KERN_INFO "BrcmNAND mfg %x %x %s %dMB\n",
                chipId->mafId, chipId->chipId, chipId->chipIdStr, (int) (flashSize >>20));

	print_config_regs();

}

#if 0

static const struct brcmnand_manufacturers brcmnand_manuf_ids[] = {
        {ONENAND_MFR_SAMSUNG, "Samsung"},
};

/**
 * brcmnand_check_maf - Check manufacturer ID
 * @param manuf         manufacturer ID
 *
 * Check manufacturer ID
 */
static int brcmnand_check_maf(int manuf)
{
	int size = ARRAY_SIZE(brcmnand_manuf_ids);
	char *name;
        int i;

        for (i = 0; i < size; i++)
                if (manuf == brcmnand_manuf_ids[i].id)
                        break;

	if (i < size)
		name = brcmnand_manuf_ids[i].name;
	else
		name = "Unknown";

        printk(KERN_DEBUG "BrcmNAND Manufacturer: %s (0x%0x)\n", name, manuf);

        return (i == size);
}

#endif

/*
 * bit 31: 	1 = OTP read-only
 * 30: 		Page Size: 0 = PG_SIZE_512, 1 = PG_SIZE_2KB
 * 28-29: 	Block size: 3=512K, 1 = 128K, 0 = 16K, 2 = 8K
 * 27:		Reserved
 * 24-26:	Device_Size
 *			0:	4MB
 *			1:	8MB
 *			2: 	16MB
 *			3:	32MB
 *			4:	64MB
 *			5:	128MB
 *			6: 	256MB
 * 23:		Dev_Width 0 = Byte8, 1 = Word16
 * 22-19: 	Reserved
 * 18:16:	Full Address Bytes
 * 15		Reserved
 * 14:12	Col_Adr_Bytes
 * 11:		Reserved
 * 10-08	Blk_Adr_Bytes
 * 07-00	Reserved
 */
 
void
brcmnand_decode_config(struct brcmnand_chip* chip, uint32_t nand_config)
{
	unsigned int chipSizeShift;
	
	//chip->chipSize = (nand_config & 0x07000000) >> (24 - 20);

	switch ((nand_config & 0x30000000) >> 28) {
		case 3:
			chip->blockSize = 512 << 10;
			break;
		case 2:
			chip->blockSize = 8 << 10;
			break;
		case 1:	
			chip->blockSize = 128 << 10;
			break;
		case 0:
			chip->blockSize = 16 << 10;
			break;
	}
	chip->erase_shift = ffs(chip->blockSize) - 1;

	switch((nand_config & 0x40000000) >> 30) {
		case 0:
			chip->pageSize= 512;
			break;
		case 1:
			chip->pageSize = 2048;
			break;
	}
	chip->page_shift = ffs(chip->pageSize) - 1;
	chip->page_mask = (1 << chip->page_shift) - 1;

	chipSizeShift = (nand_config & 0x07000000) >> 24;
	chip->chipSize = (4 << 20) << chipSizeShift;

	chip->busWidth = 1 + ((nand_config & 0x00400000) >> 23);

	printk(KERN_INFO "NAND Config: Reg=%08x, chipSize=%dMB, blockSize=%dK, erase_shift=%x\n",
		nand_config, chip->chipSize >> 20, chip->blockSize >> 10, chip->erase_shift);
	printk(KERN_INFO "busWidth=%d, pageSize=%dB, page_shift=%d, page_mask=%08x\n", 
		chip->busWidth, chip->pageSize, chip->page_shift , chip->page_mask);

}

/*
 * Adjust timing pattern if specified in chip ID list
 * Also take dummy entries, but no adjustments will be made.
 */
static void brcmnand_adjust_timings(struct brcmnand_chip *this, brcmnand_chip_Id* chip)
{
		unsigned long nand_timing1 = this->ctrl_read(BCHP_NAND_TIMING_1);
		unsigned long nand_timing1_b4;
		unsigned long nand_timing2 = this->ctrl_read(BCHP_NAND_TIMING_2);
		unsigned long nand_timing2_b4;
		
	// Adjust NAND timings:
	if (chip->timing1) {
		nand_timing1_b4 = nand_timing1;

		if (chip->timing1 & BCHP_NAND_TIMING_1_tWP_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tWP_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tWP_MASK);  
		}
		if (chip->timing1 & BCHP_NAND_TIMING_1_tWH_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tWH_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tWH_MASK);  
		}
		if (chip->timing1 & BCHP_NAND_TIMING_1_tRP_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tRP_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tRP_MASK);  
		}
		if (chip->timing1 & BCHP_NAND_TIMING_1_tREH_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tREH_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tREH_MASK);  
		}
		if (chip->timing1 & BCHP_NAND_TIMING_1_tCS_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tCS_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tCS_MASK);  
		}
		if (chip->timing1 & BCHP_NAND_TIMING_1_tCLH_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tCLH_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tCLH_MASK);  
		}
		if (chip->timing1 & BCHP_NAND_TIMING_1_tALH_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tALH_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tALH_MASK);  
		}
		if (chip->timing1 & BCHP_NAND_TIMING_1_tADL_MASK) {
			nand_timing1 &= ~BCHP_NAND_TIMING_1_tADL_MASK;
			nand_timing1 |= (chip->timing1 & BCHP_NAND_TIMING_1_tADL_MASK);  
		}

		this->ctrl_write(BCHP_NAND_TIMING_1, nand_timing1);
printk("Adjust timing1: Was %08x, changed to %08x\n", nand_timing1_b4, nand_timing1);
	}
	else {
printk("timing1 not adjusted: %08x\n", nand_timing1);
	}

	// Adjust NAND timings:
	if (chip->timing2) {
		nand_timing2_b4 = nand_timing2;

		if (chip->timing2 & BCHP_NAND_TIMING_2_tWB_MASK) {
			nand_timing2 &= ~BCHP_NAND_TIMING_2_tWB_MASK;
			nand_timing2 |= (chip->timing2 & BCHP_NAND_TIMING_2_tWB_MASK);  
		}
		if (chip->timing2 & BCHP_NAND_TIMING_2_tWHR_MASK) {
			nand_timing2 &= ~BCHP_NAND_TIMING_2_tWHR_MASK;
			nand_timing2 |= (chip->timing2 & BCHP_NAND_TIMING_2_tWHR_MASK);  
		}
		if (chip->timing2 & BCHP_NAND_TIMING_2_tREAD_MASK) {
			nand_timing2 &= ~BCHP_NAND_TIMING_2_tREAD_MASK;
			nand_timing2 |= (chip->timing2 & BCHP_NAND_TIMING_2_tREAD_MASK);  
		}

		this->ctrl_write(BCHP_NAND_TIMING_2, nand_timing2);
printk("Adjust timing2: Was %08x, changed to %08x\n", nand_timing2_b4, nand_timing2);
	}
	else {
printk("timing2 not adjusted: %08x\n", nand_timing2);
	}
}


/**
 * brcmnand_probe - [BrcmNAND Interface] Probe the BrcmNAND device
 * @param mtd		MTD device structure
 *
 * BrcmNAND detection method:
 *   Compare the the values from command with ones from register
 */
static int brcmnand_probe(struct mtd_info *mtd)
{
	struct brcmnand_chip *this = mtd->priv;
	unsigned char brcmnand_maf_id, brcmnand_dev_id;
	uint32 dev_id;
	uint32_t nand_config;
	int version_id;
	//int density;
	int i;

PRINTK("-->brcmnand_probe: this=%p, this->ctrl_read=%p\n", this, this->ctrl_read);

	/* Send the command for reading device ID from controller */
	dev_id = this->ctrl_read(BCHP_NAND_FLASH_DEVICE_ID);

	printk(KERN_INFO "brcmnand_probe: dev_id=%x\n", dev_id);
#if 1
	/* Read manufacturer and device IDs from Controller */
	brcmnand_maf_id = (dev_id >> 24) & 0xff;
	brcmnand_dev_id = (dev_id >> 16) & 0xff;

	/* Check manufacturer ID */
	//if (brcmnand_check_maf(brcmnand_maf_id))
	//	return -ENXIO;
#endif

	/* Look up in our table for infos on device */
	for (i=0; i < BRCMNAND_MAX_CHIPS; i++) {
		if (brcmnand_dev_id == brcmnand_chips[i].chipId 
			&& brcmnand_maf_id == brcmnand_chips[i].mafId)
			break;
	}

	if (i >= BRCMNAND_MAX_CHIPS) {
#ifdef CONFIG_MIPS_BCM7400A0
		printk(KERN_ERR "DevId %08x may not be supported\n", (unsigned int) dev_id);
		/* Because of the bug in the controller in the first version,
		 * if we can't identify the chip, we punt
		 */
		return (-EINVAL);
#else
		printk(KERN_WARNING"DevId %08x may not be supported.  Will use config info\n", (unsigned int) dev_id);
#endif
	}

	nand_config = this->ctrl_read(BCHP_NAND_CONFIG);

#ifdef CONFIG_MIPS_BCM7400A0
	// Workaround for bug in 7400A0 returning invalid config
	switch(i) { 
	case 0: /* SamSung NAND 1Gbit */
	case 1: /* ST NAND 1Gbit */
	case 4:
	case 5:
		/* Large page, 128K erase block
		PAGE_SIZE = 0x1 = 1b = PG_SIZE_2KB
		BLOCK_SIZE = 0x1 = 01b = BK_SIZE_128KB
		DEVICE_SIZE = 0x5 = 101b = DVC_SIZE_128MB
		DEVICE_WIDTH = 0x0 = 0b = DVC_WIDTH_8
		FUL_ADR_BYTES = 5 = 101b
		COL_ADR_BYTES = 2 = 010b
		BLK_ADR_BYTES = 3 = 011b
		*/
		nand_config &= ~0x30000000;
		nand_config |= 0x10000000; // bit 29:28 = 1 ===> 128K erase block
		//nand_config = 0x55042200; //128MB, 0x55052300  for 256MB
		this->ctrl_write(BCHP_NAND_CONFIG, nand_config);

		break;

	case 2:
	case 3:
		/* Small page, 16K erase block
		PAGE_SIZE = 0x0 = 0b = PG_SIZE_512B
		BLOCK_SIZE = 0x0 = 0b = BK_SIZE_16KB
		DEVICE_SIZE = 0x5 = 101b = DVC_SIZE_128MB
		DEVICE_WIDTH = 0x0 = 0b = DVC_WIDTH_8
		FUL_ADR_BYTES = 5 = 101b
		COL_ADR_BYTES = 2 = 010b
		BLK_ADR_BYTES = 3 = 011b
		*/
		nand_config &= ~0x70000000;
		this->ctrl_write(BCHP_NAND_CONFIG, nand_config);

		break;
	
	default:
		printk(KERN_ERR "%s: DevId %08x not supported\n", __FUNCTION__, (unsigned int) dev_id);
		BUG();
		break;
	}
#endif
	brcmnand_decode_config(this, nand_config);

	// Also works for dummy entries, but no adjustments possible
	brcmnand_adjust_timings(this, &brcmnand_chips[i]);

	/* Flash device information */
	brcmnand_print_device_info(&brcmnand_chips[i], this->chipSize);
	this->options = brcmnand_chips[i].options;
	this->device_id = dev_id;
		
	/* BrcmNAND page size & block size */	
	mtd->oobblock = this->pageSize; 		
	mtd->oobsize = mtd->oobblock >> 5; // tht - 16 byte OOB for 512B page, 64B for 2K page
	mtd->erasesize = this->blockSize;

	/* Fix me: When we have both a NOR and NAND flash on board */

	mtd->size = this->chipSize;


	/* Version ID */
	version_id = this->ctrl_read(BCHP_NAND_REVISION);
	printk(KERN_INFO "BrcmNAND version = 0x%04x %dMB @%p\n", 
		version_id, this->chipSize>>20, this->vbase);

	return 0;
}

/**
 * brcmnand_suspend - [MTD Interface] Suspend the BrcmNAND flash
 * @param mtd		MTD device structure
 */
static int brcmnand_suspend(struct mtd_info *mtd)
{
	DEBUG(MTD_DEBUG_LEVEL3, "-->%s  \n", __FUNCTION__);
	return brcmnand_get_device(mtd, FL_PM_SUSPENDED);
}

/**
 * brcmnand_resume - [MTD Interface] Resume the BrcmNAND flash
 * @param mtd		MTD device structure
 */
static void brcmnand_resume(struct mtd_info *mtd)
{
	struct brcmnand_chip *this = mtd->priv;

	DEBUG(MTD_DEBUG_LEVEL3, "-->%s  \n", __FUNCTION__);
	if (this->state == FL_PM_SUSPENDED)
		brcmnand_release_device(mtd);
	else
		printk(KERN_ERR "resume() called for the chip which is not"
				"in suspended state\n");
}


/**
 * fill_autooob_layout - [NAND Interface] build the layout for hardware ECC case
 * @mtd:	MTD device structure
 * @eccbytes:	Number of ECC bytes per page
 *
 * Build the page_layout array for NAND page handling for hardware ECC
 * handling basing on the nand_oobinfo structure supplied for the chip
 */
static int fill_autooob_layout(struct mtd_info *mtd)
{
	struct brcmnand_chip *this = mtd->priv;
	struct nand_oobinfo *oob = this->autooob;
	int oobfreesize = 0;
	int i, res = 0;
	int eccpos = 0, eccbytes = 0, cur = 0, oobcur = 0;

PRINTK("-->fill_autooob_layout\n");
	this->layout = kmalloc(HW_AUTOOOB_LAYOUT_SIZE * sizeof (struct page_layout_item), GFP_KERNEL);

	if (this->layout == NULL) {
		printk(KERN_ERR "%s: kmalloc failed\n", __FUNCTION__);
		return -ENOMEM;
	}
	else
		this->layout_allocated = 1;

	memset(this->layout, 0, HW_AUTOOOB_LAYOUT_SIZE * sizeof (struct page_layout_item));


	this->layout[0].type = ITEM_TYPE_DATA;
	this->layout[0].length = mtd->oobblock;
	DEBUG (MTD_DEBUG_LEVEL3, "fill_autooob_layout: data type, length %d\n", this->layout[0].length);

	i = 1;
	// THT: Our layout is not uniform across eccsteps, so we must scan the entire layout,
	// and we cannot replicate it.
	while (i < HW_AUTOOOB_LAYOUT_SIZE && cur < mtd->oobsize) {
		if (oob->oobfree[oobcur][0] == cur) {
			int len = oob->oobfree[oobcur][1];
			oobfreesize += this->layout[i].length;
			oobcur++;
			if (i > 0 && this->layout[i-1].type == ITEM_TYPE_OOBFREE) {
				i--;
				cur -= this->layout[i].length;
				this->layout[i].length += len;
				DEBUG (MTD_DEBUG_LEVEL3, "fill_autooob_layout: oobfree concatenated, aggregate length %d\n", this->layout[i].length);
			} else {
				this->layout[i].type = ITEM_TYPE_OOBFREE;
				this->layout[i].length = len;
				DEBUG (MTD_DEBUG_LEVEL3, "fill_autooob_layout: oobfree type, length %d\n", this->layout[i].length);
			}
		} else if (oob->eccpos[eccpos] == cur) {
			int eccpos_cur = eccpos;
			do  {
				eccpos++;
				eccbytes++;
			} while (eccbytes < oob->eccbytes && oob->eccpos[eccpos] == oob->eccpos[eccpos+1] - 1);
			eccpos++;
			eccbytes++;
			this->layout[i].type = ITEM_TYPE_ECC;
			this->layout[i].length = eccpos - eccpos_cur;
			DEBUG (MTD_DEBUG_LEVEL3, "fill_autooob_layout: ecc type, length %d\n", this->layout[i].length);
		} else {
			int len = min_t(int, oob->eccpos[eccpos], oob->oobfree[oobcur][0]);
			if (i > 0 && this->layout[i-1].type == ITEM_TYPE_OOB) {
				i--;
				cur -= this->layout[i].length;
				this->layout[i].length += len;
				DEBUG (MTD_DEBUG_LEVEL3, "fill_autooob_layout: oob concatenated, aggregate length %d\n", this->layout[i].length);
			} else {
				this->layout[i].type = ITEM_TYPE_OOB;
				this->layout[i].length = len;
				DEBUG (MTD_DEBUG_LEVEL3, "fill_autooob_layout: oob type, length %d\n", this->layout[i].length);
			}
		}
		cur += this->layout[i].length;
		i++;
	}

	return res;
}


static void fill_ecccmp_mask(struct mtd_info *mtd)
{
	struct brcmnand_chip *this = mtd->priv;
	int i, len;
	struct nand_oobinfo* oobsel = this->autooob;
	unsigned char* myEccMask = (unsigned char*) eccmask; // Defeat const

	/* 
	 * Should we rely on eccmask being zeroed out
	 */
	for (i=0; i < ARRAY_SIZE(eccmask); i++) {
		myEccMask[i] = 0;
	}
	/* Write 0xFF where there is a free byte */
	for (i = 0, len = 0; len < mtd->oobavail && len < mtd->oobsize && i < ARRAY_SIZE(oobsel->oobfree); i++) {
		int to = oobsel->oobfree[i][0];
		int num = oobsel->oobfree[i][1];

		memcpy (&myEccMask[to], ffchars, num);
		len += num;
	}
}


/**
 * brcmnand_scan - [BrcmNAND Interface] Scan for the BrcmNAND device
 * @param mtd		MTD device structure
 * @param maxchips	Number of chips to scan for
 *
 * This fills out all the not initialized function pointers
 * with the defaults.
 * The flash ID is read and the mtd/chip structures are
 * filled with the appropriate values.
 *
 * THT: For now, machips should always be 1.
 */
int brcmnand_scan(struct mtd_info *mtd , int maxchips )
{
	struct brcmnand_chip* this = (struct brcmnand_chip*) mtd->priv;
	unsigned char brcmnand_maf_id;
	int err, i;

	if (!this->ctrl_read)
		this->ctrl_read = brcmnand_ctrl_read;
	if (!this->ctrl_write)
		this->ctrl_write = brcmnand_ctrl_write;
	if (!this->ctrl_writeAddr)
		this->ctrl_writeAddr = brcmnand_ctrl_writeAddr;

/*
	if (!this->command)
		this->command = brcmnand_command;
*/
	if (!this->wait)
		this->wait = brcmnand_wait;

#if 0
	if (!this->write_byte)
		this->write_byte = brcmnand_write_byte;
	if (!this->read_byte)
		this->read_byte = brcmnand_read_byte;
	if (!this->write_word)
		this->write_word = brcmnand_write_word;
	if (!this->read_word)
		this->read_word = brcmnand_read_word;
	if (!this->block_bad)
		this->block_bad = brcmnand_block_isbad;
//	if (!this->block_markbad)
//		this->block_markbad = brcmnand_default_block_markbad;
	if (!this->write_buf)
		this->write_buf = brcmnand_write_buf;
	if (!this->read_buf)
		this->read_buf = brcmnand_read_buf;
	if (!this->verify_buf)
		this->verify_buf = brcmnand_verify_buf;
//	if (!this->scan_bbt)
//		this->scan_bbt = brcmnand_default_bbt;
#endif

	if (!this->block_markbad)
		this->block_markbad = brcmnand_default_block_markbad;
	if (!this->scan_bbt)
		this->scan_bbt = brcmnand_default_bbt;
	if (!this->erase_bbt)
		this->erase_bbt = brcmnand_erase_bbt;

PRINTK("brcmnand_scan: Calling brcmnand_probe\n");
	if (brcmnand_probe(mtd))
		return -ENXIO;

PRINTK("brcmnand_scan: Done brcmnand_probe\n");

	this->bbt_erase_shift =  ffs(mtd->erasesize) - 1;

	/* Calculate the address shift from the page size */	
	this->page_shift = ffs(mtd->oobblock) - 1;
	this->bbt_erase_shift = this->phys_erase_shift = ffs(mtd->erasesize) - 1;
	this->chip_shift = ffs(this->chipSize) - 1;

	printk(KERN_INFO "page_shift=%d, bbt_erase_shift=%d, chip_shift=%d, phys_erase_shift=%d\n",
		this->page_shift, this->bbt_erase_shift , this->chip_shift, this->phys_erase_shift);

	/* Set the bad block position */
	this->badblockpos = mtd->oobblock > 512 ? 
		NAND_LARGE_BADBLOCK_POS : NAND_SMALL_BADBLOCK_POS;

#if 0
/* THT: FOr now */
		/* Get chip options, preserve non chip based options */
		this->options &= ~NAND_CHIPOPTIONS_MSK;
		this->options |= nand_flash_ids[i].options & NAND_CHIPOPTIONS_MSK;
		/* Set this as a default. Board drivers can override it, if neccecary */
		this->options |= NAND_NO_AUTOINCR;
		/* Check if this is a not a samsung device. Do not clear the options
		 * for chips which are not having an extended id.
		 */	
		if (nand_maf_id != NAND_MFR_SAMSUNG && !nand_flash_ids[i].pagesize)
			this->options &= ~NAND_SAMSUNG_LP_OPTIONS;
#endif
#if 0
		/* Check for AND chips with 4 page planes */
		if (this->options & NAND_4PAGE_ARRAY)
			this->erase_cmd = multi_erase_cmd;
		else
			this->erase_cmd = single_erase_cmd;
#endif
#if 0
		/* Do not replace user supplied command function ! */
		if (mtd->oobblock > 512 && this->cmdfunc == nand_command)
			this->cmdfunc = nand_command_lp;
#endif


#if 0
//THT: We don't have this feature
	/* Set Sync. Burst Read after probing */
	if (this->mmcontrol) {
		printk(KERN_INFO "BrcmNAND Sync. Burst Read support\n");
		this->read_bufferram = brcmnand_sync_read_bufferram;
	}
#endif

	/* Allocate buffers, if neccecary */
	if (!this->oob_buf) {
		size_t len;
		len = mtd->oobsize << (this->phys_erase_shift - this->page_shift);
		this->oob_buf = kmalloc (len, GFP_KERNEL);
		if (!this->oob_buf) {
			printk (KERN_ERR "brcmnand_scan(): Cannot allocate oob_buf %d bytes\n", len);
			return -ENOMEM;
		}
		this->options |= NAND_OOBBUF_ALLOC;
	}

	if (!this->data_buf) {
		size_t len;
		len = mtd->oobblock + mtd->oobsize;
		this->data_buf = kmalloc (len, GFP_KERNEL);
		if (!this->data_buf) {
			if (this->options & NAND_OOBBUF_ALLOC)
				kfree (this->oob_buf);
			printk (KERN_ERR "brcmnand_scan(): Cannot allocate data_buf %d bytes\n", len);
			return -ENOMEM;
		}
		this->options |= NAND_DATABUF_ALLOC;
	}
	
	this->state = FL_READY;
	init_waitqueue_head(&this->wq);
	spin_lock_init(&this->chip_lock);

	/* The number of bytes available for the filesystem to place fs dependend
	 * oob data */
PRINTK( "Deterimining mtd->oobavail, this->autooob=%p \n", this->autooob);

	switch (mtd->oobsize) {
	case 64:
		this->autooob = &brcmnand_oob_64;
		/*
		 * Adjust oobfree for ST chips, which also uses the 6th byte,
		 * in addition to the first byte to mark a bad block
		 */
		brcmnand_maf_id = (this->device_id >> 24) & 0xff;
		if (brcmnand_maf_id == FLASHTYPE_ST) {
			BUG_ON(this->autooob->oobfree[0][1] != 4);
			this->autooob->oobfree[0][1] = 3; // Exclude 6th byte from OOB free
		}
		break;

	case 16:
		this->autooob = &brcmnand_oob_16;
		break;

	default:
		printk(KERN_WARNING "No OOB scheme defined for oobsize %d\n",
			mtd->oobsize);
		/* To prevent kernel oops */
		this->autooob = &brcmnand_oob_16;
		break;
	}

	memcpy(&mtd->oobinfo, this->autooob, sizeof(struct nand_oobinfo));
	
PRINTK("brcmnand_scan: this->autooob=%p, &brcmnand_oob_64=%p\n", this->autooob, &brcmnand_oob_64);

	mtd->oobavail = 0;
	for (i = 0; this->autooob->oobfree[i][1] && i < 8; i++) {
PRINTK("i=%d, oobfree.length=%d\n", i, this->autooob->oobfree[i][1]);
		mtd->oobavail += this->autooob->oobfree[i][1];
	}

PRINTK( "mtd->oobavail  %d\n", mtd->oobavail);

	switch (this->pageSize) {
	case 512:
		this->eccmode = NAND_ECC_HW3_512;
		this->eccbytes = 3;
		this->eccsize = 512;	
		break;
		
	case 2048:
		/* Brcm NAND controller uses NAND_ECC_HW3_512 regardless of page size
		 * may be until the next revision which uses HW6_512
		 */
		this->eccmode = NAND_ECC_HW3_512;
		this->eccbytes = 3;
		this->eccsize = 512;	
		break;
	default:
		printk(KERN_ERR "Page size %d not supported\n", this->pageSize);
	}
	mtd->eccsize = this->eccsize;
	this->eccOobSize = 16;

PRINTK( "mtd->eccsize=%d, eccbytes=%d\n", mtd->eccsize, this->eccbytes);

	/* For 2K page, eccsteps is 4 for the 4 slides that make up a page */
	this->eccsteps = mtd->oobblock / this->eccsize;
	/* We consider only layout allocation performed in nand_base */
	this->layout_allocated = 0;
	if (!this->layout && this->autooob)
		if (fill_autooob_layout(mtd) < 0)
			BUG();

	/* Set up base, based on flash size, for now hardcoded to the 1Gbit flash chip */
	if (this->chipSize == (256 << 20)) {
		this->pbase = 0x12000000;
		mtd->size = 0x20000000 - this->pbase; // THT: This is different than this->chipSize
	} else {
		/* We know that flash endAddr is 0x2000_0000 */
		this->pbase = 0x20000000 - this->chipSize;
		mtd->size = this->chipSize;
	}
	this->vbase = (void*) KSEG1ADDR(this->pbase);

	/*
	 * Initialize the eccmask array for ease of verifying OOB area.
	 */
	fill_ecccmp_mask(mtd);
	

	/* Fill in remaining MTD driver data */
	mtd->type = MTD_NANDFLASH;
	mtd->flags = MTD_CAP_NANDFLASH | MTD_ECC;
	mtd->ecctype = MTD_ECC_SW;
	mtd->erase = brcmnand_erase;
	mtd->point = NULL;
	mtd->unpoint = NULL;
	mtd->read = brcmnand_read;
	mtd->write = brcmnand_write;
	mtd->read_ecc = brcmnand_read_ecc;
	mtd->write_ecc = brcmnand_write_ecc;
	mtd->read_oob = brcmnand_read_oob;
	mtd->write_oob = brcmnand_write_oob;
	mtd->read_oobfree = brcmnand_read_oobfree;
	mtd->write_oobfree = brcmnand_write_oobfree;
	
	mtd->readv = NULL;
	mtd->readv_ecc = NULL;
	mtd->writev = brcmnand_writev;
	mtd->writev_ecc = brcmnand_writev_ecc;
	mtd->sync = brcmnand_sync;
	mtd->lock = NULL;
	mtd->unlock = brcmnand_unlock;
	mtd->suspend = brcmnand_suspend;
	mtd->resume = brcmnand_resume;
	mtd->block_isbad = brcmnand_block_isbad;
	mtd->block_markbad = brcmnand_block_markbad;
	mtd->owner = THIS_MODULE;

	/* Unlock whole block */
PRINTK("Calling mtd->unlock(ofs=0, chipSize=%08x\n", this->chipSize);
	mtd->unlock(mtd, 0x0, this->chipSize);

/*
* As a one time deal, erase the entire flash
*/
if (gClearBBT) {
  int bOffset, ret, needBBT;
  int page;
  for (bOffset=0; bOffset < this->chipSize; bOffset += mtd->erasesize) {
	unsigned long pAddr = this->pbase + bOffset;
	
	/* Skip reserved area, 7MB starting at 0x1f80_0000.
	 * Reserved area is owned by the bootloader.  Linux owns the rootfs and the BBT area
	 */
	if (pAddr  >= 0x1f800000 && pAddr < 0x1ff00000)
		continue;
	
PRINTK("Erasing block at %08x\n", bOffset);
    	this->ctrl_writeAddr(this, bOffset);
	
    	this->ctrl_write(BCHP_NAND_CMD_START, OP_BLOCK_ERASE);
    	// Wait until flash is ready
    	ret = brcmnand_write_is_complete(mtd, &needBBT);
    	if (needBBT) {
		printk(KERN_WARNING "%s: Marking bad block @%08x\n", __FUNCTION__, bOffset);
		ret = this->block_markbad(mtd, bOffset);
    	}

#if 0
    /* Erase the OOB */
    for (page = 0; page < (mtd->erasesize / mtd->oobblock); page++) {
	int retlen;

    	brcmnand_write_oob(mtd, bAddr + (page << this->page_shift), mtd->oobsize, &retlen, ffchars);
    }
#endif
  }
}
PRINTK("Scanning BBT\n");
	err =  this->scan_bbt(mtd);
PRINTK("<-- brcmnand_scan, err=%d\n", err);
	return err;

}

/**
 * brcmnand_release - [BrcmNAND Interface] Free resources held by the BrcmNAND device
 * @param mtd		MTD device structure
 */
void brcmnand_release(struct mtd_info *mtd)
{
	struct brcmnand_chip *this = mtd->priv;

#ifdef CONFIG_MTD_PARTITIONS
	/* Deregister partitions */
	del_mtd_partitions (mtd);
#endif
	/* Deregister the device */
	del_mtd_device (mtd);

	/* Buffer allocated by brcmnand_scan */
	if (this->options & NAND_DATABUF_ALLOC)
		kfree(this->data_buf);

	/* Buffer allocated by brcmnand_scan */
	if (this->options & NAND_OOBBUF_ALLOC)
		kfree(this->oob_buf);

}



