/*
 * Copyright (c) 2006-2010 by Panasonic Corporation
 *
 *  All rights reserved.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include <directfb.h>
#include <core/coretypes.h>
#include <ctype.h>
#include <fusion/shmalloc.h>
#include <fusion/arena.h>
#include <direct/memcpy.h>

#include <core/layers.h>
#include <core/screens.h>
#include <core/core_system.h>
#include <core/core.h>
#include <core/surface_pool.h>

#include <core/state.h>
#include "surfacemanager.h"

#include "dfosd.h"
#include "primary.h"

#ifndef PANASONIC_VERSION
#error "PANASONIC_VERSION undefined !!"
#endif

DFB_CORE_SYSTEM( dfosd )

DFOSD_DEVICE   dfosd_device_priv;
DFOSD_DEVICE  *dfosd_device = NULL;
CoreDFB       *dfb_dfosd_core = NULL;

extern ScreenFuncs dfosdPrimaryScreenFuncs;
extern DisplayLayerFuncs dfosdPrimaryLayerFuncs;
extern const SurfacePoolFuncs dfosdSurfacePoolFuncs;
extern int dfosd_plane_reconfig(DFOSD_PLANE                *plane, 
                                CoreLayerRegionConfig      *config, 
                                CoreLayerRegionConfigFlags  updated );

typedef struct {
  unsigned int  xres;
  unsigned int  yres;
  DFBSurfacePixelFormat pixelformat;
} SDL_FB_CONF;

static SDL_FB_CONF sdl_fb_conf[DFOSD_ID_LAST];

static void init_fb_conf()
{
  const SDL_FB_CONF dfosd_conf[DFOSD_ID_LAST] = DFOSD_SETTING;

  memcpy(&sdl_fb_conf[0], &dfosd_conf[0], sizeof(dfosd_conf));
}

static int
dfosd_init(DFOSD_DEVICE *dev);

static void
dfosd_finish(DFOSD_DEVICE *dev);

extern int
dfosd_mmap_plane(DFOSD_PLANE *plane);
static int
dfosd_munmap_plane(DFOSD_PLANE *plane);

/*
 * system_get_info
 */
static void
system_get_info( CoreSystemInfo *info )
{
  info->type = CORE_ANY;
  info->caps = CSCAPS_ACCELERATION;

  snprintf( info->name, DFB_CORE_SYSTEM_INFO_NAME_LENGTH, "DFOSD" );
}

/*
 * system_initialize
 */
static DFBResult
system_initialize( CoreDFB *core, void **data )
{
  CoreScreen *screen;
  int i;

  D_ASSERT( dfosd_device == NULL );

  D_INFO("<< Panasonic version : %d  (PID=%d %d) >>\n", PANASONIC_VERSION , (int)getpid() , (int)getppid());

 dfosd_device = (DFOSD_DEVICE*) SHCALLOC( dfb_core_shmpool(core), 1, sizeof(DFOSD_DEVICE) );
  if (!dfosd_device) {
    D_ERROR( "DirectFB/dfosd_device: Couldn't allocate shared memory!\n" );
    return D_OOSHM();
  }
  memset(&dfosd_device_priv, 0, sizeof(DFOSD_DEVICE));
  dfosd_device_priv.shared = dfosd_device;

  if ( dfosd_init(dfosd_device) ) {
    D_ERROR( "DirectFB/DFOSD: Couldn't initialize DFOSD \n");
    return DFB_INIT;
  }

  fusion_arena_add_shared_field( dfb_core_arena( core ), "dfosd_device", dfosd_device );

  dfosd_device->core = core;
  dfosd_device_priv.core = core;
  dfb_dfosd_core = core;

  if(dfb_config->video_length){
    dfosd_device->video_mem[2] = dfb_config->video_length;
  }else{
#ifdef VIDEORAM_SIZE
    dfosd_device->video_mem[2] = VIDEORAM_SIZE;
#else
    dfosd_device->video_mem[2] = 0;
#endif
  }
  if(dfb_config->video_logs){
    dfosd_device->video_mem[0] = dfb_config->video_logs;
  }else{
#ifdef VIDEORAM_LOGICAL_ADDR
    dfosd_device->video_mem[0] = VIDEORAM_LOGICAL_ADDR;
#else
    dfosd_device->video_mem[0] = 0;
#endif
  }
  if(dfb_config->video_phys){
    dfosd_device->video_mem[1] = dfb_config->video_phys;
  }else{
#ifdef VIDEORAM_PHYSICAL_ADDR
    dfosd_device->video_mem[1] = VIDEORAM_PHYSICAL_ADDR;
#else
    dfosd_device->video_mem[1] = 0;
#endif
  }

  screen = dfb_screens_register( NULL, NULL, &dfosdPrimaryScreenFuncs );

  if(dfb_config->primary_layer < 0 || dfb_config->primary_layer >= DFOSD_ID_LAST){
    dfb_config->primary_layer = DLID_PRIMARY;
  }

  dfosd_device->primary_id = dfb_config->primary_layer;

  for(i=0; i < DFOSD_ID_LAST; i++){
    dfb_layers_register( screen, &dfosd_device->planes[i],&dfosdPrimaryLayerFuncs );
    dfb_surface_pool_initialize( core, &dfosdSurfacePoolFuncs,&dfosd_device->planes[i].dfosd_pool );
  }

  *data = dfosd_device;
  return DFB_OK;
}

/*
 * system_shutdown
 */
static DFBResult
system_shutdown( bool emergency )
{
  int i;

  if(dfosd_device != NULL){
    dfosd_finish(dfosd_device);

    for(i=0; i < DFOSD_ID_LAST; i++){
      dfb_surface_pool_destroy( dfosd_device->planes[i].dfosd_pool );
    }
    SHFREE( dfb_core_shmpool( dfb_dfosd_core ), dfosd_device );
    dfosd_device = NULL ;
    dfb_dfosd_core = NULL ;
  }
  return DFB_OK;
}

/*
 * system_join
 */
static DFBResult
system_join( CoreDFB *core, void **data )
{
  CoreScreen *screen;
  int i,ret;
  DFOSD_PLANE *plane;
  int layer_swap[DFOSD_ID_LAST];

  D_ASSERT( dfosd_device == NULL );

  D_INFO("<< Panasonic version : %d  (PID=%d %d) >>\n", PANASONIC_VERSION , (int)getpid() , (int)getppid());

  fusion_arena_get_shared_field( dfb_core_arena( core ),"dfosd_device", (void *) &dfosd_device );
  memset(&dfosd_device_priv, 0, sizeof(DFOSD_DEVICE));
  dfosd_device_priv.shared = dfosd_device;

  for (i = 0; i < DFOSD_ID_LAST; i++){
   layer_swap[i] = i;
  }

  if(dfb_config->primary_layer < 0 || dfb_config->primary_layer >= DFOSD_ID_LAST){
    dfb_config->primary_layer = DLID_PRIMARY;
  }
  layer_swap[0] =  dfb_config->primary_layer;
  layer_swap[dfb_config->primary_layer] = 0;

  for (i = 0; i < DFOSD_ID_LAST; i++)
    {
      plane = &dfosd_device->planes[i];
      if ((dfb_config->layers[layer_swap[i]].init == true)){ 
        ret = dfosd_mmap_plane(plane);
        if (ret)
          return ret;
      }
    }

  dfosd_device_priv.core = core;
  dfb_dfosd_core = core;

  screen = dfb_screens_register( NULL, NULL, &dfosdPrimaryScreenFuncs );

  dfosd_device->primary_id = dfb_config->primary_layer;
  for(i=0; i < DFOSD_ID_LAST; i++){
      dfb_layers_register( screen, &dfosd_device->planes[i],&dfosdPrimaryLayerFuncs );
      dfb_surface_pool_join( core, dfosd_device->planes[i].dfosd_pool,&dfosdSurfacePoolFuncs );
  }
  *data = dfosd_device;

  return DFB_OK;
}

/*
 * system_leave
 */
static DFBResult
system_leave( bool emergency )
{
  int i;
  DFOSD_PLANE *plane;

  for (i = 0; i < DFOSD_ID_LAST; i++)
    {
      if(dfosd_device != NULL){
        dfb_surface_pool_leave(dfosd_device->planes[i].dfosd_pool);
      }
    }
  for (i = 0; i < DFOSD_ID_LAST; i++)
    {
      /* unmap video memory */
      plane = &dfosd_device->planes[i];
      dfosd_munmap_plane(plane);
    }

  usleep(100000); /* waiting memory free */

  dfosd_device = NULL ;
  dfb_dfosd_core = NULL ;

  return DFB_OK;
}

/*
 * system_suspend
 */
static DFBResult
system_suspend()
{
  return DFB_UNIMPLEMENTED;
}

/*
 * system_resume
 */
static DFBResult
system_resume()
{
  return DFB_UNIMPLEMENTED;
}

/*
 * system_map_mmio
 */
static volatile void *
system_map_mmio( unsigned int    offset,
                 int             length )
{
  return (void *)&dfosd_device_priv;
}

/*
 * system_unmap_mmio
 */
static void
system_unmap_mmio( volatile void  *addr,
                   int             length )
{
}

/*
 * system_get_accelerator
 */
static int
system_get_accelerator()
{
  return -1;
}

/*
 * system_get_modes
 */
static VideoMode *
system_get_modes()
{
  return NULL;
}

/*
 * system_get_current_mode
 */
static VideoMode *
system_get_current_mode()
{
  return NULL;
}

/*
 * system_thread_init
 */
static DFBResult
system_thread_init()
{
  return DFB_OK;
}

/*
 * system_input_filter
 */
static bool
system_input_filter( CoreInputDevice *device,
                     DFBInputEvent   *event )
{
  return false;
}

/*
 * system_video_memory_physical
 */
static unsigned long
system_video_memory_physical( unsigned int offset )
{
#if 0
  return DFB_DDR_PADDR_UNCACHE((unsigned long)offset);
#else
  return 0;
#endif
}

/*
 * system_video_memory_virtual
 */
static void *
system_video_memory_virtual( unsigned int offset )
{
  return (dfosd_device->logical_base + offset);
}

/*
 * system_videoram_length
 */
static unsigned int
system_videoram_length()
{
  return dfosd_device->video_mem[2];
}
/*
 * system_aux_memory_physical
 */
static unsigned long
system_aux_memory_physical( unsigned int offset )
{
#if 0
  return DFB_DDR_PADDR_UNCACHE((unsigned long)offset);
#else
  return 0;
#endif
}

/*
 * system_aux_memory_virtual
 */
static void *
system_aux_memory_virtual( unsigned int offset )
{
#if 0
  DFOSD_PLANE *osda = &dfosd_device->planes[DFOSD_ID0];
  DFOSD_PLANE *osdc = &dfosd_device->planes[DFOSD_ID1];
  DFOSD_PLANE *losd = &dfosd_device->planes[DFOSD_ID2];
  unsigned int osda_size = osda->smem_len;
  unsigned int osdc_size = osdc->smem_len;
  unsigned int losd_size = losd->smem_len;

  if ( osda->fb_pbase <= offset &&
       offset < osda->fb_pbase + osda_size ) {
    return osda->fb_base + offset - osda->fb_pbase;
  }
  else if ( osdc->fb_pbase <= offset &&
            offset < osdc->fb_pbase + osdc_size ) {
    return osdc->fb_base + offset - osdc->fb_pbase;
  }
  else if ( losd->fb_pbase <= offset &&
            offset < losd->fb_pbase + losd_size ) {
    return losd->fb_base + offset - losd->fb_pbase;
  }
  return (dfosd_device->logical_base + offset - dfosd_device->physical_base);
#else
  return 0;
#endif
}

/*
 * system_auxram_length
 */
static unsigned int
system_auxram_length()
{
  return 0;
}

/*
 * system_get_busid
 */
static void
system_get_busid( int *ret_bus, int *ret_dev, int *ret_func )
{
  return;
}

/*
 * system_get_deviceid
 */
static void
system_get_deviceid( unsigned int *ret_vendor_id,
                     unsigned int *ret_device_id )
{
  return;
}


/*
 * dfosd_mmap_plane
 */

int
dfosd_mmap_plane(DFOSD_PLANE *plane)
{
  int  exmem_fd = 0;
  
  exmem_fd = open(DEV_EXMEM, O_RDWR|O_SYNC);
  
  int i;
  for(i=0; i < DFOSD_MAX_SURFACE_BUFFERS; i++){
    if(!(plane->smem_buffer[i] & 0x80000000)){
      if(plane->smem_paddr[i]){
      /* adjustment memory address to MMU PageSize boundary */
        plane->fb_base[i] = (void *)mmap((caddr_t) plane->smem_buffer[i], plane->smem_len,
                                         (PROT_READ | PROT_WRITE), (MAP_SHARED | MAP_FIXED),
                                         exmem_fd, (off_t)plane->smem_paddr[i]);
        if (plane->fb_base[i] == (void *)-1) {
          perror ("ERROR: mmap FB fails!");
          close(exmem_fd);
          return -1;
        }
      }else{
        plane->fb_base[i] = NULL;
      }
    }
  }
  if(!plane->lut_pbase){
    if((8 == plane->bits_per_pixel) && (plane->lut_offset) ){
      /* adjustment memory address to MMU PageSize boundary */
      plane->lut_base[0] = (unsigned long *)mmap((caddr_t)plane->lut_pbase,
                                              plane->lut_size,
                                              PROT_READ | PROT_WRITE,
                                              MAP_SHARED | MAP_FIXED,
                                              exmem_fd,
                                              (off_t)plane->lut_offset);
      if (plane->lut_base[0] == (unsigned long *) -1) {
        plane->lut_base[0] = 0;
        perror ("ERROR: mmap CLUT fails!");
        close(exmem_fd);
        return -1;
      }
    }
  }
  close(exmem_fd);
  return 0;
}

/*
 * dfosd_munmap_plane
 */
static int
dfosd_munmap_plane(DFOSD_PLANE *plane)
{
  int i;
  for(i=0; i < DFOSD_MAX_SURFACE_BUFFERS; i++){
    if(!(plane->smem_buffer[i] & 0x80000000)){
      if(plane->fb_base[i]){
        munmap( plane->fb_base[i], plane->smem_len );
      }
      plane->fb_base[i] = 0;
    }
  }

  if(plane->lut_pbase){
    if(8 == plane->bits_per_pixel){
      munmap( plane->lut_base[0], plane->lut_size );
      plane->lut_base[0] = 0;
    }
  }
  return 0;
}

/*
 * dfosd_init
 */
static int
dfosd_init(DFOSD_DEVICE *dev)
{
  int ret;
  int i;
  DFOSD_PLANE *plane;
  SDL_FB_CONF *fb_conf;
  char    str[30];

  int layer_swap[DFOSD_ID_LAST];

  init_fb_conf();

  for (i = 0; i < DFOSD_ID_LAST; i++){
   layer_swap[i] = i;
  }

  if(dfb_config->primary_layer < 0 || dfb_config->primary_layer >= DFOSD_ID_LAST){
    dfb_config->primary_layer = 0;
  }
  layer_swap[0] =  dfb_config->primary_layer;
  layer_swap[dfb_config->primary_layer] = 0;

  memset(dev, 0, sizeof(DFOSD_DEVICE));

  for (i = 0; i < DFOSD_ID_LAST; i++){
    plane = &dev->planes[i];

    memset(plane, 0, sizeof(DFOSD_PLANE));

    /* set FSCREENINFO and VSCREENINFO */
    fb_conf = (SDL_FB_CONF *)&sdl_fb_conf[i];
    plane->pixelformat = fb_conf->pixelformat;
    plane->bits_per_pixel = DFB_BYTES_PER_PIXEL(fb_conf->pixelformat) * 8;
    plane->xres           = fb_conf->xres;
    plane->yres           = fb_conf->yres;
    plane->smem_len       = plane->xres * DFB_BYTES_PER_PIXEL(fb_conf->pixelformat);
    if(plane->smem_len % DFB_CARD_LINE_BYTE_ALIGN){
      plane->smem_len = (plane->smem_len + (DFB_CARD_LINE_BYTE_ALIGN-1)) & ~(DFB_CARD_LINE_BYTE_ALIGN-1);
    }
    plane->smem_len       = plane->smem_len * plane->yres ;
    if((plane->pixelformat == DSPF_NV12) || (plane->pixelformat == DSPF_I420) || (plane->pixelformat == DSPF_YV12)){
      plane->smem_len = (plane->smem_len * 2) + DFB_CARD_PAGE_BYTE_ALIGN;
    }
  
    if ((dfb_config->layers[layer_swap[i]].init == true)) {
      ret = dfosd_mmap_plane(plane);
      if (ret){
        return ret;
      }
    }
    sprintf(str,"Plane lock%d",i);
    fusion_skirmish_init( &plane->lock, str , dfb_core_world(dev->core) );
  }
  return 0; 
}

/*
 * dfosd_finish
 */
static void
dfosd_finish(DFOSD_DEVICE *dev)
{
  int  i;
  DFOSD_PLANE *plane;

  for (i = 0; i < DFOSD_ID_LAST; i++) 
    {
      plane = &dev->planes[i];
      dfosd_munmap_plane(plane);
      fusion_skirmish_destroy( &plane->lock );
    }
}

int dfosd_plane_reconfig(DFOSD_PLANE *plane, CoreLayerRegionConfig *config, CoreLayerRegionConfigFlags updated)
{
  if(!(updated & CLRCF_WIDTH)){
    config->width = plane->xres;
  }
  if(!(updated & CLRCF_HEIGHT)){
    config->height = plane->yres;
  }
  if(!(updated & CLRCF_FORMAT)){
    config->format = plane->pixelformat;
  }
  if(!(updated & CLRCF_MEMORY_F)){
    config->memaddr[0] = plane->smem_buffer[0];
  }
  if(!(updated & CLRCF_MEMORY_B)){
    config->memaddr[1] = plane->smem_buffer[1];
  }

  if((plane->addr_config < 3) ||
    (plane->xres != config->width) || 
    (plane->yres != config->height) || 
    (plane->pixelformat != config->format) ||
    (plane->smem_buffer[0] != config->memaddr[0]) ||
    (plane->smem_buffer[1] != config->memaddr[1]) ||
    (plane->smem_buffer[2] != config->memaddr[2])
    )
    {
      if(dfosd_munmap_plane(plane)){
        return -1;
      }
      plane->xres           = config->width;
      plane->yres           = config->height;
      plane->pixelformat = config->format;
      plane->bits_per_pixel = DFB_BYTES_PER_PIXEL(config->format) * 8;
      plane->smem_len       = plane->xres * DFB_BYTES_PER_PIXEL(config->format);
      if(plane->smem_len % DFB_CARD_LINE_BYTE_ALIGN){
        plane->smem_len = (plane->smem_len + (DFB_CARD_LINE_BYTE_ALIGN-1)) & ~(DFB_CARD_LINE_BYTE_ALIGN-1);
      }
      plane->smem_len       = plane->smem_len * plane->yres ;
      if((plane->pixelformat == DSPF_NV12) || (plane->pixelformat == DSPF_I420) || (plane->pixelformat == DSPF_YV12)){
        plane->smem_len = (plane->smem_len * 2) + DFB_CARD_PAGE_BYTE_ALIGN;
      }
      plane->smem_buffer[0] = config->memaddr[0];
      plane->smem_buffer[1] = config->memaddr[1];
      plane->smem_buffer[2] = config->memaddr[2];
      if(dfosd_mmap_plane(plane)){
        return -1;
      }
      plane->addr_config += 1;
    }
 return 0;
}
