/*
  (c) Copyright 2001-2008  The world wide DirectFB Open Source Community (directfb.org)
  (c) Copyright 2000-2004  Convergence (integrated media) GmbH

  All rights reserved.

  Written by Denis Oliver Kropp <dok@directfb.org>,
  Andreas Hundt <andi@fischlustig.de>,
  Sven Neumann <neo@directfb.org>,
  Ville Syrjälä <syrjala@sci.fi> and
  Claudio Ciccani <klan@users.sf.net>.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
*/

#include <asm/types.h>

#include <direct/debug.h>
#include <direct/mem.h>

#include <directfb.h>
#include <core/coretypes.h>
#include <core/surface_pool.h>

#include <core/state.h>
#include "surfacemanager.h"

#include <gfx/convert.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "dfosd.h"

extern DFOSD_DEVICE  *dfosd_device;
extern int dfosd_mmap_plane(DFOSD_PLANE *plane);

D_DEBUG_DOMAIN( DFOSD_Surfaces, "DFOSD/Surfaces", "OSD Surface Pool" );
D_DEBUG_DOMAIN( DFOSD_SurfLock, "DFOSD/SurfLock", "OSD Surface Pool Locks" );

/**********************************************************************************************************************/

typedef struct {
  int             magic;

  CoreDFB        *core;
} DFOSDPoolLocalData;

typedef struct {
  int   magic;

  int   offset;
  int   pitch;
  int   size;

  Chunk *chunk;
} DFOSDAllocationData;

/**********************************************************************************************************************/

static int
dfosdPoolDataSize()
{
  return sizeof(DFOSD_PLANE);
}

static int
dfosdPoolLocalDataSize()
{
  return sizeof(DFOSDPoolLocalData);
}

static int
dfosdAllocationDataSize()
{
  return sizeof(DFOSDAllocationData);
}

static int dfosd_videoram_mmap(void)
{
  int  exmem_fd = 0;
  
  if((dfosd_device->manager != NULL) && (dfosd_device->video_mem[0] != 0) && (dfosd_device->video_mem[2] != 0) ){
    exmem_fd = open(DEV_EXMEM, O_RDWR|O_SYNC);

    D_DEBUG_AT(DFOSD_Surfaces,"Videoram Area %s[%d] adr=0x%08x - 0x%08x size=0x%x\n",
               __func__,__LINE__,(int)(dfosd_device->video_mem[0] + dfosd_device->video_mem[1]),
               (int)(dfosd_device->video_mem[0] + dfosd_device->video_mem[1] + dfosd_device->video_mem[2]-1),
               (int)dfosd_device->video_mem[2]);
   if(dfosd_device->video_mem[0] && dfosd_device->video_mem[2]){
      D_INFO("   videoram logs=%#08x phys=%#08x size=%#08x\n",(int)dfosd_device->video_mem[0],
             (int)dfosd_device->video_mem[1],(int)dfosd_device->video_mem[2]);
   }

    /* adjustment memory address to MMU PageSize boundary */
    dfosd_device->manager->chunks->offset = 
      (unsigned long)mmap((caddr_t)dfosd_device->video_mem[0], (size_t)dfosd_device->video_mem[2],
                          (PROT_READ | PROT_WRITE), (MAP_SHARED | MAP_FIXED),
                          exmem_fd, (off_t)dfosd_device->video_mem[1]);
    if (dfosd_device->manager->chunks->offset == -1) {
      dfb_surfacemanager_destroy( dfosd_device->manager );
      dfosd_device->manager = NULL;
      perror ("ERROR: mmap chunk fails!");
      close(exmem_fd);
      return -1;
    }
    close(exmem_fd);
  }
  return 0;
}

static DFBResult
dfosdInitPool( CoreDFB                    *core,
               CoreSurfacePool            *pool,
               void                       *pool_data,
               void                       *pool_local,
               void                       *system_data,
               CoreSurfacePoolDescription *ret_desc )
{
  int i;
  DFOSDPoolLocalData *local = pool_local;
  int ret;

  D_DEBUG_AT( DFOSD_Surfaces, "%s()\n", __FUNCTION__ );

  D_ASSERT( core != NULL );
  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_ASSERT( local != NULL );
  D_ASSERT( ret_desc != NULL );

  for(i=0;i < DFOSD_ID_LAST; i++){
    if(dfosd_device->planes[i].manager == NULL) {
      dfb_surfacemanager_create( core, 
                                 dfosd_device->planes[i].smem_len,&dfosd_device->planes[i].manager  );
      break;
    }
  }

  ret_desc->caps              = CSPCAPS_PHYSICAL | CSPCAPS_VIRTUAL;
  ret_desc->access[CSAID_CPU] = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
  ret_desc->access[CSAID_GPU] = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
  ret_desc->types             = CSTF_LAYER |  CSTF_SHARED | CSTF_EXTERNAL;

  ret_desc->priority          = CSPP_DEFAULT;

  ret_desc->types             |= CSTF_WINDOW;
  if((dfosd_device->manager == NULL) && (dfosd_device->video_mem[0] != 0) && (dfosd_device->video_mem[2] != 0) ){
    ret = dfb_surfacemanager_create( core, dfosd_device->video_mem[2], &dfosd_device->manager );
    if (ret) return ret;
    ret = dfosd_videoram_mmap();
    if(ret) return ret;
  }

  /* For hardware layers */
  ret_desc->access[CSAID_LAYER0] = CSAF_READ;
  ret_desc->access[CSAID_LAYER1] = CSAF_READ;
  ret_desc->access[CSAID_LAYER2] = CSAF_READ;
  ret_desc->access[CSAID_LAYER3] = CSAF_READ;
  ret_desc->access[CSAID_LAYER4] = CSAF_READ;
  ret_desc->access[CSAID_LAYER5] = CSAF_READ;
  ret_desc->access[CSAID_LAYER6] = CSAF_READ;
  ret_desc->access[CSAID_LAYER7] = CSAF_READ;


  snprintf( ret_desc->name, DFB_SURFACE_POOL_DESC_NAME_LENGTH, "OSD Memory" );

  local->core = core;

  D_MAGIC_SET( local, DFOSDPoolLocalData );


  D_ASSERT( dfosd_device != NULL );


  return DFB_OK;
}

static DFBResult
dfosdJoinPool( CoreDFB                    *core,
               CoreSurfacePool            *pool,
               void                       *pool_data,
               void                       *pool_local,
               void                       *system_data )
{
  DFOSDPoolLocalData *local = pool_local;
  int ret;
  static int join_videoram_map = 1;

  D_DEBUG_AT( DFOSD_Surfaces, "%s()\n", __FUNCTION__ );

  D_ASSERT( core != NULL );
  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_ASSERT( local != NULL );

  if(join_videoram_map){
    ret = dfosd_videoram_mmap();
    if(ret) return ret;
    join_videoram_map = 0;
  }

  local->core = core;

  D_MAGIC_SET( local, DFOSDPoolLocalData );

  return DFB_OK;
}

static DFBResult
dfosdDestroyPool( CoreSurfacePool *pool,
                  void            *pool_data,
                  void            *pool_local )
{
  DFOSDPoolLocalData *local = pool_local;
  int i;

  D_DEBUG_AT( DFOSD_Surfaces, "%s()\n", __FUNCTION__ );

  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_MAGIC_ASSERT( local, DFOSDPoolLocalData );

  for(i=0;i < DFOSD_ID_LAST; i++){
    if(dfosd_device->planes[i].manager != NULL) {
      dfb_surfacemanager_destroy(dfosd_device->planes[i].manager);
      dfosd_device->planes[i].manager = NULL;
    }
  }

  if (dfosd_device->manager) {
    if (dfosd_device->manager->chunks->offset) {
      munmap( (void *)dfosd_device->manager->chunks->offset, dfosd_device->video_mem[2]);
    }
    dfb_surfacemanager_destroy( dfosd_device->manager );
    dfosd_device->manager = NULL;
  }

  D_MAGIC_CLEAR( local );

  return DFB_OK;
}

static DFBResult
dfosdLeavePool( CoreSurfacePool *pool,
                void            *pool_data,
                void            *pool_local )
{
  DFOSDPoolLocalData *local = pool_local;

  D_DEBUG_AT( DFOSD_Surfaces, "%s()\n", __FUNCTION__ );

  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_MAGIC_ASSERT( local, DFOSDPoolLocalData );

  D_MAGIC_CLEAR( local );

  return DFB_OK;
}

static DFBResult
dfosdTestConfig( CoreSurfacePool         *pool,
                 void                    *pool_data,
                 void                    *pool_local,
                 CoreSurfaceBuffer       *buffer,
                 const CoreSurfaceConfig *config )
{
  DFBResult           ret = DFB_FAILURE;
  CoreSurface        *surface;
  DFOSDPoolLocalData *local = pool_local;

  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_MAGIC_ASSERT( local, DFOSDPoolLocalData );
  D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );

  surface = buffer->surface;
  D_MAGIC_ASSERT( surface, CoreSurface );

  if (surface->type & CSTF_LAYER){
    if(!dfosd_device->planes[surface->resource_id].manager){
      return DFB_FAILURE;
    }else{
      return DFB_OK;
      /*   ret = dfb_surfacemanager_allocate( local->core, 
           dfosd_device->planes[surface->resource_id].manager,buffer, NULL, NULL );*/
    }
  }else if(surface->type & CSTF_EXTERNAL){
    if(!dfosd_device->manager){
      return DFB_FAILURE;
    }else{
      ret = dfb_surfacemanager_allocate( local->core, dfosd_device->manager, buffer, NULL, NULL );
    }
  }

  D_DEBUG_AT( DFOSD_Surfaces, "  -> %s\n", DirectFBErrorString(ret) );

  return ret;
}

static DFBResult
dfosdAllocateBuffer( CoreSurfacePool       *pool,
                     void                  *pool_data,
                     void                  *pool_local,
                     CoreSurfaceBuffer     *buffer,
                     CoreSurfaceAllocation *allocation,
                     void                  *alloc_data )
{
  DFBResult           ret;
  CoreSurface         *surface;
  DFOSDAllocationData *alloc = alloc_data;
  DFOSDPoolLocalData *local = pool_local;
  int buffer_id;

  surface = buffer->surface;

  if (surface->type & CSTF_LAYER)  {
    DFOSD_PLANE *plane =  &dfosd_device->planes[surface->resource_id];
    alloc->offset = 0;
    for(buffer_id = 0; buffer_id < DFOSD_MAX_SURFACE_BUFFERS; buffer_id++){
      if(surface->buffers[buffer_id] == buffer){
        if (!plane->fb_base[buffer_id]){
           dfosd_mmap_plane(plane);
        }
        alloc->pitch  = plane->xres * DFB_BYTES_PER_PIXEL(plane->pixelformat);
        alloc->size   = plane->yres * alloc->pitch;
        alloc->offset = (unsigned long)plane->fb_base[buffer_id];
        if(!alloc->offset){
          if(plane->addr_config > 2)
            D_INFO("No implementation plane video memory ( resource_id = %d [ %d ] )\n", (int)surface->resource_id, buffer_id);
        }
        break;
      }
    }
  }
  else {
    Chunk *chunk;

    ret = dfb_surfacemanager_allocate( local->core,dfosd_device->manager,buffer, allocation, &chunk );
    if (ret)
      return ret;

    D_MAGIC_ASSERT( chunk, Chunk );

    alloc->offset = chunk->offset;
    alloc->pitch  = chunk->pitch;
    alloc->size   = chunk->length;

    alloc->chunk  = chunk;
  }

  allocation->size   = alloc->size;
  allocation->offset = alloc->offset;

  D_MAGIC_SET(alloc, DFOSDAllocationData);

  return DFB_OK;
}

static DFBResult
dfosdDeallocateBuffer( CoreSurfacePool       *pool,
                       void                  *pool_data,
                       void                  *pool_local,
                       CoreSurfaceBuffer     *buffer,
                       CoreSurfaceAllocation *allocation,
                       void                  *alloc_data )
{
  DFOSDAllocationData *alloc = alloc_data;
  int i;

  D_DEBUG_AT( DFOSD_Surfaces, "%s( %p )\n", __FUNCTION__, buffer );

  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
  D_MAGIC_ASSERT( alloc, DFOSDAllocationData );

  if (buffer->surface->type & CSTF_LAYER)  {
    i  = buffer->surface->resource_id;
    if (alloc->chunk)
      dfb_surfacemanager_deallocate( dfosd_device->planes[i].manager, alloc->chunk );
  }else{
    if (alloc->chunk)
      dfb_surfacemanager_deallocate( dfosd_device->manager, alloc->chunk );
  }

  D_MAGIC_CLEAR( alloc );

  return DFB_OK;
}

static DFBResult
dfosdLock( CoreSurfacePool       *pool,
           void                  *pool_data,
           void                  *pool_local,
           CoreSurfaceAllocation *allocation,
           void                  *alloc_data,
           CoreSurfaceBufferLock *lock )
{
  DFOSDAllocationData *alloc = alloc_data;

  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
  D_MAGIC_ASSERT( alloc, DFOSDAllocationData );
  D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

  D_DEBUG_AT( DFOSD_SurfLock, "%s( %p )\n", __FUNCTION__, lock->buffer );
  lock->pitch = alloc->pitch;
  lock->offset = alloc->offset;
  lock->addr = (unsigned long *)alloc->offset;
  lock->phys = (unsigned long) lock->addr ;

  return DFB_OK;
}

static DFBResult
dfosdUnlock( CoreSurfacePool       *pool,
             void                  *pool_data,
             void                  *pool_local,
             CoreSurfaceAllocation *allocation,
             void                  *alloc_data,
             CoreSurfaceBufferLock *lock )
{
  DFOSDAllocationData *alloc = alloc_data;

  D_MAGIC_ASSERT( pool, CoreSurfacePool );
  D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
  D_MAGIC_ASSERT( alloc, DFOSDAllocationData );
  D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

  D_DEBUG_AT( DFOSD_SurfLock, "%s( %p )\n", __FUNCTION__, lock->buffer );

  (void) alloc;

  return DFB_OK;
}

const SurfacePoolFuncs dfosdSurfacePoolFuncs = {
 PoolDataSize:       dfosdPoolDataSize,
 PoolLocalDataSize:  dfosdPoolLocalDataSize,
 AllocationDataSize: dfosdAllocationDataSize,

 InitPool:           dfosdInitPool,
 JoinPool:           dfosdJoinPool,
 DestroyPool:        dfosdDestroyPool,
 LeavePool:          dfosdLeavePool,

 TestConfig:         dfosdTestConfig,
 AllocateBuffer:     dfosdAllocateBuffer,
 DeallocateBuffer:   dfosdDeallocateBuffer,

 Lock:               dfosdLock,
 Unlock:             dfosdUnlock
};

