/*
 * Copyright (C) 2010 Panasonic Corporation
 * All Rights Reserved.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <command.h>
#include <malloc.h>
#include <net.h>
#include <miiphy.h>

#include <asm/arch/ethernet.h>

static int rx_pos = 0;	/* Current RX Descriptor pos. */
static int rx_siz = 0;	/* RX buf size */
static int rx_off = 0;	/* RX offset */
static int tx_off = 0;	/* TX offset */
static int tx_num = 0;	/* number of txed packets */

/* used when copy needed */
static uchar  tx_adj_PacketBuf[PKTSIZE_ALIGN + PKTALIGN];
static uchar *tx_adj_buf;

static ulong mn_avev3_readreg( struct eth_device *dev,  ulong addr)
{
	ulong val;
	val = MN_AVEV3_REG(dev->iobase, addr);
	return val;
}

static void mn_avev3_writereg( struct eth_device *dev, ulong addr,  ulong val)
{
	MN_AVEV3_REG(dev->iobase, addr) = val;
}

	/* (descriptor r/w) i:index  j:0=CMDSTS, 1=BUFPTR */
static void mn_avev3_write_rxdm( struct eth_device *dev, int i, int j, ulong val)
{
	mn_avev3_writereg(dev, MN_AVEV3_RXDM + i*8 + j*4, val);
}
static void mn_avev3_write_txdm( struct eth_device *dev, int i, int j, ulong val)
{
	mn_avev3_writereg(dev, MN_AVEV3_TXDM + i*8 + j*4, val);
}

static ulong mn_avev3_read_rxdm( struct eth_device *dev, int i, int j)
{
	return mn_avev3_readreg(dev, MN_AVEV3_RXDM + i*8 + j*4);
}
static ulong mn_avev3_read_txdm( struct eth_device *dev, int i, int j)
{
	return mn_avev3_readreg(dev, MN_AVEV3_TXDM + i*8 + j*4);
}

static void mn_avev3_halt(struct eth_device *dev)
{
	ulong val;
	/*-- stop Tx Rx and GRST (w/o PHY RST) --*/
	val = mn_avev3_readreg(dev, MN_AVEV3_GRR);
	if(val == 0) {	/* when GRR all cleared */
		val = mn_avev3_readreg(dev, MN_AVEV3_RXCR);
		val &= ~MN_AVEV3_RXCR_RXEN;
		mn_avev3_writereg(dev, MN_AVEV3_RXCR, val);
		mn_avev3_writereg(dev, MN_AVEV3_DESCC, MN_AVEV3_DESCC_RXDSTP);
		do {
			val = mn_avev3_readreg(dev, MN_AVEV3_DESCC);
		} while((val & MN_AVEV3_DESCC_RXDSTPST) == 0);
		mn_avev3_writereg(dev, MN_AVEV3_GRR, MN_AVEV3_GRR_GRST);
	}
	udelay(MN_AVEV3_GRST_DELAY_USEC);
}

static void mn_avev3_reset(struct eth_device *dev)
{
	mn_avev3_halt(dev);

	mn_avev3_writereg(dev, MN_AVEV3_GRR, MN_AVEV3_GRR_CLR);
	rx_pos = 0;
	rx_off = 0;
	tx_off = 0;
	tx_num = 0;
	tx_adj_buf = &tx_adj_PacketBuf[0] + (PKTALIGN -1);
	tx_adj_buf -= (ulong)tx_adj_buf % PKTALIGN;
}


static int mn_avev3_init(struct eth_device *dev, bd_t * bd)
{
	ulong val;
	int i;

	mn_avev3_reset(dev);
#if __BYTE_ORDER == __BIG_ENDIAN
	val = mn_avev3_readreg(dev, 0x4124);
	val |= 0x00010000;
	mn_avev3_writereg(dev, 0x4124, val);
	val = mn_avev3_readreg(dev, 0x4160);
	val |= 0x00010000;
	mn_avev3_writereg(dev, 0x4160, val);
#endif	/*__BYTE_ORDER*/

	/* set config */
	val = MN_AVEV3_EMCR_CONF;
	mn_avev3_writereg(dev, MN_AVEV3_EMCR, val);
	if(val & MN_AVEV3_EMCR_ROFE) {
		rx_off = 2;
	}
	if(val & MN_AVEV3_EMCR_TOFE) {
		tx_off = 2;
	}
	rx_siz = (PKTSIZE_ALIGN - rx_off);
	/* set num descriptors
	 * use       1 tx descriptor and
	 *   PKTBUFSRX rx descriptors
	 */
	/* set a tx descriptor */
	mn_avev3_writereg(dev, MN_AVEV3_TXDC, MN_AVEV3_TXDC_SIZE(1));
	mn_avev3_write_txdm(dev, 0, MN_AVEV3_TXDM_CMDSTS, 0);

	/* set rx descriptors */
	mn_avev3_writereg(dev, MN_AVEV3_RXDC, MN_AVEV3_RXDC_SIZE(PKTBUFSRX));
	for(i = 0; i < PKTBUFSRX; i++) {
		val = (ulong) NetRxPackets[i];
		//MN_AVEV3_CACHE_INV_B(val, rx_siz + rx_off);
		MN_AVEV3_CACHE_FLS(val, rx_siz + rx_off);
		val = MN_AVEV3_NCADDR(val);
		mn_avev3_write_rxdm(dev, i, MN_AVEV3_RXDM_BUFPTR, val);
		mn_avev3_write_rxdm(dev, i, MN_AVEV3_RXDM_CMDSTS, rx_siz);
	}

	/* set MAC address */
	val = ((dev->enetaddr[0] << 8)|(dev->enetaddr[1]));
	mn_avev3_writereg(dev, MN_AVEV3_RXMAC2R, val);
	val = ((dev->enetaddr[2] << 24)|(dev->enetaddr[3] << 16)
	      |(dev->enetaddr[4] <<  8)|(dev->enetaddr[5]));
	mn_avev3_writereg(dev, MN_AVEV3_RXMAC1R, val);
	/* clr interrupt, set interrupt mask */
	mn_avev3_writereg(dev, MN_AVEV3_GISR, MN_AVEV3_GISR_CLR);
	mn_avev3_writereg(dev, MN_AVEV3_GIMR, MN_AVEV3_GIMR_CLR);

	/* set RXEN, Full Duplex, MTU value */
	val = (MN_AVEV3_RXCR_RXEN | MN_AVEV3_RXCR_FDUP | MN_AVEV3_RXCR_MTU);
	mn_avev3_writereg(dev, MN_AVEV3_RXCR, val);

	/* enable TX/RX descriptor */
	mn_avev3_writereg(dev, MN_AVEV3_DESCC, (MN_AVEV3_DESCC_RDE|MN_AVEV3_DESCC_TDE));
	return 0;
}

static int mn_avev3_send(struct eth_device *dev, volatile void *packet, int length)
{
	ulong algn;
	ulong val;
	volatile uchar *ptr = packet;	/* use uchar pointer */

	/* adjust alignment for tx_off */
	algn = (((ulong)ptr) & 0x3);
	if(algn != tx_off) {
		MN_AVEV3_MEMCPY(tx_adj_buf + tx_off, ptr, length);
		ptr = tx_adj_buf;
	} else {
		ptr -= tx_off;
	}
	/*-- check length --*/
	if(length < MN_AVEV3_MIN_XMITSIZE) {
		volatile uchar *pad = ptr + tx_off + length;
		while(length < MN_AVEV3_MIN_XMITSIZE) {
			*pad++ = 0;
			++length;
		}
	}
	/* check ownership and wait for previous xmit done */
	do {
		val = mn_avev3_read_txdm(dev, 0, MN_AVEV3_TXDM_CMDSTS);
	} while(val & MN_AVEV3_TXDM_OWN);

	MN_AVEV3_CACHE_FLS(ptr, length + tx_off);
	/* set addr */
	val = MN_AVEV3_NCADDR((ulong)ptr);
	
	mn_avev3_write_txdm(dev, 0, MN_AVEV3_TXDM_BUFPTR, val);
	/* set size and kick xmit */
	val = (MN_AVEV3_TXDM_OWN | length);
	mn_avev3_write_txdm(dev, 0, MN_AVEV3_TXDM_CMDSTS, val);
	++tx_num;

	/* wait for xmit */
	do {
		val = mn_avev3_read_txdm(dev, 0, MN_AVEV3_TXDM_CMDSTS);
	} while(val & MN_AVEV3_TXDM_OWN);

	if((val & MN_AVEV3_TXDM_OK) == 0) {
		printf("%s:INFO:%d bad send sts=0x%lux\n", dev->name, tx_num, val);
	}

	return 0;
}

static int mn_avev3_recv(struct eth_device *dev)
{
	volatile uchar *ptr;
	int len;
	ulong sts;

	for(;;) {
		sts = mn_avev3_read_rxdm(dev, rx_pos, MN_AVEV3_RXDM_CMDSTS);
		if((sts & MN_AVEV3_RXDM_OWN) == 0) {
			/* hardware ownership : no rec'ved packets */
			break;
		}

		ptr = NetRxPackets[rx_pos] + rx_off;
		if(sts & MN_AVEV3_RXDM_OK) {
			len = sts & MN_AVEV3_RXDM_SIZE_MASK;
			/* invalidate After DMA done */
			MN_AVEV3_CACHE_INV_A(ptr, len);
			NetReceive(ptr, len);
		} else {
			printf("%s: bad packet [%d] sts=0x%lux ptr=%p (dropped)\n",
				dev->name, rx_pos, sts, ptr);
		}

		/* invalidate Before DMA start */
		//MN_AVEV3_CACHE_INV_B(NetRxPackets[rx_pos], rx_siz + rx_off);
		MN_AVEV3_CACHE_FLS(NetRxPackets[rx_pos], rx_siz + rx_off);
		/* set hardware ownership (kick recv) */
		mn_avev3_write_rxdm(dev, rx_pos, MN_AVEV3_RXDM_CMDSTS, rx_siz);

		/* set to next rec'v position */
		if(++rx_pos >= PKTBUFSRX) {
			rx_pos = 0;
		}
	}
	return 0;
}

int util_mac_read(unsigned char *mac_buf);

int mn_avev3_initialize(int dev_num, int base_addr)
{
	struct eth_device *dev;
	ulong val;
	uchar enetaddr[6];

	dev = malloc(sizeof(*dev));
	if (!dev) {
		return -1;
	}
	memset(dev, 0, sizeof(*dev));

	dev->iobase = base_addr;

	/* check ID */
	val = mn_avev3_readreg(dev, MN_AVEV3_IDR);
	if(val != MN_AVEV3_IDR_VALUE) {
		printf("%s: IDR fail(base=%#x)\n", __func__, base_addr);
		free(dev);
		return -1;
	}

	/* get MAC address */
	if(0 == util_mac_read(enetaddr)){
		eth_setenv_enetaddr("ethaddr", enetaddr);
	}
	else{
		eth_getenv_enetaddr("ethaddr", enetaddr);
	}
	memcpy(dev->enetaddr, enetaddr, 6);

	dev->init = mn_avev3_init;
	dev->halt = mn_avev3_halt;
	dev->send = mn_avev3_send;
	dev->recv = mn_avev3_recv;
	sprintf(dev->name, "%s-%hu", DRIVERNAME, dev_num);

	eth_register(dev);

	return 1;
}
