/*
 * Copyright (C) 2011 Panasonic Corporation
 * All Rights Reserved.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
#include <common.h>
#include <asm/io.h>
#include <asm/arch/l2ca-regs.h>
#include <asm/arch/cacheflush.h>
#include <asm/arch/memory_layout.h>

#define	L2PCMD_INV		L2COPA_CM_INV
#define	L2PCMD_WB		L2COPA_CM_WB
#define	L2PCMD_WBINV		L2COPA_CM_WBINV
#define	L2PCMD_L2CPREFETCH	L2COPA_CM_PF_L2C
#define	L2PCMD_DPREFETCH	L2COPA_CM_PF_PFBUF
#define	L2PCMD_IPREFETCH	(L2COPA_CM_PF_PFBUF | L2COPA_TID)
#define	L2PCMD_TOUCH		L2COPA_CM_TOUCH
#define	L2PCMD_TOUCH_ZERO	L2COPA_CM_TOUCH_ZERO
#define	L2PCMD_TOUCH_DIRTY	L2COPA_CM_TOUCH_DIRTY
#define	L2PCMD_SYNC		L2COPE_CM_SYNC
#define	L2PCMD_PBFLUSH		L2COPE_CM_FLSH_PFBUF

#define	L2QCMD_INV		L2COQM_CM_INV
#define	L2QCMD_WB		L2COQM_CM_WB
#define	L2QCMD_WBINV		L2COQM_CM_WBINV
#define	L2QCMD_L2CPREFETCH	L2COQM_CM_PF_L2C
#define	L2QCMD_TOUCH		L2COQM_CM_TOUCH
#define	L2QCMD_TOUCH_ZERO	L2COQM_CM_TOUCH_ZERO
#define	L2QCMD_TOUCH_DIRTY	L2COQM_CM_TOUCH_DIRTY

#define	MAXAREAPURGE_L2C	0x00020000

static inline void l2ca_writel(unsigned long val, unsigned long offset)
{
	writel(val, UNIPHIER_L2C_BASE + offset);
}

static inline unsigned long l2ca_readl(unsigned long offset)
{
	return readl(UNIPHIER_L2C_BASE + offset);
}

#define	sync_opt_ex_bus() \
do { \
	unsigned long extra; \
	unsigned long tmp; \
	__asm__("mrc p15, 0, %0, c1, c0, 0" : "=r" (extra)); \
	if(extra & 0x1) { \
		tmp = *(volatile unsigned long *)UNIPHIER_KERNEL_UNCACHE_BASE; \
	} else { \
		tmp = *(volatile unsigned long *)UNIPHIER_SDRAM_BASE; \
	} \
} while (0)

static void l2ca_sync(void);

static void _set_l2cache_qcmd(unsigned long start, unsigned long size, int cmd)
{
unsigned long	l2coqm_val;
unsigned long	l2co_start, l2co_end, l2co_size;

    l2coqm_val = (size <= MAXAREAPURGE_L2C ? L2COQM_S_ADDRESS : L2COQM_S_ALL);
    l2coqm_val |= L2COQM_CE | L2COQM_CW_FLAG | (cmd & L2COQM_CM);

    l2co_start = start & L2_CACHE_ADRS_MASK;
    l2co_end   = (start + size + (L2_CACHE_BYTES - 1)) & L2_CACHE_ADRS_MASK;
    l2co_size  = (l2co_end - l2co_start) & L2COQSZ_SIZE;

    sync_opt_ex_bus();
    disable_interrupts();

    do {
	enable_interrupts();

	// Enter an local atomic section
	disable_interrupts();

	// Get a right to use L2-cache operation
	l2ca_writel(l2coqm_val, L2COQM_OFFSET);

	// Set address
	l2ca_writel(l2co_start, L2COQAD_OFFSET);

	// Set size
	l2ca_writel(l2co_size, L2COQSZ_OFFSET);

    } while ((l2ca_readl(L2COPPQSEF_OFFSET) & (L2COPPQSEF_FE | L2COPPQSEF_OE)) != 0x00);

    enable_interrupts();

    /* wait till all queue commands are completed */
    while ((l2ca_readl(L2COLPQS_OFFSET) & L2COLPQS_QST) == L2COLPQS_QST);
    l2ca_writel(L2COLPQS_EF, L2COLPQS_OFFSET);

    l2ca_sync();
    sync_opt_ex_bus();
}

static void _set_l2cache_pcmd(unsigned long start, int cmd)
{
unsigned long	dummy;

    if ((cmd == L2PCMD_SYNC)) {
	disable_interrupts();
	__asm__ __volatile__(
	"dmb\n"
	"str	%1, [%2]\n"
	"ldr	%0, [%2]\n"
	"dmb\n"
	: "=r"(dummy)
	: "r"(cmd & L2COPE_CM), "r"(UNIPHIER_L2C_BASE + L2COPE_OFFSET)
	: "memory"
	);
	enable_interrupts();
    } else if (cmd == L2PCMD_PBFLUSH) {
	/* Set L2Cache primitive command */
	l2ca_writel((cmd & L2COPE_CM), L2COPE_OFFSET);
	/* Dummy read for synchronization */
	dummy	= l2ca_readl(L2COPE_OFFSET);

    } else {
	/* Set target address and L2Cache primitive command */
	l2ca_writel((start & L2COPA_ADDRESS) | (cmd & L2COPA_TID_CM), L2COPA_OFFSET);
	/* Dummy read for synchronization */
	dummy	= l2ca_readl(L2COPA_OFFSET);
    }
}

static void l2ca_sync(void) {
	_set_l2cache_pcmd(0, L2PCMD_PBFLUSH);
	_set_l2cache_pcmd(0, L2PCMD_SYNC);
}

void __attribute__((naked)) flush_L2_dcache_all(void)
{
	asm volatile(
"	dsb									\n"
"	mov	r0, %14		@ L2QCMD_WBINV					\n"
"	orr	r0, r0, %0	@ L2COQM_S_ALL | L2COQM_CE | L2COQM_CW_FLAG	\n"
"	movw	r1, %1		@ L2COQM					\n"
"	movt	r1, %2								\n"
"	movw	r2, %3		@ L2COPPQSEF 					\n"
"	movt	r2, %4								\n"
"1:	str	r0, [r1]							\n"
"	ldr	r3, [r2]							\n"
"	tst	r3, %5		@ L2COPPQSEF_FE | L2COPPQSEF_OE			\n"
"	bne	1b								\n"
"	movw	r0, %6		@ L2COLPQS					\n"
"	movt	r0, %7								\n"
"2:	ldr	r1, [r0]							\n"
"	tst	r1, %8		@ L2COLPQS_QST					\n"
"	bne	2b								\n"
"	mov	r1, %9		@ L2COLPQS_EF					\n"
"	str	r1, [r0]							\n"
"	mov	r0, %10		@ L2PCMD_SYNC					\n"
"	movw	r1, %11		@ L2COPE					\n"
"	movt	r1, %12								\n"
"	str	r0, [r1]							\n"
"	ldr	r0, [r1]							\n"
"	mov	r0, %13								\n"
"	str	r0, [r1]	@ L2PCMD_PBFLUSH				\n"
"	ldr	r0, [r1]							\n"
"	dsb									\n"
"	bx	lr								\n"
	:
	: "I" (L2COQM_S_ALL | L2COQM_CE | L2COQM_CW_FLAG),
	  "i" ( (UNIPHIER_L2C_BASE + L2COQM_OFFSET)        & 0xffff),
	  "i" (((UNIPHIER_L2C_BASE + L2COQM_OFFSET) >> 16) & 0xffff),
	  "i" ( (UNIPHIER_L2C_BASE + L2COPPQSEF_OFFSET)        & 0xffff),
	  "i" (((UNIPHIER_L2C_BASE + L2COPPQSEF_OFFSET) >> 16) & 0xffff),
	  "I" (L2COPPQSEF_FE | L2COPPQSEF_OE),
	  "i" ( (UNIPHIER_L2C_BASE + L2COLPQS_OFFSET)        & 0xffff),
	  "i" (((UNIPHIER_L2C_BASE + L2COLPQS_OFFSET) >> 16) & 0xffff),
	  "I" (L2COLPQS_QST),
	  "I" (L2COLPQS_EF),
	  "I" (L2PCMD_SYNC),
	  "i" ( (UNIPHIER_L2C_BASE + L2COPE_OFFSET)        & 0xffff),
	  "i" (((UNIPHIER_L2C_BASE + L2COPE_OFFSET) >> 16) & 0xffff),
	  "I" (L2PCMD_PBFLUSH),
	  "I" (L2QCMD_WBINV)
	: "cc", "memory", "r0", "r1", "r2", "r3"
	);
}

void invalidate_L2_dcache_range(unsigned long start, unsigned long stop)
{
	if (stop > start) {
		if (stop - start <= MAXAREAPURGE_L2C) {
			_set_l2cache_qcmd(start, stop - start, L2QCMD_INV);
		} else {
			_set_l2cache_qcmd(start, stop - start, L2QCMD_WBINV);
		}
	}
}

void flush_L2_dcache_range(unsigned long start, unsigned long stop)
{
	if (stop > start) {
		_set_l2cache_qcmd(start, stop - start, L2QCMD_WBINV);
	}
}
