/*
 * Copyright (C) 2010 Panasonic Corporation
 * All Rights Reserved.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/arch/timer-regs.h>

static unsigned long timestamp = 0;	// precision: msec
static unsigned long lastinc = 0;	// val of counter when timestamp was increased

#define TIMER_LOAD_VAL 0xFFFFFFFF
#define MASK_16 0xFFFF

#define get_counter()	(~(*TM45BC))

int timer_init (void)
{
	timestamp = 0;
	lastinc = 0;
	*TM5BR = (TIMER_LOAD_VAL >> 16) & MASK_16;
	*TM5MD = (TM5MD_INIT_COUNTER | TM5MD_SRC_TM4CASCADE);
	*TM4BR = (TIMER_LOAD_VAL & MASK_16);
	*TM4MD = TM4MD_INIT_COUNTER;
	*TM5MD = (TM5MD_COUNT_ENABLE | TM5MD_SRC_TM4CASCADE);
	*TM4MD = (TM4MD_COUNT_ENABLE);

	return 0;
}

/*
 * timer without interrupts
 */
void reset_timer(void)
{
	reset_timer_masked();
}

ulong get_timer (ulong base)
{
	return get_timer_masked() - base;
}

void set_timer (ulong t)
{
	timestamp = t;
}

/* delay x useconds */
void __udelay (unsigned long usec)
{
	long tmo = usec * ((IOCLK / 1000) / 1000);
	unsigned long now, last = get_counter();

	while (tmo > 0) {
		now = get_counter();
		if (last > now) {/* count up timer overflow */
			tmo -= TIMER_LOAD_VAL - last + now;
		} else {
			tmo -= now - last;
		}
		last = now;
	}
	return;
}

void reset_timer_masked(void)
{
	/* reset time, capture current incrementer value time */
	lastinc = get_counter();
	timestamp = 0;		/* start "advancing" time stamp from 0 */
	return;
}

ulong get_timer_masked(void)
{
	/* current counter value */
	ulong now = get_counter();
	ulong buf = (now - lastinc) / (IOCLK / CONFIG_SYS_HZ);

	if (buf) {
		if (now >= lastinc) {	/* normal mode (non roll) */
			/* move stamp fordward with absoulte diff ticks */
			timestamp += buf;
		} else {		/* we have rollover of incrementer */
			timestamp += ((TIMER_LOAD_VAL - lastinc + now)
				      / (IOCLK / CONFIG_SYS_HZ));
		}
		lastinc = now;
	}
	return timestamp;
}

unsigned long long get_ticks(void)
{
	return get_timer(0);
}

ulong get_tbclk (void)
{
	return CONFIG_SYS_HZ;;
}
