#!/bin/sh
set -e

BOARD_NAME="MN2WS0220_ref"

TOOLCHAIN="arm-none-linux-gnueabi"
OBJCOPY="${TOOLCHAIN}-objcopy"

KNLDIR="${BUILDROOT}/usr/include/linux-base"
if [ -f ${KNLDIR}/include/generated/autoconf.h ]; then
	KNLAUTOCONF="${KNLDIR}/include/generated/autoconf.h"
else
	KNLAUTOCONF="${KNLDIR}/include/linux/autoconf.h"
fi
VMLINUX="${KNLDIR}/vmlinux"
COMPRESSDEF="./include/configs/${BOARD_NAME}.h"

CONFIG_ENABLE_SECURE_BOOT=`sed -n -e 's/.*CONFIG_ENABLE_SECURE_BOOT=//p' ${MAKETOP}/usr/include/linux-base/.config`

create_uImage()
{
#### create binary from vmlinux ####
    LINUX_BIN="./linux.bin"

    ${OBJCOPY} -O binary -R .note.gnu.build-id -R .comment -S ${VMLINUX} ${LINUX_BIN}

    RET=$?
    if [ "${RET}" -ne 0 ]
	then
	echo "error in objcopy!"
	return ${RET}
    fi

#### create compressed binary from binary ####
    COMMPRESS_CONF=`awk '/CONFIG_KERNEL_COMPRESS/ {print $3;}' ${COMPRESSDEF}`

    case "$COMMPRESS_CONF" in
        "CONFIG_LZMA") 
			COMPRESS="lzma"
			COMPRESS_SUFFIX="lzma"
			COMPRESS_TYPE="lzma"
			LEVEL="-9";;
        "CONFIG_LZO")  
			# -U option means delete input files after successful (like gzip )
			COMPRESS="$(BUILDROOT)/tools/lzop -U"
			COMPRESS_SUFFIX="lzo"
			COMPRESS_TYPE="lzo"
			LEVEL="-9";;
        *) 
			echo "default gzip!"
			COMPRESS="gzip"
			COMPRESS_SUFFIX="gz"
			COMPRESS_TYPE="gzip"
			LEVEL="-9";;
    esac

    LINUX_CMP="${LINUX_BIN}.${COMPRESS_SUFFIX}"

    echo "${COMPRESS} ${LEVEL} ${LINUX_BIN} ${LINUX_CMP}"

	${COMPRESS} -f ${LEVEL} ${LINUX_BIN}
	RET=$?
	
	if [ "${RET}" -ne 0 ]
		then
		echo "error in compress!"
		rm "${LINUX_BIN}"
		return ${RET}
	fi


#### create uImage from compressed binary ####
    MKIMAGE="./tools/mkimage"
    ARCH="arm"
    KERNEL_TEXT_OFFSET="0x00008000"
    KERNEL_RAM_BASE_ADDR=`awk '/CONFIG_KERNEL_RAM_BASE_ADDRESS\>/ {print $3;}' ${KNLAUTOCONF}`
    KERNEL_TEXT_ADDR=`printf '%#x\n' $((${KERNEL_RAM_BASE_ADDR}+${KERNEL_TEXT_OFFSET}))`
    UIMAGE="uImage"

    ${MKIMAGE} -A ${ARCH} -O linux -T kernel -C ${COMPRESS_TYPE} -a ${KERNEL_TEXT_ADDR} -e ${KERNEL_TEXT_ADDR} -n "Linux Kernel Image" -d ${LINUX_CMP} ${UIMAGE}

    RET=$?
    echo "${MKIMAGE} -A ${ARCH} -O linux -T kernel -C ${COMPRESS_TYPE} -a ${KERNEL_TEXT_ADDR} -e ${KERNEL_TEXT_ADDR} -n \"Linux Kernel Image\" -d ${LINUX_CMP} ${UIMAGE}"

    if [ "${RET}" -ne 0 ]
	then
	echo "error in mkimage!"
	return ${RET}
    fi

    if [ "${CONFIG_ENABLE_SECURE_BOOT}" = "y" ]; then
        SIGN_TOOLS_DIR=../secure/tool
        SIGN_TOOL="${SIGN_TOOLS_DIR}/gensign -s ${SIGN_TOOLS_DIR}/app_ek.bin"
        SIGN_TMP=sign.bin
        ${OBJCOPY} -I binary -O binary --pad-to=0x1FFF00 ${UIMAGE}
        ${SIGN_TOOL} ${UIMAGE} ${SIGN_TMP}
        cat ${SIGN_TMP} >> ${UIMAGE}
        rm ${SIGN_TMP}
    fi

    mv ${UIMAGE} ${KNLDIR}/arch/arm/boot/
    rm ${LINUX_CMP}
    return 0
}

make clean
make ${BOARD_NAME}_config
make tools
create_uImage
cp -v -p ${KNLDIR}/arch/arm/boot/${UIMAGE} ${BUILDROOT}/usr/target/
make
cp -v -p u-boot.bin ${BUILDROOT}/usr/target/
RET=$?
#echo "${RET}"
exit ${RET}
