#include <common.h>
#include <mmc.h>
#include <malloc.h>
#include <asm/io.h>
#include <asm/errno.h>
#include <asm/byteorder.h>
#include "mn2ws_mmc.h"
#include <asm/cache.h>
#include <asm/arch/cacheflush.h>

#if 0
#define DBG_PRT_REG(fmt, args...) printf(fmt, ##args)
#else
#define DBG_PRT_REG(fmt, args...)
#endif

#if 0
#define DBG_PRT_MMC_GEN(fmt, args...) printf(fmt, ##args)
#else
#define DBG_PRT_MMC_GEN(fmt, args...)
#endif
#if 0
#define DBG_PRT_MMC_FUNC(fmt, args...)	printf("%s: " fmt, __FUNCTION__, ## args)
#else
#define DBG_PRT_MMC_FUNC(fmt, args...)
#endif


static u32 read_32x1(u32 addr)
{
 	u32 ret;

	DBG_PRT_REG("R , addr:0x%08x\n", addr);

	ret = *(volatile u32 *)addr;

	DBG_PRT_REG("R , ret value:0x%08x\n", ret);

	return ret;
}

static u32 maskread_32x1(u32 addr, u32 mask)
{
	u32 ret;

	DBG_PRT_REG("MR, addr:0x%08x, mask:0x%08x\n", addr, mask);

	ret = read_32x1(addr) & mask;
	DBG_PRT_REG("MR, ret value:0x%08x\n", ret);

	return ret;
}

static u32 write_32x1(u32 addr, u32 data)
{
	DBG_PRT_REG("W , addr:0x%08x, data:0x%08x\n", addr, data);

	*(volatile u32 *)addr = data;

	DBG_PRT_REG("W , ret value:0x%08x\n", data);

	/* temp */
	udelay(100);

	return data;
}

static u32 maskwrite_32x1(u32 addr, u32 mask, u32 data)
{
	u32 value;
	
	DBG_PRT_REG("MW, addr:0x%08x, mask:0x%08x, data:0x%08x\n", addr, mask, data);

	value = (read_32x1(addr) & ~(mask)) | (data & mask);

	write_32x1(addr, value);

	DBG_PRT_REG("MW, ret value:0x%08x\n", value);

	return value;
}

static int mio_dma_status(void)
{
	return 0;
}

static void mio_dma_set_adrmode(int rw)
{
	u32 ChSrcAmode = 0;
	u32 ChDstAmode = 0;

	/* Ring buffer setting */
	if(rw == MMC_DATA_READ){
		/* MMC -> DDR */
		ChSrcAmode |= DMAC_CDMIO_CHAMODE_AUPDT2;
		ChDstAmode |= DMAC_CDMIO_CHAMODE_ENDIAN0;	/* Little-Endian */
		ChDstAmode |= DMAC_CDMIO_CHAMODE_TYPE1;
	}else{
		/* DDR -> MMC */
		ChSrcAmode |= DMAC_CDMIO_CHAMODE_ENDIAN0;	/* Little-Endian */
		ChSrcAmode |= DMAC_CDMIO_CHAMODE_TYPE1;
		ChDstAmode |= DMAC_CDMIO_CHAMODE_AUPDT2;
	}

	ChSrcAmode |= 0x400;
	ChDstAmode |= 0x400;

	write_32x1(DMAC_CDMIO_CH4SRCAMODE, ChSrcAmode);
	write_32x1(DMAC_CDMIO_CH4DSTAMODE, ChDstAmode);
}

static void mio_dma_set_rb(u32 Start, u32 Count)
{
	u32 CountByte;
	u32 end;

	CountByte = Count;
	end = Start + CountByte;

	write_32x1(DMAC_CDMIO_RB4BGNADRS, Start);
	write_32x1(DMAC_CDMIO_RB4ENDADRS, end);

	write_32x1(DMAC_CDMIO_RB4RDPTR, Start);
	write_32x1(DMAC_CDMIO_RB4WRPTR, Start);

	return;
}

static void mio_dma_set_transfer(int rw, u32 Addr, u32 Count)
{
	u32 CountByte;

	CountByte = Count;

	write_32x1(DMAC_CDMIO_CH4IE, DMAC_CDMIO_CH_ALLCLR);
	maskwrite_32x1(DMAC_CDMIO_CH4IR, DMAC_CDMIO_CH_CH1, DMAC_CDMIO_CH_CH1);
	maskwrite_32x1(DMAC_CDMIO_CH4IE, DMAC_CDMIO_CH_CH1, DMAC_CDMIO_CH_CH1);

	if(rw == MMC_DATA_READ){
		/* MMC -> DDR */
		write_32x1(DMAC_CDMIO_CH4SRCSTRTADRS, 0);
		write_32x1(DMAC_CDMIO_CH4DSTSTRTADRS, Addr);
	}else{
		/* DDR -> eMMC */
		write_32x1(DMAC_CDMIO_CH4SRCSTRTADRS, Addr);
		write_32x1(DMAC_CDMIO_CH4DSTSTRTADRS, 0);
	}

	write_32x1(DMAC_CDMIO_CH4SIZE, CountByte);

	return;
}

static void mio_dma_start(void)
{
	u32 Strt0 = 0;

	Strt0 |= DMAC_CDMIO_STRT0_REQ;
	Strt0 &= ~DMAC_CDMIO_STRT0_STOP;

	/* DMA start */
	write_32x1(DMAC_CDMIO_STRT0, Strt0);

	return;
}

static int mio_dma_checkdmaend(void)
{
	int   count = 0;
	int  ret = 1;
	u32 value;

	for(count = 0;count < 1000;count++){
		value = read_32x1(DMAC_CDMIO_CH4IR);
		if(value & DMAC_CDMIO_CH_CH1){
			ret = 0;
			maskwrite_32x1(DMAC_CDMIO_CH4IR, DMAC_CDMIO_CH_CH1, DMAC_CDMIO_CH_CH1);
			break;
		}
	}

	return ret;
}

static int mn2ws_mmc_wait_buffer_read_enable(struct mmc *mmc)
{
	volatile u32 sd_info2;
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;

	/* Check SD_INFO2 bit8 BRE */
	while(1){
		sd_info2 = maskread_32x1((u32)&mn2ws->sd_info2, SD_INFO2_ERROR_ALL | SD_INFO2_INFO_BRE);

		if(sd_info2 & SD_INFO2_ERROR_ALL){
			while(1);
			return 1;
		}else if(sd_info2 & SD_INFO2_INFO_BRE){
			break;
		}
	}

	write_32x1((u32)&mn2ws->sd_info2, ~SD_INFO2_INFO_BRE);	// Clear SD_INFO2 bit8 BRE

	return 0;
}

static int mn2ws_mmc_read_buf(struct mmc *mmc, u32 size, u16 *buf)
{
	int   ret = 0;
	int   i, j;
	u32 tmp_data;
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;
	u16 tmp_buf;

	for(i = 0; i < (int)(size / mmc->read_bl_len); i++){
		/* Check SD_INFO2 bit8 BRE */
		if(mn2ws_mmc_wait_buffer_read_enable(mmc)){
			ret = 1;
			goto end;
		}
		for(j = 0; j < mmc->read_bl_len / 4; j++){
			tmp_data = read_32x1((u32)&mn2ws->sd_buf);
			*buf =  tmp_data & 0xFFFF;
			/* *buf++; */
			tmp_buf = *buf++;	/* for warning */
			*buf = (tmp_data >> 16) & 0xFFFF;
			/* *buf++; */
			tmp_buf = *buf++;	/* for warning */
		}
	}


end:
	return ret;
}

static int mn2ws_mmc_wait_buffer_write_enable(struct mmc *mmc)
{
	volatile u32 sd_info2;
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;

	/* Check SD_INFO2 bit9 BWE */
	while(1){
		sd_info2 = maskread_32x1((u32)&mn2ws->sd_info2, SD_INFO2_ERROR_ALL | SD_INFO2_INFO_BWE);

		if(sd_info2 & SD_INFO2_ERROR_ALL){
			while(1);
			return 1;
		}else if(sd_info2 & SD_INFO2_INFO_BWE){
			break;
		}
	}

	write_32x1((u32)&mn2ws->sd_info2, ~SD_INFO2_INFO_BWE);	// Clear SD_INFO2 bit9 BWE

	return 0;
}

static int mn2ws_mmc_write_buf(struct mmc *mmc, u32 size, const u16 *buf)
{
	int   ret = 0;
	int   i, j;
	u32 tmp_data;
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;
	u16 tmp_buf;

	for(i = 0; i < (int)(size / mmc->read_bl_len); i++){
		/* Check SD_INFO2 bit9 BWE */
		if(mn2ws_mmc_wait_buffer_write_enable(mmc)){
			ret = 1;
			goto end;
		}
		for(j = 0; j < mmc->read_bl_len / 4; j++){
			tmp_data  =  (*buf) & 0xFFFF;
			/* *buf++; */
			tmp_buf = *buf++;	/* for warning */
			tmp_data |= ((*buf) & 0xFFFF) << 16;
			/* *buf++; */
			tmp_buf = *buf++;	/* for warning */
			write_32x1((u32)&mn2ws->sd_buf, tmp_data);
		}
	}

end:
	return ret;
}

static int mn2ws_mmc_data_end(struct mmc *mmc)
{
	volatile u32 sd_info1;
	volatile u32 sd_info2;
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;

	/* Check SD_INFO1 bit2 */
	while(1){
		sd_info1 = maskread_32x1((u32)&mn2ws->sd_info1, SD_INFO1_INFO_RWA_CMP);
		sd_info2 = maskread_32x1((u32)&mn2ws->sd_info2, SD_INFO2_ERROR_ALL);

		if(sd_info2 != 0x0000){
			while(1);
			return 1;
		}else if(sd_info1 == SD_INFO1_INFO_RWA_CMP){
			break;
		}
	}

	write_32x1((u32)&mn2ws->sd_info1, ~SD_INFO1_INFO_RWA_CMP);	// Clear SD_INFO1 bit2

	return 0;
}

static int mn2ws_send_command_main(struct mmc *mmc, struct mmc_cmd *cmd, struct mmc_data *data)
{
	volatile u32 sd_info1 = 0x0;
	volatile u32 sd_info2 = 0x0;
	int   ret = 0;
	u32 command = 0;
	int dma;
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;

	/* Enable DMA only read block or write block command */
	if (cmd->cmdidx != 17 && cmd->cmdidx != 18 && cmd->cmdidx != 24 && cmd->cmdidx != 25)
		dma = 0;
	else
#if defined(CONFIG_MMC_TEST)
		dma = test_dma_ena;
#else
		dma = 1;
#endif

	/* Check Command Busy Status */
	while(maskread_32x1((u32)&mn2ws->sd_info2, SD_INFO2_INFO_CBSY) == SD_INFO2_INFO_CBSY);

	/* Clear All IRQ */
	write_32x1((u32)&mn2ws->sd_info1, 0x0000);
	write_32x1((u32)&mn2ws->sd_info2, 0x0000);

	if(data){
		write_32x1((u32)&mn2ws->sd_size, data->blocksize);

		if(dma){
			/* MMC buffer DMA transfer set(enable) */
			maskwrite_32x1((u32)&mn2ws->cc_ext_mode, 0x0002, 0x0002);
		}else{
			/* MMC buffer DMA transfer set(disable) */
			maskwrite_32x1((u32)&mn2ws->cc_ext_mode, 0x0002, 0x0000);
		}

		if(data->blocks > 1){
			/* Enables SD_SECCNT */
			maskwrite_32x1((u32)&mn2ws->sd_stop, 0x0100, 0x0100);
			/* Set SD_SECCNT */
			write_32x1((u32)&mn2ws->sd_seccnt, data->blocks);
		}else{
			/* Disables SD_SECCNT */
			maskwrite_32x1((u32)&mn2ws->sd_stop, 0x0100, 0x0000);
		}

		if(dma){
			mio_dma_status();
			mio_dma_set_adrmode(data->flags);
			if(data->flags == MMC_DATA_READ){
				mio_dma_set_rb((u32)data->dest, (data->blocks * data->blocksize));
				mio_dma_set_transfer(data->flags, (u32)data->dest, (data->blocks * data->blocksize));
			}else{
				vaddr_purge_cache( (u32)data->src, (data->blocks * data->blocksize), PURGE_CACHE_D_PURGE_INV );	//L1/L2 cache purge inv
				mio_dma_set_rb((u32)data->src,  (data->blocks * data->blocksize));
				mio_dma_set_transfer(data->flags, (u32)data->src,  (data->blocks * data->blocksize));
			}
			mio_dma_start();
		}
	}

	write_32x1((u32)&mn2ws->sd_arg0, cmd->cmdarg);

	if(cmd->resp_type != SD_CMD_RESP_NORMAL){
		switch(cmd->cmdidx) {
			case 0:
				command = SD_CMD_SR_ADTC_R1;
				break;
			case 1:
				command = SD_CMD_NODATA_R3;
				break;
			case 3:
				command = SD_CMD_NODATA_R1;
				break;
			case 6:
				command = SD_CMD_NODATA_R1B;
				break;
			case 7:
				if(cmd->resp_type == SD_CMD_RESP_R1){
					command = SD_CMD_NODATA_R1;
				}else if(cmd->resp_type == SD_CMD_RESP_R1B){
					command = SD_CMD_NODATA_NORESP;
				}
				break;
			case 8:
				command = SD_CMD_SR_ADTC_R1;
				break;
			case 12:
				command = SD_CMD_NODATA_NORESP;
				break;
			case 14:
				command = SD_CMD_SR_ADTC_R1;
				break;
			case 19:
				command = SD_CMD_SW_ADTC_R1;
				break;
			case 62:
				command = SD_CMD_NODATA_R1B;
				break;
		}
	}
	command |= cmd->cmdidx;
	write_32x1((u32)&mn2ws->sd_cmd,  command);

	/* Wait Command response */
	while(1){
		sd_info1 = maskread_32x1((u32)&mn2ws->sd_info1, SD_INFO1_INFO_CMD_CMP);
		sd_info2 = maskread_32x1((u32)&mn2ws->sd_info2, SD_INFO2_ERROR_ALL);

		if(sd_info2 != 0x0000){
			DBG_PRT_MMC_GEN("Wait command response error! 0x%x\n", sd_info2);
			return sd_info2;
		}else if(sd_info1 == SD_INFO1_INFO_CMD_CMP){
			break;
		}
	}

	/* Clear Command response */
	write_32x1((u32)&mn2ws->sd_info1, ~SD_INFO1_INFO_CMD_CMP);

	/* Get Response */
	cmd->response[0] = ((read_32x1((u32)&mn2ws->sd_rsp1) & 0xFFFF) << 16) | (read_32x1((u32)&mn2ws->sd_rsp0) & 0xFFFF);
	cmd->response[1] = ((read_32x1((u32)&mn2ws->sd_rsp3) & 0xFFFF) << 16) | (read_32x1((u32)&mn2ws->sd_rsp2) & 0xFFFF);
	cmd->response[2] = ((read_32x1((u32)&mn2ws->sd_rsp5) & 0xFFFF) << 16) | (read_32x1((u32)&mn2ws->sd_rsp4) & 0xFFFF);
	cmd->response[3] = ((read_32x1((u32)&mn2ws->sd_rsp7) & 0xFFFF) << 16) | (read_32x1((u32)&mn2ws->sd_rsp6) & 0xFFFF);

	if(data){
		if(!dma){
			if(data->flags == MMC_DATA_READ){
				if(mn2ws_mmc_read_buf(mmc, (data->blocks * data->blocksize), (u16 *)data->dest)){
					printf("mn2ws_mmc_read_buf error\n");
					goto err;
				}
			}else{
				if(mn2ws_mmc_write_buf(mmc, (data->blocks * data->blocksize), (u16 *)data->src)){
					printf("mn2ws_mmc_write_buf error\n");
					goto err;
				}
			}
		}
		/* Wait All accesses complete */
		if(mn2ws_mmc_data_end(mmc)){
			printf("mn2ws_mmc_data_end error\n");
			goto err;
		}
	}

err:

	if(data){
		if(dma){
			/* eMMC buffer DMA transfer set(disable) */
			maskwrite_32x1((u32)&mn2ws->cc_ext_mode, 0x0002, 0x0000);
			if (mio_dma_checkdmaend()) {
				printf("mio_dma_checkdmaend error\n");
			}
			if(data->flags == MMC_DATA_READ){
				vaddr_purge_cache( (u32)data->dest, (data->blocks * data->blocksize), PURGE_CACHE_D_INV );	//L1/L2 cache inv
			}
		}
	}

	return ret;
}

static void mn2ws_set_ios(struct mmc *mmc)
{
	uchar div;
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;

	DBG_PRT_MMC_FUNC("\n");

	/* Set bus width */
	if (mmc->bus_width == 8) 
		maskwrite_32x1((u32)&mn2ws->sd_option, 0x0000a000, 0x00006000);
	else if (mmc->bus_width == 4)
		maskwrite_32x1((u32)&mn2ws->sd_option, 0x0000a000, 0x00004000);
	else
		maskwrite_32x1((u32)&mn2ws->sd_option, 0x0000a000, 0x0000c000);

	/* Set the clock speed */
	if (mmc->f_max % (2 * mmc->clock) == 0)
		div = (uchar)(mmc->f_max / (2 * mmc->clock) - 1);
	else
		div = (uchar)(mmc->f_max / (2 * mmc->clock));
	DBG_PRT_MMC_GEN("div %d set_clk %d max_clk %d\n", div, mmc->clock, mmc->f_max);

	write_32x1((u32)&mn2ws->sd_clk_ctrl, 0x00000100 | (div & 0xff));
}

static int mn2ws_send_cmd(struct mmc *mmc, struct mmc_cmd *cmd, struct mmc_data *data)
{
	int ret = 0;
	int status;
	
	DBG_PRT_MMC_FUNC("\n");

	switch (cmd->resp_type) {
		case MMC_RSP_R1:
			cmd->resp_type = SD_CMD_RESP_R1;
			break;
		case MMC_RSP_R1b:
			cmd->resp_type = SD_CMD_RESP_R1B;
			break;
		case MMC_RSP_R2:
			cmd->resp_type = SD_CMD_RESP_R2;
			break;
		case MMC_RSP_R3:
			cmd->resp_type = SD_CMD_RESP_R3;
			break;
		case MMC_RSP_NONE:
			cmd->resp_type = SD_CMD_RESP_NORMAL;
			break;
		default:
			cmd->resp_type = SD_CMD_RESP_NORMAL;
			break;
	}
	if (cmd->cmdidx == MMC_CMD_APP_CMD)
		cmd->resp_type = SD_CMD_RESP_NORMAL;

	status = mn2ws_send_command_main(mmc, cmd, data);
	if (status & 0x0040) {
		DBG_PRT_MMC_GEN("time out error!\n");
		return TIMEOUT;
	}

	if (cmd->resp_type == SD_CMD_RESP_R2) {
		uint response[4];
		/* bit104-127 */
		response[0] = ((uint)cmd->response[3] << 8) | ((uint)cmd->response[2] & 0xFF000000 >> 24);
		/* bit72-103 */
		response[1] = ((uint)cmd->response[2] << 8) | ((uint)cmd->response[1] & 0xFF000000 >> 24);
		/* bit40-71 */
		response[2] = ((uint)cmd->response[1] << 8) | ((uint)cmd->response[0] & 0xFF000000 >> 24);
		/* bit8-39 */
		response[3] = ((uint)cmd->response[0] << 8);
		DBG_PRT_MMC_GEN("response[0-3] 0x%08x 0x%08x 0x%08x 0x%08x\n", 
					response[0], response[1], response[2], response[3]);
		cmd->response[0] = response[0];
		cmd->response[1] = response[1];
		cmd->response[2] = response[2];
		cmd->response[3] = response[3];
	}
	else {
		DBG_PRT_MMC_GEN("response[0] 0x%08x\n", cmd->response[0]); 
	}

	return ret;
}

/* Entered into mmc structure during driver init */
static int mn2ws_init(struct mmc *mmc)
{
	mn2ws_mmc_t *mn2ws = (mn2ws_mmc_t *)mmc->priv;
	
	DBG_PRT_MMC_FUNC("\n");

	write_32x1((u32)&mn2ws->soft_rst, 0x00000007);        /* reset clear */
	write_32x1((u32)&mn2ws->soft_rst, 0x00000000);        /* reset */
	write_32x1((u32)&mn2ws->soft_rst, 0x00000007);        /* reset clear */
	
	/* set SDR */
	maskwrite_32x1((u32)&mn2ws->sdif_mode, 0x00000001, 0x00000000);
	
	/* Set Clock Enable , Change Clock to BaseClock/256 */
	write_32x1((u32)&mn2ws->sd_clk_ctrl, 0x00000140);

	write_32x1((u32)&mn2ws->sd_size, 0x0000);
	
	return 0;
}

static int mn2ws_initialize(bd_t *bis)
{
	struct mmc *mmc = NULL;
	struct mn2ws_mmc *regs = (struct mn2ws_mmc *)SDIP_BASEADDR;
	
	DBG_PRT_MMC_FUNC("\n");

	mmc = malloc(sizeof(struct mmc));

	if (!mmc)
		return -ENOMEM;

	strcpy(mmc->name, "MN2WS eMMC");

	mmc->priv = regs;
	
	/* functions */
	mmc->send_cmd = mn2ws_send_cmd;
	mmc->set_ios = mn2ws_set_ios;
	mmc->init = mn2ws_init;

	mmc->host_caps = MMC_MODE_4BIT;

	mmc->voltages = MMC_VDD_32_33 | MMC_VDD_33_34;	/* 3.3V */

	mmc->f_min = 43400;		/* 43.40KHz */
	mmc->f_max = 100000000;	/* 100MHz */

	mmc_register(mmc);

#if defined(CONFIG_MMC_TEST)
	test_mul_read_ena = 1;
//	printf("Multi block read command is enable.\n");
	test_dma_ena = 1;
//	printf("DMA transfer is enable.\n");
#endif

	return 0;
}

int mn2ws_mmc_init(bd_t *bis)
{
	return mn2ws_initialize(bis);
}
