/* $Id: nand-gpbc.c,v 1.1.1.1 2012/09/25 23:51:31 txbsd Exp $
 * nand-gpbc.c: MTD NAND flash driver for gpbc
 *
 * Copyright (C) 2010-2011 Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

// ### caution ###
// If you use oobbuf in read/write, confirm the usage of "ops.ooblen".
// If kernel version is change, confirm the usage of "ops.ooblen" in "nand_base.c nand_bbt.c nand-XXX.c".

#define MTD_NAND_MN2WS_GPBC_ECC_BUG
//#define MTD_NAND_GPBC_TEST
//#define MTD_NAND_GPBC_ERASEALL

#include <common.h>
#include <malloc.h>
#include <nand.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/compat.h>
#include <linux/mtd/nand_ecc.h>
#include <linux/err.h>
#include <asm/errno.h>
#include <asm/io.h>

#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
	#include <asm/cache.h>
	#include <asm/arch/cacheflush.h>
#endif

#include "nand-gpbc-unit.h"
#include "nand-gpbc.h"
#include "nand-gpbc-reg.h"

#if !((0 < NAND_GPBC_FLASH_BANKS) && (NAND_GPBC_FLASH_BANKS <= NAND_GPBC_MAX_FLASH_BANKS))
#error "Error: Invalid NAND_GPBC_FLASH_BANKS."
#endif

static uint8_t nand_gpbc_bbt_pattern[] = NAND_GPBC_UNIT_BBT_PATTERN_MASTER;
static struct nand_bbt_descr nand_gpbc_BbtMasterDesc = {
	.options	= NAND_GPBC_BBT_OPTION,
	.offs		= NAND_GPBC_BBT_OFFS,
	.len		= NAND_GPBC_BBT_LEN,
	.veroffs	= NAND_GPBC_BBT_VEROFFS,
	.maxblocks	= NAND_GPBC_UNIT_BBT_MAXBLOCKS,
	.pattern	= nand_gpbc_bbt_pattern
};
static uint8_t nand_gpbc_mirror_pattern[] = NAND_GPBC_UNIT_BBT_PATTERN_MIRROR;
static struct nand_bbt_descr nand_gpbc_BbtMirrorDesc = {
	.options	= NAND_GPBC_BBT_OPTION,
	.offs		= NAND_GPBC_BBT_OFFS,
	.len		= NAND_GPBC_BBT_LEN,
	.veroffs	= NAND_GPBC_BBT_VEROFFS,
	.maxblocks	= NAND_GPBC_UNIT_BBT_MAXBLOCKS,
	.pattern	= nand_gpbc_mirror_pattern
};

static struct nand_ecclayout nand_gpbc_EccLayout = {
	.eccbytes	= 0,
	.eccpos		= NAND_GPBC_UNIT_OOB_ECCPOS,
	.oobfree	= NAND_GPBC_UNIT_OOB_OOBFREE
};

#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
static uint8_t nand_gpbc_bbt_flag_ff_pattern[] = NAND_GPBC_BBT_FLAG_FF_PATTERN;
static struct nand_bbt_descr nand_gpbc_BbtFlagDesc = {
	.options	= NAND_GPBC_BBT_OPTION,
	.offs		= NAND_GPBC_BBT_FLAG_OFFS,
	.len		= NAND_GPBC_BBT_FLAG_LEN,
	.pattern	= nand_gpbc_bbt_flag_ff_pattern
};
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */

static uint8_t nand_gpbc_scan_ff_pattern[] = NAND_GPBC_BBTSCAN_FF_PATTERN;
static struct nand_bbt_descr nand_gpbc_BbPatternDesc = {
	.options	= NAND_GPBC_BBT_OPTION,
	.offs		= NAND_GPBC_BBTSCAN_OFFS,
	.len		= NAND_GPBC_BBTSCAN_LEN,
	.pattern	= nand_gpbc_scan_ff_pattern
};

#ifdef	CONFIG_MTD_NAND_MN2WS_BB
	static struct nand_bbm	nand_gpbc_ABbm[CONFIG_SYS_NAND_MAX_CHIPS];
#endif	//CONFIG_MTD_NAND_MN2WS_BB


//----------------------------
//------ layer 1 func --------
//----------------------------


static void nand_gpbc_index32( struct nand_gpbc_info *PGpbc, u32 cmd, u32 data )
{
	*(volatile u32 *)PGpbc->memBase = cmd;
	*(volatile u32 *)(PGpbc->memBase + 0x10) = data;
}


static void nand_gpbc_write32( u32 data, void *addr )
{
	*(volatile u32 *)addr = data;
}


static u32 nand_gpbc_read32( void *addr )
{
	u32 data;
	
	data = *(volatile u32 *)addr;
	return data;
}


static void nand_gpbc_clearIntStatus(struct nand_gpbc_info *PGpbc, int bank)
{
	nand_gpbc_write32( NAND_GPBC_NI__MASK, PGpbc->regBase + PGpbc->nist[bank] );
}


#ifndef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
	static u32 nand_gpbc_pollIrq( struct nand_gpbc_info *PGpbc, int bank, u32 mask )
	{
		u32		result;
		unsigned long		timeo;
		
		//sync
		(void)nand_gpbc_read32( PGpbc->regBase + PGpbc->nie[bank] );
		
		reset_timer();
		do{
			result = nand_gpbc_read32( PGpbc->regBase + PGpbc->nist[bank] );
			if( result & mask ){
				return result;
			}
			timeo = get_timer(0);
		}while( ((u64)NAND_GPBC_CMD_TIMEOUT * CONFIG_SYS_HZ + 999) / 1000 > timeo );
		result = nand_gpbc_read32( PGpbc->regBase + PGpbc->nist[bank] );
		if( !(result & mask) ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}
		
		return result;
	}
#endif


#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
	static irqreturn_t nand_gpbc_irq( int this_irq, void *dev_id )
	{
		struct nand_gpbc_info *PGpbc = (struct nand_gpbc_info *)dev_id;
		
		nand_gpbc_write32( 0, PGpbc->regBase + PGpbc->nie[0] );	//disable irq
		nand_gpbc_write32( 0, PGpbc->regBase + PGpbc->nie[1] );	//disable irq
		(void)nand_gpbc_read32( PGpbc->regBase + PGpbc->nie[1] );	//sync
		PGpbc->event = 1;
		wake_up( &PGpbc->queue );
		
		return IRQ_HANDLED;
	}
	
	
	static u32 nand_gpbc_waitIrq( struct nand_gpbc_info *PGpbc, int bank, u32 mask )
	{
		long	status;
		u32		result;
		
		PGpbc->event = 0;
		nand_gpbc_write32( mask, PGpbc->regBase + PGpbc->nie[bank] );	//sync & enable irq
		status = wait_event_timeout( PGpbc->queue, (PGpbc->event == 1)
				, msecs_to_jiffies(NAND_GPBC_IRQ_TIMEOUT) );
		nand_gpbc_write32( 0, PGpbc->regBase + PGpbc->nie[bank] );	//disable irq : abnormal condition
		result = nand_gpbc_read32( PGpbc->regBase + PGpbc->nist[bank] );
		if( !status && (PGpbc->event != 1) ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}
		
		return result;
	}
#endif

#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
static void nand_gpbc_set_cmd_desc_read_page(struct nand_gpbc_cmd_desc_t *cmd_desc,
					     u32 flash_page, u16 cmd_flags, u64 mem_ptr)
{
	cmd_desc->next_ptr = 0;
	cmd_desc->flash_page = flash_page;
	cmd_desc->reserved0 = 0;
	cmd_desc->cmd_type = 0x2001;	//read-ahead
	cmd_desc->cmd_flags = cmd_flags;
	cmd_desc->sync_arg = 0;
	cmd_desc->mem_ptr = mem_ptr;
	cmd_desc->status = 0;
	cmd_desc->reserved1 = 0;
	cmd_desc->reserved2 = 0;
	cmd_desc->sync_flag_ptr = 0;
	cmd_desc->mem_copy_addr = 0;
	cmd_desc->meta_data_addr = 0;
}

static u32 nand_gpbc_poll_cmd_dma(struct nand_gpbc_info *PGpbc)
{
	volatile struct nand_gpbc_cmd_desc_t *cmd_desc = PGpbc->cmd_desc;
	while (1) {
		if (cmd_desc->status & NAND_GPBC_CDESC_STAT_CMP) {
			break;
		}
	}
	return (u32)cmd_desc->status;
}
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */

static int nand_gpbc_getDevice( struct nand_gpbc_info *PGpbc, int newState )
{
	struct nand_chip	*PChip = &PGpbc->chip;
	
	PChip->state = newState;
	return 0;
}


static void nand_gpbc_releaseDevice( struct nand_gpbc_info *PGpbc )
{
	struct nand_chip *PChip = &PGpbc->chip;
	
	PChip->state = FL_READY;
}


#ifndef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
	static u32 nand_gpbc_checkIrq( struct nand_gpbc_info *PGpbc, int bank )
	{
		u32		result;
		
		result = nand_gpbc_read32( PGpbc->regBase + PGpbc->nist[bank] );
		
		return result;
	}
#endif


static u32 nand_gpbc_getCorrectedNum( struct nand_gpbc_info *PGpbc, int bank )
{
	u32 correctedNum;
	
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
	correctedNum = (PGpbc->cmd_desc->status & NAND_GPBC_CDESC_STAT_MAX_ERR) >>
		NAND_GPBC_CDESC_STAT_MAX_ERR_SHIFT;
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
	correctedNum = (nand_gpbc_read32( PGpbc->regBase + NAND_GPBC_NECCINF ) >> (8 * bank)) & 0x7f;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
	
	return correctedNum;
}


static u8 *nand_gpbc_transferOob( struct nand_gpbc_info *PGpbc, uint8_t *pDestBuf,
				  struct mtd_oob_ops *POps, size_t len )
{
	struct nand_chip	*PChip = &PGpbc->chip;
	
	switch(POps->mode)
	{
		case MTD_OOB_PLACE:
		case MTD_OOB_RAW:
			memcpy(pDestBuf, PChip->oob_poi + POps->ooboffs, len);
			return pDestBuf + len;
		case MTD_OOB_AUTO:
		{
			struct nand_oobfree *PFree = PChip->ecc.layout->oobfree;
			u32	boffs = 0;
			u32	roffs = POps->ooboffs;
			size_t bytes = 0;
			
			for(; PFree->length && len; PFree++, len -= bytes)
			{
				if( unlikely(roffs) ){
					if (roffs >= PFree->length) {
						roffs -= PFree->length;
						continue;
					}
					boffs = PFree->offset + roffs;
					bytes = min_t(size_t, len,
						      (PFree->length - roffs));
					roffs = 0;
				}else{
					bytes = min_t(size_t, len, PFree->length);
					boffs = PFree->offset;
				}
				memcpy(pDestBuf, PChip->oob_poi + boffs, bytes);
				pDestBuf += bytes;
			}
			return pDestBuf;
		}
		default:
			BUG();
	}
	return NULL;
}


static u8 *nand_gpbc_fillOob( struct nand_gpbc_info *PGpbc, u8 *pOob, struct mtd_oob_ops *POps )
{
	struct mtd_info		*PMtd = PGpbc->mtd;
	struct nand_chip	*PChip = &PGpbc->chip;
	size_t len = POps->ooblen;
	
	memset( PChip->oob_poi, 0xff, PMtd->oobsize );
	
	switch( POps->mode )
	{
		case MTD_OOB_PLACE:
		case MTD_OOB_RAW:
			#ifdef MTD_NAND_GPBC_TEST
				if( POps->ooboffs + len > PMtd->oobsize ){
					len = PMtd->oobsize - POps->ooboffs;
				}
			#endif
			memcpy( PChip->oob_poi + POps->ooboffs, pOob, len );
			return pOob + len;
		case MTD_OOB_AUTO:
		{
			struct nand_oobfree *PFree = PChip->ecc.layout->oobfree;
			u32 boffs = 0, woffs = POps->ooboffs;
			size_t bytes = 0;
			
			#ifdef MTD_NAND_GPBC_TEST
				if( POps->ooboffs + len > PMtd->oobavail ){
					len = PMtd->oobavail - POps->ooboffs;
				}
			#endif
			for( ; PFree->length && len; PFree++, len -= bytes )
			{
				// Write request not from offset 0 ?
				if( unlikely(woffs) ){
					if( woffs >= PFree->length ){
						woffs -= PFree->length;
						continue;
					}
					boffs = PFree->offset + woffs;
					bytes = min_t( size_t, len, (PFree->length - woffs) );
					woffs = 0;
				}else{
					bytes = min_t( size_t, len, PFree->length );
					boffs = PFree->offset;
				}
				memcpy( PChip->oob_poi + boffs, pOob, bytes );
				pOob += bytes;
			}
			return pOob;
		}
		default:
			BUG();
	}
	return NULL;
}


//----------------------------
//------ layer 2 func --------
//----------------------------


static s32 nand_gpbc_checkBank(struct nand_gpbc_info *PGpbc, int bank)
{
	if (bank > NAND_GPBC_FLASH_BANKS - 1) {
		return -1;
	}
	return 0;
}


#if 0 /* This function is not used. When need, enable this function. */
static u32 nand_gpbc_resetBank( struct nand_gpbc_info *PGpbc, int bank )
{
	u32		irqMask;
	u32		result;
	
	nand_gpbc_clearIntStatus( PGpbc, bank );
	nand_gpbc_write32( (NAND_GPBC_NDRST__B0RST<<bank), PGpbc->regBase + NAND_GPBC_NDRST );
	irqMask = NAND_GPBC_NI__RSTCMP | NAND_GPBC_NI__ERR;
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
		result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
	#else
		result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
	#endif
	if( NAND_GPBC_NI__ERR & result ){
		return result;
	}
	
	return 0;
}
#endif


#if !defined(CONFIG_MTD_NAND_MN2WS_BB) && !defined(MTD_NAND_GPBC_ERASEALL)
	static int nand_gpbc_isBadPage( struct nand_gpbc_info *PGpbc, loff_t ofs )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = PMtd->priv;
		struct nand_bbt_descr	*PBp = PChip->badblock_pattern;
		u32	irqMask;
		u32	result;
		int	bank = ofs >> PChip->chip_shift;
		u32	bankSel = bank << NAND_GPBC_BANK_SEL;
		u32 bankPage = (ofs >> PChip->page_shift) & PChip->pagemask;
		u32	cnt;
		
		nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
		#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
		#endif
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_SPARE );	//transfer mode
		nand_gpbc_clearIntStatus( PGpbc, bank );
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, 0x2001 );	//read-ahead
		irqMask = NAND_GPBC_NI__LDCMP | NAND_GPBC_NI__ERR;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		if( NAND_GPBC_NI__ERR & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}
		nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
		for( cnt = 0; cnt < PMtd->oobsize; cnt += 4 )
		{
			*(u32 *)(PChip->oob_poi + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
		}
		
		for( cnt = 0; cnt < PBp->len; cnt++ )
		{
			if( PBp->pattern[cnt] !=  PChip->oob_poi[PBp->offs + cnt] ){
				return 1;
			}
		}
		
		return 0;
	}
#endif


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static u32 nand_gpbc_markBadPage( struct nand_gpbc_info *PGpbc, loff_t ofs )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	cnt;
		u32	irqMask;
		u32	result;
		int	bank = ofs >> PChip->chip_shift;
		u32	bankSel = bank << NAND_GPBC_BANK_SEL;
		u32 bankPage = (ofs >> PChip->page_shift) & PChip->pagemask;
		
		memset( PChip->oob_poi, 0xff, PMtd->oobsize );
		memset( PChip->oob_poi + PChip->badblock_pattern->offs, 0, PChip->badblock_pattern->len );
		nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
		nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
		#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
		#endif
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_SPARE );	//transfer mode
		nand_gpbc_clearIntStatus( PGpbc, bank );
		nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
		for( cnt = 0; cnt < PMtd->oobsize; cnt += 4 )
		{
			 nand_gpbc_write32( *(u32 *)(PChip->oob_poi + cnt), PGpbc->memBase + 0x10 );
		}
		irqMask = NAND_GPBC_NI__PROCMP | NAND_GPBC_NI__ERR;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
		if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}else if( NAND_GPBC_NI__PROFAIL & result ){
			return result;
		}
		
		return 0;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		// Guarantee that "pPageBuf" is not used by others.
		static int nand_gpbc_isErasedPage( struct nand_gpbc_info *PGpbc, int page )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	irqMask;
			u32	result;
			u32	cnt;
			int	bank = page >> (PChip->chip_shift - PChip->page_shift);
			u32	bankSel = bank << NAND_GPBC_BANK_SEL;
			u32 bankPage = page & PChip->pagemask;
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			u32	result2;
			struct nand_gpbc_cmd_desc_t *cmd_desc = PGpbc->cmd_desc;
			u32 cmd_desc_phys = PGpbc->cmd_desc_phys;
			u32 flash_page;
			u16 cmd_flags;
			u64 mem_ptr;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			flash_page = (NAND_GPBC_MAP10 | bankSel | bankPage);
			cmd_flags = (NAND_GPBC_CMD_FLAGS_TRANS_MAIN_SPARE |
				     NAND_GPBC_CMD_FLAGS_INT |
				     NAND_GPBC_CMD_FLAGS_BURST_LEN);
			mem_ptr = (u64)PGpbc->pageBufPhys;
			nand_gpbc_set_cmd_desc_read_page(cmd_desc, flash_page,
							 cmd_flags, mem_ptr);
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_clearIntStatus( PGpbc, bank );
			
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_INV );	//L1/L2 cache inv
			
			nand_gpbc_write32( NAND_GPBC_NDME__FLAG, PGpbc->regBase + NAND_GPBC_NDME );	//DMA ENABLE
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			nand_gpbc_write32(NAND_GPBC_MAP10, PGpbc->memBase);
			nand_gpbc_write32(NAND_GPBC_MAP10_BEAT0_CMD_DMA, PGpbc->memBase + 0x10);	//BEAT0
			nand_gpbc_write32(cmd_desc_phys, PGpbc->memBase + 0x10);	//BEAT1
			nand_gpbc_write32(0, PGpbc->memBase + 0x10);	//BEAT2
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_write32( NAND_GPBC_MAP10 | bankSel | bankPage, PGpbc->memBase );	//DMA ADDR
			nand_gpbc_write32( NAND_GPBC_MAP10_BEAT0_READ, PGpbc->memBase + 0x10 );	//BEAT0
			nand_gpbc_write32( PGpbc->pageBufPhys, PGpbc->memBase + 0x10 );	//BEAT1
			nand_gpbc_write32( 0, PGpbc->memBase + 0x10 );	//BEAT2
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			irqMask = NAND_GPBC_NI__DMA_CMD_COMP | NAND_GPBC_NI__TOUT;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			result2 = nand_gpbc_poll_cmd_dma(PGpbc);
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NDME );	//DMA DISABLE
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_INV );	//care of data prefetch
			if( NAND_GPBC_NI__ERR & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			if (result2 & NAND_GPBC_CDESC_STAT_ERR) {
				printk("\n## ERROR %s %s %d: result2 = 0x%08X ##\n",
				       __FILE__, __func__, __LINE__, result2);
				BUG();
			}
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			for( cnt = 0; cnt < PMtd->writesize; cnt += 4 )
			{
				if( 0xffffffff != *(u32 *)(PGpbc->pPageBuf + cnt) ){
					return 0;
				}
			}
			for( cnt = PGpbc->oobSkipSize; cnt < PGpbc->oobSkipSize + PGpbc->sectSize3 + PGpbc->chip.ecc.bytes; cnt++ )
			{
				if( 0xff != *(PGpbc->pPageBuf + PMtd->writesize + cnt) ){
					return 0;
				}
			}

			bbt_set_erased_flag(PChip);
			return 1;
		}
	#else
		// Guarantee that "pPageBuf" is not used by others.
		static int nand_gpbc_isErasedPage( struct nand_gpbc_info *PGpbc, int page )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	irqMask;
			u32	result;
			u32	cnt;
			int	bank = page >> (PChip->chip_shift - PChip->page_shift);
			u32	bankSel = bank << NAND_GPBC_BANK_SEL;
			u32 bankPage = page & PChip->pagemask;
			
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
			nand_gpbc_clearIntStatus( PGpbc, bank );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, 0x2001 );	//read-ahead
			irqMask = NAND_GPBC_NI__LDCMP | NAND_GPBC_NI__ERR;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
			if( NAND_GPBC_NI__ERR & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}
			nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
			for( cnt = 0; cnt < PMtd->writesize + PMtd->oobsize; cnt += 4 )
			{
				*(u32 *)(PGpbc->pPageBuf + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			}
			for( cnt = 0; cnt < PMtd->writesize; cnt += 4 )
			{
				if( 0xffffffff != *(u32 *)(PGpbc->pPageBuf + cnt) ){
					return 0;
				}
			}
			for( cnt = PGpbc->oobSkipSize; cnt < PGpbc->oobSkipSize + PGpbc->sectSize3 + PGpbc->chip.ecc.bytes; cnt++ )
			{
				if( 0xff != *(PGpbc->pPageBuf + PMtd->writesize + cnt) ){
					return 0;
				}
			}

			bbt_set_erased_flag(PChip);
			return 1;
		}
	#endif
#else
	// Guarantee that "pPageBuf" is not used by others.
	static int nand_gpbc_isErasedPage( struct nand_gpbc_info *PGpbc, int page )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	irqMask;
		u32	result;
		u32	cnt;
		int	bank = page >> (PChip->chip_shift - PChip->page_shift);
		u32	bankSel = bank << NAND_GPBC_BANK_SEL;
		
		nand_gpbc_clearIntStatus( PGpbc, bank );
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_CMD, NAND_CMD_READ0 );
		if( 2 == NAND_GPBC_UNIT_CA_BYTES && 3 == NAND_GPBC_UNIT_PA_BYTES ){
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, 0 );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, 0 );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, page & 0xff );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, (page >> 8) & 0xff );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, (page >> 16) & 0xff );
		}else{
			BUG();
		}
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_CMD, NAND_CMD_READSTART );
		irqMask = NAND_GPBC_NI__INTACT | NAND_GPBC_NI__ERR;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		if( NAND_GPBC_NI__ERR & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}
		for( cnt = 0; cnt < PMtd->writesize + PMtd->oobsize; cnt += 4 )
		{
			nand_gpbc_write32( NAND_GPBC_MAP00 | bankSel | cnt, PGpbc->memBase );
			*(u32 *)(PGpbc->pPageBuf + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
		}
		for( cnt = 0; cnt < PMtd->writesize; cnt += 4 )
		{
			if( 0xffffffff != *(u32 *)(PGpbc->pPageBuf + cnt) ){
				return 0;
			}
		}
		for( cnt = PGpbc->oobSkipSize; cnt < PGpbc->oobSkipSize + PGpbc->sectSize3 + PGpbc->chip.ecc.bytes; cnt++ )
		{
			if( 0xff != *(PGpbc->pPageBuf + PMtd->writesize + cnt) ){
				return 0;
			}
		}

		bbt_set_erased_flag(PChip);
		return 1;
	}
#endif

static void nand_gpbc_restoreErasedData( struct nand_gpbc_info *PGpbc, u8 *pBuf )
{
	struct nand_chip	*PChip = &PGpbc->chip;
	u32	sect;
	
	//It is not certain whether this process is necessary in non-8bit Ecc, 
	for( sect = 0; sect < PChip->ecc.steps; sect++ )
	{
		*(u32 *)(pBuf + PChip->ecc.size * sect + 0x2c0) = 0xffffffff;
	}
}


static u32 nand_gpbc_eraseBlock( struct nand_gpbc_info *PGpbc, int page )
{
	struct nand_chip	*PChip = &PGpbc->chip;
	u32	irqMask;
	u32	result;
	int bank;
	u32 bankSel;
	u32 bankPage;
	
	#ifdef CONFIG_MTD_NAND_MN2WS_BB
		page = bbt_translateBb( PGpbc->mtd, page );
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
	bank = page >> (PChip->chip_shift - PChip->page_shift);
	bankSel = bank << NAND_GPBC_BANK_SEL;
	bankPage = page & PChip->pagemask;
	
	nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
	nand_gpbc_clearIntStatus( PGpbc, bank );
	nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_ERASE );
	irqMask = NAND_GPBC_NI__ERACMP | NAND_GPBC_NI__ERR;
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
		result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
	#else
		result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
	#endif
	nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
	if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__ERAFAIL & result ){
		printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
		BUG();
	}else if( NAND_GPBC_NI__ERAFAIL & result ){
		return result;
	}
	
	return 0;
}


#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
	static u32 nand_gpbc_readPageDataEcc( struct nand_gpbc_info *PGpbc, u8 *pBuf, int page )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	irqMask;
		u32	result, result2 = 0;
		int	status;
		int	bank;
		u32	bankSel;
		u32	bankPage;
		u32	correctedNum;
		u32	physAddr;
		u32	alignFlag;
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
		struct nand_gpbc_cmd_desc_t *cmd_desc = PGpbc->cmd_desc;
		u32 cmd_desc_phys = PGpbc->cmd_desc_phys;
		u32 flash_page;
		u16 cmd_flags;
		u64 mem_ptr;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		
		#ifdef CONFIG_MTD_NAND_MN2WS_BB
			page = bbt_translateBb( PMtd, page );
		#endif	//CONFIG_MTD_NAND_MN2WS_BB
		bank = page >> (PChip->chip_shift - PChip->page_shift);
		bankSel = bank << NAND_GPBC_BANK_SEL;
		bankPage = page & PChip->pagemask;
		
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NTS );	//spare
		#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
			nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
		#endif
		
		if( (u32)pBuf % PGpbc->dmaAlignSize ){
			physAddr = PGpbc->pageBufPhys;
			alignFlag = 0;
		}else{
			physAddr = virt_to_phys( pBuf );
			alignFlag = 1;
		}
		
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
		flash_page = (NAND_GPBC_MAP10 | bankSel | bankPage);
		cmd_flags = (NAND_GPBC_CMD_FLAGS_TRANS_DEFAULT |
			     NAND_GPBC_CMD_FLAGS_INT |
			     NAND_GPBC_CMD_FLAGS_BURST_LEN);
		mem_ptr = (u64)physAddr;
		nand_gpbc_set_cmd_desc_read_page(cmd_desc, flash_page,
						 cmd_flags, mem_ptr);
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_DEFAULT );	//transfer mode
#endif /* !CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		nand_gpbc_clearIntStatus( PGpbc, bank );
		
		if( alignFlag ){
			vaddr_purge_cache( (u32)pBuf, PMtd->writesize, PURGE_CACHE_D_INV );	//L1/L2 cache inv
		}else{
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize, PURGE_CACHE_D_INV );	//L1/L2 cache inv
		}
		
		nand_gpbc_write32( NAND_GPBC_NDME__FLAG, PGpbc->regBase + NAND_GPBC_NDME );	//DMA ENABLE
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
		nand_gpbc_write32(NAND_GPBC_MAP10, PGpbc->memBase);
		nand_gpbc_write32(NAND_GPBC_MAP10_BEAT0_CMD_DMA, PGpbc->memBase + 0x10);	//BEAT0
		nand_gpbc_write32(cmd_desc_phys, PGpbc->memBase + 0x10);	//BEAT1
		nand_gpbc_write32(0, PGpbc->memBase + 0x10);	//BEAT2
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		nand_gpbc_write32( NAND_GPBC_MAP10 | bankSel | bankPage, PGpbc->memBase );	//DMA ADDR
		nand_gpbc_write32( NAND_GPBC_MAP10_BEAT0_READ, PGpbc->memBase + 0x10 );	//BEAT0
		nand_gpbc_write32( physAddr, PGpbc->memBase + 0x10 );	//BEAT1
		nand_gpbc_write32( 0, PGpbc->memBase + 0x10 );	//BEAT2
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		
		irqMask = NAND_GPBC_NI__DMA_CMD_COMP | NAND_GPBC_NI__TOUT;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
		result2 = nand_gpbc_poll_cmd_dma(PGpbc);
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NDME );	//DMA DISABLE
		if( alignFlag ){
			vaddr_purge_cache( (u32)pBuf, PMtd->writesize, PURGE_CACHE_D_INV );	//care of data prefetch
		}else{
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize, PURGE_CACHE_D_INV );	//care of data prefetch
		}
		if( NAND_GPBC_NI__ERR & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
		if (result2 & NAND_GPBC_CDESC_STAT_ERR) {
			printk("\n## ERROR %s %s %d: result2 = 0x%08X ##\n",
			       __FILE__, __func__, __LINE__, result2);
			BUG();
		}
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		
		if( !alignFlag ){
			memcpy( pBuf, PGpbc->pPageBuf, PMtd->writesize );
		}
		
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
		result2 &= NAND_GPBC_CDESC_STAT_FAIL;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
		result2 |= (result & NAND_GPBC_NI__ECCUNCOR);
		if (result2) {
			status = nand_gpbc_isErasedPage( PGpbc, page );
			if( status ){
				nand_gpbc_restoreErasedData( PGpbc, pBuf );
			}else{
				PMtd->ecc_stats.failed++;
				printk( "\n\n## Warning ## %s %s %d: ECCUNCOR ##\n\n", __FILE__, __func__, __LINE__ );
				return result;
			}
		}else{
			correctedNum = nand_gpbc_getCorrectedNum( PGpbc, bank );
			if( NAND_GPBC_CORRECTION_THRESHOLD <= correctedNum ){
				PMtd->ecc_stats.corrected++;
			}
		}
		
		return 0;
	}
#else	//CONFIG_MTD_NAND_MN2WS_GPBC_DMA
	static u32 nand_gpbc_readPageDataEcc( struct nand_gpbc_info *PGpbc, u8 *pBuf, int page )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	irqMask;
		u32	result;
		u32	cnt;
		int	status;
		int	bank;
		u32	bankSel;
		u32	bankPage;
		u32	correctedNum;
		
		#ifdef CONFIG_MTD_NAND_MN2WS_BB
			page = bbt_translateBb( PMtd, page );
		#endif	//CONFIG_MTD_NAND_MN2WS_BB
		bank = page >> (PChip->chip_shift - PChip->page_shift);
		bankSel = bank << NAND_GPBC_BANK_SEL;
		bankPage = page & PChip->pagemask;
		
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NTS );	//spare
		#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
			nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
		#endif
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_DEFAULT );	//transfer mode
		nand_gpbc_clearIntStatus( PGpbc, bank );
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, 0x2001 );	//read-ahead
		irqMask = NAND_GPBC_NI__LDCMP | NAND_GPBC_NI__ERR;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		if( NAND_GPBC_NI__ERR & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}
		nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
		for( cnt = 0; cnt < PMtd->writesize; cnt += 4 )
		{
			*(u32 *)(pBuf + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
		}
		
		result = nand_gpbc_checkIrq( PGpbc, bank );
		if( NAND_GPBC_NI__ECCUNCOR & result ){
			status = nand_gpbc_isErasedPage( PGpbc, page );
			if( status ){
				nand_gpbc_restoreErasedData( PGpbc, pBuf );
			}else{
				PMtd->ecc_stats.failed++;
				printk( "\n\n## Warning ## %s %s %d: ECCUNCOR ##\n\n", __FILE__, __func__, __LINE__ );
				return result;
			}
		}else{
			correctedNum = nand_gpbc_getCorrectedNum( PGpbc, bank );
			if( NAND_GPBC_CORRECTION_THRESHOLD <= correctedNum ){
				PMtd->ecc_stats.corrected++;
			}
		}
		
		return 0;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_GPBC_DMA


#ifdef MTD_NAND_MN2WS_GPBC_ECC_BUG
	static u32 nand_gpbc_readPageAllEcc( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page );
	static u32 nand_gpbc_readPageAllRaw( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page );
	
	static u32 nand_gpbc_readPageAll( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page, int raw )
	{
		u32		result;
		
		if( !raw ){
			result = nand_gpbc_readPageAllEcc( PGpbc, pBuf, page );
		}else{
			result = nand_gpbc_readPageAllRaw( PGpbc, pBuf, page );
		}
		
		return result;
	}
	
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		static u32 nand_gpbc_readPageAllEcc( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	irqMask;
			u32	result, result2 = 0;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			u32	correctedNum;
			int	status;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			struct nand_gpbc_cmd_desc_t *cmd_desc = PGpbc->cmd_desc;
			u32 cmd_desc_phys = PGpbc->cmd_desc_phys;
			u32 flash_page;
			u16 cmd_flags;
			u64 mem_ptr;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			flash_page = (NAND_GPBC_MAP10 | bankSel | bankPage);
			cmd_flags = (NAND_GPBC_CMD_FLAGS_TRANS_MAIN_SPARE |
				     NAND_GPBC_CMD_FLAGS_INT |
				     NAND_GPBC_CMD_FLAGS_BURST_LEN);
			mem_ptr = (u64)PGpbc->pageBufPhys;
			nand_gpbc_set_cmd_desc_read_page(cmd_desc, flash_page,
							 cmd_flags, mem_ptr);
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_clearIntStatus( PGpbc, bank );
			
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_INV );	//L1/L2 cache inv
			nand_gpbc_write32( NAND_GPBC_NDME__FLAG, PGpbc->regBase + NAND_GPBC_NDME );	//DMA ENABLE
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			nand_gpbc_write32(NAND_GPBC_MAP10, PGpbc->memBase);
			nand_gpbc_write32(NAND_GPBC_MAP10_BEAT0_CMD_DMA, PGpbc->memBase + 0x10);	//BEAT0
			nand_gpbc_write32(cmd_desc_phys, PGpbc->memBase + 0x10);	//BEAT1
			nand_gpbc_write32(0, PGpbc->memBase + 0x10);	//BEAT2
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_write32( NAND_GPBC_MAP10 | bankSel | bankPage, PGpbc->memBase );	//DMA ADDR
			nand_gpbc_write32( NAND_GPBC_MAP10_BEAT0_READ, PGpbc->memBase + 0x10 );	//BEAT0
			nand_gpbc_write32( PGpbc->pageBufPhys, PGpbc->memBase + 0x10 );	//BEAT1
			nand_gpbc_write32( 0, PGpbc->memBase + 0x10 );	//BEAT2
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			irqMask = NAND_GPBC_NI__DMA_CMD_COMP | NAND_GPBC_NI__TOUT;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			result2 = nand_gpbc_poll_cmd_dma(PGpbc);
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NDME );	//DMA DISABLE
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_INV );	//care of data prefetch
			if( NAND_GPBC_NI__ERR & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			if (result2 & NAND_GPBC_CDESC_STAT_ERR) {
				printk("\n## ERROR %s %s %d: result2 = 0x%08X ##\n",
				       __FILE__, __func__, __LINE__, result2);
				BUG();
			}
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->chip.oob_poi, PGpbc->pPageBuf + pagePos, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			result2 &= NAND_GPBC_CDESC_STAT_FAIL;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			result2 |= (result & NAND_GPBC_NI__ECCUNCOR);
			if (result2) {
				status = nand_gpbc_isErasedPage( PGpbc, page );
				if( status ){
					nand_gpbc_restoreErasedData( PGpbc, pBuf );
				}else{
					PMtd->ecc_stats.failed++;
					printk( "\n\n## Warning ## %s %s %d: ECCUNCOR ##\n\n", __FILE__, __func__, __LINE__ );
					return result;
				}
			}else{
				correctedNum = nand_gpbc_getCorrectedNum( PGpbc, bank );
				if( NAND_GPBC_CORRECTION_THRESHOLD <= correctedNum ){
					PMtd->ecc_stats.corrected++;
				}
			}
			
			return 0;
		}
	
	#else
	
		static u32 nand_gpbc_readPageAllEcc( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	irqMask;
			u32	result;
			u32	cnt;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			u32	correctedNum;
			int	status;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
			nand_gpbc_clearIntStatus( PGpbc, bank );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, 0x2001 );	//read-ahead
			irqMask = NAND_GPBC_NI__LDCMP | NAND_GPBC_NI__ERR;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
			if( NAND_GPBC_NI__ERR & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}
			nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
			for( cnt = 0; cnt < PMtd->writesize + PMtd->oobsize; cnt += 4 )
			{
				*(u32 *)(PGpbc->pPageBuf + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			}
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->chip.oob_poi, PGpbc->pPageBuf + pagePos, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
			result = nand_gpbc_checkIrq( PGpbc, bank );
			if( NAND_GPBC_NI__ECCUNCOR & result ){
				status = nand_gpbc_isErasedPage( PGpbc, page );
				if( status ){
					nand_gpbc_restoreErasedData( PGpbc, pBuf );
				}else{
					PMtd->ecc_stats.failed++;
					printk( "\n\n## Warning ## %s %s %d: ECCUNCOR ##\n\n", __FILE__, __func__, __LINE__ );
					return result;
				}
			}else{
				correctedNum = nand_gpbc_getCorrectedNum( PGpbc, bank );
				if( NAND_GPBC_CORRECTION_THRESHOLD <= correctedNum ){
					PMtd->ecc_stats.corrected++;
				}
			}
			
			return 0;
		}
	
	#endif
	
	static u32 nand_gpbc_readPageAllRaw( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	irqMask;
		u32	result;
		u32	cnt;
		int	bank;
		u32	bankSel;
		u32	dataPos;
		u32	oobPos;
		u32	sect;
		u32	pagePos;
		
		#ifdef CONFIG_MTD_NAND_MN2WS_BB
			page = bbt_translateBb( PMtd, page );
		#endif	//CONFIG_MTD_NAND_MN2WS_BB
		bank = page >> (PChip->chip_shift - PChip->page_shift);
		bankSel = bank << NAND_GPBC_BANK_SEL;
		
		nand_gpbc_clearIntStatus( PGpbc, bank );
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_CMD, NAND_CMD_READ0 );
		if( 2 == NAND_GPBC_UNIT_CA_BYTES && 3 == NAND_GPBC_UNIT_PA_BYTES ){
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, 0 );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, 0 );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, page & 0xff );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, (page >> 8) & 0xff );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, (page >> 16) & 0xff );
		}else{
			BUG();
		}
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_CMD, NAND_CMD_READSTART );
		irqMask = NAND_GPBC_NI__INTACT | NAND_GPBC_NI__ERR;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		if( NAND_GPBC_NI__ERR & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}
		for( cnt = 0; cnt < PMtd->writesize + PMtd->oobsize; cnt += 4 )
		{
			nand_gpbc_write32( NAND_GPBC_MAP00 | bankSel | cnt, PGpbc->memBase );
			*(u32 *)(PGpbc->pPageBuf + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
		}
		
		pagePos = 0;
		dataPos = 0;
		oobPos = PGpbc->oobSkipSize;
		for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
		{
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.size );
			pagePos += PGpbc->chip.ecc.size;
			dataPos += PGpbc->chip.ecc.size;
			memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.bytes );
			pagePos += PGpbc->chip.ecc.bytes;
			oobPos += PGpbc->chip.ecc.bytes;
		}
		memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize2 );
		pagePos += PGpbc->sectSize2;
		dataPos += PGpbc->sectSize2;
		memcpy( PGpbc->chip.oob_poi, PGpbc->pPageBuf + pagePos, PGpbc->oobSkipSize );
		pagePos += PGpbc->oobSkipSize;
		memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize3 );
		pagePos += PGpbc->sectSize3;
		memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
		
		return 0;
	}
#else
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		static u32 nand_gpbc_readPageAll( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page, int raw )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	irqMask;
			u32	result, result2 = 0;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			u32	correctedNum;
			int	status;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			struct nand_gpbc_cmd_desc_t *cmd_desc = PGpbc->cmd_desc;
			u32 cmd_desc_phys = PGpbc->cmd_desc_phys;
			u32 flash_page;
			u16 cmd_flags;
			u64 mem_ptr;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			if( raw ){
				nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}else{
				nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			flash_page = (NAND_GPBC_MAP10 | bankSel | bankPage);
			cmd_flags = (NAND_GPBC_CMD_FLAGS_TRANS_MAIN_SPARE |
				     NAND_GPBC_CMD_FLAGS_INT |
				     NAND_GPBC_CMD_FLAGS_BURST_LEN);
			mem_ptr = (u64)PGpbc->pageBufPhys;
			nand_gpbc_set_cmd_desc_read_page(cmd_desc, flash_page,
							 cmd_flags, mem_ptr);
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_clearIntStatus( PGpbc, bank );
			
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_INV );	//L1/L2 cache inv
			nand_gpbc_write32( NAND_GPBC_NDME__FLAG, PGpbc->regBase + NAND_GPBC_NDME );	//DMA ENABLE
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			nand_gpbc_write32(NAND_GPBC_MAP10, PGpbc->memBase);
			nand_gpbc_write32(NAND_GPBC_MAP10_BEAT0_CMD_DMA, PGpbc->memBase + 0x10);	//BEAT0
			nand_gpbc_write32(cmd_desc_phys, PGpbc->memBase + 0x10);	//BEAT1
			nand_gpbc_write32(0, PGpbc->memBase + 0x10);	//BEAT2
#else /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_write32( NAND_GPBC_MAP10 | bankSel | bankPage, PGpbc->memBase );	//DMA ADDR
			nand_gpbc_write32( NAND_GPBC_MAP10_BEAT0_READ, PGpbc->memBase + 0x10 );	//BEAT0
			nand_gpbc_write32( PGpbc->pageBufPhys, PGpbc->memBase + 0x10 );	//BEAT1
			nand_gpbc_write32( 0, PGpbc->memBase + 0x10 );	//BEAT2
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			irqMask = NAND_GPBC_NI__DMA_CMD_COMP | NAND_GPBC_NI__TOUT;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			result2 = nand_gpbc_poll_cmd_dma(PGpbc);
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NDME );	//DMA DISABLE
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_INV );	//care of data prefetch
			if( NAND_GPBC_NI__ERR & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
			if (result2 & NAND_GPBC_CDESC_STAT_ERR) {
				printk("\n## ERROR %s %s %d: result2 = 0x%08X ##\n",
				       __FILE__, __func__, __LINE__, result2);
				BUG();
			}
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->chip.oob_poi, PGpbc->pPageBuf + pagePos, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
			if( !raw ){
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
				result2 &= NAND_GPBC_CDESC_STAT_FAIL;
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
				result2 |= (result & NAND_GPBC_NI__ECCUNCOR);
				if (result2) {
					status = nand_gpbc_isErasedPage( PGpbc, page );
					if( status ){
						nand_gpbc_restoreErasedData( PGpbc, pBuf );
					}else{
						PMtd->ecc_stats.failed++;
						printk( "\n\n## Warning ## %s %s %d: ECCUNCOR ##\n\n", __FILE__, __func__, __LINE__ );
						return result;
					}
				}else{
					correctedNum = nand_gpbc_getCorrectedNum( PGpbc, bank );
					if( NAND_GPBC_CORRECTION_THRESHOLD <= correctedNum ){
						PMtd->ecc_stats.corrected++;
					}
				}
			}
			
			return 0;
		}
	#else
		static u32 nand_gpbc_readPageAll( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page, int raw )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	irqMask;
			u32	result;
			u32	cnt;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			u32	correctedNum;
			int	status;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			if( raw ){
				nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}else{
				nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
			nand_gpbc_clearIntStatus( PGpbc, bank );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, 0x2001 );	//read-ahead
			irqMask = NAND_GPBC_NI__LDCMP | NAND_GPBC_NI__ERR;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
			if( NAND_GPBC_NI__ERR & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}
			nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
			for( cnt = 0; cnt < PMtd->writesize + PMtd->oobsize; cnt += 4 )
			{
				*(u32 *)(PGpbc->pPageBuf + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			}
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->chip.oob_poi, PGpbc->pPageBuf + pagePos, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( pBuf + dataPos, PGpbc->pPageBuf + pagePos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->chip.oob_poi + oobPos, PGpbc->pPageBuf + pagePos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
			if( !raw ){
				result = nand_gpbc_checkIrq( PGpbc, bank );
				if( NAND_GPBC_NI__ECCUNCOR & result ){
					status = nand_gpbc_isErasedPage( PGpbc, page );
					if( status ){
						nand_gpbc_restoreErasedData( PGpbc, pBuf );
					}else{
						PMtd->ecc_stats.failed++;
						printk( "\n\n## Warning ## %s %s %d: ECCUNCOR ##\n\n", __FILE__, __func__, __LINE__ );
						return result;
					}
				}else{
					correctedNum = nand_gpbc_getCorrectedNum( PGpbc, bank );
					if( NAND_GPBC_CORRECTION_THRESHOLD <= correctedNum ){
						PMtd->ecc_stats.corrected++;
					}
				}
			}
			
			return 0;
		}
	#endif
#endif

#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
	static u32 nand_gpbc_writePageDataEcc( struct nand_gpbc_info *PGpbc, const u8 *pBuf, int page )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	irqMask;
		u32	result;
		int bank;
		u32	bankSel;
		u32	bankPage;
		u32	physAddr;
		u32	alignFlag;
		
		#ifdef CONFIG_MTD_NAND_MN2WS_BB
			page = bbt_translateBb( PMtd, page );
		#endif	//CONFIG_MTD_NAND_MN2WS_BB
		bank = page >> (PChip->chip_shift - PChip->page_shift);
		bankSel = bank << NAND_GPBC_BANK_SEL;
		bankPage = page & PChip->pagemask;
		
		nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NTS );	//spare
		#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
			nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
		#endif
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_DEFAULT );	//transfer mode
		nand_gpbc_clearIntStatus( PGpbc, bank );
		
		if( (u32)pBuf % PGpbc->dmaAlignSize ){
			physAddr = PGpbc->pageBufPhys;
			alignFlag = 0;
		}else{
			physAddr = virt_to_phys( (void *)pBuf );
			alignFlag = 1;
		}
		
		if( alignFlag ){
			vaddr_purge_cache( (u32)pBuf, PMtd->writesize, PURGE_CACHE_D_PURGE_INV );	//L1/L2 cache clean
		}else{
			memcpy( PGpbc->pPageBuf, pBuf, PMtd->writesize );
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize, PURGE_CACHE_D_PURGE_INV );	//L1/L2 cache clean
		}
		
		nand_gpbc_write32( NAND_GPBC_NDME__FLAG, PGpbc->regBase + NAND_GPBC_NDME );	//DMA ENABLE
		nand_gpbc_write32( NAND_GPBC_MAP10 | bankSel | bankPage, PGpbc->memBase );	//DMA ADDR
		nand_gpbc_write32( NAND_GPBC_MAP10_BEAT0_WRITE, PGpbc->memBase + 0x10 );	//BEAT0
		nand_gpbc_write32( physAddr, PGpbc->memBase + 0x10 );	//BEAT1
		nand_gpbc_write32( 0, PGpbc->memBase + 0x10 );	//BEAT2
		
		irqMask = NAND_GPBC_NI__DMA_CMD_COMP | NAND_GPBC_NI__TOUT;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NDME );	//DMA DISABLE
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
		if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}else if( NAND_GPBC_NI__PROFAIL & result ){
			return result;
		}
		
		return 0;
	}
#else
	static u32 nand_gpbc_writePageDataEcc( struct nand_gpbc_info *PGpbc, const u8 *pBuf, int page )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	irqMask;
		u32	result;
		u32	cnt;
		int bank;
		u32	bankSel;
		u32	bankPage;
		
		#ifdef CONFIG_MTD_NAND_MN2WS_BB
			page = bbt_translateBb( PMtd, page );
		#endif	//CONFIG_MTD_NAND_MN2WS_BB
		bank = page >> (PChip->chip_shift - PChip->page_shift);
		bankSel = bank << NAND_GPBC_BANK_SEL;
		bankPage = page & PChip->pagemask;
		
		nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NTS );	//spare
		#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
			nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
		#endif
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_DEFAULT );	//transfer mode
		nand_gpbc_clearIntStatus( PGpbc, bank );
		nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
		for( cnt = 0; cnt < PMtd->writesize; cnt += 4 )
		{
			nand_gpbc_write32( *(u32 *)(pBuf + cnt), PGpbc->memBase + 0x10 );
		}
		irqMask = NAND_GPBC_NI__PROCMP | NAND_GPBC_NI__ERR;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
		if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}else if( NAND_GPBC_NI__PROFAIL & result ){
			return result;
		}
		
		return 0;
	}
#endif


#ifdef MTD_NAND_MN2WS_GPBC_ECC_BUG
	static u32 nand_gpbc_writePageAllEcc( struct nand_gpbc_info *PGpbc, const uint8_t *pBuf, int page );
	static u32 nand_gpbc_writePageAllRaw( struct nand_gpbc_info *PGpbc, const uint8_t *pBuf, int page );
	
	static u32 nand_gpbc_writePageAll( struct nand_gpbc_info *PGpbc, uint8_t *pBuf, int page, int raw )
	{
		u32		result;
		
		if( !raw ){
			result = nand_gpbc_writePageAllEcc( PGpbc, pBuf, page );
		}else{
			result = nand_gpbc_writePageAllRaw( PGpbc, pBuf, page );
		}
		
		return result;
	}
	
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		static u32 nand_gpbc_writePageAllEcc( struct nand_gpbc_info *PGpbc, const uint8_t *pBuf, int page )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
			u32	irqMask;
			u32 result;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
			nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
			nand_gpbc_clearIntStatus( PGpbc, bank );
			
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_PURGE_INV );	//L1/L2 cache clean
			nand_gpbc_write32( NAND_GPBC_NDME__FLAG, PGpbc->regBase + NAND_GPBC_NDME );	//enable DMA
			nand_gpbc_write32( NAND_GPBC_MAP10 | bankSel | bankPage, PGpbc->memBase );	//DMA ADDR
			nand_gpbc_write32( NAND_GPBC_MAP10_BEAT0_WRITE, PGpbc->memBase + 0x10 );	//BEAT0
			nand_gpbc_write32( PGpbc->pageBufPhys, PGpbc->memBase + 0x10 );	//BEAT1
			nand_gpbc_write32( 0, PGpbc->memBase + 0x10 );	//BEAT2
			
			irqMask = NAND_GPBC_NI__DMA_CMD_COMP | NAND_GPBC_NI__TOUT;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NDME );	//disable DMA
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
			if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}else if( NAND_GPBC_NI__PROFAIL & result ){
				return result;
			}
			
			return 0;
		}
	#else
		static u32 nand_gpbc_writePageAllEcc( struct nand_gpbc_info *PGpbc, const uint8_t *pBuf, int page )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
			u32	irqMask;
			u32 result;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
			nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
			nand_gpbc_clearIntStatus( PGpbc, bank );
			nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
			for( pagePos = 0; pagePos < PMtd->writesize + PMtd->oobsize; pagePos += 4 )
			{
				nand_gpbc_write32( *(u32 *)(PGpbc->pPageBuf + pagePos), PGpbc->memBase + 0x10 );
			}
			irqMask = NAND_GPBC_NI__PROCMP | NAND_GPBC_NI__ERR;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
			if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}else if( NAND_GPBC_NI__PROFAIL & result ){
				return result;
			}
			
			return 0;
		}
	#endif
	
	static u32 nand_gpbc_writePageAllRaw( struct nand_gpbc_info *PGpbc, const uint8_t *pBuf, int page )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		u32	dataPos;
		u32	oobPos;
		u32	sect;
		u32	pagePos;
		u32	irqMask;
		u32 result;
		int	bank;
		u32	bankSel;
		u32	bankPage;
		u32	cnt;
		
		#ifdef CONFIG_MTD_NAND_MN2WS_BB
			page = bbt_translateBb( PMtd, page );
		#endif	//CONFIG_MTD_NAND_MN2WS_BB
		bank = page >> (PChip->chip_shift - PChip->page_shift);
		bankSel = bank << NAND_GPBC_BANK_SEL;
		bankPage = page & PChip->pagemask;
		
		pagePos = 0;
		dataPos = 0;
		oobPos = PGpbc->oobSkipSize;
		for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
		{
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->chip.ecc.size );
			pagePos += PGpbc->chip.ecc.size;
			dataPos += PGpbc->chip.ecc.size;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PGpbc->chip.ecc.bytes );
			pagePos += PGpbc->chip.ecc.bytes;
			oobPos += PGpbc->chip.ecc.bytes;
		}
		memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize2 );
		pagePos += PGpbc->sectSize2;
		dataPos += PGpbc->sectSize2;
		memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi, PGpbc->oobSkipSize );
		pagePos += PGpbc->oobSkipSize;
		memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize3 );
		pagePos += PGpbc->sectSize3;
		memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
		
		nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
		nand_gpbc_clearIntStatus( PGpbc, bank );
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_CMD, NAND_CMD_SEQIN );
		if( 2 == NAND_GPBC_UNIT_CA_BYTES && 3 == NAND_GPBC_UNIT_PA_BYTES ){
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, 0 );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, 0 );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, page & 0xff );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, (page >> 8) & 0xff );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, (page >> 16) & 0xff );
		}else{
			BUG();
		}
		for( cnt = 0; cnt < PMtd->writesize + PMtd->oobsize; cnt += 4 )
		{
			nand_gpbc_write32( NAND_GPBC_MAP00 | bankSel | cnt, PGpbc->memBase );
			nand_gpbc_write32( *(u32 *)(PGpbc->pPageBuf + cnt), PGpbc->memBase + 0x10 );
		}
		nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_CMD, NAND_CMD_PAGEPROG );
		irqMask = NAND_GPBC_NI__INTACT | NAND_GPBC_NI__ERR;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
		#endif
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
		if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
			printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
			BUG();
		}else if( NAND_GPBC_NI__PROFAIL & result ){
			return result;
		}
		
		return 0;
	}
#else
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		static u32 nand_gpbc_writePageAll( struct nand_gpbc_info *PGpbc, const uint8_t *pBuf, int page, int raw )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
			u32	irqMask;
			u32 result;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
			nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			if( raw ){
				nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}else{
				nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
			nand_gpbc_clearIntStatus( PGpbc, bank );
			
			vaddr_purge_cache( (u32)PGpbc->pPageBuf, PMtd->writesize + PMtd->oobsize, PURGE_CACHE_D_PURGE_INV );	//L1/L2 cache clean
			nand_gpbc_write32( NAND_GPBC_NDME__FLAG, PGpbc->regBase + NAND_GPBC_NDME );	//enable DMA
			nand_gpbc_write32( NAND_GPBC_MAP10 | bankSel | bankPage, PGpbc->memBase );	//DMA ADDR
			nand_gpbc_write32( NAND_GPBC_MAP10_BEAT0_WRITE, PGpbc->memBase + 0x10 );	//BEAT0
			nand_gpbc_write32( PGpbc->pageBufPhys, PGpbc->memBase + 0x10 );	//BEAT1
			nand_gpbc_write32( 0, PGpbc->memBase + 0x10 );	//BEAT2
			
			irqMask = NAND_GPBC_NI__DMA_CMD_COMP | NAND_GPBC_NI__TOUT;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NDME );	//disable DMA
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
			if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}else if( NAND_GPBC_NI__PROFAIL & result ){
				return result;
			}
			
			return 0;
		}
	#else
		static u32 nand_gpbc_writePageAll( struct nand_gpbc_info *PGpbc, const uint8_t *pBuf, int page, int raw )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			u32	dataPos;
			u32	oobPos;
			u32	sect;
			u32	pagePos;
			u32	irqMask;
			u32 result;
			int	bank;
			u32	bankSel;
			u32	bankPage;
			
			#ifdef CONFIG_MTD_NAND_MN2WS_BB
				page = bbt_translateBb( PMtd, page );
			#endif	//CONFIG_MTD_NAND_MN2WS_BB
			bank = page >> (PChip->chip_shift - PChip->page_shift);
			bankSel = bank << NAND_GPBC_BANK_SEL;
			bankPage = page & PChip->pagemask;
			
			pagePos = 0;
			dataPos = 0;
			oobPos = PGpbc->oobSkipSize;
			for( sect = 0; sect < PGpbc->chip.ecc.steps - 1; sect++ )
			{
				memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->chip.ecc.size );
				pagePos += PGpbc->chip.ecc.size;
				dataPos += PGpbc->chip.ecc.size;
				memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PGpbc->chip.ecc.bytes );
				pagePos += PGpbc->chip.ecc.bytes;
				oobPos += PGpbc->chip.ecc.bytes;
			}
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize2 );
			pagePos += PGpbc->sectSize2;
			dataPos += PGpbc->sectSize2;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi, PGpbc->oobSkipSize );
			pagePos += PGpbc->oobSkipSize;
			memcpy( PGpbc->pPageBuf + pagePos, pBuf + dataPos, PGpbc->sectSize3 );
			pagePos += PGpbc->sectSize3;
			memcpy( PGpbc->pPageBuf + pagePos, PGpbc->chip.oob_poi + oobPos, PMtd->oobsize - PGpbc->oobSkipSize - PGpbc->sectSize3 );
			
			nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
			nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
			if( raw ){
				nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}else{
				nand_gpbc_write32( NAND_GPBC_NECCE__ECCEN, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
			}
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_MAIN_SPARE );	//transfer mode
			nand_gpbc_clearIntStatus( PGpbc, bank );
			nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
			for( pagePos = 0; pagePos < PMtd->writesize + PMtd->oobsize; pagePos += 4 )
			{
				nand_gpbc_write32( *(u32 *)(PGpbc->pPageBuf + pagePos), PGpbc->memBase + 0x10 );
			}
			irqMask = NAND_GPBC_NI__PROCMP | NAND_GPBC_NI__ERR;
			#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
				result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
			#else
				result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
			#endif
			nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
			if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
				printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
				BUG();
			}else if( NAND_GPBC_NI__PROFAIL & result ){
				return result;
			}
			
			return 0;
		}
	#endif
#endif


static u32 nand_gpbc_readPageOob( struct nand_gpbc_info *PGpbc, int page )
{
	struct mtd_info		*PMtd = PGpbc->mtd;
	struct nand_chip	*PChip = &PGpbc->chip;
	u32	irqMask;
	u32	result;
	u32	cnt;
	u8	*pBuf;
	int	bank;
	u32	bankSel;
	u32	bankPage;
	
	#ifdef CONFIG_MTD_NAND_MN2WS_BB
		page = bbt_translateBb( PMtd, page );
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
	bank = page >> (PChip->chip_shift - PChip->page_shift);
	bankSel = bank << NAND_GPBC_BANK_SEL;
	bankPage = page & PChip->pagemask;
	
	nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
	#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
	#endif
	nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_SPARE );	//transfer mode
	nand_gpbc_clearIntStatus( PGpbc, bank );
	nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, 0x2001 );	//read-ahead
	irqMask = NAND_GPBC_NI__LDCMP | NAND_GPBC_NI__ERR;
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
		result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
	#else
		result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
	#endif
	if( NAND_GPBC_NI__ERR & result ){
		printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
		BUG();
	}
	nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
	pBuf = PChip->oob_poi;
	for( cnt = 0; cnt < PMtd->oobsize; cnt += 4 )
	{
		*(u32 *)(pBuf + cnt) = nand_gpbc_read32( PGpbc->memBase + 0x10 );
	}
	
	return 0;
}


static u32 nand_gpbc_writePageOob( struct nand_gpbc_info *PGpbc, int page )
{
	struct mtd_info		*PMtd = PGpbc->mtd;
	struct nand_chip	*PChip = &PGpbc->chip;
	u32	irqMask;
	u32	result;
	u32	cnt;
	int	bank;
	u32	bankSel;
	u32	bankPage;
	
	#ifdef CONFIG_MTD_NAND_MN2WS_BB
		page = bbt_translateBb( PMtd, page );
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
	bank = page >> (PChip->chip_shift - PChip->page_shift);
	bankSel = bank << NAND_GPBC_BANK_SEL;
	bankPage = page & PChip->pagemask;
	
	nand_gpbc_write32( NAND_GPBC_NWPD__WPD, PGpbc->regBase + NAND_GPBC_NWPD );	//disable WP
	nand_gpbc_write32( NAND_GPBC_NTS__TS, PGpbc->regBase + NAND_GPBC_NTS );	//spare
	#ifndef MTD_NAND_MN2WS_GPBC_ECC_BUG
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NECCE );	//ecc
	#endif
	nand_gpbc_index32( PGpbc, NAND_GPBC_MAP10 | bankSel | bankPage, NAND_GPBC_MAP10_SPARE );	//transfer mode
	nand_gpbc_clearIntStatus( PGpbc, bank );
	nand_gpbc_write32( NAND_GPBC_MAP01 | bankSel | bankPage, PGpbc->memBase );
	for( cnt = 0; cnt < PMtd->oobsize; cnt += 4 )
	{
		 nand_gpbc_write32( *(u32 *)(PGpbc->chip.oob_poi + cnt), PGpbc->memBase + 0x10 );
	}
	irqMask = NAND_GPBC_NI__PROCMP | NAND_GPBC_NI__ERR;
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
		result = nand_gpbc_waitIrq( PGpbc, bank, irqMask );
	#else
		result = nand_gpbc_pollIrq( PGpbc, bank, irqMask );
	#endif
	nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );	//enable WP
	if( NAND_GPBC_NI__ERR & ~NAND_GPBC_NI__PROFAIL & result ){
		printk( "\n## ERROR %s %s %d: result = 0x%08X ##\n", __FILE__, __func__, __LINE__, result );
		BUG();
	}else if( NAND_GPBC_NI__PROFAIL & result ){
		return result;
	}
	
	return 0;
}


//----------------------------
//------ layer 3 func --------
//----------------------------

#ifndef MTD_NAND_GPBC_ERASEALL
	#ifndef CONFIG_MTD_NAND_MN2WS_BB
		static int nand_gpbc_isBad( struct nand_gpbc_info *PGpbc, loff_t ofs, int getchip, int allowbbt )
		{
			struct mtd_info		*PMtd = PGpbc->mtd;
			struct nand_chip	*PChip = &PGpbc->chip;
			int		status;
			
			if( !PChip->bbt ){
				if( getchip ){
					nand_gpbc_getDevice( PGpbc, FL_READING );
				}
				
				status = nand_gpbc_isBadPage( PGpbc, ofs );
				
				if (getchip){
					nand_gpbc_releaseDevice( PGpbc );
				}
				
				return status;
			}
			
			return nand_isbad_bbt( PMtd, ofs, allowbbt );
		}
	#else	//CONFIG_MTD_NAND_MN2WS_BB
		static int nand_gpbc_isBad( struct nand_gpbc_info *PGpbc, loff_t ofs, int getchip, int allowbbt )
		{
			return nand_isbad_bbt( PGpbc->mtd, ofs, allowbbt );
		}
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
#else
	static int nand_gpbc_isBad( struct nand_gpbc_info *PGpbc, loff_t ofs, int getchip, int allowbbt )
	{
		return 0;
	}
#endif

#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_markBad( struct nand_gpbc_info *PGpbc, loff_t ofs )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		int block;
		int ret;
		u32	result;
		
		block = (int)(ofs >> PChip->bbt_erase_shift);
		if( PChip->bbt ){
			PChip->bbt[block >> 2] |= 0x01 << ((block & 0x03) << 1);
		}
		
		if( PChip->options & NAND_USE_FLASH_BBT ){
			ret = nand_update_bbt( PMtd, ofs );
		}else{
			nand_gpbc_getDevice( PGpbc, FL_WRITING );
			
			result = nand_gpbc_markBadPage( PGpbc, ofs );
			
			nand_gpbc_releaseDevice( PGpbc );
			
			if( result ){
				ret = -EIO;
			}else{
				ret = 0;
			}
		}
		
		if( !ret ){
			PMtd->ecc_stats.badblocks++;
		}
		
		return ret;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_markBad( struct nand_gpbc_info *PGpbc, loff_t ofs )
	{
		struct mtd_info		*PMtd = PGpbc->mtd;
		struct nand_chip	*PChip = &PGpbc->chip;
		int		status;
		
		
		nand_gpbc_getDevice( PGpbc, FL_WRITING );
		
		status = bbt_registerBb( PMtd, ofs >> (PChip->page_shift) );
		
		nand_gpbc_releaseDevice( PGpbc );
		
		return status;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


//----------------------------
//-------- ecc func ----------
//----------------------------


static void nand_gpbc_ecc_controlEccBug( struct mtd_info* PMtd, int mode )
{
	BUG();
}


static int nand_gpbc_ecc_calculateEccBug( struct mtd_info *PMtd, const unsigned char *buf, unsigned char *code)
{
	BUG();
	
	return 0;
}


static int nand_gpbc_ecc_correctEccBug( struct mtd_info *PMtd, unsigned char *buf, unsigned char *read_ecc, unsigned char *calc_ecc )
{
	BUG();
	
	return 0;
}


static int nand_gpbc_ecc_readPageBug( struct mtd_info *mtd, struct nand_chip *chip, uint8_t *buf, int page)
{
	BUG();
	
	return 0;
}


static int nand_gpbc_ecc_readPageRawBug(struct mtd_info *PMtd, struct nand_chip *PChip, uint8_t *pBuf, int page)
{
	BUG();
	
	return 0;
}


static void nand_gpbc_ecc_writePageRawBug( struct mtd_info *PMtd, struct nand_chip *PChip, const uint8_t *pBuf )
{
	BUG();
}


static void nand_gpbc_ecc_writePageBug( struct mtd_info *PMtd, struct nand_chip *PChip, const uint8_t *pBuf )
{
	BUG();
}


static int nand_gpbc_ecc_readOobBug(struct mtd_info *PMtd, struct nand_chip *PChip, int page, int sndcmd)
{
	BUG();
	
	return 0;
}


static int nand_gpbc_ecc_writeOobBug( struct mtd_info *PMtd, struct nand_chip *PChip, int page )
{
	BUG();
	
	return 0;
}


//----------------------------
//------- chip func ----------
//----------------------------


static u8 nand_gpbc_chip_readByteBug( struct mtd_info* PMtd )
{
	BUG();
	
	return 0;
}


static u16 nand_gpbc_chip_readWordBug( struct mtd_info *PMtd)
{
	BUG();
	
	return 0;
}


static void nand_gpbc_chip_writeBufBug( struct mtd_info *mtd, const uint8_t *pBuf, int len )
{
	BUG();
}


static void nand_gpbc_chip_readBufBug( struct mtd_info *mtd, uint8_t *pBuf, int len )
{
	BUG();
}


static int nand_gpbc_chip_verifyBufBug(struct mtd_info *mtd, const uint8_t *buf, int len)
{
	BUG();
	
	return 0;
}


static void nand_gpbc_chip_selectChipBug( struct mtd_info *mtd, int chip )
{
	BUG();
}


static int nand_gpbc_chip_checkBadBug( struct mtd_info *PMtd, loff_t ofs, int getchip)
{
	BUG();
	
	return 0;
}


static int nand_gpbc_chip_markBadBug( struct mtd_info *PMtd, loff_t ofs )
{
	BUG();
	
	return 0;
}


void nand_gpbc_chip_controlChipBug( struct mtd_info* PMtd, int cmd, unsigned int ctrl )
{
	BUG();
}


static int nand_gpbc_chip_checkReadyBug(struct mtd_info* PMtd)
{
	BUG();
	
	return 0;
}


static void nand_gpbc_chip_cmdFuncBug(struct mtd_info *PMtd, unsigned int cmd, int col, int page)
{
	BUG();
}


static int nand_gpbc_chip_waitFuncBug(struct mtd_info *PMtd, struct nand_chip *chip)
{
	BUG();
	return 0;
}


static void nand_gpbc_chip_eraseCmdBug( struct mtd_info *PMtd, int page )
{
	BUG();
}


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_chip_writePageBug( struct mtd_info *PMtd, struct nand_chip *PChip, const uint8_t *pBuf, int page, int cached, int raw )
	{
		BUG();
		
		return 0;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_chip_writePageBug( struct mtd_info *PMtd, struct nand_chip *PChip, const uint8_t *pBuf, int page, int cached, int raw, int *pRetPage )
	{
		BUG();
		
		return 0;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_chip_eraseOps( struct mtd_info *PMtd, struct erase_info *PEinfo, int allowbbt )
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int page, ret;
		loff_t len;
		u32		result;
		int	status;
		
		if( PEinfo->addr & ((1 << PChip->phys_erase_shift) - 1) ){
			return -EINVAL;
		}
		
		if( PEinfo->len & ((1 << PChip->phys_erase_shift) - 1) ){
			return -EINVAL;
		}
		
		if( (PEinfo->len + PEinfo->addr) > PMtd->size ){
			return -EINVAL;
		}
		
		PEinfo->fail_addr = MTD_FAIL_ADDR_UNKNOWN;
		
		nand_gpbc_getDevice( PGpbc, FL_ERASING );
		
		page = (int)(PEinfo->addr >> PChip->page_shift);
		len = PEinfo->len;
		PEinfo->state = MTD_ERASING;
		
		while( len )
		{
			status = nand_gpbc_isBad( PGpbc, ((loff_t)page) << PChip->page_shift, 0, allowbbt );
			if( status ){
				printk(KERN_WARNING "%s: attempt to erase a bad block at page 0x%08x\n", __func__, page);
				PEinfo->state = MTD_ERASE_FAILED;
				goto erase_exit;
			}
			
			if( page <= PChip->pagebuf && PChip->pagebuf < (page + PGpbc->pagesPerBlock) ){
				PChip->pagebuf = -1;
			}
			
			result = nand_gpbc_eraseBlock( PGpbc, page );
			if( result ){
				if( NAND_GPBC_NI__ERAFAIL & result ){
					PEinfo->state = MTD_ERASE_FAILED;
					PEinfo->fail_addr = ((loff_t)page << PChip->page_shift);
					goto erase_exit;
				}else{
					PEinfo->state = MTD_ERASE_FAILED;
					goto erase_exit;
				}
			}
			
			len -= PMtd->erasesize;
			page += PGpbc->pagesPerBlock;
		}
		PEinfo->state = MTD_ERASE_DONE;
		
	erase_exit:
		
		ret = PEinfo->state == MTD_ERASE_DONE ? 0 : -EIO;
		
		nand_gpbc_releaseDevice( PGpbc );
		
		if( !ret ){
			mtd_erase_callback(PEinfo);
		}
		
		return ret;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
/*
 * bbt_masked_page_ret and rewrite_bbt are not used in this function.
 * It makes the same arguments with nand_erase_nand().
 */
	static int nand_gpbc_chip_eraseOps(struct mtd_info *PMtd, struct erase_info *PEinfo,
			    int allowbbt, unsigned int *bbt_masked_page_ret, loff_t *rewrite_bbt)
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int page, ret;
		loff_t len;
		u32		result;
		int	status;
		
		PEinfo->fail_addr = MTD_FAIL_ADDR_UNKNOWN;
		page = (int)(PEinfo->addr >> PChip->page_shift);
		len = PEinfo->len;
		PEinfo->state = MTD_ERASING;
		
		while( len )
		{
			status = nand_gpbc_isBad( PGpbc, ((loff_t)page) << PChip->page_shift, 0, allowbbt );
			if( status ){
				printk(KERN_WARNING "%s: attempt to erase a bad block at page 0x%08x\n", __func__, page);
				PEinfo->state = MTD_ERASE_FAILED;
				goto erase_exit;
			}
			
			if( page <= PChip->pagebuf && PChip->pagebuf < (page + PGpbc->pagesPerBlock) ){
				PChip->pagebuf = -1;
			}
			
			result = nand_gpbc_eraseBlock( PGpbc, page );
			if( result ){
				if( NAND_GPBC_NI__ERAFAIL & result ){
					PEinfo->state = MTD_ERASE_FAILED;
					PEinfo->fail_addr = ((loff_t)page << PChip->page_shift);
					goto erase_exit;
				}else{
					PEinfo->state = MTD_ERASE_FAILED;
					goto erase_exit;
				}
			}
			
			len -= PMtd->erasesize;
			page += PGpbc->pagesPerBlock;
		}
		PEinfo->state = MTD_ERASE_DONE;
		
	erase_exit:
		
		ret = PEinfo->state == MTD_ERASE_DONE ? 0 : -EIO;
		
		return ret;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


static int nand_gpbc_chip_readOps(struct mtd_info *PMtd, loff_t from, struct mtd_oob_ops *POps)
{
	struct nand_chip 		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
	int page, col, bytes, aligned;
	struct mtd_ecc_stats stats;
	u32 readlen;
	u32 oobreadlen;
	u8 *pBufpoi, *pOob, *pBuf;
	#ifdef MTD_NAND_GPBC_TEST
		u32 ooboffs;
	#endif
	
	stats = PMtd->ecc_stats;
	
	page = (int)(from >> PChip->page_shift);
	
	col = (int)(from & (PMtd->writesize - 1));
	
	pBuf = POps->datbuf;
	pOob = POps->oobbuf;
	readlen = POps->len;
	oobreadlen = POps->ooblen;
	
	#ifdef MTD_NAND_GPBC_TEST
		ooboffs = POps->ooboffs;
	#endif
	while(1)
	{
		bytes = min(PMtd->writesize - col, readlen);
		aligned = (bytes == PMtd->writesize);
		bbt_clear_erased_flag(PChip);
		
		// Is the current page in the buffer ?
		if( page != PChip->pagebuf || pOob ){
			pBufpoi = aligned ? pBuf : PChip->buffers->databuf;
			
			if( unlikely(POps->mode == MTD_OOB_RAW) ){
				nand_gpbc_readPageAll( PGpbc, pBufpoi, page, 1 );
			}else{
				if( !POps->oobbuf ){
					nand_gpbc_readPageDataEcc( PGpbc, pBufpoi, page );
				}else{
					nand_gpbc_readPageAll( PGpbc, pBufpoi, page, 0 );
				}
			}
			
			if( !aligned ){
				if( !pOob ){
					PChip->pagebuf = page;
				}
				memcpy( pBuf, PChip->buffers->databuf + col, bytes );
			}
			
			pBuf += bytes;
			
			if( unlikely(pOob) ){
				#ifndef MTD_NAND_GPBC_TEST
					if( POps->mode != MTD_OOB_RAW ){
						int toread = min( oobreadlen, PChip->ecc.layout->oobavail );
						if( toread ){
							pOob = nand_gpbc_transferOob( PGpbc, pOob, POps, toread );
							oobreadlen -= toread;
						}
					}else{
							pBuf = nand_gpbc_transferOob( PGpbc, pBuf, POps, PMtd->oobsize );
					}
				#else
					if( POps->mode == MTD_OOB_AUTO ){
						int toread = min( oobreadlen, PChip->ecc.layout->oobavail - POps->ooboffs );
						if( toread ){
							pOob = nand_gpbc_transferOob( PGpbc, pOob, POps, toread );
							oobreadlen -= toread;
						}
					}else{
						pOob = nand_gpbc_transferOob( PGpbc, pOob, POps, PMtd->oobsize - POps->ooboffs );
						oobreadlen -= PMtd->oobsize - POps->ooboffs;
					}
					POps->ooboffs = 0;
				#endif
			}
		}else{
			memcpy(pBuf, PChip->buffers->databuf + col, bytes);
			pBuf += bytes;
		}
		
		readlen -= bytes;
		
		if( !readlen ){
			break;
		}
		
		/* For subsequent reads align to page boundary. */
		col = 0;
		/* Increment page address */
		page++;
	}
	
	#ifdef MTD_NAND_GPBC_TEST
		POps->ooboffs = ooboffs;
	#endif
	POps->retlen = POps->len - (size_t) readlen;
	if( pOob ){
		POps->oobretlen = POps->ooblen - oobreadlen;
	}
	
	if( PMtd->ecc_stats.failed - stats.failed ){
		return -EBADMSG;
	}
	
	return  PMtd->ecc_stats.corrected - stats.corrected ? -EUCLEAN : 0;
}


static int nand_gpbc_chip_readOob( struct mtd_info *PMtd, loff_t from, struct mtd_oob_ops *POps )
{
	struct nand_chip 		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
	int page;
	int readlen = POps->ooblen;
	int len;
	uint8_t *pBuf = POps->oobbuf;
	#ifdef MTD_NAND_GPBC_TEST
		u32 ooboffs;
	#endif
	
	if( POps->mode == MTD_OOB_AUTO ){
		len = PChip->ecc.layout->oobavail;
	}else{
		len = PMtd->oobsize;
	}
	
	if( unlikely(POps->ooboffs >= len) ){
		return -EINVAL;
	}
	
	if( unlikely(from >= PMtd->size ||
		    POps->ooboffs + readlen > ((PMtd->size >> PChip->page_shift) -
			(from >> PChip->page_shift)) * len) ){
		return -EINVAL;
	}
	
	page = (int)(from >> PChip->page_shift);
	#ifdef MTD_NAND_GPBC_TEST
		ooboffs = POps->ooboffs;
	#endif
	while(1)
	{
		nand_gpbc_readPageOob( PGpbc, page );
		
		#ifndef MTD_NAND_GPBC_TEST
			len = min(len, readlen);
			pBuf = nand_gpbc_transferOob( PGpbc, pBuf, POps, len );
		#else
			if( POps->mode == MTD_OOB_AUTO ){
				len = min( (int)(PChip->ecc.layout->oobavail - POps->ooboffs), readlen );
			}else{
				len = min( (int)(PMtd->oobsize - POps->ooboffs), readlen );
			}
			pBuf = nand_gpbc_transferOob( PGpbc, pBuf, POps, len );
			POps->ooboffs = 0;
		#endif
		
		readlen -= len;
		if( !readlen ){
			break;
		}
		
		page++;
	}
	
	#ifdef MTD_NAND_GPBC_TEST
		POps->ooboffs = ooboffs;
	#endif
	POps->oobretlen = POps->ooblen;
	return 0;
}


#define NOTALIGNED(x)	((x) & (PChip->subpagesize - 1)) != 0


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_chip_writeOps(struct mtd_info *PMtd, loff_t to, struct mtd_oob_ops *POps)
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int page, column;
		u32 writelen;
		uint8_t *pOob = POps->oobbuf;
		uint8_t *pBuf = POps->datbuf;
		int ret, subpage;
		u32	result;
		
		POps->retlen = 0;
		
		writelen = POps->len;
		if( !writelen ){
			return 0;
		}
		
		// check subpage align
		if( NOTALIGNED(to) || NOTALIGNED(POps->len) ){
			printk( KERN_NOTICE "%s: Attempt to write not page aligned data\n", __func__ );
			return -EINVAL;
		}
		
		column = to & (PMtd->writesize - 1);
		subpage = column || (writelen & (PMtd->writesize - 1));
		if( subpage && pOob ){
			return -EINVAL;
		}
		
		page = (int)(to >> PChip->page_shift);
		
		// Invalidate the page cache, when we write to the cached page
		if( (to <= (PChip->pagebuf << PChip->page_shift)) &&
				((PChip->pagebuf << PChip->page_shift) < (to + POps->len)) ){
			PChip->pagebuf = -1;
		}
		
		while(1)
		{
			int	bytes = PMtd->writesize;
			u8	*wbuf = pBuf;
			
			// Partial page write ?
			if( unlikely(column || writelen < (PMtd->writesize - 1)) ){
				bytes = min_t( int, bytes - column, (int)writelen );
				PChip->pagebuf = -1;
				memset( PChip->buffers->databuf, 0xff, PMtd->writesize );
				memcpy( &PChip->buffers->databuf[column], pBuf, bytes );
				wbuf = PChip->buffers->databuf;
			}
			
			if( !pOob ){
				if( MTD_OOB_RAW != POps->mode ){
					result = nand_gpbc_writePageDataEcc( PGpbc, wbuf, page );
				}else{
					memset( PChip->oob_poi, 0xff, PMtd->oobsize );
					result = nand_gpbc_writePageAll( PGpbc, wbuf, page, 1 );
				}
			}else{
				pOob = nand_gpbc_fillOob( PGpbc, pOob, POps );
				if( MTD_OOB_RAW != POps->mode ){
					result = nand_gpbc_writePageAll( PGpbc, wbuf, page, 0 );
				}else{
					result = nand_gpbc_writePageAll( PGpbc, wbuf, page, 1 );
				}
			}
			if( NAND_GPBC_NI__ERR & result ){
				ret = -EIO;
				break;
			}
			
			writelen -= bytes;
			if( !writelen ){
				ret = 0;
				break;
			}
			
			column = 0;
			pBuf += bytes;
			page++;
		}
		
		POps->retlen = POps->len - writelen;
		if( unlikely(pOob) ){
			POps->oobretlen = POps->ooblen;
		}
		
		return ret;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_chip_writeOps( struct mtd_info *PMtd, loff_t to, struct mtd_oob_ops *POps
					, int *pRetPage )
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int page, column;
		u32 writelen;
		uint8_t *pOob = POps->oobbuf;
		uint8_t *pBuf = POps->datbuf;
		int ret, subpage;
		u32	result;
		
		
		*pRetPage = -1;
		POps->retlen = 0;
		
		writelen = POps->len;
		if( !writelen ){
			return 0;
		}
		
		// check subpage align
		if( NOTALIGNED(to) || NOTALIGNED(POps->len) ){
			printk( KERN_NOTICE "%s: Attempt to write not page aligned data\n", __func__ );
			return -EINVAL;
		}
		
		column = to & (PMtd->writesize - 1);
		subpage = column || (writelen & (PMtd->writesize - 1));
		if( subpage && pOob ){
			return -EINVAL;
		}
		
		page = (int)(to >> PChip->page_shift);
		
		// Invalidate the page cache, when we write to the cached page
		if( (to <= (PChip->pagebuf << PChip->page_shift)) &&
				((PChip->pagebuf << PChip->page_shift) < (to + POps->len)) ){
			PChip->pagebuf = -1;
		}
		
		while(1)
		{
			int	bytes = PMtd->writesize;
			u8	*wbuf = pBuf;
			
			// Partial page write ?
			if( unlikely(column || writelen < (PMtd->writesize - 1)) ){
				bytes = min_t( int, bytes - column, (int)writelen );
				PChip->pagebuf = -1;
				memset( PChip->buffers->databuf, 0xff, PMtd->writesize );
				memcpy( &PChip->buffers->databuf[column], pBuf, bytes );
				wbuf = PChip->buffers->databuf;
			}
			
			if( !pOob ){
				if( MTD_OOB_RAW != POps->mode ){
					result = nand_gpbc_writePageDataEcc( PGpbc, wbuf, page );
				}else{
					memset( PChip->oob_poi, 0xff, PMtd->oobsize );
					result = nand_gpbc_writePageAll( PGpbc, wbuf, page, 1 );
				}
			}else{
				pOob = nand_gpbc_fillOob( PGpbc, pOob, POps );
				if( MTD_OOB_RAW != POps->mode ){
					result = nand_gpbc_writePageAll( PGpbc, wbuf, page, 0 );
				}else{
					result = nand_gpbc_writePageAll( PGpbc, wbuf, page, 1 );
				}
			}
			if( NAND_GPBC_NI__ERR & result ){
				if( NAND_GPBC_NI__PROFAIL & result ){
					*pRetPage = page;
				}
				ret = -EIO;
				break;
			}
			
			writelen -= bytes;
			if( !writelen ){
				ret = 0;
				break;
			}
			
			column = 0;
			pBuf += bytes;
			page++;
		}
		
		POps->retlen = POps->len - writelen;
		if( unlikely(pOob) ){
			POps->oobretlen = POps->ooblen;
		}
		
		return ret;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_chip_writeOob( struct mtd_info *PMtd, loff_t to, struct mtd_oob_ops *POps )
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int page, len;
		u32	result;
		
		if( POps->mode == MTD_OOB_AUTO ){
			len = PChip->ecc.layout->oobavail;
		}else{
			len = PMtd->oobsize;
		}
		
		if( (POps->ooboffs + POps->ooblen) > len ){
			return -EINVAL;
		}
		
		if( unlikely(POps->ooboffs >= len) ){
			return -EINVAL;
		}
		
		if( unlikely(to >= PMtd->size || POps->ooboffs + POps->ooblen >
				((PMtd->size >> PChip->page_shift) - (to >> PChip->page_shift)) * len) ){
			return -EINVAL;
		}
		
		page = (int)(to >> PChip->page_shift);
		if( page == PChip->pagebuf ){
			PChip->pagebuf = -1;
		}
		
		nand_gpbc_fillOob( PGpbc, POps->oobbuf, POps );
		result = nand_gpbc_writePageOob( PGpbc, page );
		if( result ){
			return -EIO;
		}
		
		POps->oobretlen = POps->ooblen;
		
		return 0;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_chip_writeOob( struct mtd_info *PMtd, loff_t to, struct mtd_oob_ops *POps
					, int *pRetPage )
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int page, len;
		u32	result;
		
		*pRetPage = -1;
		
		if( POps->mode == MTD_OOB_AUTO ){
			len = PChip->ecc.layout->oobavail;
		}else{
			len = PMtd->oobsize;
		}
		
		if( (POps->ooboffs + POps->ooblen) > len ){
			return -EINVAL;
		}
		
		if( unlikely(POps->ooboffs >= len) ){
			return -EINVAL;
		}
		
		if( unlikely(to >= PMtd->size || POps->ooboffs + POps->ooblen >
				((PMtd->size >> PChip->page_shift) - (to >> PChip->page_shift)) * len) ){
			return -EINVAL;
		}
		
		page = (int)(to >> PChip->page_shift);
		if( page == PChip->pagebuf ){
			PChip->pagebuf = -1;
		}
		
		nand_gpbc_fillOob( PGpbc, POps->oobbuf, POps );
		result = nand_gpbc_writePageOob( PGpbc, page );
		if( result ){
			if( NAND_GPBC_NI__PROFAIL & result ){
				*pRetPage = page;
			}
			return -EIO;
		}
		
		POps->oobretlen = POps->ooblen;
		
		return 0;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


//----------------------------
//--------- mtd func ---------
//----------------------------


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_erase( struct mtd_info *PMtd, struct erase_info *PEinfo )
	{
		int status;
		
		status = nand_gpbc_chip_eraseOps( PMtd, PEinfo, 0 );
		
		return status;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_erase(struct mtd_info *PMtd, struct erase_info *PEinfo)
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int status;
		
		if( PEinfo->addr & ((1 << PChip->phys_erase_shift) - 1) ){
			return -EINVAL;
		}
		
		if( PEinfo->len & ((1 << PChip->phys_erase_shift) - 1) ){
			return -EINVAL;
		}
		
		if( (PEinfo->len + PEinfo->addr) > PMtd->size ){
			return -EINVAL;
		}
		
		nand_gpbc_getDevice( PGpbc, FL_ERASING );
		
		while(1)
		{
			int		status2;
			
			status = nand_gpbc_chip_eraseOps( PMtd, PEinfo, 0, NULL, NULL );
			if( MTD_FAIL_ADDR_UNKNOWN == PEinfo->fail_addr ){
				break;
			}
			
			status2 = bbt_alternateBb( PMtd, (PEinfo->fail_addr >> PChip->page_shift) );
			if( 0 < status2 ){
				break;
			}else if( 0 > status2 ){
				BUG();
			}
		}
		
		nand_gpbc_releaseDevice( PGpbc );
		
		if( !status ){
			mtd_erase_callback(PEinfo);
		}
		
		return status;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


static int nand_gpbc_mtd_read(struct mtd_info *PMtd, loff_t from, size_t len, size_t *retlen, uint8_t *buf)
{
	struct nand_chip		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
	int ret;
	
	if( (from + len) > PMtd->size ){
		return -EINVAL;
	}
	if( !len ){
		#ifdef MTD_NAND_GPBC_TEST
			*retlen = 0;
		#endif
		return 0;
	}
	
	nand_gpbc_getDevice( PGpbc, FL_READING );
	
	PChip->ops.len = len;
	PChip->ops.datbuf = buf;
	PChip->ops.oobbuf = NULL;
	
	ret = nand_gpbc_chip_readOps( PMtd, from, &PChip->ops );
	
	*retlen = PChip->ops.retlen;
	
	nand_gpbc_releaseDevice( PGpbc );
	
	return ret;
}


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_write(struct mtd_info *PMtd, loff_t to, size_t len, size_t *retlen, const uint8_t *pBuf)
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int ret;
		
		/* Do not allow reads past end of device */
		if ((to + len) > PMtd->size)
			return -EINVAL;
		if( !len ){
			#ifdef MTD_NAND_GPBC_TEST
				*retlen = 0;
			#endif
			return 0;
		}
		
		nand_gpbc_getDevice( PGpbc, FL_WRITING );
		
		PChip->ops.len = len;
		PChip->ops.datbuf = (uint8_t *)pBuf;
		PChip->ops.oobbuf = NULL;
		
		ret = nand_gpbc_chip_writeOps(PMtd, to, &PChip->ops);
		
		*retlen = PChip->ops.retlen;
		
		nand_gpbc_releaseDevice( PGpbc );
		
		return ret;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_write( struct mtd_info *PMtd, loff_t to, size_t len
					, size_t *retlen, const uint8_t *pBuf )
	{
		struct nand_chip		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int ret;
		
		if( (to + len) > PMtd->size ){
			return -EINVAL;
		}
		if( !len ){
			#ifdef MTD_NAND_GPBC_TEST
				*retlen = 0;
			#endif
			return 0;
		}
		
		nand_gpbc_getDevice( PGpbc, FL_WRITING );
		
		while(1)
		{
			int		status;
			int		retPage;
			
			
			PChip->ops.len = len;
			PChip->ops.datbuf = (uint8_t *)pBuf;
			PChip->ops.oobbuf = NULL;
			
			ret = nand_gpbc_chip_writeOps( PMtd, to, &PChip->ops, &retPage );
			if( -1 == retPage ){
				break;
			}
			
			status = bbt_replaceBb( PMtd, retPage );
			//if retPage belongs beside the normal area
			if( 0 < status ){
				break;
			}else if( 0 > status ){
				BUG();
			}
		}
		
		*retlen = PChip->ops.retlen;
		
		nand_gpbc_releaseDevice( PGpbc );
		
		return ret;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


static int nand_gpbc_mtd_readOob( struct mtd_info *PMtd, loff_t from, struct mtd_oob_ops *POps )
{
	struct nand_chip 		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
	int ret;
	
	POps->retlen = 0;
	
	if( POps->datbuf && (from + POps->len) > PMtd->size ){
		return -EINVAL;
	}
	
	nand_gpbc_getDevice( PGpbc, FL_READING );

	switch( POps->mode )
	{
		case MTD_OOB_PLACE:
		case MTD_OOB_AUTO:
		case MTD_OOB_RAW:
			break;
		default:
			BUG();
	}
	
	if( !POps->datbuf ){
		ret = nand_gpbc_chip_readOob( PMtd, from, POps );
	}else{
		ret = nand_gpbc_chip_readOps( PMtd, from, POps );
	}
	
	nand_gpbc_releaseDevice( PGpbc );
	return ret;
}


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_writeOob( struct mtd_info *PMtd, loff_t to, struct mtd_oob_ops *POps )
	{
		struct nand_chip 		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int ret;
		
		POps->retlen = 0;
		
		if( POps->datbuf && (to + POps->len) > PMtd->size ){
			return -EINVAL;
		}
		
		nand_gpbc_getDevice( PGpbc, FL_WRITING );
		
		switch( POps->mode )
		{
			case MTD_OOB_PLACE:
			case MTD_OOB_AUTO:
			case MTD_OOB_RAW:
				break;
			default:
				BUG();
		}
		
		if( !POps->datbuf ){
			ret = nand_gpbc_chip_writeOob( PMtd, to, POps );
		}else{
			ret = nand_gpbc_chip_writeOps( PMtd, to, POps );
		}
		
//	out:
		nand_gpbc_releaseDevice( PGpbc );
		return ret;
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_writeOob(struct mtd_info *PMtd, loff_t to,
				  struct mtd_oob_ops *POps)
	{
		struct nand_chip 		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int ret;
		
		POps->retlen = 0;
		
		if( POps->datbuf && (to + POps->len) > PMtd->size ){
			return -EINVAL;
		}
		
		nand_gpbc_getDevice( PGpbc, FL_WRITING );
		
		switch(POps->mode)
		{
			case MTD_OOB_PLACE:
			case MTD_OOB_AUTO:
			case MTD_OOB_RAW:
				break;
			default:
				BUG();
		}
		
		while(1)
		{
			int		status;
			int		retPage;
			
			
			if( !POps->datbuf ){
				ret = nand_gpbc_chip_writeOob( PMtd, to, POps, &retPage );
			}else{
				ret = nand_gpbc_chip_writeOps( PMtd, to, POps, &retPage );
			}
			
			if( -1 == retPage ){
				break;
			}
			
			status = bbt_replaceBb( PMtd, retPage );
			if( 0 < status ){
				break;
			}else if( 0 > status ){
				BUG();
			}
		}
		
		nand_gpbc_releaseDevice( PGpbc );
		return ret;
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


static void nand_gpbc_mtd_sync( struct mtd_info *PMtd )
{
	struct nand_chip 		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
	
	nand_gpbc_getDevice( PGpbc, FL_SYNCING );
	nand_gpbc_releaseDevice( PGpbc );
}


static int nand_gpbc_mtd_suspend(struct mtd_info *PMtd)
{
	struct nand_chip 		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;

	return nand_gpbc_getDevice( PGpbc, FL_PM_SUSPENDED );
}


static void nand_gpbc_mtd_resume(struct mtd_info *PMtd)
{
	struct nand_chip 		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;

	if( FL_PM_SUSPENDED == PChip->state ){
		nand_gpbc_releaseDevice( PGpbc );
	}else{
		printk(KERN_ERR "%s called for a chip which is not in suspended state\n", __func__);
	}
}


static int nand_gpbc_mtd_isBad( struct mtd_info *PMtd, loff_t offs )
{
	struct nand_chip 		*PChip = PMtd->priv;
	struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
	
	if( offs > PMtd->size ){
		return -EINVAL;
	}
	
	return nand_gpbc_isBad( PGpbc, offs, 1, 0 );
}


#ifndef CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_markBad( struct mtd_info *PMtd, loff_t ofs )
	{
		struct nand_chip 		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		int status;
		
		status = nand_gpbc_mtd_isBad( PMtd, ofs );
		if( status ){
			if( status > 0 ){
				return 0;
			}
			return status;
		}
		
		return nand_gpbc_markBad( PGpbc, ofs );
	}
#else	//CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_mtd_markBad(struct mtd_info *PMtd, loff_t ofs)
	{
		struct nand_chip 		*PChip = PMtd->priv;
		struct nand_gpbc_info	*PGpbc = (struct nand_gpbc_info *)PChip->priv;
		
		return nand_gpbc_markBad( PGpbc, ofs );
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


//----------------------------
//-------- init func ---------
//----------------------------


static int nand_gpbc_initHw1(struct nand_gpbc_info *PGpbc)
{
	u32		cnt;
	u32		temp32;
	
	#if 0
	{
		u32		irqMask;
		u32		result;

		//Wait for the completion of this controller initialization
		irqMask = NAND_GPBC_NI__RSTCMP;
		#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
			result = nand_gpbc_waitIrq( PGpbc, 0, irqMask );
		#else
			result = nand_gpbc_pollIrq( PGpbc, 0, irqMask );
		#endif
		if( NAND_GPBC_NI__ERR & result ){
			return -1;
		}
	}
	#endif
	
	#ifdef MTD_NAND_MN2WS_GPBC_ECC_BUG
		temp32 = nand_gpbc_read32( PGpbc->regBase + NAND_GPBC_NECCE );
		if( !temp32 ){
			printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
			BUG();
		}
	#endif
	
	//This driver has been tested only on 8bit ECC.
	temp32 = nand_gpbc_read32( PGpbc->regBase + NAND_GPBC_NECCC );
	if( 8 != temp32 ){
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		BUG();
	}
	
	//enable WP
	nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NWPD );
	
	//Enable R/B pin of all banks
	nand_gpbc_write32( NAND_GPBC_NRBE__MASK, PGpbc->regBase + NAND_GPBC_NRBE );
	
	//Check the number of available banks
	PGpbc->banks = 1;
	for( cnt = 1; cnt < NAND_GPBC_MAX_FLASH_BANKS; cnt++ ){
		if (nand_gpbc_checkBank(PGpbc, cnt)) {
			break;
		}
		PGpbc->banks++;
	}
	
	return 0;
}

static void nand_read_id(struct nand_gpbc_info *gpbc, u8 *nand_id, int num)
{
	u32 bank = 0 << NAND_GPBC_BANK_SEL;
	int i = 0;

	nand_gpbc_index32(gpbc, NAND_GPBC_MAP11 | bank | NAND_GPBC_MAP11_CMD, NAND_CMD_READID);
	nand_gpbc_index32(gpbc, NAND_GPBC_MAP11 | bank | NAND_GPBC_MAP11_ADR, 0);
	printk("[READ ID]\n");
	for (i = 0; i < num; i++) {
		nand_gpbc_write32(NAND_GPBC_MAP11 | bank | NAND_GPBC_MAP11_DAT, gpbc->memBase);
		nand_id[i] = (u8)(nand_gpbc_read32(gpbc->memBase + 0x10) & 0xff);
		printk("  ID%d : 0x%x\n", i, (unsigned int)nand_id[i]);
	}
}


static void nand_gpbc_getId(struct nand_gpbc_info *PGpbc)
{
	/*
	 * This driver get parameters with IDREAD command.
	 * This driver cannot get parameters from onfi standard.
	 */
	nand_read_id(PGpbc, PGpbc->aId, MTD_NAND_MN2WS_ID_READ_NUM);

#ifdef PANA_SAMPLE_BOARD	/* disable nand-id check */
	if( !((NAND_GPBC_UNIT_MID == PGpbc->aId[0]) && (NAND_GPBC_UNIT_DID == PGpbc->aId[1])) ){
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		BUG();
		return;
	}
#endif	
	//pulse
	nand_gpbc_write32( NAND_GPBC_UNIT_NCDWR, PGpbc->regBase + NAND_GPBC_NCDWR );
	nand_gpbc_write32( NAND_GPBC_UNIT_NCAAD, PGpbc->regBase + NAND_GPBC_NCAAD );
	nand_gpbc_write32( NAND_GPBC_UNIT_NRW, PGpbc->regBase + NAND_GPBC_NRW );
	nand_gpbc_write32( NAND_GPBC_UNIT_NRDCP, PGpbc->regBase + NAND_GPBC_NRDCP );
	nand_gpbc_write32( NAND_GPBC_UNIT_NRWLP, PGpbc->regBase + NAND_GPBC_NRWLP );
	nand_gpbc_write32( NAND_GPBC_UNIT_NRWHP, PGpbc->regBase + NAND_GPBC_NRWHP );
	nand_gpbc_write32( NAND_GPBC_UNIT_NCES, PGpbc->regBase + NAND_GPBC_NCES );
	nand_gpbc_write32( NAND_GPBC_UNIT_NRR, PGpbc->regBase + NAND_GPBC_NRR );
	
	#if 0
		{
			u32		aTemp32[5];
			u32		temp32;
			u32		bankSel = 0 << NAND_GPBC_BANK_SEL;
			
			temp32 = nand_gpbc_read32( PGpbc->regBase + NAND_GPBC_NONDNOL );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_CMD, NAND_CMD_READID );
			nand_gpbc_index32( PGpbc, NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_ADR, 0 );
			nand_gpbc_write32( NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_DAT, PGpbc->memBase );
			aTemp32[0] = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			nand_gpbc_write32( NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_DAT, PGpbc->memBase );
			aTemp32[1] = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			nand_gpbc_write32( NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_DAT, PGpbc->memBase );
			aTemp32[2] = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			nand_gpbc_write32( NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_DAT, PGpbc->memBase );
			aTemp32[3] = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			nand_gpbc_write32( NAND_GPBC_MAP11 | bankSel | NAND_GPBC_MAP11_DAT, PGpbc->memBase );
			aTemp32[4] = nand_gpbc_read32( PGpbc->memBase + 0x10 );
			printk( "ID1 : 0x%08X\n", temp32[0] );
			printk( "ID2 : 0x%08X\n", temp32[1] );
			printk( "ID3 : 0x%08X\n", temp32[2] );
			printk( "ID4 : 0x%08X\n", temp32[3] );
			printk( "ID5 : 0x%08X\n", temp32[4] );
		}
	#endif
}


static void nand_gpbc_initHw2(struct nand_gpbc_info *PGpbc)
{
	struct mtd_info		*PMtd = PGpbc->mtd;
	u32		temp32;
	
	//
	// If the chip doesn't support ONFI,
	//   set default values of registers classified into AUTOCONF reg.
	
	//Set SPARE_AREA_SKIP_BYTES
	nand_gpbc_write32( NAND_GPBC_OOB_SKIP_SIZE, PGpbc->regBase + NAND_GPBC_NSASK );
	
	//Set SPARE_AREA_MARKER
	nand_gpbc_write32( 0xffff, PGpbc->regBase + NAND_GPBC_NSAMK );
	
	//Set PAGES_PER_BLOCK
	temp32 = PMtd->erasesize / PMtd->writesize;
	nand_gpbc_write32( temp32, PGpbc->regBase + NAND_GPBC_NBPNUM );
	
	//Set DEVICE_MAIN_AREA_SIZE
	nand_gpbc_write32( PMtd->writesize, PGpbc->regBase + NAND_GPBC_NMASZ );
	
	//Set DEVICE_SPARE_AREA_SIZE
	nand_gpbc_write32( PMtd->oobsize, PGpbc->regBase + NAND_GPBC_NSASZ );

	//Set MANUFACTURE_ID
	nand_gpbc_write32(PGpbc->aId[0], PGpbc->regBase + NAND_GPBC_NMID);

	//Set DEVICE_WIDTH
	nand_gpbc_write32(0, PGpbc->regBase + NAND_GPBC_NDW);

	//Set DEVICE_NUMBER
	nand_gpbc_write32(0x00000001, PGpbc->regBase + NAND_GPBC_NDNUM);
}


static int nand_gpbc_getInfoFromId( struct nand_gpbc_info *PGpbc )
{
	struct mtd_info			*PMtd = PGpbc->mtd;
	struct nand_chip		*PChip = &PGpbc->chip;
	struct nand_flash_dev	*PType;
	u32		mafId;
	u32		mafIdx;
	u32		devId;
	u32		cnt;
	
	mafId = PGpbc->aId[0];
	for( mafIdx = 0; nand_manuf_ids[mafIdx].id != 0x0; mafIdx++ ){
		if( nand_manuf_ids[mafIdx].id == mafId ){
			break;
		}
	}
	if( 0 == nand_manuf_ids[mafIdx].id ){
		return -ENODEV;
	}
	
	devId = PGpbc->aId[1];
	PType = NULL;
	for( cnt = 0; nand_flash_ids[cnt].name != NULL; cnt++ ){
		if( devId == nand_flash_ids[cnt].id ){
			PType =  &nand_flash_ids[cnt];
			break;
		}
	}
	if( !PType ){
		return -ENODEV;
	}
	
	PMtd->name = PType->name;
	PChip->chipsize = (uint64_t)PType->chipsize << 20;
	
	if( !PType->pagesize ){
		// Newer devices have all the information in additional id bytes
		int extid;
		
		PChip->cellinfo = PGpbc->aId[2];
		PChip->numchips = 1 << (PChip->cellinfo & 0x03);
		extid = PGpbc->aId[3];
		PMtd->writesize = 1024 << (extid & 0x3);
		PMtd->oobsize = PMtd->writesize / 32;
		extid >>= 4;
		PMtd->erasesize = (64 * 1024) << (extid & 0x03);
	}else{
		// Old devices have chip data hardcoded in the device id table
		PChip->cellinfo = 0;
		PChip->numchips = 1;
		PMtd->writesize = PType->pagesize;
		PMtd->oobsize = PMtd->writesize / 32;
		PMtd->erasesize = PType->erasesize;
		if( NAND_BUSWIDTH_16 & PType->options ){
			return -ENODEV;
		}
	}
	if( PGpbc->banks < PChip->numchips ){
		BUG();
	}else if( PGpbc->banks > PChip->numchips ){
		PChip->numchips = PGpbc->banks;
	}
	
	if( NAND_GPBC_MAX_FLASH_BANKS < PChip->numchips ){
		return -ENODEV;
	}else if( CONFIG_SYS_NAND_MAX_CHIPS < PChip->numchips ){
		return -ENODEV;
	}
	if( (NAND_GPBC_ECCSIZE > PMtd->writesize) || (PMtd->writesize % NAND_GPBC_ECCSIZE) ){
		return -ENODEV;
	}
	
	PMtd->size = PChip->numchips * PChip->chipsize;
	PChip->page_shift = ffs(PMtd->writesize) - 1;
	PChip->pagemask = (PChip->chipsize >> PChip->page_shift) - 1;
	PChip->bbt_erase_shift = PChip->phys_erase_shift = ffs(PMtd->erasesize) - 1;
	if( PChip->chipsize & 0xffffffff ){
		//if chipsize is less than 4GB
		PChip->chip_shift = ffs((unsigned)PChip->chipsize) - 1;
	}else{
		PChip->chip_shift = ffs((unsigned)(PChip->chipsize >> 32)) + 32 - 1;
	}
	PChip->badblockpos = NAND_LARGE_BADBLOCK_POS;
	
	PChip->options &= ~NAND_CHIPOPTIONS_MSK;
	PChip->options |= PType->options & NAND_CHIPOPTIONS_MSK;
	//Set chip as a default. Board drivers can override it, if necessary
	PChip->options |= NAND_NO_AUTOINCR;
	if( NAND_MFR_SAMSUNG != mafId && !PType->pagesize ){
		PChip->options &= ~NAND_SAMSUNG_LP_OPTIONS;
	}
	if( NAND_4PAGE_ARRAY & PType->options ){
		return -ENODEV;
	}
	
	printk( KERN_INFO "NAND device: Manufacturer ID: 0x%02x, Chip ID: 0x%02x (%s %s)\n"
			, mafId, devId, nand_manuf_ids[mafIdx].name, PType->name );
	printk( KERN_INFO "%d NAND chips detected\n", PChip->numchips );
	
	return 0;
}


static int nand_gpbc_initInfo2( struct nand_gpbc_info *PGpbc )
{
	struct mtd_info		*PMtd = PGpbc->mtd;
	struct nand_chip	*PChip = &PGpbc->chip;
	int		status;
	u32		cnt;
	
	PChip->options = NAND_GPBC_OPTIONS;
	if( NAND_BUSWIDTH_16 & PChip->options ){
		printk(KERN_WARNING "NAND_BUSWIDTH_16 is not supported.\n");
		return -EINVAL;
	}
	if( NAND_OWN_BUFFERS & PChip->options ){
		printk(KERN_WARNING "NAND_OWN_BUFFERS is not supported.\n");
		return -EINVAL;
	}
	
	status = nand_gpbc_getInfoFromId( PGpbc );
	if( status ){
		return status;
	}
	//the chip spec has been reflected in "options" on nand_gpbc_getInfoFromId.
	//So, clear the options not supported by the controller.
	PGpbc->chip.options &= ~(NAND_NO_PADDING | NAND_CACHEPRG | NAND_COPYBACK);
	
	PGpbc->pagesPerBlock = PMtd->erasesize / PMtd->writesize;
	
	PMtd->priv = &PGpbc->chip;
	
	PChip->priv = PGpbc;
	PChip->IO_ADDR_R = 0;
	PChip->IO_ADDR_W = 0;
	PChip->dev_ready = nand_gpbc_chip_checkReadyBug;
	PChip->cmd_ctrl = nand_gpbc_chip_controlChipBug;
	PChip->erase_cmd = nand_gpbc_chip_eraseCmdBug;
	PChip->errstat = NULL;		//unused func
	PChip->fErase = nand_gpbc_chip_eraseOps;
	#ifdef	CONFIG_MTD_NAND_MN2WS_BB
		PChip->fReadOps = nand_gpbc_chip_readOps;
		PChip->fWriteOps = nand_gpbc_chip_writeOps;
		PChip->fReadOob = nand_gpbc_chip_readOob;
		PChip->fWriteOob = nand_gpbc_chip_writeOob;
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
	PChip->write_page = nand_gpbc_chip_writePageBug;
	PChip->chip_delay = 0;	//unused param
	PChip->cmdfunc = nand_gpbc_chip_cmdFuncBug;
	PChip->waitfunc = nand_gpbc_chip_waitFuncBug;
	PChip->select_chip = nand_gpbc_chip_selectChipBug;
	PChip->read_byte = nand_gpbc_chip_readByteBug;
	PChip->read_word = nand_gpbc_chip_readWordBug;
	PChip->block_bad = nand_gpbc_chip_checkBadBug;
	PChip->block_markbad = nand_gpbc_chip_markBadBug;
	PChip->write_buf = nand_gpbc_chip_writeBufBug;
	PChip->read_buf = nand_gpbc_chip_readBufBug;
	PChip->verify_buf = nand_gpbc_chip_verifyBufBug;
	PChip->scan_bbt = nand_default_bbt;
	PChip->controller = &PChip->hwcontrol;
	PGpbc->chip.bbt_td = &nand_gpbc_BbtMasterDesc;
	PGpbc->chip.bbt_md = &nand_gpbc_BbtMirrorDesc;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	PGpbc->chip.bbt_flag = &nand_gpbc_BbtFlagDesc;
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	PGpbc->chip.badblock_pattern = &nand_gpbc_BbPatternDesc;
	PGpbc->chip.ecc.mode = NAND_ECC_HW_SYNDROME;
	PGpbc->chip.ecclayout = &nand_gpbc_EccLayout;
	PGpbc->chip.ecc.layout = PGpbc->chip.ecclayout;
	PGpbc->chip.ecc.bytes = NAND_GPBC_ECCBYTES;
	PGpbc->chip.ecc.calculate = nand_gpbc_ecc_calculateEccBug;
	PGpbc->chip.ecc.correct = nand_gpbc_ecc_correctEccBug;
	PGpbc->chip.ecc.hwctl = nand_gpbc_ecc_controlEccBug;
	PGpbc->chip.ecc.size = NAND_GPBC_ECCSIZE;
	PGpbc->chip.ecc.prepad = 0;
	PGpbc->chip.ecc.postpad = 0;
	PGpbc->chip.ecc.read_page = nand_gpbc_ecc_readPageBug;
	PGpbc->chip.ecc.read_page_raw = nand_gpbc_ecc_readPageRawBug;
	PGpbc->chip.ecc.write_page = nand_gpbc_ecc_writePageBug;
	PGpbc->chip.ecc.write_page_raw = nand_gpbc_ecc_writePageRawBug;
	PGpbc->chip.ecc.read_oob = nand_gpbc_ecc_readOobBug;
	PGpbc->chip.ecc.write_oob = nand_gpbc_ecc_writeOobBug;
	PGpbc->chip.ecc.read_subpage = NULL;	//unused func
	
	PGpbc->sectSize3 = NAND_GPBC_ECCBYTES * (PGpbc->mtd->writesize / NAND_GPBC_ECCSIZE - 1);
	PGpbc->sectSize2 = NAND_GPBC_ECCSIZE - PGpbc->sectSize3;
	PGpbc->oobSkipSize = NAND_GPBC_OOB_SKIP_SIZE;
	
	PChip->oob_poi = PChip->buffers->databuf + PMtd->writesize;
	
	if( PMtd->writesize < PChip->ecc.size ){
		BUG();
	}
	
	PChip->ecc.layout->oobavail = 0;
	for( cnt = 0; PChip->ecc.layout->oobfree[cnt].length
			&& cnt < ARRAY_SIZE(PChip->ecc.layout->oobfree); cnt++ )
	{
		PChip->ecc.layout->oobavail += PChip->ecc.layout->oobfree[cnt].length;
	}
	PMtd->oobavail = PChip->ecc.layout->oobavail;
	
	PChip->ecc.steps = PMtd->writesize / PChip->ecc.size;
	if(PChip->ecc.steps * PChip->ecc.size != PMtd->writesize) {
		printk(KERN_WARNING "Invalid ecc parameters\n");
		BUG();
	}
	PChip->ecc.total = PChip->ecc.steps * PChip->ecc.bytes;
	
	PMtd->subpage_sft = 0;
	PChip->subpagesize = PMtd->writesize;
	
	PChip->state = FL_READY;
	
	/* Invalidate the pagebuffer reference */
	PChip->pagebuf = -1;
	
	/* Fill in remaining MTD driver data */
	PMtd->type = MTD_NANDFLASH;
	PMtd->flags = MTD_CAP_NANDFLASH;
	PMtd->erase = nand_gpbc_mtd_erase;
	PMtd->point = NULL;
	PMtd->unpoint = NULL;
	PMtd->read = nand_gpbc_mtd_read;
	PMtd->write = nand_gpbc_mtd_write;
	PMtd->read_oob = nand_gpbc_mtd_readOob;
	PMtd->write_oob = nand_gpbc_mtd_writeOob;
	PMtd->sync = nand_gpbc_mtd_sync;
	PMtd->lock = NULL;
	PMtd->unlock = NULL;
	PMtd->suspend = nand_gpbc_mtd_suspend;
	PMtd->resume = nand_gpbc_mtd_resume;
	PMtd->block_isbad = nand_gpbc_mtd_isBad;
	PMtd->block_markbad = nand_gpbc_mtd_markBad;
	#ifdef CONFIG_MTD_NAND_MN2WS_BB
		PMtd->getBbmInfo = bbt_getBbmInfo;
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
	
	/* propagate ecc.layout to mtd_info */
	PMtd->ecclayout = PChip->ecc.layout;
	
	return 0;
}


#ifdef	CONFIG_MTD_NAND_MN2WS_BB
	static int nand_gpbc_allocBbmResources( struct nand_gpbc_info *PGpbc )
	{
		u32		cnt;
		s32		aBtBlocks[CONFIG_SYS_NAND_MAX_CHIPS] = NAND_GPBC_UNIT_BTBLOCKS;
		s32		aMtBlocks[CONFIG_SYS_NAND_MAX_CHIPS] = NAND_GPBC_UNIT_MTBLOCKS;
		
		PGpbc->chip.pBbmBuf = NULL;
		PGpbc->chip.pBlockBuf = NULL;
		PGpbc->chip.psBbm = nand_gpbc_ABbm;
		for( cnt = 0; cnt < CONFIG_SYS_NAND_MAX_CHIPS; cnt++ )
		{
			struct nand_bbm		*psBbm;
			
			psBbm = &PGpbc->chip.psBbm[cnt];
			psBbm->btBlocks = aBtBlocks[cnt];
			psBbm->mtBlocks = aMtBlocks[cnt];
			psBbm->bbmPages = -1;
			psBbm->bbMapSize = -1;
			psBbm->bbListSize = -1;
			psBbm->usedMtBlocks = -1;
			psBbm->altBlocks = -1;
			psBbm->pBbMap = NULL;
			psBbm->psBbList = NULL;
		}
		
		for( cnt = 0; cnt < PGpbc->chip.numchips; cnt++ )
		{
			struct nand_bbm		*psBbm;
			
			psBbm = &PGpbc->chip.psBbm[cnt];
			psBbm->bbMapSize = ((1 << (PGpbc->chip.chip_shift - PGpbc->chip.bbt_erase_shift)) + 8 * 4 - 1)
						/ (8 * 4) * 4;
			psBbm->bbListSize = sizeof(struct nand_bbList) * psBbm->mtBlocks;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			psBbm->bbmPages = (psBbm->bbMapSize + psBbm->bbListSize + 4 * 2
						+ sizeof(PGpbc->chip.bbt_td->pages[0]) + sizeof(PGpbc->chip.bbt_md->pages[0])
						+ (1 << PGpbc->chip.page_shift) + sizeof(PGpbc->chip.bbt_td->len)
						+ roundup(sizeof(PGpbc->chip.bbt_td->version[0]), sizeof(u32)) - 1)
						/ (1 << PGpbc->chip.page_shift);
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			psBbm->bbmPages = (psBbm->bbMapSize + psBbm->bbListSize + 4 * 2
						+ sizeof(PGpbc->chip.bbt_td->pages[0]) + sizeof(PGpbc->chip.bbt_md->pages[0])
						+ (1 << PGpbc->chip.page_shift) - 1) / (1 << PGpbc->chip.page_shift);
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG*/
			psBbm->pBbMap = kmalloc( psBbm->bbMapSize, GFP_KERNEL );
			if( !psBbm->pBbMap ){
				return -ENOMEM;
			}
			psBbm->psBbList = kmalloc( psBbm->bbListSize, GFP_KERNEL );
			if( !psBbm->psBbList ){
				return -ENOMEM;
			}
		}
		PGpbc->chip.pBbmBuf = vmalloc( PGpbc->chip.psBbm[0].bbmPages * (PGpbc->mtd->writesize + PGpbc->mtd->oobsize) );
		if( !PGpbc->chip.pBbmBuf ){
			return -ENOMEM;
		}
		PGpbc->chip.pBlockBuf = vmalloc( (1 << (PGpbc->chip.bbt_erase_shift - PGpbc->chip.page_shift))
					* (PGpbc->mtd->writesize + PGpbc->mtd->oobsize) );
		if( !PGpbc->chip.pBlockBuf ){
			return -ENOMEM;
		}
		
		return 0;
	}


	static void nand_gpbc_freeBbmResources( struct nand_gpbc_info *PGpbc )
	{
		u32		cnt;
		
		if( PGpbc->chip.pBlockBuf ){
			vfree( PGpbc->chip.pBlockBuf );
		}
		if( PGpbc->chip.pBbmBuf ){
			vfree( PGpbc->chip.pBbmBuf );
		}
		for( cnt = 0; cnt < PGpbc->chip.numchips; cnt++ )
		{
			struct nand_bbm		*psBbm;
			
			psBbm = &PGpbc->chip.psBbm[cnt];
			if( psBbm->pBbMap ){
				kfree(psBbm->pBbMap);
			}
			if( psBbm->psBbList ){
				kfree(psBbm->psBbList);
			}
		}
	}
#endif	//CONFIG_MTD_NAND_MN2WS_BB


static void nand_gpbc_initInfo1( struct nand_gpbc_info *PGpbc )
{
	PGpbc->nist[0] = NAND_GPBC_NIST0;
	PGpbc->nist[1] = NAND_GPBC_NIST1;
	PGpbc->nie[0] = NAND_GPBC_NIE0;
	PGpbc->nie[1] = NAND_GPBC_NIE1;
}


int board_nand_init_gpbc( struct mtd_info *PMtd )
{
	struct nand_chip	*PChip;
	int err = 0;
	struct nand_gpbc_info *PGpbc;
	#ifdef	CONFIG_MTD_CMDLINE_PARTS
		int num_parts;
	#endif	/* CONFIG_MTD_CMDLINE_PARTS */
	int		status;
	
	
	PGpbc = kzalloc( sizeof(struct nand_gpbc_info), GFP_KERNEL );
	if( !PGpbc ){
		err = -ENOMEM;
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		goto out;
	}
	
	PGpbc->mtd = PMtd;
	PChip = &PGpbc->chip;
	PChip->flags = 0;
	PChip->buffers = kmalloc( sizeof(*PChip->buffers), GFP_KERNEL );
	if( !PChip->buffers ){
		err = -ENOMEM;
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		goto out_err;
	}
	
	PGpbc->regBase = (void *)NAND_GPBC_UNIT_REG_BASE;
	if( !PGpbc->regBase ){
		err = -ENOMEM;
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		goto out_err;
	}
	PGpbc->memBase = (void *)NAND_GPBC_UNIT_MEM_BASE;
	if( !PGpbc->memBase ){
		err = -ENOMEM;
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		goto out_err;
	}
	
	nand_gpbc_initInfo1( PGpbc );
	
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
		PGpbc->irq = NAND_GPBC_IRQ_ID;
		PGpbc->event = 0;
		init_waitqueue_head( &PGpbc->queue );
		nand_gpbc_write32( 0, PGpbc->regBase + PGpbc->nie[0] );
		nand_gpbc_write32( 0, PGpbc->regBase + PGpbc->nie[1] );
		nand_gpbc_write32( NAND_GPBC_NGINTE__INTEN, PGpbc->regBase + NAND_GPBC_NGINTE );
		mn_intc_clear( PGpbc->irq );
		status = request_irq( PGpbc->irq, nand_gpbc_irq, 0, "nand-gpbc", PGpbc );
		if( status ){
			err = status;
			printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
			goto out_irq;
		}
	#endif
	
	status = nand_gpbc_initHw1( PGpbc );
	if( status ){
		err = status;
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		goto out_err2;
	}
	
	nand_gpbc_getId( PGpbc );
	
	err = nand_gpbc_initInfo2( PGpbc );
	if( err ){
		err = status;
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		goto out_err2;
	}
	
	#ifndef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		PGpbc->pPageBuf = kmalloc( PMtd->writesize + PMtd->oobsize, GFP_KERNEL );
		if( !PGpbc->pPageBuf ){
			err = -ENOMEM;
			printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
			goto out_err2;
		}
	#else
		if( NAND_GPBC_DMA_BURST < cache_line_size() ){
			if( cache_line_size() % NAND_GPBC_DMA_BURST ){
				BUG();
			}
			PGpbc->dmaAlignSize = cache_line_size();
		}else{
			if( NAND_GPBC_DMA_BURST % cache_line_size() ){
				BUG();
			}
			PGpbc->dmaAlignSize = NAND_GPBC_DMA_BURST;
		}
		PGpbc->pPageBufRaw = kmalloc( ALIGN(PMtd->writesize + PMtd->oobsize, PGpbc->dmaAlignSize) + PGpbc->dmaAlignSize, GFP_KERNEL );
		if( !PGpbc->pPageBufRaw ){
			err = -ENOMEM;
			printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
			goto out_err2;
		}
		
#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ
		PGpbc->cmd_desc = (struct nand_gpbc_cmd_desc_t *)CONFIG_NAND_CMD_DESC_ADDR;
		if (!PGpbc->cmd_desc) {
			err = -ENOMEM;
			printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
			goto out_err2;
		}
		PGpbc->cmd_desc_phys = (u32)virt_to_phys(PGpbc->cmd_desc);
#endif /* CONFIG_MTD_NAND_MN2WS_GPBC_CDMA_READ */
	#endif
	
	nand_gpbc_initHw2( PGpbc );
	
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		if( PAGE_SIZE < cache_line_size() ){
			BUG();
		}
		if( PAGE_SIZE % cache_line_size() ){
			BUG();
		}
		if( PMtd->writesize < cache_line_size() ){
			BUG();
		}
		if( PMtd->writesize % cache_line_size() ){
			BUG();
		}
		if( PAGE_SIZE < PMtd->writesize ){
			BUG();
		}
		if( PAGE_SIZE % PMtd->writesize ){
			BUG();
		}
		PGpbc->pPageBuf = (void *)ALIGN((u32)PGpbc->pPageBufRaw, PGpbc->dmaAlignSize);
		if( (u32)PGpbc->pPageBuf % cache_line_size() ){
			BUG();
		}
		PGpbc->pageBufPhys = virt_to_phys( PGpbc->pPageBuf );
	#endif
	
	#ifdef CONFIG_MTD_NAND_MN2WS_BB
		err = nand_gpbc_allocBbmResources( PGpbc );
		if( err ){
			printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
			goto out_bbm;
		}
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
	
	if( !(PChip->options & NAND_SKIP_BBTSCAN) ){
		err = nand_default_bbt( PMtd );
		if( err ){
			printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
			goto out_bbt;
		}
	}
	
	#ifdef CONFIG_MTD_PARTITIONS
		#ifdef	CONFIG_MTD_CMDLINE_PARTS
			num_parts = parse_mtd_partitions(PGpbc->mtd,
							 PData->chip.part_probe_types,
							 &PGpbc->parts, 0);
			if (num_parts > 0) {
				err = add_mtd_partitions(PGpbc->mtd, PGpbc->parts, num_parts);
				if( err ){
					printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
					goto out_part;
				}
			}else{
				err = -EINVAL;
				printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
				goto out_part;
			}
		#else	//CONFIG_MTD_CMDLINE_PARTS
			/* platform specific function to set partitions */
			if (PData->chip.set_parts) {
				PData->chip.set_parts(PGpbc->mtd->size, &PData->chip);
			}
			if (PData->chip.partitions) {
				PGpbc->parts = PData->chip.partitions;
				err = add_mtd_partitions(PGpbc->mtd, PGpbc->parts,
							 PData->chip.nr_partitions);
				if( err ){
					printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
					goto out_part;
				}
			}
		#endif	//CONFIG_MTD_CMDLINE_PARTS
	#endif	//CONFIG_MTD_PARTITIONS
	
	status = add_mtd_device(PGpbc->mtd);
	if( status ){
		err = -1;
		printk( "\n## ERROR %s %s %d ##\n", __FILE__, __func__, __LINE__ );
		goto out_mtd;
	}
	goto out;
	
out_mtd:
	#ifdef CONFIG_MTD_PARTITIONS
		/* Deregister partitions */
		del_mtd_partitions( PMtd );
	out_part:
		#ifdef CONFIG_MTD_CMDLINE_PARTS
			/* freeing partition info if necessary */
			if( PGpbc->parts && PGpbc->parts != PData->chip.partitions) {
				kfree( PGpbc->parts );
			}
		#endif //CONFIG_MTD_CMDLINE_PARTS
	#endif	//CONFIG_MTD_PARTITIONS
out_bbt:
	if( PChip->bbt ){
		kfree( PChip->bbt );
	}
	#ifdef	CONFIG_MTD_NAND_MN2WS_BB
	out_bbm:
		nand_gpbc_freeBbmResources( PGpbc );
	#endif	//CONFIG_MTD_NAND_MN2WS_BB
out_err2:
	#ifndef CONFIG_MTD_NAND_MN2WS_GPBC_DMA
		if( PGpbc->pPageBuf ){
			kfree( PGpbc->pPageBuf );
		}
	#else
		if( PGpbc->pPageBufRaw ){
			kfree( PGpbc->pPageBufRaw );
		}
	#endif
	#ifdef CONFIG_MTD_NAND_MN2WS_GPBC_IRQ
		free_irq( PGpbc->irq, PGpbc );
	out_irq:
		nand_gpbc_write32( 0, PGpbc->regBase + NAND_GPBC_NGINTE );
	#endif
out_err:
	if( PChip->buffers ){
		kfree( PChip->buffers );
	}
	if( PGpbc ){
		kfree( PGpbc );
	}
out:
	
	return err;
}


