/* Simple IPv4 Link-Local addressing (see <http://www.zeroconf.org/>)
 * @(#)llip.c, 1.5, Copyright 2003 by Arthur van Hoff (avh@strangeberry.com)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * See <http://www.gnu.org/copyleft/lesser.html>
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 */
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <sys/wait.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <net/ethernet.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <linux/if_packet.h>
#include <linux/sockios.h>
#include <time.h>
#include <assert.h>
#include <sys/ioctl.h>

#include <errno.h>

#include "aip.h"
#include "AVH-IPv4LL.h"

#define LINKLOCAL_ADDR          0xa9fe0000
#define LINKLOCAL_MASK          0xFFFF0000
#define FAILURE_INTERVAL        (14000)

#define DEFAULT_INTERFACE       "eth0"

/*** constant definition of RFC3927 */
#define PROBE_WAIT		    (1000)
#define PROBE_NUM		    (3)
#define PROBE_MIN		    (1000)
#define PROBE_MAX		    (2000)
#define ANNOUNCE_NUM		(2)
#define ANNOUNCE_INTERVAL	(2000)
#define ANNOUNCE_WAIT		(2000)
#define MAX_COLLISIONS		(10)
#define RATE_LIMIT_INTERVAL	(60000) /* msec */
#define DEFEND_INTERVAL     (10)    /* sec */

#define PIPE_FD_INDEX (0)
#define ARP_FD_INDEX  (1)

#define AVH_MSG_BUF_SIZE (256)

static struct in_addr null_ip = {0};
static struct ether_addr null_addr = {{0, 0, 0, 0, 0, 0}};
static struct ether_addr broadcast_addr = {{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}};

/**
 * ARP packet.
 */
struct arp_packet {
	struct ether_header hdr;
	struct arphdr arp;
	struct ether_addr source_addr;
	struct in_addr source_ip;
	struct ether_addr target_addr; 
	struct in_addr target_ip;
	unsigned char pad[18];
} __attribute__ ((__packed__));

typedef enum _avh_state_
{
	AVH_INIT,
	AVH_PROBE,
	AVH_ANNOUNCE,
	AVH_READY
}dAVHState_e;

typedef struct _svh_info_
{
	struct sockaddr saddr;
	struct ether_addr addr;
	struct pollfd fds[2];
	int32 conflicts;
	int32 defend;
	int32 iNextTimeout;
	struct timeval tTimeoutStart;
	struct timeval tInitTv;
	dAVHState_e eState;
	int nprobes;
	int nannounces;
	uint8 aucIfName[IFNAMSIZ + 1];
	uint8 aucInitAddr[IPV4_ADDR_LEN];
	boolean fHaveInitAddr;
}dAVHInfo_t;

static dAVHInfo_t tAvhInfo;

static void vRunAutoIp(void);
static void vAutoIpMain(int iPoolRet,uint32 uiAssignedAddr);
static int socket_arp(struct sockaddr *saddr, char *intf);
static void millisleep(unsigned int ms);
static void arp(int fd, 
				struct sockaddr *saddr, 
				int op,
				struct ether_addr *source_addr, struct in_addr source_ip,
				struct ether_addr *target_addr, struct in_addr target_ip);
static int elapse_time(struct timeval *start);
static void update_timeout(void);
static void set_timeout(int iTimeout);
static int rand_time(int from, int to);


void *vAutoIpTask(void *pvArg)
{
	int32 iPipeFd;
	
	iPipeFd = *(int32*)pvArg;

    tAvhInfo.fds[PIPE_FD_INDEX].fd     = iPipeFd;
    tAvhInfo.fds[PIPE_FD_INDEX].events = POLLIN | POLLERR;

	vRunAutoIp();

	/* ˤϤʤ*/

	return NULL;
}

void vAvhInitState(void)
{
	tAvhInfo.eState     = AVH_INIT;
	tAvhInfo.nprobes    = 0;
	tAvhInfo.nannounces = 0;
}

int iAvhInitParams(char *pucInterFace)
{
	struct ifreq ifr;

	int fd;

	fd = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ARP));
	if ( fd  < 0) 
	{
		eDbgPrintf(DBG_FATAL_ERROR,"<AVH> iAvhInitParams sock open err=%d\n",errno);
		return S_FAILURE;;
	}
	memset(tAvhInfo.aucIfName,0,IFNAMSIZ + 1);
	strncpy((char*)tAvhInfo.aucIfName,(char*)pucInterFace,IFNAMSIZ);

	/* get the ethernet address of the interface */
	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_name, pucInterFace, sizeof(ifr.ifr_name));

	if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0) 
	{
		eDbgPrintf(DBG_FATAL_ERROR,"ioctl failed(SIOCGIFHWADDR) err=%d\n",errno);	
		close(fd);
		return S_FAILURE;
	}
	memcpy(&(tAvhInfo.addr), &ifr.ifr_hwaddr.sa_data, ETHER_ADDR_LEN);

    /* initialize pseudo random selection of IP addresses */
	srandom((tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-4] << 24) |
			(tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-3] << 16) |
			(tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-2] <<  8) |
			(tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-1] <<  0));

	tAvhInfo.conflicts             = 0;
	tAvhInfo.defend                = 0;
	tAvhInfo.fds[PIPE_FD_INDEX].fd = -1;
	tAvhInfo.fds[ARP_FD_INDEX].fd  = -1;
	tAvhInfo.eState                = AVH_INIT;
	tAvhInfo.nprobes               = 0;
	tAvhInfo.nannounces            = 0;
	tAvhInfo.fHaveInitAddr         = FALSE;
	memset(tAvhInfo.aucInitAddr,0,IPV4_ADDR_LEN);

	gettimeofday(&(tAvhInfo.tInitTv), NULL);

	close(fd);
	
	return S_SUCCESS;
}

char* pcAvhGetIfName( void )
{
	return tAvhInfo.aucIfName;
}

int iAvhInitParams2(char *pucInterFace)
{
	struct ifreq ifr;

	int fd;

	fd = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ARP));
	if ( fd  < 0) 
	{
		eDbgPrintf(DBG_FATAL_ERROR,"<AVH> iAvhInitParams sock open err=%d\n",errno);
		return S_FAILURE;;
	}
	memset(tAvhInfo.aucIfName,0,IFNAMSIZ + 1);
	strncpy((char*)tAvhInfo.aucIfName,(char*)pucInterFace,IFNAMSIZ);

	/* get the ethernet address of the interface */
	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_name, pucInterFace, sizeof(ifr.ifr_name));

	if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0) 
	{
		eDbgPrintf(DBG_FATAL_ERROR,"ioctl failed(SIOCGIFHWADDR) err=%d\n",errno);	
		close(fd);
		return S_FAILURE;
	}
	memcpy(&(tAvhInfo.addr), &ifr.ifr_hwaddr.sa_data, ETHER_ADDR_LEN);

    /* initialize pseudo random selection of IP addresses */
	srandom((tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-4] << 24) |
			(tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-3] << 16) |
			(tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-2] <<  8) |
			(tAvhInfo.addr.ether_addr_octet[ETHER_ADDR_LEN-1] <<  0));

	tAvhInfo.conflicts             = 0;
	tAvhInfo.defend                = 0;
//	tAvhInfo.fds[PIPE_FD_INDEX].fd = -1;
//	tAvhInfo.fds[ARP_FD_INDEX].fd  = -1;
	tAvhInfo.eState                = AVH_INIT;
	tAvhInfo.nprobes               = 0;
	tAvhInfo.nannounces            = 0;
	tAvhInfo.fHaveInitAddr         = FALSE;
	memset(tAvhInfo.aucInitAddr,0,IPV4_ADDR_LEN);

	gettimeofday(&(tAvhInfo.tInitTv), NULL);

	close(fd);
	
	return S_SUCCESS;
}

int iOpenArpSocket()
{

	if( tAvhInfo.fds[ARP_FD_INDEX].fd != -1)
	{/* Ǥ arp Ѥ socketopen  */
		eDbgPrintf(DBG_ERROR,"<AVH>: arp socket dubole open\n");
		return -1;
	}

	socket_arp(&(tAvhInfo.saddr), tAvhInfo.aucIfName);
    tAvhInfo.fds[ARP_FD_INDEX].events = POLLIN | POLLERR;

	return tAvhInfo.fds[ARP_FD_INDEX].fd;
}

void vCloseArpSocket()
{
	if(-1 != tAvhInfo.fds[ARP_FD_INDEX].fd)
	{
		close(tAvhInfo.fds[ARP_FD_INDEX].fd);
	}
	else
	{
		eDbgPrintf(DBG_ERROR,"<AVH>: vCloseArpSocket called but ARP FD is -1\n");
	}
	tAvhInfo.fds[ARP_FD_INDEX].fd = -1;

	return;
}

/**
 * Pick a random link local IP address.
 */
static void pick(struct in_addr *ip)
{
	/* 
	   256Ȳ256ϻѤǤʤ(RFC3927 2.1)
	   0x0000-0xfd00δ֤randäƺǸ0x0100ϰϤ碌Ƥ
	*/
	ip->s_addr = htonl(LINKLOCAL_ADDR | ((abs(random()) % 0xFD00) + 0x0100));

	vAvhPrintfMessageWithTime("pickup addr %s\n",inet_ntoa(*ip));

}

/**
 * Send out an ARP packet.
 */
static void arp(int fd, 
				struct sockaddr *saddr, 
				int op,
				struct ether_addr *source_addr, struct in_addr source_ip,
				struct ether_addr *target_addr, struct in_addr target_ip)
{
	struct arp_packet p;

	memset(&p, 0, sizeof(p));

	/* ether header */
	p.hdr.ether_type = htons(ETHERTYPE_ARP);
	memcpy(p.hdr.ether_shost, source_addr, ETH_ALEN);
	memcpy(p.hdr.ether_dhost, &broadcast_addr, ETH_ALEN);

	/* arp request */
	p.arp.ar_hrd = htons(ARPHRD_ETHER);
	p.arp.ar_pro = htons(ETHERTYPE_IP);
	p.arp.ar_hln = ETH_ALEN;
	p.arp.ar_pln = 4;
	p.arp.ar_op = htons(op);
	memcpy(&p.source_addr, source_addr, ETH_ALEN);
	memcpy(&p.source_ip, &source_ip, sizeof(p.source_ip));
	memcpy(&p.target_addr, target_addr, ETH_ALEN);
	memcpy(&p.target_ip, &target_ip, sizeof(p.target_ip));

	if (sendto(fd, &p, sizeof(p), 0, saddr, sizeof(struct sockaddr)) < 0) 
	{
		perror("sendto failed");
	}
}


/**
 * create socket and bind for arp
 */
static int socket_arp(struct sockaddr *saddr, char *intf)
{
	int fd;

	assert(	(tAvhInfo.fds[ARP_FD_INDEX].fd == -1) );

	if ((fd = socket(PF_PACKET, SOCK_PACKET, htons(ETH_P_ARP))) < 0) 
	{
		eDbgPrintf(DBG_FATAL_ERROR,"<AVH> sock open err=%d\n",errno);
		return -1;
	}

	memset(saddr,0,sizeof(*saddr));
	strncpy(saddr->sa_data,intf,sizeof(saddr->sa_data));
	
	/* bind to the ARP socket */
	if (bind(fd, saddr, sizeof(*saddr)) < 0) 
	{
		eDbgPrintf(DBG_FATAL_ERROR,"<AVH> bind failed err=%d\n",errno);
		return -1;
    }

	tAvhInfo.fds[ARP_FD_INDEX].fd = fd;
	
	return fd;
}

/**
 * main program
 */
static void vRunAutoIp(void)
{
	int32 iFdNum = 1;	
	int32 iPoolRet;
	
	set_timeout(-1);

	eDbgPrintf(DBG_LOG,"<AVH> auto ip task start\n");

	while (TRUE)
	{
		boolean isReady;

		tAvhInfo.fds[PIPE_FD_INDEX].events = POLLIN | POLLERR;
		tAvhInfo.fds[PIPE_FD_INDEX].revents = 0;
		
		tAvhInfo.fds[ARP_FD_INDEX].events = POLLIN | POLLERR;
		tAvhInfo.fds[ARP_FD_INDEX].revents = 0;

		iPoolRet = poll(tAvhInfo.fds, iFdNum, tAvhInfo.iNextTimeout);

		if(1 == iPoolRet)
		{
			if ((tAvhInfo.fds[PIPE_FD_INDEX].revents & POLLIN) != 0) 
			{   /* Message from other task */
				uint8 aucTmpBuf[1];
				vAvhPrintfMessageWithTime("recv control message\n");

				read(tAvhInfo.fds[PIPE_FD_INDEX].fd, aucTmpBuf, sizeof(aucTmpBuf));/* dumy read */
			}
		}

		
		if( fAipStateCheck( &isReady, 
							&(tAvhInfo.fHaveInitAddr),
							tAvhInfo.aucInitAddr) )
		{/* fAipStateCheckdisable֤λsocketcloseԤ */
			uint32 uiAssignedAddr = 0;
			if( tAvhInfo.fHaveInitAddr )
			{/* ̤ꤵ줿ɥ쥹μФ */

				if(AVH_INIT != tAvhInfo.eState  )
				{
					eDbgPrintf(DBG_ERROR,"<AVH> have init address but tAvhInfo.eState  != AVH_INIT \n");
				}
#ifdef ENDIAN_IS_LITTEL
				uiAssignedAddr = tAvhInfo.aucInitAddr[3];
				uiAssignedAddr = (uiAssignedAddr << 8 ) | tAvhInfo.aucInitAddr[2]; 
				uiAssignedAddr = (uiAssignedAddr << 8 ) | tAvhInfo.aucInitAddr[1]; 
				uiAssignedAddr = (uiAssignedAddr << 8 ) | tAvhInfo.aucInitAddr[0]; 
#else /* d ENDIAN_IS_LITTEL. BIG ENDIAN */
				uiAssignedAddr = tAvhInfo.aucInitAddr[0];
				uiAssignedAddr = (uiAssignedAddr << 8 ) | tAvhInfo.aucInitAddr[1]; 
				uiAssignedAddr = (uiAssignedAddr << 8 ) | tAvhInfo.aucInitAddr[2]; 
				uiAssignedAddr = (uiAssignedAddr << 8 ) | tAvhInfo.aucInitAddr[3]; 
#endif /* d ENDIAN_IS_LITTEL */
			}
			if(isReady)
			{
				tAvhInfo.eState = AVH_READY;
			}
			iFdNum = 2;
			vAutoIpMain(iPoolRet,uiAssignedAddr);
		}
		else
		{
			set_timeout(-1);
			iFdNum = 1;
		}
	}
}
				
static void vAutoIpMain(int iPoolRet,uint32 uiAssignedAddr)
{			
	static struct arp_packet recv_arp;
	static struct in_addr ip = {0};
	static boolean fFirstPicked = FALSE;

	int iPool;

	if(0 != uiAssignedAddr)
	{
		fFirstPicked = TRUE;
		ip.s_addr = uiAssignedAddr;
		vAvhPrintfMessageWithTime("<AVH> set aip address with enable %s\n",inet_ntoa(ip));
	}
	else if(!fFirstPicked)
	{
		fFirstPicked = TRUE;
		pick(&ip);
	}
	if(AVH_INIT == 	tAvhInfo.eState) 
	{
		int32 timeout;
		timeout = rand_time(0, PROBE_WAIT);
		vAvhPrintfMessageWithTime("AVH State is Init.sleep %u msec before send probe\n",timeout);

		millisleep(timeout);
		iPool = 0;
	}
	else
	{
		iPool = iPoolRet;
	}
	if( tAvhInfo.fds[ARP_FD_INDEX].fd == -1 )
	{
		vAvhPrintfMessageWithTime("fd error\n");
		millisleep(PROBE_WAIT);
		return;
	}

	switch (iPool) 
	{
	case 0:/* timeout */
		if (tAvhInfo.nprobes < PROBE_NUM) 
		{
			/* ARP probe */
			vAvhPrintfMessageWithTime("ARP probe %s\n",inet_ntoa(ip));

			arp( tAvhInfo.fds[ARP_FD_INDEX].fd, 
				 (struct sockaddr *)&(tAvhInfo.saddr), 
				 ARPOP_REQUEST, 
				 &(tAvhInfo.addr), 
				 null_ip, 
				 &null_addr, 
				 ip);
				
			if( 0 == tAvhInfo.nprobes)
			{
				tAvhInfo.eState = AVH_PROBE;
			}
			(tAvhInfo.nprobes)++;
			if (tAvhInfo.nprobes < PROBE_NUM)
			{/* PROBE_NUMPROBEѤ ARPꤲ*/
				set_timeout( rand_time(PROBE_MIN, PROBE_MAX) );
			}
			else 
			{/* probe եνλ Announce ե */
				set_timeout(ANNOUNCE_WAIT);
			}
		}
		else if (tAvhInfo.nannounces < ANNOUNCE_NUM) 
		{/* ARP announce */
			vAvhPrintfMessageWithTime("ARP announce %s\n", inet_ntoa(ip));
			
			if( 0 == tAvhInfo.nannounces )
			{
				tAvhInfo.eState = AVH_ANNOUNCE;
			}
			arp( tAvhInfo.fds[ARP_FD_INDEX].fd,
				 (struct sockaddr *)&(tAvhInfo.saddr),
				 ARPOP_REQUEST,
				 &(tAvhInfo.addr),
				 ip,
				 &(tAvhInfo.addr),
				 ip);
			
			(tAvhInfo.nannounces)++;
				
			if (ANNOUNCE_NUM == tAvhInfo.nannounces)/* address fix */
			{
				uint8 aucRet[IPV4_ADDR_LEN];
				tAvhInfo.conflicts = 0;
				set_timeout(-1);
#ifdef ENDIAN_IS_LITTEL
				aucRet[0] = (ip.s_addr)       & 0x000000FF;
				aucRet[1] = (ip.s_addr >> 8)  & 0x000000FF;
				aucRet[2] = (ip.s_addr >> 16) & 0x000000FF;
				aucRet[3] = (ip.s_addr >> 24) & 0x000000FF;
#else /* d ENDIAN_IS_LITTEL. BIG ENDIAN */
				aucRet[0] = (ip.s_addr >> 24) & 0x000000FF;
				aucRet[1] = (ip.s_addr >> 16) & 0x000000FF;
				aucRet[2] = (ip.s_addr >> 8)  & 0x000000FF;
				aucRet[3] = (ip.s_addr)       & 0x000000FF;
#endif /* d ENDIAN_IS_LITTEL */

				vAipNotifyAddress(aucRet,IPV4_ADDR_LEN);
				tAvhInfo.eState = AVH_READY;
			}
			else
			{
				set_timeout(ANNOUNCE_INTERVAL);
			}
		}
		else 
		{/* 
		  * state error
		  * If not probe/announce/init state, timeout will not occure
		  * 
		  */
			vAvhPrintfMessageWithTime("state error(state is).use %s\n",inet_ntoa(ip),tAvhInfo.eState);
			set_timeout(-1);
		}
		break;
	case 1:/* i/o event */
		update_timeout(); /* set next timeout*/
		if ( (tAvhInfo.fds[ARP_FD_INDEX].revents & POLLIN) == 0) 
		{
			if (tAvhInfo.fds[ARP_FD_INDEX].revents & POLLERR) 
			{
				eDbgPrintf(DBG_FATAL_ERROR,"<AVH>I/O error\n");
			}
			break;
		}
		
		/* read ARP packet */
		if (recv(tAvhInfo.fds[ARP_FD_INDEX].fd, &recv_arp, sizeof(recv_arp), 0) < 0) 
		{
			/* How to handle */
			eDbgPrintf(DBG_FATAL_ERROR,"<AVH>recv failed\n");
			break;
		}

		if ( ntohs(recv_arp.hdr.ether_type) != ETHERTYPE_ARP )
		{/* Ether type ARPʳ */
			break;
		}

		if ( (ntohs(recv_arp.arp.ar_op) != ARPOP_REQUEST) && 
			 (ntohs(recv_arp.arp.ar_op) != ARPOP_REPLY) )
		{/* ARPRequest/Replayʳ */
			break;
		}

		/* confilict  */
		if( ( recv_arp.source_ip.s_addr == ip.s_addr) && 
			( memcmp(&(tAvhInfo.addr), &(recv_arp.source_addr), ETH_ALEN) != 0)) 
		{
			uint8 aucZeroAddr[4] = {0,0,0,0};
			vAvhPrintfMessageWithTime(" ARP conflict %s\n", inet_ntoa(ip));
			
			if (AVH_READY == tAvhInfo.eState) {
				time_t now = time(0);
				
				if ((tAvhInfo.defend + DEFEND_INTERVAL) < now) 
				{
					vAvhPrintfMessageWithTime(" try send anounce ARP %s\n", inet_ntoa(ip));
					tAvhInfo.defend = now;
					arp( tAvhInfo.fds[ARP_FD_INDEX].fd, 
						 (struct sockaddr *)&(tAvhInfo.saddr), 
						 ARPOP_REQUEST, 
						 &(tAvhInfo.addr), 
						 ip, 
						 &(tAvhInfo.addr), 
						 ip);

					set_timeout(-1);
					break;
				}

				tAvhInfo.defend = now;
				
				vAipNotifyAddress(aucZeroAddr,4);
			}
			(tAvhInfo.conflicts)++;

			tAvhInfo.eState = AVH_INIT;

			if (tAvhInfo.conflicts >= MAX_COLLISIONS)
			{/* confilictĶ */
				vAvhPrintfMessageWithTime("autoip max collision over\n");
				vAipForceDisable();
				vAipNotifyAddress(aucZeroAddr,4);
				set_timeout(-1);
				tAvhInfo.nprobes    = 0;
				tAvhInfo.nannounces = 0;
				tAvhInfo.conflicts = 0;
				tAvhInfo.defend = 0;
			}
			else
			{
				pick(&ip);

				vAvhPrintfMessageWithTime("Next pickup for conflict %s\n", inet_ntoa(ip));
				set_timeout(0);
				tAvhInfo.nprobes       = 0;
				tAvhInfo.nannounces    = 0;
			}
		}
		break;
	default:
		eDbgPrintf(DBG_FATAL_ERROR,"<AVH> poll failed\n");
		break;
	}
	
	return;
}

/**
 * generate a random time.
 */
static int rand_time(int from, int to)
{
	if (from == to)
	{
		return from;
	}
	else
	{
		return from + (random() % (1 + to - from));
	}
}


static void set_timeout(int iTimeout)
{
	tAvhInfo.iNextTimeout = iTimeout;
	if( iTimeout > 0 )
	{
		gettimeofday( &(tAvhInfo.tTimeoutStart), NULL);
	}
	
	return;
}

static void update_timeout(void)
{
	int diff;
	if ( tAvhInfo.iNextTimeout <= 0)
	{
		/* no need update */
		return;
	}
	diff = tAvhInfo.iNextTimeout  - elapse_time(&(tAvhInfo.tTimeoutStart));
	if(diff > 0)
	{
		tAvhInfo.iNextTimeout = diff;
	}
	else
	{
		tAvhInfo.iNextTimeout = 0;
	}
}

static void millisleep(unsigned int ms)
{
    struct timespec req;
    struct timespec rem;
    
    req.tv_sec = ms / 1000;
    req.tv_nsec = (ms % 1000) * 1000000;
    while (nanosleep(&req, &rem) == -1)
	{
		if (errno == EINTR) 
		{
			req = rem;
			continue;
		} 
		else
		{
			break;
		}
    }
}

static int elapse_time(struct timeval *start)
{
	struct timeval tv;
	int ms;

	gettimeofday(&tv,NULL);

	if( tv.tv_usec < start->tv_usec )
	{
		tv.tv_sec--;
		tv.tv_usec += 1000000;
	}
	
	tv.tv_sec  -= start->tv_sec;
	tv.tv_usec -= start->tv_usec;
	ms = tv.tv_sec * 1000 + tv.tv_usec / 1000;

	return ms;
}


void vAvhPrintfMessageWithTime(char *pcFmt,...)
{
	va_list list;
	char acMessageBuf[AVH_MSG_BUF_SIZE];
	int elapse;
	va_start(list, pcFmt); 

	vsnprintf(acMessageBuf,AVH_MSG_BUF_SIZE, pcFmt, list); 

	elapse = elapse_time( &(tAvhInfo.tInitTv) );
	eDbgPrintf(DBG_INFO,"<AVH>%08lu:%s",elapse,acMessageBuf);

	va_end(list); 
}


