/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/****************************************************************************/
#include "fmb_hw.h"
#include "fmb_sys_abs.h"

PREPARE_DEBUG_FOR_FUNC();

/********************************************************/
/*                  GROBAL VALUES                       */
/********************************************************/
static struct fmb_core_private  s_Fmb_core_private_list[FMB_MAX_CARDS];

/********************************************************/
/*                  FUNCTION PROTOTYPE                  */
/********************************************************/
static void fmb_core_private_data_release( struct fmb_core_private* priv_p );

/********************************************************/
/*                                                      */
/*  LSI core register processing function               */
/*                                                      */
/********************************************************/


/********************************************************/
/*                                                      */
/*  Message buffer processing function                  */
/*                                                      */
/********************************************************/


/********************************************************/
/*                                                      */
/*  LSI core dependence Interrupt process function      */
/*                                                      */
/********************************************************/

/**
*   @brief      MB86H57 Codec TACK interrupt TaskLet
*   @param[in]  data           the pointer to the HW private data
*   @param[in]  cack_id        the value of command ack id
*   @param[in]  sub_cack_id    the value of sub command ack id
*   @return     None
*   @note       None
*   @attention  None
*/
void fmb_core_tack_func( unsigned long data, unsigned short cack_id, unsigned short sub_cack_id, unsigned char continuity_counter ) /* call interrupt message in when recived ack message */
{
    struct fmb_hard_private*        hw_priv_p;
    struct fmb_core_private*        priv_p;
    int                             minor;

    hw_priv_p = (struct fmb_hard_private *)data;
    PRIV_P_CHECK_VOID( hw_priv_p, -ENODEV );

    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK_VOID( priv_p, -ENODEV );
    minor = priv_p->minor;
    MSG( INTERNAL_FUNC, minor, "START" );

    if (priv_p->sys_cmd_send_flag == FMB_SYS_CMD_SEND_ASYNC) {
        /* not implemented */
    }
    else {
        priv_p->sys_cmd_ack_info.id         = cack_id;
        priv_p->sys_cmd_ack_info.sub_id     = sub_cack_id;
        priv_p->sys_cmd_ack_info_ccnt       = continuity_counter;
        priv_p->sys_cmd_ack_rcv_flag        = FMB_SYS_CMD_ACK_RECEIVE;

        if(sub_cack_id == 0x80){
            /* ACK ERROR*/
            MSG( DEBUG_LVL, minor, "ACK ERROR:cack_id = %02x, sub_cack_id = %02x",cack_id, sub_cack_id);
        }
        FMB_SYS_ABS_wake_up_interruptible( &priv_p->sys_cmd_ack_wait_sync );
    }

    priv_p->sys_cmd_send_flag = FMB_SYS_CMD_SEND_NONE;

    MSG( INTERNAL_FUNC, minor, "END" );
}

/********************************************************/
/*                          */
/*  LSI core dependence initialization function */
/*                          */
/********************************************************/
/**
 *  @brief      Initializes the core private data.
 *  @param[in]  hw_priv_p   the pointer to the HW private data
 *  @param[in]  priv_p      the pointer to the core private data
 *  @return     0   success
 *  @return     -1  fail
 *  @note       None
 *  @attention  None
 */
static int fmb_core_private_data_init(  struct fmb_hard_private*    hw_priv_p,
                                        struct fmb_core_private*    priv_p )
{
    int ret;
    int minor;
    int minor_num;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    PRIV_P_CHECK( priv_p,    -ENODEV );
    
    ret = 0;
    
    minor     = hw_priv_p->minor;
    minor_num = hw_priv_p->minor_num;
    MSG( INTERNAL_FUNC, minor, "START" );
#ifdef TV_LINUX_FUNCTION
    priv_p->lock_core_priv          = (spinlock_t)SPIN_LOCK_UNLOCKED;
#else
    priv_p->lock_core_priv          = SPIN_LOCK_UNLOCKED;
#endif
#if 0 /* During a function add */
    priv_p->encode_mode             = FMB_DATA_ENCODE_TOP;
    priv_p->notify_mode             = FMB_DATA_NOTIFY_NONE;
#endif
    priv_p->sys_cmd_ack_rcv_flag    = FMB_SYS_CMD_ACK_NONE;
    priv_p->sys_cmd_send_flag       = FMB_SYS_CMD_SEND_NONE;
    priv_p->fact_irq_type           = FMB_FACT_IRQ_NO_CANCEL;
    priv_p->fact_irq_stop_flag      = FMB_FACT_IRQ_NO_CANCEL;
    init_waitqueue_head( &priv_p->sys_cmd_ack_wait_sync );
    init_waitqueue_head( &priv_p->sys_msg_wait );
    init_waitqueue_head( &priv_p->eject_event_wait );
    init_waitqueue_head( &priv_p->fact_irq_wait );
    init_waitqueue_head( &priv_p->wait_bit_wait );
#ifdef MB86E61_FUNCTION
    init_waitqueue_head( &priv_p->boot_event_wait );
#endif /* MB86E61_FUNCTION */
    init_MUTEX( &priv_p->sys_cmd_sync_send_mutex );
    init_MUTEX( &priv_p->wait_fact_mutex );
    init_MUTEX( &priv_p->async_info_mutex );

    priv_p->minor                   = minor;
    priv_p->minor_num               = minor_num;
    priv_p->hw_priv_p               = hw_priv_p;
    priv_p->sys_cmd_continuity_counter = 0;

    MSG( INTERNAL_FUNC, minor, "END" );
    return 0;
}

/**
 *  @brief      Releases the core private data.
 *  @param[in]  priv_p  the pointer to the core private data
 *  @return     None
 *  @note       None
 *  @attention  None
 */
static void fmb_core_private_data_release( struct fmb_core_private* priv_p )
{
    int minor;

    PRIV_P_CHECK_VOID( priv_p, -ENODEV );
    
    minor = priv_p->minor;
    MSG( INTERNAL_FUNC, minor, "START" );

    MSG( INTERNAL_FUNC, minor, "END" );
}


/**
 *  @brief      The MB86H57-LSI part of the probe function.
 *  @param[in]  hw_priv_p   the pointer to the HW private data
 *  @return     0   success
 *  @return     -1  fail
 *  @note       None
 *  @attention  None
 */
int Fmb_core_probe( struct fmb_hard_private*    hw_priv_p )
{
    struct fmb_core_private*    priv_p;
    int                         ret;
    int                         minor;

    priv_p = NULL;
    ret = 0;
    minor = hw_priv_p->minor;
    
    MSG( INTERNAL_FUNC, minor, "START" );

    priv_p = &s_Fmb_core_private_list[hw_priv_p->minor_num];

    if ( priv_p->minor != -1 ) {
        MSG( INTERNAL_ERR_LVL, minor,
             "Specified minor has already been used. priv_p->minor = %d",
             priv_p->minor );
        return -1;
    }

    ret = fmb_core_private_data_init( hw_priv_p, priv_p );
    if (ret != 0) {
        MSG( INTERNAL_ERR_LVL, minor, "fmb_core_private_data_init() NG ret = %d", ret );
        return -1;
    }
    hw_priv_p->core_priv_p = priv_p;

    MSG( INTERNAL_FUNC, minor, "END" );

    return 0;
}


/**
 *  @brief      Kills tasklets and releases the core private data.
 *  @param[in]  priv_p  the pointer to the core private data
 *  @return     None
 *  @note       None
 *  @attention  None
 */
void Fmb_core_remove( struct fmb_core_private* priv_p )
{
    struct fmb_hard_private*    hw_priv_p;
    int                         minor;
    int                         minor_num;
    
    PRIV_P_CHECK_VOID( priv_p, -ENODEV );

    hw_priv_p = priv_p->hw_priv_p;
    minor     = priv_p->minor;
    minor_num = priv_p->minor_num;
    MSG( INTERNAL_FUNC, minor, "START" );

    fmb_core_private_data_release( priv_p );

    /* initialize the core private data */
    memset( &s_Fmb_core_private_list[minor_num], 0, sizeof( struct fmb_core_private ) );
    s_Fmb_core_private_list[minor_num].minor     = -1;
    s_Fmb_core_private_list[minor_num].minor_num = -1;

    MSG( INTERNAL_FUNC, minor, "END" );
}


/**
 *  @brief      The MB86H57-LSI part of the init_module function.
 *  @param      None
 *  @return     None
 *  @note       None
 *  @attention  None
 */
void Fmb_core_init_module( void )
{
    int cnt   = 0;
    int minor;

    minor = -1;

    MSG( INTERNAL_FUNC, minor, "START" );

    memset( s_Fmb_core_private_list, 0x00, sizeof( s_Fmb_core_private_list ) );

    for ( cnt = 0; cnt < FMB_MAX_CARDS; cnt++ ) {
        s_Fmb_core_private_list[cnt].minor = -1;
    }

    MSG( INTERNAL_FUNC, minor, "END" );
}
/**
 *  @brief      The MB86H57-LSI part of the irq status Check function.
 *  @param      None
 *  @return     None
 *  @note       None
 *  @attention  None
 */
int Fmb_core_lsi_get_irq_status( struct fmb_hard_private* hw_priv_p, unsigned long *val_status )
{
    int ret   = 0;
    int minor;
//    unsigned long val_status;
    unsigned short val_st1;
    unsigned short val_st1_mask;
    unsigned short val_st2;
    unsigned short val_xerror;

    int result_internal = 0;

    minor = -1;

    MSG( INTERNAL_FUNC, minor, "START" );

         /* get irq status */
        val_st1      = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_API_IRQST,  &result_internal ); if ( result_internal < 0 ) return result_internal;
        val_st2      = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_API_IRQST2, &result_internal ); if ( result_internal < 0 ) return result_internal;
        if ( ( val_st1 & FMB_FACT_STATE ) != 0 )
        {
            val_xerror = Fmb_hw_reg_read_lsi( hw_priv_p, FMB_REG_STATE, &result_internal );    if ( result_internal < 0 ) return result_internal;
            if ( val_xerror == 0xffff ) val_xerror = 1;
            else    val_xerror = 0;
        }
        else    val_xerror = 0;
        
        if ( val_xerror == 1 )
        {
            /* xerror : disable FMB_FACT_STATE */
            val_st1_mask = FMB_FACT_STATE;
            val_st1      &= ~val_st1_mask;
        }
        *val_status   = (unsigned long)val_st2 << ( 8 * 2 ) ;
        *val_status  += (unsigned long)val_st1;
        *val_status  &= (unsigned long)(~FMB_FACT_XERROR); /* erase IRQEN2_GPIO_BIT */

        if ( val_xerror == 1 ) *val_status  |= FMB_FACT_XERROR;

    
    MSG( INTERNAL_FUNC, minor, "END" );
    
    return ret;
}
