/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/****************************************************************************/
#include "fmb_api.h"

#ifndef VERSION
#define VERSION "?.?.?.?"
#endif

/********************************************************/
/*                  FUNCTION PROTOTYPE                  */
/********************************************************/

PREPARE_DEBUG();

MODULE_LICENSE( "GPL" );
MODULE_DESCRIPTION( "MB86H57_Driver" );
DECLARE_MUTEX( sm );

/********************************************************/
/*      GROBAL VALUES               */
/********************************************************/
static struct fmb_hard_private  s_Fmb_hard_private_list[FMB_MAX_CARDS];
static int                      s_Fmb_cards_active = 0;
static spinlock_t               s_Fmb_hw_lock   = SPIN_LOCK_UNLOCKED;



/********************************************************/
/*                          */
/*   register processing function       */
/*                          */
/********************************************************/

/**
*   @brief      Register Read to LSI
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*	@param[out] result          Processing result 0:normal (<0):abnormal     
*   @return     val             register read value
*   @note       None
*   @attention  None
*/  

unsigned short Fmb_hw_reg_read_lsi(struct fmb_hard_private* priv_p, unsigned long addr, int *result )
{
    unsigned short ret;
    
    ret = fmb_hw_device_reg_read( priv_p, addr, result );
    
    return ret;
}

/**
*   @brief      Register Read to LSI( 32bit )
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*	@param[out] result          Processing result 0:normal (<0):abnormal     
*   @return     val             register read value
*   @note       None
*   @attention  None
*/  

unsigned long Fmb_hw_reg_read_lsi32(struct fmb_hard_private* priv_p, unsigned long addr, int *result )
{
    unsigned long ret;
    
    ret = fmb_hw_device_reg_read32( priv_p, addr, result );
    
    return ret;
}


/**
*   @brief      Register Write to LSI
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*   @param[in]  data            register write value
*	@param[out] result          Processing result 0:normal (<0):abnormal     
*   @return     None
*   @note       None
*   @attention  None
*/  

void Fmb_hw_reg_write_lsi(struct fmb_hard_private* priv_p, unsigned long addr, unsigned short data, int *result )
{
    fmb_hw_device_reg_write( priv_p, addr, data, result );
    
    return;
}


#if 0 /* 1102 The support schedule when the future.  */
/**
*   @brief      Register Read to LSI Enhancing master
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI Enhancing master register address
*   @return     val             register read value
*   @note       None
*   @attention  None
*/  

unsigned long Fmb_hw_reg_read_lsi_master(struct fmb_hard_private* priv_p, unsigned long addr)
{
    unsigned long ret;
    
    ret = fmb_hw_device_reg_read_master( priv_p, addr );
    
    return ret;
}

/**
*   @brief      Register Write to LSI Enhancing master
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI Enhancing master register address
*   @param[in]  data            register write value
*   @return     None
*   @note       None
*   @attention  None
*/  
void Fmb_hw_reg_write_lsi_master(struct fmb_hard_private* priv_p, unsigned long addr, unsigned long data)
{
    fmb_hw_device_reg_write_master( priv_p, addr, data );
    
    return;
}

#endif /* 1102 The support schedule when the future.  */

/**
*   @brief      Register Read Modfi Write to LSI 32bit
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*   @param[in]  mask            mask value at the time of writing
*   @param[in]  data            register write value
*   @return     val             register read value
*   @note       None
*   @attention  None
*/
int Fmb_hw_reg_rmw_lsi32(struct fmb_hard_private* priv_p, unsigned long addr,
                          unsigned long mask, unsigned long data)
{
    int ret;
    
    ret = fmb_hw_device_reg_rmw32( priv_p, addr, mask, data );
    
    return ret;
}
/**
*   @brief  Check access area for lsi
*   @param[in]  priv_p          private data pointer
*   @param[in]  offset_addr     offset address to specified usb
*   @param[in]  access_size     Size of access area
*   @return     0               Normal end
*   @return     -EINVAL         The content of the parameter is illegal.
*   @note       None
*   @attention  None
*/

int Fmb_hw_check_addr_lsi(struct fmb_hard_private* priv_p, unsigned long offset_addr, unsigned long access_size)
{
    int             minor;
    unsigned long   tail_addr;

    PRIV_P_CHECK( priv_p, -ENODEV );
    minor = priv_p->minor;

    MSG(INTERNAL_FUNC, minor, "START");

    tail_addr = (offset_addr + access_size)*2;
    if ((offset_addr*2) > FMB_CORE_LSI_SIZE) {
        MSG(INTERNAL_ERR_LVL, minor, "Specified address is abnormal.");
        return -EINVAL;
    }
    if (tail_addr > FMB_CORE_LSI_SIZE) {
        MSG(INTERNAL_ERR_LVL, minor, "Specified address is abnormal.");
        return -EINVAL;
    }
    MSG(INTERNAL_FUNC, minor, "END");
    return 0;
}


struct fmb_hard_private*
fmb_hw_private_alloc()
{
    struct fmb_hard_private *priv_p;
    unsigned long   flags = 0;
    int             minor = 0;
    int             i;
    
    spin_lock_irqsave(&s_Fmb_hw_lock, flags);
    if (s_Fmb_cards_active >= FMB_MAX_CARDS) {
        spin_unlock(&s_Fmb_hw_lock);
        MSG(INTERNAL_ERR_LVL, minor, "No remainder of private data.");
        return NULL;
    }
    for(i=0;i<FMB_MAX_CARDS;i++)
    {
        if ( s_Fmb_hard_private_list[i].minor_num == -1 )  break;
    }

    if ( i < FMB_MAX_CARDS )
    {
        priv_p = &s_Fmb_hard_private_list[i];
        priv_p->minor_num = i;
        s_Fmb_cards_active++;
    }
    else
    {
        MSG(INTERNAL_ERR_LVL, minor, "No remainder of private data.(abnormal)");
        priv_p = NULL;
    }
    
    spin_unlock_irqrestore(&s_Fmb_hw_lock, flags);        

    return priv_p;
}

void
fmb_hw_private_free()
{
    s_Fmb_cards_active--;
    
    return;
}

/********************************************************/
/*                          */
/*  Hardware dependence initialization function */
/*                          */
/********************************************************/

/**
*   @brief      FMB Hardware Private Data initialize
*   @param[in]  priv_p          private data pointer
*   @return     None
*   @note       None
*   @attention  None
*/

void fmb_hw_private_data_init(struct fmb_hard_private* priv_p)
{
    int             minor;
    minor = priv_p->minor;

    MSG(INTERNAL_LOG, minor, "START");

    MSG(INTERNAL_LOG, minor, "END");
}

/**
*   @brief      FMB  Hardware Private Data release
*   @param[in]  priv_p          private data pointer
*   @return     None
*   @note       None
*   @attention  None
*/

void fmb_hw_private_data_release(struct fmb_hard_private* priv_p)
{
    int             minor;
    
    PRIV_P_CHECK_VOID( priv_p, -ENODEV );

    minor = priv_p->minor;

    MSG(INTERNAL_LOG, minor, "START");
    
    /* initialize the core private data */
    memset( priv_p, 0, sizeof(struct fmb_hard_private));
    priv_p->minor = -1;
    priv_p->minor_num = -1;

    MSG(INTERNAL_LOG, minor, "END");
}

/********************************************************/
/*  systemcall basic API Function           */
/********************************************************/

/**
*   @brief      FMB system call open()
*   @param[in]  *ip         Information in filesystem
*   @param[in]  *fp         File pointer
*   @return     0
*   @note       None
*   @attention  None
*/

static int fmb_hw_open(struct inode* ip_p, struct file* fp_p)
{
    int             minor;
    int ret;
    int i;
    
    minor = MINOR(ip_p->i_rdev);

    if ( minor < 0 ) return -ENODEV;
    
    MSG(DRIVER_FUNC_LVL, minor, "START");

    for(i=0;i<FMB_MAX_CARDS;i++)
    {
        if ( s_Fmb_hard_private_list[i].minor == minor ) break;
    }
    
    if ( i >= FMB_MAX_CARDS )
    {
        fp_p->private_data = NULL;
        return -ENODEV;
    }
    fp_p->private_data = &s_Fmb_hard_private_list[i];
    
    ret = fmb_hw_device_open( ip_p, fp_p );
    if ( ret != 0 ) return ret;

    MSG(DRIVER_FUNC_LVL, minor, "END");
    return 0;
}


/**
*   @brief      FMB system call close()
*   @param[in]  *ip         Information in filesystem
*   @param[in]  *fp         File pointer
*   @return     0
*   @note       None
*   @attention  None
*/

static int fmb_hw_release(struct inode* ip_p, struct file* fp_p)
{
    int             minor;
    struct fmb_hard_private*    priv_p = (struct fmb_hard_private*)fp_p->private_data;
    int ret;

    PRIV_P_CHECK( priv_p, -ENODEV );
    
    minor = priv_p->minor;

    MSG(DRIVER_FUNC_LVL, minor, "START");

//    ret = fmb_hw_usb_release( ip_p, fp_p );
    ret = fmb_hw_device_release( ip_p, fp_p );
    if ( ret != 0 ) return ret;

    MSG(DRIVER_FUNC_LVL, minor, "END");
    return 0;
}

/**
*   @brief      FMB system call ioctl()
*   @param[in]  *ip     Information in filesystem
*   @param[in]  *fp     File pointer
*   @param[in]  cmd     Command number from application
*   @param[in]  arg     Argument from application 
*   @return     0
*   @note       None
*   @attention  None
*/

static int fmb_hw_ioctl(struct inode* ip_p, struct file* fp_p,
             unsigned int cmd, unsigned long arg)
{
    int             minor;
    int             rc; 
    struct fmb_hard_private*    priv_p;

    priv_p = (struct fmb_hard_private*)fp_p->private_data;
    PRIV_P_CHECK( priv_p, -ENODEV );
    
    minor = priv_p->minor;
    rc = 0;
    
    MSG(DRIVER_FUNC_LVL, minor, "START");

    rc = fmb_hw_device_ioctl( priv_p, cmd, arg );
    if ( rc == -EINVAL )
    {
        /* ioctl() to FMB Codec */ 
        rc = Fmb_api_ioctl(priv_p, cmd, arg);
    }
    
    MSG(DRIVER_FUNC_LVL, minor, "END");
    return rc;
}

/********************************************************/
/*                          */
/*  Driver module initialization function       */
/*                          */
/********************************************************/
/*------------------------------------------------------*/
/* Data for driver registration                 */
/*------------------------------------------------------*/

struct fmb_hw_obj fmb_hw_obj_link = 
{
    .fops.ioctl   = fmb_hw_ioctl,
    .fops.open    = fmb_hw_open,
    .fops.release = fmb_hw_release,
};

/**
*   @brief      FMB Hardware initialize module
*   @param[in]  None
*   @return     0           Normal end
*   @return     -EBUSY      Can not module registe
                            Can not interrupt reuest
*   @note       None
*   @attention  None
*/  

static int __init fmb_hw_init_module(void)
{
    int         minor;
    int         cnt;
    int         rc;

    minor = -1;
    
    MSG(DRIVER_FUNC_LVL, minor, "START");

    memset( s_Fmb_hard_private_list, 0x00, sizeof( s_Fmb_hard_private_list ) );
    for (cnt = 0; cnt < FMB_MAX_CARDS; cnt++){
        s_Fmb_hard_private_list[cnt].minor     = -1;
        s_Fmb_hard_private_list[cnt].minor_num = -1;
    }
    
    Fmb_core_init_module();

    
    rc = fmb_hw_device_register();
    if(rc < 0){
        MSG(KERN_ERR_LVL, minor, "USB driver registration error");
        return -EBUSY;      
    }
    MSG(DRIVER_FUNC_LVL, minor, "END");
    return rc;
}

/**
*   @brief      FMB Hardware cleanup module
*   @param[in]  None
*   @return     None
*   @note       None
*   @attention  None
*/  

static void __exit fmb_hw_cleanup_module(void)
{
    int         minor;
    
    minor = -1;

    MSG(DRIVER_FUNC_LVL, minor, "START");
    
    
    fmb_hw_device_unregister();

    MSG(DRIVER_FUNC_LVL, minor, "END");
}

module_init(fmb_hw_init_module);
module_exit(fmb_hw_cleanup_module);

