/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/****************************************************************************/
#include "fmb_hw.h"
#include "fmb_core.h"
#include "fmb_hw_usb.h"
#include "fmb_drv.h"
#include "fmb_sys_abs.h"

#ifdef USE_UPPER_OS
#include <linux/smp_lock.h>
#include <linux/slab.h>
#endif /* USE_UPPER_OS */

#ifdef DEBUGON_USB_ACCESS
        char dbuf[1024];
#endif /* DEBUGON_USB_ACCESS */

static void fmb_hw_usb_polling_cleanup(struct fmb_hw_usb *dev);
static int fmb_hw_usb_polling_thread( void  * arg);

//PREPARE_DEBUG();
PREPARE_DEBUG_FOR_FUNC();

struct usb_driver   *fmb_hw_usb_get_driver( void );
int                 fmb_hw_usb_probe_endpoint_free( struct fmb_hw_usb * dev );
int                 fmb_hw_usb_mmaparea_free( struct fmb_hw_usb *dev );

#ifndef MULTIENDIAN_FUNCTION
void
local_swap( unsigned char *a_1byte, unsigned char  *b_1byte )
{
        unsigned char   work;

        work     = *a_1byte;
        *a_1byte = *b_1byte;
        *b_1byte = work;
    
        return;
}

void
local_cvt_lowdata32( unsigned long *data_4byte )
{
        unsigned char *work;

        work = (unsigned char *)data_4byte;
        local_swap( work + 0, work + 3 );
        local_swap( work + 1, work + 2 );
    
        return;
}

unsigned char
local_picup_byte( unsigned long *pwork, int pos )
{
    unsigned char *work = (unsigned char *)pwork;
    
    return( *(work+pos) );
}
unsigned char
local_picup64_byte( unsigned long long *pwork, int pos )
{
    unsigned char *work = (unsigned char *)pwork;
    
    return( *(work+pos) );
}

void
local_4byte_out( unsigned long *pwork )
{
    printk("[%02x %02x %02x %02x]",
        local_picup_byte( pwork, 0 ),
        local_picup_byte( pwork, 1 ),
        local_picup_byte( pwork, 2 ),
        local_picup_byte( pwork, 3 )
    );
}
void
local_8byte_out( unsigned long long *pwork )
{
    printk("[%02x %02x %02x %02x _ %02x %02x %02x %02x]",
        local_picup64_byte( pwork, 0 ),
        local_picup64_byte( pwork, 1 ),
        local_picup64_byte( pwork, 2 ),
        local_picup64_byte( pwork, 3 ),
        local_picup64_byte( pwork, 4 ),
        local_picup64_byte( pwork, 5 ),
        local_picup64_byte( pwork, 6 ),
        local_picup64_byte( pwork, 7 )
    );
}
unsigned char
local_get_1from8( unsigned long long in_data, int pos_byte )
{
    unsigned long long mask_base = 0xff;
    unsigned long long mask;
    unsigned char ret;

    mask = mask_base << ( pos_byte * 8 );
    in_data &= mask;
    ret = in_data >> ( pos_byte * 8 );

    return ret;
}

unsigned char
local_get_1from4( unsigned long in_data, int pos_byte )
{
    unsigned long mask_base = 0xff;
    unsigned long mask;
    unsigned char ret;

    mask = mask_base << ( pos_byte * 8 );
    in_data &= mask;
    ret = in_data >> ( pos_byte * 8 );

    return ret;
}

unsigned char
local_get_1from2( unsigned short in_data, int pos_byte )
{
    unsigned short mask_base = 0xff;
    unsigned short mask;
    unsigned char ret;

    mask = mask_base << ( pos_byte * 8 );
    in_data &= mask;
    ret = in_data >> ( pos_byte * 8 );

    return ret;
}

void
local_set_HLdata64( void *out_64bitdata, unsigned long long in_data )
{
    unsigned char *work;

    work = (unsigned char *)out_64bitdata;

    *(work+0) = local_get_1from8( in_data, 5);
    *(work+1) = local_get_1from8( in_data, 4);
    *(work+2) = local_get_1from8( in_data, 7);
    *(work+3) = local_get_1from8( in_data, 6);
    *(work+4) = local_get_1from8( in_data, 1);
    *(work+5) = local_get_1from8( in_data, 0);
    *(work+6) = local_get_1from8( in_data, 3);
    *(work+7) = local_get_1from8( in_data, 2);
}
#if 0 /* future support */
void
local_set_HLdata32( void *out_32bitdata, unsigned long in_data )
{
        unsigned char *work;

        work = (unsigned char *)out_32bitdata;

        *(work+0) = local_get_1from4( in_data, 3);
        *(work+1) = local_get_1from4( in_data, 2);
        *(work+2) = local_get_1from4( in_data, 1);
        *(work+3) = local_get_1from4( in_data, 0);
}
void
local_set_LHdata32( void *out_32bitdata, unsigned long in_data )
{
        unsigned char *work;

        work = (unsigned char *)out_32bitdata;

        *(work+0) = local_get_1from4( in_data, 0);
        *(work+1) = local_get_1from4( in_data, 1);
        *(work+2) = local_get_1from4( in_data, 2);
        *(work+3) = local_get_1from4( in_data, 3);
}
void
local_set_HLdata16( void *out_16bitdata, unsigned short in_data )
{
        unsigned char *work;

        work = (unsigned char *)out_16bitdata;

        *(work+0) = local_get_1from2( in_data, 1);
        *(work+1) = local_get_1from2( in_data, 0);
}
void
local_set_LHdata16( void *out_16bitdata, unsigned short in_data )
{
        unsigned char *work;

        work = (unsigned char *)out_16bitdata;

        *(work+0) = local_get_1from2( in_data, 0);
        *(work+1) = local_get_1from2( in_data, 1);
}
#endif
#endif

#ifdef DEBUGON_USB_ACCESS

int
fmb_usb_debug_msg_init( struct fmb_usb_debug_msg *usb_dmsg )
{
    usb_dmsg->flag     = FMB_USB_DEBUGMSG_DEFAULT;
    usb_dmsg->oldflag  = FMB_USB_DEBUGMSG_DEFAULT;
    usb_dmsg->cnt      = 0;
    usb_dmsg->mask     = 0;
    
    return 0;
}

void
debug_data_dump( char *buf, void *addr, size_t size, int mode )
{
#define DEBUG_DUMP_SIZE 16
    char area[DEBUG_DUMP_SIZE+5];
    int i;
    int flg = 0;
    
    if ( size > DEBUG_DUMP_SIZE )
    {
        size = DEBUG_DUMP_SIZE;
        flg  = 1;
    }
    
    for(i=0;i<size;i++)
    {
        if ( mode != 0 )
        {
            if ( i > mode ) break;
        }
        sprintf( area, " %02x", (unsigned char)*((unsigned char *)(addr + i)) );
        strcat( buf, area );
    }
    if ( flg == 1 ) strcat( buf, "..." );
    
    return;
}

int
fmb_usb_debugmsg_check( struct fmb_hw_usb* usb_priv, unsigned long flag )
{
    int ret = 0;
    
    unsigned long work;
    if ( usb_priv->usb_debug_msg.cnt > 0 )
    {
        work = usb_priv->usb_debug_msg.oldflag;
        usb_priv->usb_debug_msg.cnt--;
    }
    else
    {
        work = usb_priv->usb_debug_msg.flag;
    }
    work = work & flag;
    if ( work != 0 )
    {
        ret = 0;
    }
    else
    {
        ret = -1;
    }
    return ret;
}
#endif /* DEBUGON_USB_ACCESS */
/********************************************************/
/*                          */
/*   register processing function       */
/*                          */
/********************************************************/

/**
*   @brief      Register Read to LSI
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*   @return     val             register read value
*   @note       None
*   @attention  None
*/  

unsigned short fmb_hw_device_reg_read(struct fmb_hard_private* priv_p, unsigned long addr, int *result )
{
    int             minor;
    int             retry;
#ifdef MULTIENDIAN_FUNCTION
    unsigned short  val;
#else
    unsigned short  *val;
    unsigned char   data[2];
    unsigned char   data_work;
#endif
    unsigned long   fRetryError;
    int ret;
    struct fmb_hw_usb* usb_priv;

    *result = 0;
    
    PRIV_P_CHECK( priv_p, /*-ENODEV*/0 );
    minor = priv_p->minor;
    retry = 5;
    fRetryError = 0;
    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, /*-ENODEV*/0 );
    
    MSG(INTERNAL_FUNC, minor, "START");

    mutex_lock(&(usb_priv->io_mutex));
#ifdef MULTIENDIAN_FUNCTION
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_RW, USB_CMDTYPE_IN, addr, &val, 2, 3*HZ );
    val = be16_to_cpu(val);
#else
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_RW, USB_CMDTYPE_IN, addr, &data, 2, 3*HZ );
    data_work = data[0];
    data[0]   = data[1];
    data[1]   = data_work;
    val = (unsigned short *)data;
#endif
    mutex_unlock(&(usb_priv->io_mutex));

    if ( ret < 0 )
    {
        *result = ret;
        MSG(ERR_LVL, minor, "register read error(By way of USB) code=%d", ret ); /* It is scheduled to correspond by return value in the future. */
    }

    MSG(INTERNAL_FUNC, minor, "END");
#ifdef MULTIENDIAN_FUNCTION
    return val;
#else
    return (unsigned short)*val;
#endif
}

/**
*   @brief      Register Read to LSI (32bit)
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*   @return     val             register read value
*   @note       None
*   @attention  None
*/  
unsigned long fmb_hw_device_reg_read32(struct fmb_hard_private* priv_p, unsigned long addr, int *result )
{
    int             minor;
    int             ret = 0;
    struct fmb_hw_usb* usb_priv;
    unsigned long   val = 0;
#ifndef MULTIENDIAN_FUNCTION
    unsigned char   data_work[4];
#endif
    *result = 0;
    
    PRIV_P_CHECK( priv_p, /*-ENODEV*/0 );
    minor = priv_p->minor;
    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, /*-ENODEV*/0 );
    
    MSG(INTERNAL_FUNC, minor, "START");

#ifndef MULTIENDIAN_FUNCTION
    memset( &data_work[0], 0, 4 );
#endif
    
    mutex_lock(&(usb_priv->io_mutex));
#ifdef MULTIENDIAN_FUNCTION
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_RW, USB_CMDTYPE_IN, addr, &val, 4, 3*HZ );
    val = be32_to_cpu(val);
#else
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_RW, USB_CMDTYPE_IN, addr, &data_work[0], 4, 3*HZ );
    val  = data_work[3];
    val <<= 8;
    val += data_work[2];
    val <<= 8;
    val += data_work[1];
    val <<= 8;
    val += data_work[0];
#endif

    mutex_unlock(&(usb_priv->io_mutex));

    if ( ret < 0 )
    {
        *result = ret;
        MSG(ERR_LVL, minor, "register read error(By way of USB) code=%d", ret ); /* It is scheduled to correspond by return value in the future. */
    }

    MSG(INTERNAL_FUNC, minor, "END");
    return val;
}



/**
*   @brief      Register Write to LSI
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*   @param[in]  data            register write value
*   @return     None
*   @note       None
*   @attention  None
*/  

void fmb_hw_device_reg_write(struct fmb_hard_private* priv_p, unsigned long addr, unsigned short data, int *result )
{
    int             minor;
    int             retry;
    unsigned long   val;
#ifndef MULTIENDIAN_FUNCTION
    unsigned char   data_work[2];
#endif
    unsigned long   fRetryError;
    int ret;
    struct fmb_hw_usb* usb_priv;

    *result = 0;
    
    PRIV_P_CHECK_VOID( priv_p, -ENODEV );

    minor = priv_p->minor;
    retry = 5;
    val = 0;
    fRetryError = 0;
    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK_VOID( priv_p, -ENODEV );

    MSG(DEBUG_LVL, minor, "START");

    mutex_lock(&(usb_priv->io_mutex));
#ifdef MULTIENDIAN_FUNCTION
    data = cpu_to_be16( data );
#else
    *(unsigned short *)data_work = data;
    data = ( (unsigned short)data_work[0] << 8 ) + data_work[1];
#endif
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_RW, USB_CMDTYPE_OUT, addr, &data, 2, 3*HZ );
    mutex_unlock(&(usb_priv->io_mutex));
    
    if ( ret < 0 )
    {
        *result = ret;
        MSG(ERR_LVL, minor, "register write error(By way of USB) code=%d", ret ); /* It is scheduled to correspond by return value in the future. */
    }

    MSG(DEBUG_LVL, minor, "END");
}

#if 0 /* 1102 The support schedule when the future.  */

/**
*   @brief      Register Read to LSI Enhancing master
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI Enhancing master register address
*   @return     val             register read value
*   @note       None
*   @attention  None
*/  

unsigned long fmb_hw_device_reg_read_master(struct fmb_hard_private* priv_p, unsigned long addr)
{
    int             minor;
    int             retry;
    unsigned long   val;
    unsigned long   fRetryError;
    int ret;
    struct fmb_hw_usb* usb_priv;

    PRIV_P_CHECK( priv_p, -ENODEV );

    minor = priv_p->minor;
    retry = 5;
    val = 0;
    fRetryError = 0;
    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    
    MSG(INTERNAL_LOG, minor, "START");

    mutex_lock(&(usb_priv->io_mutex));
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_RW, USB_CMDTYPE_IN, addr, &val, 4, 3*HZ );
#ifdef MULTIENDIAN_FUNCTION
    val = be32_to_cpu(val);
#endif
    mutex_unlock(&(usb_priv->io_mutex));

    MSG(INTERNAL_LOG, minor, "END");
    return val;
}


/**
*   @brief      Register Write to LSI Enhancing master
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI Enhancing master register address
*   @param[in]  data            register write value
*   @return     None
*   @note       None
*   @attention  None
*/  

void fmb_hw_device_reg_write_master(struct fmb_hard_private* priv_p, unsigned long addr, unsigned long data)
{
    int             minor;
    int             retry;
    unsigned long   fRetryError;
    int ret;
    struct fmb_hw_usb* usb_priv;

    PRIV_P_CHECK_VOID( priv_p, -ENODEV );

    minor = priv_p->minor;
    retry = 5;
    fRetryError = 0;
    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK_VOID( usb_priv, -ENODEV );

    MSG(INTERNAL_LOG, minor, "START");

    mutex_lock(&(usb_priv->io_mutex));
#ifdef MULTIENDIAN_FUNCTION
    data = cpu_to_be32( data );
#endif
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_RW, USB_CMDTYPE_OUT, addr, &data, 4, 3*HZ );
    mutex_unlock(&(usb_priv->io_mutex));

    MSG(INTERNAL_LOG, minor, "END");
}

#endif /* 1102 The support schedule when the future.  */

/**
*   @brief      Register Read Modfi Write to LSI 32bit
*   @param[in]  priv_p          private data pointer
*   @param[in]  addr            LSI register address
*   @param[in]  mask            mask value at the time of writing
*   @param[in]  data            register write value
*   @return     val             register read value
*   @note       None
*   @attention  None
*/

int fmb_hw_device_reg_rmw32(struct fmb_hard_private* priv_p, unsigned long addr,
                          unsigned long mask, unsigned long data )
{
    int             minor;
    int             retry;
#ifdef MULTIENDIAN_FUNCTION
    unsigned long   val[2];
#else
    unsigned long long  val, w_val;
#endif
    long            val_size; 
    unsigned long   fRetryError;
    int ret = 0;
    struct fmb_hw_usb* usb_priv;

    PRIV_P_CHECK( priv_p, -ENODEV );

    minor = priv_p->minor;
    retry = 5;
    fRetryError = 0;
    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    
    MSG(INTERNAL_FUNC, minor, "START");

    
    val_size = sizeof( val );

    mutex_lock(&(usb_priv->io_mutex));

#ifdef MULTIENDIAN_FUNCTION
    val[0]   = data;
    val[1]   = mask;
    
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_FiledW, USB_CMDTYPE_OUT, addr, val, val_size, 3*HZ );
#else
    w_val   = data;
    w_val <<= 4 * 8;
    w_val  += mask;
    local_set_HLdata64( &val, w_val );
    
    ret = fmb_hw_usb_ep_control_reg( usb_priv, USB_AMODE_FiledW, USB_CMDTYPE_OUT, addr, &val, val_size, 3*HZ );
#endif
    mutex_unlock(&(usb_priv->io_mutex));
    
    MSG(INTERNAL_FUNC, minor, "END");
    return ret;
}

/**
*   @brief      bulk read endpoint address of READ API is set.
*   @param[in]  priv_p          private data pointer
*   @param[in]  endpoint        endpoint no
*   @return     0:OK -1:NG
*   @note       None
*   @attention  None
*/  
int
fmb_hw_usb_set_read_epaddr( struct fmb_hard_private* priv_p, int endpoint )
{
    int ret = 0;
    struct fmb_hw_usb *usb_priv;
    
    PRIV_P_CHECK( priv_p, -ENODEV );
    
    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    
    mutex_lock(&(usb_priv->i_mutex));
    mutex_lock(&(usb_priv->io_mutex));
    switch( endpoint )
    {
    case 1: /* bulk-in 1 */
        usb_priv->bulk_in_size_read         = usb_priv->bulk_in_size;
        usb_priv->bulk_in_endpointAddr_read = usb_priv->bulk_in_endpointAddr;
        usb_priv->bulk_in_buffer_read       = usb_priv->bulk_in_buffer;
        break;
    case 4: /* bulk-in 4 */
        usb_priv->bulk_in_size_read         = usb_priv->bulk_in_size2;
        usb_priv->bulk_in_endpointAddr_read = usb_priv->bulk_in_endpointAddr2;
        usb_priv->bulk_in_buffer_read       = usb_priv->bulk_in_buffer2;
        break;
    default:
        ret = -1;   /* error */
        break;
    }
    mutex_unlock(&(usb_priv->io_mutex));
    mutex_unlock(&(usb_priv->i_mutex));
    return ret;
}

#ifdef DEBUGON_USB_ACCESS
/**
*  @brief           fmb_hw_usb_set_debug_msg() Command Function
*  @param[in]       fmb_hard_private_data_p     hard private data pointer
*  @param[in,out]   arg         Argument from application
*  @param[in]       (struct fmb_usb_debug_msg *) =arg
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -ENODEV     no device
*  @note            None
*  @attention       When DEBUGON_USB_ACCESS macro is defined, this function is effective.
*/
int
fmb_hw_usb_set_debug_msg( struct fmb_hard_private* priv_p, unsigned long argp)
{
    int ret   = 0;
    int minor = -1;
    unsigned long work;
    unsigned long work_mask;
    
    struct fmb_usb_debug_msg usb_debug_msg;
    struct fmb_hw_usb *usb_priv;
    
    PRIV_P_CHECK( priv_p, -ENODEV );

    usb_priv = (struct fmb_hw_usb*)priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    
    if (copy_from_user(&usb_debug_msg, (void*)argp, sizeof(struct fmb_usb_debug_msg))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_usb_debug_msg)");
        return -EFAULT;
    }
    
    
    if ( usb_debug_msg.cnt > 0 )
    {
        usb_priv->usb_debug_msg.oldflag = usb_priv->usb_debug_msg.flag;
    }
    work_mask = usb_debug_msg.flag & usb_debug_msg.mask;
    work = usb_priv->usb_debug_msg.flag & ~(usb_debug_msg.mask);
    usb_priv->usb_debug_msg.flag = work | work_mask;
    
    usb_priv->usb_debug_msg.cnt   = usb_debug_msg.cnt;
    
    return ret;
}
#endif /* DEBUGON_USB_ACCESS */

int
fmb_usb_spirom_get_version( struct fmb_hard_private *hw_priv_p )
{
    int minor;
    struct fmb_hw_usb* usb_priv;
#ifndef MB86E61_FUNCTION /* (UNDEF)MB86E61_FUNCTION */
    unsigned long offset = FMB_REG_SPIROM_VER;
#ifdef MULTIENDIAN_FUNCTION
    unsigned long work_val;
#else
    union uwork
    {
        unsigned long work_val;
        char          work_bin[4];
    } work;
#endif
    int result;
#endif /* (UNDEF)MB86E61_FUNCTION */
    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );

#ifdef MB86E61_FUNCTION
    
    /* boot of internal-ROM : version information doesn't exist : the endpoint composition has only control transfers*/
    memset( hw_priv_p->spirom_ver, 0x20, FMB_REG_SPIROM_VER_BYTE );
    
#else  /* MB86E61_FUNCTION */

    if ( (usb_priv->bulk_out_endpointAddr == 0 ) && (usb_priv->bulk_in_endpointAddr == 0) 
        && (usb_priv->bulk_in_endpointAddr2 == 0)
        && (usb_priv->interrupt_in_endpointAddr == 0) )
    {
        /* boot of internal-ROM : version information doesn't exist : the endpoint composition has only control transfers*/
        memset( hw_priv_p->spirom_ver, 0x20, FMB_REG_SPIROM_VER_BYTE );
    }
    else
    {
        /* boot of SPI-ROM */
#ifdef MULTIENDIAN_FUNCTION
        work_val = fmb_hw_device_reg_read32( hw_priv_p, offset, &result );
        work_val = cpu_to_be32(work_val);
        memcpy( hw_priv_p->spirom_ver, &work_val, 4 );
        
        work_val = fmb_hw_device_reg_read32( hw_priv_p, offset+4, &result );
        work_val = cpu_to_be32(work_val);
        memcpy( hw_priv_p->spirom_ver + 4, &work_val, 4 );

        work_val = fmb_hw_device_reg_read32( hw_priv_p, offset+8, &result );
        work_val = cpu_to_be32(work_val);
        memcpy( hw_priv_p->spirom_ver + 8, &work_val, 2 );
#else
        work.work_val = fmb_hw_device_reg_read32( hw_priv_p, offset, &result );
        memcpy( hw_priv_p->spirom_ver, work.work_bin, 4 );
        
        work.work_val = fmb_hw_device_reg_read32( hw_priv_p, offset+4, &result );
        memcpy( hw_priv_p->spirom_ver + 4, work.work_bin, 4 );

        work.work_val = fmb_hw_device_reg_read32( hw_priv_p, offset+8, &result );
        memcpy( hw_priv_p->spirom_ver + 8, work.work_bin, 2 );
#endif
    }
    
#endif /* MB86E61_FUNCTION */
    
    return 0;
}
/**
*   @brief      Device ids of USB is registered.
*   @param[in]  d         usb device Ids struct  pointer
*   @return     0:OK (other):error
*   @note       None
*   @attention  None
*/  
int
fmb_hw_usb_register( long *d )
{
    return usb_register( (struct usb_driver *)d );
}

/**
*   @brief      Device ids of USB is unregistered.
*   @param[in]  d         usb device Ids struct  pointer
*   @return     None
*   @note       None
*   @attention  None
*/  
void
fmb_hw_usb_unregister( long *d )
{
    usb_deregister( (struct usb_driver *)d );
    
    return;
}
#ifdef USB_URB_FUNCTION
/**
*   @brief      fmb_hw_usb_cmd_completion
*   @param[in]  urb  urb buffer pointer
*   @return     None
*   @note       None
*   @attention  None
*/  
#ifdef TV_LINUX_FUNCTION
static void fmb_hw_usb_cmd_completion(struct urb *urb, struct pt_regs *regs)
#else
static void fmb_hw_usb_cmd_completion(struct urb *urb)
#endif
{
    complete((struct completion *)urb->context);
}
#endif

/**
*    @brief        fmb_hw_usb_ep_interrupt_read
*    @param[in]    dev           usb device Structure pointer
*    @param[in]    data          recive data               (interrupt trans)
*    @param[in]    size          recive data size          (interrupt trans)
*    @param[in]    actual_length recive data actual length (interrupt trans)
*    @param[in]    timeout     time out
*    @return       0:OK -1:NG
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_ep_interrupt_read( struct fmb_hw_usb *dev, void *data, int size, int *actual_length, int timeout )
{
    int endpoint = dev->interrupt_in_endpointAddr;
    int ret;
#ifdef USB_URB_FUNCTION
    int          minor = -1;
	int			status;
	struct usb_host_endpoint *ep;
	unsigned int pipe;
#endif

    PRIV_P_CHECK( dev,  -ENODEV );
    PARAM_CHECK(  data, -EINVAL );
    PARAM_CHECK(  actual_length, -EINVAL );
    
#ifdef USB_URB_FUNCTION
    pipe = usb_rcvintpipe( dev->udev, endpoint );

	ep = (usb_pipein(pipe) ? dev->udev->ep_in : dev->udev->ep_out)
			[usb_pipeendpoint(pipe)];
	if (!ep ){
		/* device removed */
        // MSG(KERN_ERR_LVL, minor, "endpoint error!!\n");
		return -EINVAL;
	}
	dev->interrupt_urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!dev->interrupt_urb){
        MSG(KERN_ERR_LVL, minor, "urb allocate error!!\n");
		return -ENOMEM;
	}
    
	usb_fill_int_urb(dev->interrupt_urb, dev->udev, pipe, 
				(char *) data, size, 
			    fmb_hw_usb_cmd_completion, NULL,
			    ep->desc.bInterval);
			     
	init_completion(&dev->interrupt_urb_done); 	
	dev->interrupt_urb->context = &dev->interrupt_urb_done;
	dev->interrupt_urb->actual_length = 0;
	status = usb_submit_urb(dev->interrupt_urb, GFP_KERNEL);
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "usb_submit_urb error!! status=%d\n",status);
		usb_free_urb(dev->interrupt_urb);
		dev->interrupt_urb = NULL;
		return status;
	}
	

	wait_for_completion_timeout(&dev->interrupt_urb_done,timeout);
	status = dev->interrupt_urb->status;

	if ( status == -EINPROGRESS ){
		return status;
	}
	
	if ( status != 0 ){
		if ( status != -EPROTO && status != -ESHUTDOWN ){ /* device removed */
			MSG(KERN_ERR_LVL, minor, "wait_for_completion_timeout status=%d!!\n",status);
		}
		usb_free_urb(dev->interrupt_urb);
		dev->interrupt_urb = NULL;
		return status;
	}
	
	*actual_length = dev->interrupt_urb->actual_length;
	ret = dev->interrupt_urb->actual_length;
	
	usb_free_urb(dev->interrupt_urb);
	dev->interrupt_urb = NULL;
#else
    ret = usb_interrupt_msg( dev->udev, usb_rcvintpipe( dev->udev, endpoint ), (char *) data, size, actual_length, timeout );
#endif
#ifdef DEBUGON_USB_ACCESS
    {
        if ( fmb_usb_debugmsg_check( dev, FMB_USB_DEBUGMSG_INT ) == 0 )
        {
            if ( ret == 0 )
            {
                int minor = -1;
                FMB_GET_HW_MINOR( minor, dev->hw_priv_p );
                sprintf( dbuf, "EA:%02x DS:%05d ADS:%05d TO:%05d ", endpoint, size, *actual_length, timeout );
                strcat( dbuf, " data(RCV):" );
                debug_data_dump( dbuf, data, *actual_length, 0 );
                info( "%03d:usbINT:%s", minor, dbuf );
            }
        }
    }
#endif /* DEBUGON_USB_ACCESS */

    return ret;
}
#ifdef USB_URB_FUNCTION
/**
*    @brief        fmb_hw_usb_ep_interrupt_wait_for_completion
*    @param[in]    dev           usb device Structure pointer
*    @param[in]    data          recive data               (interrupt trans)
*    @param[in]    size          recive data size          (interrupt trans)
*    @param[in]    actual_length recive data actual length (interrupt trans)
*    @param[in]    timeout     time out
*    @return       0:OK -1:NG
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_ep_interrupt_wait_for_completion( struct fmb_hw_usb *dev, void *data, int size, int *actual_length, int timeout)
{
    int ret;

    int          minor = -1;
	int			status;

    PRIV_P_CHECK( dev,  -ENODEV );
    PARAM_CHECK(  data, -EINVAL );
    PARAM_CHECK(  actual_length, -EINVAL );

    PARAM_CHECK(  dev->interrupt_urb, -EINVAL );

	wait_for_completion_timeout(&dev->interrupt_urb_done,timeout);
	
	status = dev->interrupt_urb->status;
	
	if ( status == -EINPROGRESS ){
		return status;
	}
	
	if ( status != 0 ){
		if ( status != -EPROTO && status != -ESHUTDOWN ){ /* device removed */
			MSG(KERN_ERR_LVL, minor, "wait_for_completion_timeout status=%d!!\n",status);
		}
		usb_free_urb(dev->interrupt_urb);
		dev->interrupt_urb = NULL;
		return status;
	}
	
	*actual_length = dev->interrupt_urb->actual_length;
	ret = dev->interrupt_urb->actual_length;
	
	usb_free_urb(dev->interrupt_urb);
	dev->interrupt_urb = NULL;

    return ret;
}
#endif

/**
*    @brief        fmb_hw_usb_set_api_cmd
*    @param[in]    dev         usb device Structure pointer
*    @param[in]    cmd_param   SET_API_CMD (struct fmb_hw_usb_setapicmd_param)
*    @param[in]    timeout     time out
*    @return        0:OK -1:NG
*    @note        None
*    @attention    None
*/
int
fmb_hw_usb_ep_set_api_cmd( struct fmb_hw_usb *dev, struct fmb_hw_usb_setapicmd_param *cmd_param, int timeout )
{
    int ret;
    struct fmb_hw_usb_setup_packet  s_packet;
    int endpoint;
    
    PRIV_P_CHECK( dev,          -ENODEV );
    PARAM_CHECK(  cmd_param,    -EINVAL );
    
    endpoint               = dev->control_out_endpointAddr;
    s_packet.bmRequestType = USB_CMDTYPE_OUT;
    s_packet.bRequest      = USB_CMDREQ_SET_API_CMD;
#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue        = 0x0000;
    s_packet.wIndex        = 0x0000;
    s_packet.wLength       = sizeof(struct fmb_hw_usb_setapicmd_param);
#else
    s_packet.wValueLH      = 0x0000;
    s_packet.wIndexLH      = 0x0000;
    s_packet.wLengthLH     = sizeof(struct fmb_hw_usb_setapicmd_param);
#endif
    
    cmd_param->reserve0    = 0;
    
#ifdef MULTIENDIAN_FUNCTION
    ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, (long*)cmd_param, s_packet.wLength, timeout, USB_CMDTYPE_OUT );
#else
    ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, (long*)cmd_param, s_packet.wLengthLH, timeout, USB_CMDTYPE_OUT );
#endif
    
    return ret;

    /* reply endpoint 3 */
}


/**
*    @brief        fmb_hw_usb_ep_control_reg
*    @param[in]    dev         usb device Structure pointer
*    @param[in]    acs_mode    access mode USB_AMODE_RW/USB_AMODE_FiledW
*    @param[in]    io_mode     in/out mode USB_CMDTYPE_IN/USB_CMDTYPE_OUT
*    @param[in]    addr        LSI address
*    @param[in/out]    data        data address (AMODE_RW:1-64byte, USB_AMODE_FiledW:struct fmb_hw_usb_control_reg_data)
*    @param[in]    size        data size
*    @param[in]    timeout     time out
*    @return        0:OK -1:NG
*    @note        None
*    @attention    None
*/
int
fmb_hw_usb_ep_control_reg( struct fmb_hw_usb *dev, int acs_mode, int io_mode, unsigned long addr, void *data, long data_size, int timeout )
{
    int ret;
    struct fmb_hw_usb_setup_packet  s_packet;
    int endpoint;
    int          minor = -1;
    
    PRIV_P_CHECK( dev, -ENODEV );
    PARAM_CHECK( data, -EINVAL );
    
    endpoint = dev->control_out_endpointAddr;
    if ( acs_mode != USB_AMODE_RW && acs_mode != USB_AMODE_FiledW )
    {
        /* access mode error */
        MSG(KERN_ERR_LVL, minor, "access mode error (%d)\n", io_mode );
        return -1;
    }
    else if ( acs_mode == USB_AMODE_RW && data_size > 64 )
    {
        /* size error */
        MSG(KERN_ERR_LVL, minor, "size over(>64) error (%d)\n", io_mode );
        return -1;
    }
    else if ( acs_mode == USB_AMODE_FiledW && data_size > 8 )
    {
        /* size error data size 4 or 8 */
        MSG(KERN_ERR_LVL, minor, "size over(>8) error (%d)\n", io_mode );
        return -1;
    }
    
    switch( io_mode )
    {
    case USB_CMDTYPE_OUT:
        endpoint = dev->control_out_endpointAddr;
        break;
    case USB_CMDTYPE_IN:
        endpoint = dev->control_in_endpointAddr;
        break;
    default:
        MSG(KERN_ERR_LVL, minor, "io mode error (%d)\n", io_mode );
        return -1; /* io mode error */
    }
        
#if 0
    if ( data_size == 0 )
    {
        /* size error */
        return -1;
    }
    if ( io_mode == USB_CMDTYPE_IN && acs_mode == USB_AMODE_FiledW )
    {
        /* mode error */
        return -1;
    }
#endif
    
    s_packet.bmRequestType = io_mode;
    s_packet.bRequest      = USB_CMDREQ_CONTROL_REG;
#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue        = addr >> 16;
    s_packet.wValue        <<= 8;
    s_packet.wValue        += acs_mode & 0x00ff;
    s_packet.wIndex        = addr & 0xffff;
    s_packet.wLength       = data_size;
    
    ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLength, timeout, io_mode );
#else
    s_packet.wValueL       = acs_mode;
    s_packet.wValueH       = addr >> 16;
    s_packet.wIndexLH      = addr & 0xffff;
    s_packet.wLengthLH     = data_size;
    
    ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLengthLH, timeout, io_mode );
#endif
    
    
    
    return ret;
}

/**
*    @brief        fmb_hw_usb_ep_control_i2c
*    @param[in]    dev         usb device Structure pointer
*    @param[in]    acs_mode    access mode USB_IMODE_NORMAL/USB_IMODE_WITHOUT_STOP/USB_IMODE_REPEATED_START/USB_IMODE_STOP/USB_IMODE_STATUS
*    @param[in]    io_mode     in/out mode USB_CMDTYPE_IN/USB_CMDTYPE_OUT
*    @param[in]    addr        i2c sub-address
*    @param[in/out]    data    data address
*    @param[in]    size        data size (write/read:<=64, stop:1, status:2)
*    @param[in]    timeout     time out
*    @return       0:OK -1:NG
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_ep_control_i2c( struct fmb_hw_usb *dev, unsigned char acs_mode, int io_mode, unsigned char addr, void *data, unsigned short data_size, int timeout )
{
    int ret;
#ifdef DEBUGON_I2C_ACCESS
    int i;
#endif /* DEBUGON_I2C_ACCESS */
    struct fmb_hw_usb_setup_packet  s_packet;
    int endpoint;
    int          minor = -1;
#ifdef DEBUGON_I2C_ACCESS
    char ibuf[1024];
#endif
    
    PRIV_P_CHECK( dev, -ENODEV );
    PARAM_CHECK( data, -EINVAL );
    
    FMB_GET_HW_MINOR( minor, dev->hw_priv_p );
    
	if(
	   !((acs_mode == USB_IMODE_NORMAL)         && (io_mode == USB_CMDTYPE_OUT) && (data_size<=64) && (data_size > 0) ) && // i2c write
	   !((acs_mode == USB_IMODE_NORMAL)         && (io_mode == USB_CMDTYPE_IN)  && (data_size<=64) && (data_size > 0) ) && // i2c read
	   !((acs_mode == USB_IMODE_WITHOUT_STOP)   && (io_mode == USB_CMDTYPE_OUT) && (data_size<=64) && (data_size > 0) ) && // i2c write without stop
	   !((acs_mode == USB_IMODE_REPEATED_START) && (io_mode == USB_CMDTYPE_IN)  && (data_size<=64) && (data_size > 0) ) && // i2c read repeated start
	   !((acs_mode == USB_IMODE_STOP)           && (io_mode == USB_CMDTYPE_OUT) && (data_size==1) ) &&  // i2c stop
	   !((acs_mode == USB_IMODE_STATUS)         && (io_mode == USB_CMDTYPE_IN)  && (data_size==2) )     // i2c status
	  )
    {
        /* parameter error */
        MSG(KERN_ERR_LVL, minor, "input parameter combination error (adrs:%02x, acs:%02x, io_mode:%02x, data_size:%d)\n", addr, acs_mode, io_mode, data_size );
        return -1;
    }
    
    switch( io_mode )
    {
    case USB_CMDTYPE_OUT:
        endpoint = dev->control_out_endpointAddr;
        break;
    case USB_CMDTYPE_IN:
        endpoint = dev->control_in_endpointAddr;
        break;
    default:
        MSG(KERN_ERR_LVL, minor, "io mode error (%d)\n", io_mode );
        return -1; /* io mode error */
    }
        
    s_packet.bmRequestType = io_mode;
    s_packet.bRequest      = USB_CMDREQ_CONTROL_I2C;
#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue        = acs_mode & 0x00ff;
    s_packet.wIndex        = addr;
    s_packet.wIndex        <<= 8;
    s_packet.wLength       = (data_size >> 8) & 0x00ff;
    s_packet.wLength       <<= 8;
    s_packet.wLength       += data_size & 0x00ff;
#else
    s_packet.wValueL       = acs_mode;
    s_packet.wValueH       = 0x00;
    s_packet.wIndexL       = 0x00;
    s_packet.wIndexH       = addr;
    s_packet.wLengthL      = data_size & 0xff;
    s_packet.wLengthH      = (data_size >> 8) & 0xff;
#endif

#ifdef DEBUGON_I2C_ACCESS
    sprintf( ibuf, "input parameter (adrs:%02x acs:%02x, io_mode:%02x, data_size:%d)", addr, acs_mode, io_mode, data_size );
    info("%03d:i2c:%s", minor, ibuf );
    for(i=0;i<data_size;i++){
		sprintf(ibuf, "  data[%d]:%02x", i, *(((unsigned char*)data)+i));
		info("%s", ibuf);
	}
#endif
#ifdef MULTIENDIAN_FUNCTION
    ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLength, timeout, io_mode );
#else
    ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLengthLH, timeout, io_mode );
#endif
    if ( ret < 0 ) MSG(ERR_LVL, minor, "register read error(By way of USB) code=%d", ret ); /* It is scheduled to correspond by return value in the future. */
#ifdef DEBUGON_I2C_ACCESS
    sprintf( ibuf, "output ret:%d", ret );
    info("%03d:i2c:%s", minor, ibuf );
	if ( io_mode == USB_CMDTYPE_IN ){
	    for(i=0;i<data_size;i++){
			sprintf(ibuf, "  data[%d]:%02x", i, *(((unsigned char*)data)+i));
			info("%s", ibuf);
		}
	}
#endif
    
    return ret;
}

#ifdef MB86E61_FUNCTION
/**
*    @brief        fmb_hw_usb_ep_control_cmd
*    @param[in]    dev          usb device Structure pointer
*    @param[in]    cmd_ctl_info GET_DESCRIPTOR (struct fmb_cmd_ctl_info)
*    @param[in]    timeout      time out
*    @return        0:OK -1:NG
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_ep_control_cmd( struct fmb_hw_usb *dev, struct fmb_cmd_ctl_info *cmd_ctl_info, int timeout )
{
    int ret;
    struct fmb_cmd_ctl_info *cmd_ctl_info_work;
    struct fmb_hw_usb_setup_packet  s_packet;
    int endpoint;
    int cnt;
    int minor = -1;
    int get_size;
    int total_get_size;
    unsigned char data[3];
    int retry = 0;
    int offset = 0;
    int i = 0;
    int req_cnt_base;
    int req_cnt_remain;

    PRIV_P_CHECK( dev,             -ENODEV );
    PARAM_CHECK(  cmd_ctl_info,    -EINVAL );

    FMB_GET_HW_MINOR( minor, dev->hw_priv_p );

    cmd_ctl_info_work = dev->cmd_ctl_info_ep;

    endpoint               = dev->control_in_endpointAddr;
    s_packet.bmRequestType = 0x80;
    s_packet.bRequest      = 6 /*GET_DESCRIPTOR*/;
#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue        = 0x03F1;
    s_packet.wIndex        = FMB_SCMD_CTL_GET_SIZE;
    s_packet.wLength       = 3;
#else
    s_packet.wValueLH      = 0x03F1;
    s_packet.wIndexLH      = FMB_SCMD_CTL_GET_SIZE;
    s_packet.wLengthLH     = 3;
#endif

    /* notify start of control command */
    do{
        memset(data, 0x00, 3);
#ifdef MULTIENDIAN_FUNCTION
        ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLength, timeout, USB_CMDTYPE_IN );
#else
        ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLengthLH, timeout, USB_CMDTYPE_IN );
#endif
        if((ret < 0) || (retry > FMB_SCMD_CTL_RETRY_NUM )){
            cmd_ctl_info->error_code = data[2];
            MSG(KERN_ERR_LVL, minor, "[1]usb endpoint0 read error : ret=%d err_code=0x%x", ret, cmd_ctl_info->error_code );
            return ret;
        }
        if(data[2] == 0x80) retry++;
        if(data[2] == 0x00) break;
    }while(1);

#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue        = 0x03F0;
    s_packet.wLength       = 3;
#else
    s_packet.wValueLH      = 0x03F0;
    s_packet.wLengthLH     = 3;
#endif
    memcpy(cmd_ctl_info_work, (void*)cmd_ctl_info, sizeof(struct fmb_cmd_ctl_info));

    /* send scontrol command data */
    for(cnt=0; cnt<cmd_ctl_info->cmd_size/2; cnt++){
#ifdef MULTIENDIAN_FUNCTION
        s_packet.wIndex = (cmd_ctl_info_work->cmd_data[cnt*2+1] << 8) | cmd_ctl_info_work->cmd_data[cnt*2];
#else
        s_packet.wIndexLH = (cmd_ctl_info_work->cmd_data[cnt*2+1] << 8) | cmd_ctl_info_work->cmd_data[cnt*2];
#endif
        retry = 0;
        do{
            memset(data, 0x00, 3);
#ifdef MULTIENDIAN_FUNCTION
            ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLength, timeout, USB_CMDTYPE_IN );
#else
            ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, data, s_packet.wLengthLH, timeout, USB_CMDTYPE_IN );
#endif
            if((ret < 0) || (data[2] == 0x04)  || (retry > FMB_SCMD_CTL_RETRY_NUM )){
                cmd_ctl_info->error_code = data[2];
                MSG(KERN_ERR_LVL, minor, "[2]usb endpoint0 read error : ret=%d err_code=0x%x", ret, cmd_ctl_info->error_code );
                return ret;
            }
            if(data[2] == 0x80) retry++;
            if(data[2] == 0x00) break;
        }while(1);
    }

#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue      = 0x03F2;
    s_packet.wIndex      = 0;
    s_packet.wLength     = 6;
#else
    s_packet.wValueLH      = 0x03F2;
    s_packet.wIndexLH      = 0;
    s_packet.wLengthLH     = 6;
#endif
    memcpy(cmd_ctl_info_work, (void*)cmd_ctl_info, sizeof(struct fmb_cmd_ctl_info));
    retry = 0;

    cmd_ctl_info->get_actual_size = 0;
    cmd_ctl_info->get_data_exist = 0;

    /* execute control command and get result data */
    do{
#ifdef MULTIENDIAN_FUNCTION
        ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, cmd_ctl_info_work->get_data, s_packet.wLength, timeout, USB_CMDTYPE_IN );
#else
        ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, cmd_ctl_info_work->get_data, s_packet.wLengthLH, timeout, USB_CMDTYPE_IN );
#endif
        if((ret < 0) || (retry > FMB_SCMD_CTL_RETRY_NUM ) || ((cmd_ctl_info_work->get_data[2] & 0x0f) != 0x0)){
            cmd_ctl_info->error_code = cmd_ctl_info_work->get_data[2];
            MSG(KERN_ERR_LVL, minor, "[3]usb endpoint0 read error : ret=%d err_code=0x%x", ret, cmd_ctl_info->error_code );
            return ret;
        }
        if(cmd_ctl_info_work->get_data[2] == 0x80) retry++;
        if(cmd_ctl_info_work->get_data[2] == 0x00) break;
    }while(1);

    cmd_ctl_info->get_data[3] = cmd_ctl_info_work->get_data[3];
    if(cmd_ctl_info_work->get_data[3] == 0x1){
        get_size = 0;
        total_get_size = 0;

        req_cnt_base = cmd_ctl_info_work->get_req_size / FMB_SCMD_CTL_GET_SIZE_USB;
        req_cnt_remain = cmd_ctl_info_work->get_req_size % FMB_SCMD_CTL_GET_SIZE_USB;

        for(i=0; i<req_cnt_base; i++){
            retry = 0;
            do{
#ifdef MULTIENDIAN_FUNCTION
                s_packet.wIndex      = cmd_ctl_info_work->get_offset;
                s_packet.wLength     = FMB_SCMD_CTL_GET_SIZE_USB + 6;
#else
                s_packet.wIndexLH      = cmd_ctl_info_work->get_offset;
                s_packet.wLengthLH     = FMB_SCMD_CTL_GET_SIZE_USB + 6;
#endif

#ifdef MULTIENDIAN_FUNCTION
                ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, cmd_ctl_info_work->get_data, s_packet.wLength, timeout, USB_CMDTYPE_IN );
#else
                ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, cmd_ctl_info_work->get_data, s_packet.wLengthLH, timeout, USB_CMDTYPE_IN );
#endif
                if((ret < 0) || (retry > FMB_SCMD_CTL_RETRY_NUM ) || ((cmd_ctl_info_work->get_data[2] & 0x0f) != 0x0)){
                    cmd_ctl_info->error_code = cmd_ctl_info_work->get_data[2];
                    MSG(KERN_ERR_LVL, minor, "[4-1]usb endpoint0 read error : ret=%d err_code=0x%x", ret, cmd_ctl_info->error_code );
                    return ret;
                }
                if(cmd_ctl_info_work->get_data[2] == 0x80){
                    retry++;
                }
                if(cmd_ctl_info_work->get_data[2] == 0x00){
                    break;
                }
            }while(1);

            cmd_ctl_info_work->get_offset += FMB_SCMD_CTL_GET_SIZE_USB;

            get_size = (cmd_ctl_info_work->get_data[5] << 8) | cmd_ctl_info_work->get_data[4];
            for(cnt=0; cnt<get_size; cnt++){
                cmd_ctl_info->get_data[offset] = cmd_ctl_info_work->get_data[6+cnt];
                offset++;
                if(offset > FMB_SCMD_CTL_GET_SIZE){
                    MSG(KERN_ERR_LVL, minor, "get_data[] overflow!\n");
                    break;
                }
            }
            total_get_size += get_size;
            cmd_ctl_info->get_actual_size = total_get_size;
            cmd_ctl_info->get_data_exist = cmd_ctl_info_work->get_data[3];
        }

        if(req_cnt_remain != 0){
            retry = 0;
            do{
#ifdef MULTIENDIAN_FUNCTION
                s_packet.wIndex      = cmd_ctl_info_work->get_offset;
                s_packet.wLength     = req_cnt_remain + 6;

                ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, cmd_ctl_info_work->get_data, s_packet.wLength, timeout, USB_CMDTYPE_IN );
#else
                s_packet.wIndexLH      = cmd_ctl_info_work->get_offset;
                s_packet.wLengthLH     = req_cnt_remain + 6;

                ret = fmb_hw_usb_cmd( dev, &s_packet, endpoint, cmd_ctl_info_work->get_data, s_packet.wLengthLH, timeout, USB_CMDTYPE_IN );
#endif
                if((ret < 0) || (retry > FMB_SCMD_CTL_RETRY_NUM ) || ((cmd_ctl_info_work->get_data[2] & 0x0f) != 0x0)){
                    cmd_ctl_info->error_code = cmd_ctl_info_work->get_data[2];
                    MSG(KERN_ERR_LVL, minor, "[4-2]usb endpoint0 read error : ret=%d err_code=0x%x", ret, cmd_ctl_info->error_code );
                    return ret;
                }
                if(cmd_ctl_info_work->get_data[2] == 0x80){
                    retry++;
                }
                if(cmd_ctl_info_work->get_data[2] == 0x00){
                    break;
                }
            }while(1);

            cmd_ctl_info_work->get_offset += req_cnt_remain;

            get_size = (cmd_ctl_info_work->get_data[5] << 8) | cmd_ctl_info_work->get_data[4];
            for(cnt=0; cnt<get_size; cnt++){
                cmd_ctl_info->get_data[offset] = cmd_ctl_info_work->get_data[6+cnt];
                offset++;
                if(offset > FMB_SCMD_CTL_GET_SIZE){
                    MSG(KERN_ERR_LVL, minor, "get_data[] overflow!\n");
                    break;
                }
            }
            total_get_size += get_size;
            cmd_ctl_info->get_actual_size = total_get_size;
            cmd_ctl_info->get_data_exist = cmd_ctl_info_work->get_data[3];
        }
    }

    cmd_ctl_info->error_code = cmd_ctl_info_work->get_data[2];
    return ret;
}
#endif /* MB86E61_FUNCTION */

/**
*    @brief        base API of usb control message.
*    @param[in]    dev          usb device Structure pointer
*    @param[in]    setup_packet packet info pointer
*    @param[in]    endpoint     endpoint no
*    @param[io]    data         data pointer
*    @param[in]    data_size    data size
*    @param[in]    timeout      time out
*    @param[in]    io_mode      USB_CMDTYPE_OUT or USB_CMDTYPE_IN
*    @return        0:OK -1:NG
*    @note        None
*    @attention    None
*/
int
fmb_hw_usb_cmd( struct fmb_hw_usb *dev, struct fmb_hw_usb_setup_packet *setup_packet, int endpoint, void *data, long data_size, int timeout, int io_mode )
{
    int          ret;
    unsigned int pipe;
    
    int          minor = -1;
#ifdef USB_URB_FUNCTION
	struct urb *urb;
    struct usb_ctrlrequest *dr;
	struct completion	done;
	int			status;
#endif
    
    FMB_GET_HW_MINOR( minor, dev->hw_priv_p );

    switch( io_mode )
    {
    case USB_CMDTYPE_OUT:
        pipe = usb_sndctrlpipe( dev->udev, endpoint );
        break;
    case USB_CMDTYPE_IN:
        pipe = usb_rcvctrlpipe( dev->udev, endpoint );
        break;
    default:
        MSG(KERN_ERR_LVL, minor, "io mode error (%d)\n", io_mode );
        return -1;
    }
#ifdef USB_URB_FUNCTION
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb){
        MSG(KERN_ERR_LVL, minor, "urb allocate error!!\n");
		return -ENOMEM;
	}
	
  	dr = kmalloc(sizeof(struct usb_ctrlrequest), GFP_KERNEL);
	if (!dr){
        MSG(KERN_ERR_LVL, minor, "usb_ctrlrequest allocate error!!\n");
		return -ENOMEM;
	}

	dr->bRequestType= setup_packet->bmRequestType;
	dr->bRequest = setup_packet->bRequest;
	dr->wValue = cpu_to_le16(setup_packet->wValue);
	dr->wIndex = cpu_to_le16(setup_packet->wIndex);
	dr->wLength = cpu_to_le16(data_size);

	usb_fill_control_urb(urb, dev->udev, pipe, (unsigned char *)dr, data,
			     data_size, fmb_hw_usb_cmd_completion, NULL);
			     
	init_completion(&done); 	
	urb->context = &done;
	urb->actual_length = 0;
	status = usb_submit_urb(urb, GFP_KERNEL);
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "usb_submit_urb status=%d!!\n",status);
		usb_free_urb(urb);
	    kfree( dr );
		return status;
	}
	
	wait_for_completion_timeout(&done,timeout);
	status = urb->status;
	
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "wait_for_completion_timeout status=%d!!\n",status);
		usb_free_urb(urb);
	    kfree( dr );
		return status;
	}
	
	ret = urb->actual_length;
	
	usb_free_urb(urb);
    kfree( dr );
#else
#ifdef MULTIENDIAN_FUNCTION
    ret = usb_control_msg( dev->udev, pipe, setup_packet->bRequest, setup_packet->bmRequestType, setup_packet->wValue, setup_packet->wIndex, data, data_size,  timeout );
#else
    ret = usb_control_msg( dev->udev, pipe, setup_packet->bRequest, setup_packet->bmRequestType, setup_packet->wValueLH, setup_packet->wIndexLH, data, data_size,  timeout );
#endif
#endif
    if ( ret >= 0 && ret != data_size )
    {
        MSG(KERN_ERR_LVL, minor, "The size of request and result is different.  [CTRL]EP:0x%02x,RQ:0x%02x,RT:0x%02x,DS:%ld,RET:%d"
            ,endpoint, setup_packet->bRequest, setup_packet->bmRequestType, data_size, ret  );
    }
#ifdef DEBUGON_USB_ACCESS
    {
        if ( fmb_usb_debugmsg_check( dev, FMB_USB_DEBUGMSG_CTL ) == 0 )
        {
#ifdef MULTIENDIAN_FUNCTION
            sprintf( dbuf, "EA:%02x RQ:%02x RT:%02x VAL:%04x IDX:%04x ret:%d DS:% 5ld ", endpoint, setup_packet->bRequest, setup_packet->bmRequestType, setup_packet->wValue, setup_packet->wIndex, ret, data_size );
#else
            sprintf( dbuf, "EA:%02x RQ:%02x RT:%02x VAL:%04x IDX:%04x ret:%d DS:% 5ld ", endpoint, setup_packet->bRequest, setup_packet->bmRequestType, setup_packet->wValueLH, setup_packet->wIndexLH, ret, data_size );
#endif
            switch( io_mode )
            {
            case USB_CMDTYPE_IN:
                strcat( dbuf, " data(I):" );
                break;
            case USB_CMDTYPE_OUT:
                strcat( dbuf, " data(O):" );
                break;
            default:
                strcat( dbuf, " data(err):" );
                break;
            }
            debug_data_dump( dbuf, data, data_size, 0 );
            info( "%03d:usbCTL:%s", minor, dbuf );
        }
    }
#endif /* DEBUGON_USB_ACCESS */
    return ret;
}


/**
*    @brief        delete(free) of private data.
*    @param[in]    kref   kref pointer
*    @return       None
*    @note         None
*    @attention    None
*/
static void fmb_hw_usb_delete(struct kref *kref)
{
    struct fmb_hard_private *hw_priv_p;
    struct fmb_hw_usb       *dev;
    int minor = -1;
    
    hw_priv_p = to_fmb_hard_private_ptr( kref );
    if ( hw_priv_p == NULL ) MSG(INTERNAL_ERR_LVL, minor, "hard private data cannot be gotten.(abnormal)");
    PRIV_P_CHECK_VOID( hw_priv_p, -ENODEV );
    minor = hw_priv_p->minor;
    
    dev       = (struct fmb_hw_usb *)hw_priv_p->dev_priv;
    if ( dev == NULL ) MSG(INTERNAL_ERR_LVL, minor, "usb private data cannot be gotten.(abnormal)");
    PRIV_P_CHECK_VOID( dev, -ENODEV );
   
    mutex_destroy(&dev->io_mutex);
    mutex_destroy(&dev->o_mutex);
    mutex_destroy(&dev->i2_mutex);
    mutex_destroy(&dev->i_mutex);

    usb_put_dev(dev->udev);

    fmb_hw_usb_mmaparea_free( dev );
    fmb_hw_usb_probe_endpoint_free( dev );
#ifdef MB86E61_FUNCTION
    if ( dev->cmd_ctl_info_apl != NULL )
    {
        kfree( dev->cmd_ctl_info_apl );
        dev->cmd_ctl_info_apl = NULL;
    }

    if ( dev->cmd_ctl_info_ep != NULL )
    {
        kfree( dev->cmd_ctl_info_ep );
        dev->cmd_ctl_info_ep = NULL;
    }
#endif /* MB86E61_FUNCTION */

    if ( dev->device_info != NULL )
    {
        kfree( dev->device_info );
        dev->device_info = NULL;
    }
    kfree( dev );
    
    Fmb_core_remove(hw_priv_p->core_priv_p);
    
    fmb_hw_private_data_release(hw_priv_p);    

    fmb_hw_private_free();
  
}

/**
*    @brief        usb device open.
*    @param[in]    inode   linux fops struct inode pointer
*    @param[in]    file    linux fops struct file  pointer
*    @return       
*    @note         None
*    @attention    None
*/
int fmb_hw_usb_open(struct inode *inode, struct file *file)
{
    struct fmb_hw_usb *dev;
    struct usb_interface *interface;
    struct fmb_hard_private*    priv_p;
    int subminor;
    int retval = 0;

    subminor = iminor(inode);

    interface = usb_find_interface(fmb_hw_usb_get_driver(), subminor);
    if (!interface) {
        err ("%s - error, can't find device for minor %d",
             __FUNCTION__, subminor);
        retval = -ENODEV;
        goto exit;
    }

    priv_p = usb_get_intfdata(interface);
    if (!priv_p) {
        retval = -ENODEV;
        goto exit;
    }
    if (!priv_p->dev_priv) {
        retval = -ENODEV;
        goto exit;
    }
    dev = priv_p->dev_priv;
    
    /* increment our usage count for the device */
    kref_get(&priv_p->kref);

#ifdef TV_LINUX_FUNCTION
#if defined CONFIG_PM
    /* prevent the device from being autosuspended */
    retval = usb_autopm_get_interface(interface);
    if (retval) {
        kref_put(&priv_p->kref, fmb_hw_usb_delete);
        goto exit;
    }
#endif
#endif

    /* save our object in the file's private structure */
    file->private_data = priv_p;

exit:
    return retval;
}

/**
*    @brief        usb device release(close).
*    @param[in]    inode   linux fops struct inode pointer
*    @param[in]    file    linux fops struct file  pointer
*    @return       
*    @note         None
*    @attention    None
*/
int fmb_hw_usb_release(struct inode *inode, struct file *file)
{
    struct fmb_hw_usb *dev;
    struct fmb_hard_private*    priv_p;

    priv_p = (struct fmb_hard_private *)file->private_data;
    if (priv_p == NULL)
        return -ENODEV;

    dev = priv_p->dev_priv;
    if (dev == NULL)
        return -ENODEV;

    /* allow the device to be autosuspended */
    mutex_lock(&dev->i_mutex);
    mutex_lock(&dev->i2_mutex);
    mutex_lock(&dev->o_mutex);
    mutex_lock(&dev->io_mutex);
#ifdef TV_LINUX_FUNCTION
#if defined CONFIG_PM
    if (dev->interface)
        usb_autopm_put_interface(dev->interface);
#endif
#endif
    mutex_unlock(&dev->io_mutex);
    mutex_unlock(&dev->o_mutex);
    mutex_unlock(&dev->i2_mutex);
    mutex_unlock(&dev->i_mutex);

    /* decrement the count on our device */
    kref_put(&priv_p->kref, fmb_hw_usb_delete);
    return 0;
}

/**
*    @brief        Translate from user_addr to krn_addr and get offset in block.
*    @param[in]    usb_priv_p   private data pointer
*    @param[in]    user_addr    check address
*    @return       NULL:not kernel address (other):kernel address
*    @note         None
*    @attention    None
*/
static unsigned char *fmb_hw_usb_trans_get_buf_by_useraddr(struct fmb_hw_usb *usb_priv_p, unsigned char *user_addr)
{
    int i;
    
    for(i = 0; i < FMB_USB_TRANS_BUFF_NUM; i++){
        if( (usb_priv_p->buffers_useraddr[i] <= user_addr) && (usb_priv_p->buffers_useraddr[i]+FMB_USB_TRANS_BUFF_SIZE > user_addr) ){
            return usb_priv_p->buffers[i] + (user_addr - usb_priv_p->buffers_useraddr[i]);
        }
    }

    return NULL;
}
/**
*    @brief        thumbnail read  API
*    @param[in]    file   linux fops struct file
*    @param[out]   buffer read data buffer
*    @param[in]    count  read data buffer size
*    @param[in]    ppos   linux fops struct ppos
*    @return       (0<=):actual read size (0>):error
*    @note         None
*    @attention    ( endpoint control API: ioctl:FMB_SET_READ_EPADDR )
*/
int fmb_hw_usb_thumbnail_read(struct fmb_hard_private *priv_p, struct fmb_read_thumbnail *read_thumbnail )
{
    struct fmb_hw_usb *dev;
    int retval;
    int bytes_read = 0;
//    struct fmb_hard_private*    priv_p;
    unsigned char *buffer_addr;
    size_t read_count;
#ifdef USB_URB_FUNCTION
    int          minor = -1;
	struct urb *urb;
	struct completion	done;
	int			status;
	unsigned int pipe;
#endif
    
//    priv_p = (struct fmb_hard_private *)file->private_data;
    PRIV_P_CHECK( priv_p, -ENODEV );
    dev    = priv_p->dev_priv;
    PRIV_P_CHECK( dev, -ENODEV );
    
    PARAM_CHECK( read_thumbnail->buffer, -EINVAL );
    

    read_thumbnail->actual_size = 0;
    
    if ( dev->bulk_in_endpointAddr2 == 0 ) return -EPERM;
    
    mutex_lock(&dev->i2_mutex);
    if (!dev->interface) {        /* disconnect() was called */
        retval = -ENODEV;
        goto exit;
    }

    buffer_addr = fmb_hw_usb_trans_get_buf_by_useraddr(dev, (unsigned char *)read_thumbnail->buffer);
    if(buffer_addr == NULL)
    {
        read_count  = min(dev->bulk_in_size2, read_thumbnail->count);
        buffer_addr = dev->bulk_in_buffer2;
    }
    else
    {
        read_count = read_thumbnail->count;
    }

    
#ifdef USB_URB_FUNCTION
    pipe = usb_rcvbulkpipe(dev->udev, dev->bulk_in_endpointAddr2);
    
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb){
        MSG(KERN_ERR_LVL, minor, "urb allocate error!!\n");
        retval = -ENOMEM;
        goto exit;
	}
	
	usb_fill_bulk_urb(urb, dev->udev, pipe, 
				(char *) buffer_addr, read_count, 
			    fmb_hw_usb_cmd_completion, NULL);
			     
	init_completion(&done); 	
	urb->context = &done;
	urb->actual_length = 0;
	status = usb_submit_urb(urb, GFP_KERNEL);
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "usb_submit_urb error!! status=%d\n",status);
		usb_free_urb(urb);
        retval = status;
        goto exit;
	}
	
	wait_for_completion_timeout(&done,USB_BULK_MES_TIMEOUT_THUMBNAIL);
	status = urb->status;
	
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "wait_for_completion_timeout status=%d!!\n",status);
		usb_free_urb(urb);
        retval = status;
        goto exit;
	}
	
	bytes_read = urb->actual_length;
	retval = bytes_read;
	
	usb_free_urb(urb);

#else
    /* do a blocking bulk read to get data from the device */
    retval = usb_bulk_msg(dev->udev,
                  usb_rcvbulkpipe(dev->udev, dev->bulk_in_endpointAddr2),
                  buffer_addr,
                  read_count,
                  &bytes_read, USB_BULK_MES_TIMEOUT_THUMBNAIL);
#endif
#ifdef DEBUGON_USB_ACCESS
    {
        if ( fmb_usb_debugmsg_check( dev, FMB_USB_DEBUGMSG_BULKIN ) == 0 )
        {
            if ( retval == 0 )
            {   
                int minor = -1;
                FMB_GET_HW_MINOR( minor, priv_p );
                sprintf( dbuf, "EA:%02x DS:%05d RDS:%05d", dev->bulk_in_endpointAddr2, read_count, bytes_read );
                strcat( dbuf, " data(RCV):" );
                debug_data_dump( dbuf, buffer_addr, bytes_read, 10 );
                info( "%03d:usbBLK:%s", minor, dbuf );
            }
        }
    }
#endif /* DEBUGON_USB_ACCESS */

    /* if the read was successful, copy the data to userspace */
    if (bytes_read > 0) {
        if ( buffer_addr == dev->bulk_in_buffer2 )
        {
            if (copy_to_user(read_thumbnail->buffer, dev->bulk_in_buffer2, bytes_read))
                retval = -EFAULT;
            else
                retval = bytes_read;
        }else{
            retval = bytes_read;
        }
    }

exit:
    read_thumbnail->actual_size = bytes_read;
    mutex_unlock(&dev->i2_mutex);
    if ( retval < 0 && bytes_read <= 0 ) return retval;
    else return 0;
}


/**
*    @brief        get device info API
*    @param[out]   device information
*    @return       
*    @note         None
*    @attention    ()
*/
int fmb_hw_usb_get_device_info( struct fmb_hard_private *priv_p,  unsigned long argp )
{
    int             minor = -1;
    struct fmb_hw_usb *dev;
    int             ret;
    
    PRIV_P_CHECK( priv_p, -ENODEV );
    dev    = priv_p->dev_priv;
    minor  = priv_p->minor;
    PRIV_P_CHECK( dev, -ENODEV );
    
    PARAM_CHECK( (void *)argp, -EINVAL );
    
    ret = copy_to_user( (void*)argp, dev->device_info, sizeof(struct fmb_device_info));
    if(ret != 0){
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_device_info)");
        return -EFAULT;
    }

    return 0;

}

/**
*    @brief        get write spi-rom state API
*    @param[out]   state  write spi-rom state
*    @return       
*    @note         None
*    @attention    ()
*/
int fmb_hw_usb_get_write_spi_state( struct fmb_hard_private *priv_p,  unsigned long argp )
{
    int             minor = -1;
    struct fmb_hw_usb *dev;
    int             retval;
    int             ret;
    struct fmb_write_spi_state_info write_spi_state_info;
    struct fmb_hw_usb_setup_packet  s_packet;
    int             endpoint;
    int             io_mode;
    long            data_size;
    
    PRIV_P_CHECK( priv_p, -ENODEV );
    dev    = priv_p->dev_priv;
    minor  = priv_p->minor;
    PRIV_P_CHECK( dev, -ENODEV );
    
    PARAM_CHECK( (void *)argp, -EINVAL );
    
    endpoint  = dev->control_in_endpointAddr;
    io_mode   = USB_CMDTYPE_IN;
    data_size = 1;
    s_packet.bmRequestType = io_mode;
    s_packet.bRequest      = USB_CMDREQ_WRITE_SPI_STATE;
#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue        = 0x0000;
    s_packet.wIndex        = 0x0000;
    s_packet.wLength       = data_size;
#else
    s_packet.wValueL       = 0x00;
    s_packet.wValueH       = 0x00;
    s_packet.wIndexL       = 0x00;
    s_packet.wIndexH       = 0x00;
    s_packet.wLengthL      = data_size;
    s_packet.wLengthH      = 0x00;
#endif
    
    write_spi_state_info.state = FMB_W_SPI_ST_SYS_ERR;

    retval = fmb_hw_usb_cmd( dev, &s_packet, endpoint, &(write_spi_state_info.state), data_size, 3*HZ, io_mode );
    
    ret = copy_to_user( (void*)argp, &write_spi_state_info, sizeof(struct fmb_write_spi_state_info));
    if(ret != 0){
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_write_spi_state_info)");
        return -EFAULT;
    }
    
	if ( retval > 0 ) retval = 0;

    return retval;

}

/**
*    @brief        load firm API
*    @param[in]   buffer  write spi-rom data address/firm data
*    @param[in]   size    firm data size
*    @return       
*    @note         None
*    @attention    ()
*/
int fmb_hw_usb_load_firm( struct fmb_hard_private *priv_p,  unsigned long argp )
{
    int minor = -1;
    struct fmb_hw_usb *dev;
    int retval;
    int ret;
    struct fmb_load_firm_info       load_firm_info;
    struct fmb_hw_usb_setup_packet  s_packet;
    unsigned char *buf;
#ifndef MULTIENDIAN_FUNCTION
    unsigned long mask;
#endif
    unsigned long work_size;
    int             endpoint;
    int             io_mode;
    
    PRIV_P_CHECK( priv_p, -ENODEV );
    dev    = priv_p->dev_priv;
    minor  = priv_p->minor;
    PRIV_P_CHECK( dev, -ENODEV );
    
    PARAM_CHECK( (void *)argp, -EINVAL );

    ret = copy_from_user( &load_firm_info, (void*)argp, sizeof(struct fmb_load_firm_info));
    if(ret != 0){
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_load_firm_info)");
        return -EFAULT;
    }
    
    buf = (unsigned char *)kmalloc( load_firm_info.size, GFP_KERNEL );
    if ( buf == NULL )
    {
        err( "%03d:[USB TRANS] Error: Out of memory", minor );
        return -ENOMEM;
    }

    ret = copy_from_user( buf, load_firm_info.buffer, load_firm_info.size );
    if(ret != 0){
        kfree( buf );
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_load_firm_info.buffer)");
        return -EFAULT;
    }
    
    endpoint = dev->control_out_endpointAddr;
    io_mode  = USB_CMDTYPE_OUT;
    s_packet.bmRequestType = io_mode;
    s_packet.bRequest      = USB_CMDREQ_LOAD_FIRM;
#ifdef MULTIENDIAN_FUNCTION
    s_packet.wValue        = 0x0000;
    
    work_size = load_firm_info.size;
    s_packet.wIndex       =  (work_size & 0xff000000 ) >> 24;
    s_packet.wIndex       <<= 8;
    s_packet.wIndex       += (work_size & 0x00ff0000 ) >> 16;
    s_packet.wLength      =  (work_size & 0x0000ff00 ) >>  8;
    s_packet.wLength      <<= 8;
    s_packet.wLength      += (work_size & 0x000000ff ) >>  0;
#else
    s_packet.wValueL       = 0x00;
    s_packet.wValueH       = 0x00;
    
    mask = 0xff000000;
    work_size = load_firm_info.size;
    s_packet.wIndexH       = (work_size & mask ) >> 24;
    mask >>= 8;
    s_packet.wIndexL       = (work_size & mask ) >> 16;
    mask >>= 8;
    s_packet.wLengthH      = (work_size & mask ) >>  8;
    mask >>= 8;
    s_packet.wLengthL      = (work_size & mask ) >>  0;
#endif
    
#if 1 /* normal */
    retval = fmb_hw_usb_cmd( dev, &s_packet, endpoint, buf, work_size, 3*HZ, io_mode );
#else /* debug */
    printk("load_firm : debug mode ( usb no access )\n");
    printk("load_firm : dev=0x%x,endpoint=0x%x,io_mode=0x%x\n", (unsigned int)dev, endpoint, io_mode );
#ifdef MULTIENDIAN_FUNCTION
    printk("load_firm : bmRequestType=0x%x,bRequest=0x%x,wValue=0x%x\n",
        s_packet.bmRequestType, s_packet.bRequest, s_packet.wValue );
    printk("load_firm : wIndexL=0x%x,wIndexH=0x%x,wLength=0x%x\n",
        s_packet.wIndexL, s_packet.wIndexH , s_packet.wLength );
#else
    printk("load_firm : bmRequestType=0x%x,bRequest=0x%x,wValueL=0x%x,wValueH=0x%x\n",
        s_packet.bmRequestType, s_packet.bRequest, s_packet.wValueL, s_packet.wValueH );
    printk("load_firm : wIndexL=0x%x,wIndexH=0x%x,wLengthL=0x%x,wLengthH=0x%x\n",
        s_packet.wIndexL, s_packet.wIndexH , s_packet.wLengthL, s_packet.wLengthH );
#endif
    printk("load_firm : size=0x%x,dataAddr=0x%x\n", (unsigned int)work_size, (unsigned int)buf );
    printk("load_firm : indata dump 0x0000:%02x %02x %02x %02x %02x\n"
        , *(load_firm_info.buffer+0), *(load_firm_info.buffer+1), *(load_firm_info.buffer+2), *(load_firm_info.buffer+3), *(load_firm_info.buffer+4) );

    retval = 0;
#endif

    kfree( buf );

    if ( retval > 0 ) retval = 0;
    
    return retval;

}

/**
*    @brief        read API
*    @param[in]    file   linux fops struct file
*    @param[out]   buffer read data buffer
*    @param[in]    count  read data buffer size
*    @param[in]    ppos   linux fops struct ppos
*    @return       (0<=):actual read size (0>):error
*    @note         None
*    @attention    ( endpoint control API: ioctl:FMB_SET_READ_EPADDR )
*/
static ssize_t fmb_hw_usb_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
    struct fmb_hw_usb *dev;
    int retval;
    int bytes_read = 0;
    struct fmb_hard_private*    priv_p;
    unsigned char *buffer_addr;
    size_t read_count;
#ifdef USB_URB_FUNCTION
    int          minor = -1;
	struct urb *urb;
	struct completion	done;
	int			status;
	unsigned int pipe;
#endif
    
    priv_p = (struct fmb_hard_private *)file->private_data;
    PRIV_P_CHECK( priv_p, -ENODEV );
    dev    = priv_p->dev_priv;
    PRIV_P_CHECK( dev, -ENODEV );
    
    PARAM_CHECK( buffer, -EINVAL );
    
    if ( dev->bulk_in_endpointAddr_read == 0 ) return -EPERM;
    
    mutex_lock(&dev->i_mutex);
    if (!dev->interface) {        /* disconnect() was called */
        retval = -ENODEV;
        goto exit;
    }

    buffer_addr = fmb_hw_usb_trans_get_buf_by_useraddr(dev, (unsigned char *)buffer);
    if(buffer_addr == NULL)
    {
		read_count  = min(dev->bulk_in_size_read, count);
        buffer_addr = dev->bulk_in_buffer_read;
    }
    else
    {
		read_count = count;
	}

    
#ifdef USB_URB_FUNCTION
    pipe = usb_rcvbulkpipe(dev->udev, dev->bulk_in_endpointAddr_read);
    
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb){
        MSG(KERN_ERR_LVL, minor, "urb allocate error!!\n");
        retval = -ENOMEM;
        goto exit;
	}
	
	usb_fill_bulk_urb(urb, dev->udev, pipe, 
				(char *) buffer_addr, read_count, 
			    fmb_hw_usb_cmd_completion, NULL);
			     
	init_completion(&done); 	
	urb->context = &done;
	urb->actual_length = 0;
	status = usb_submit_urb(urb, GFP_KERNEL);
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "usb_submit_urb error!! status=%d\n",status);
		usb_free_urb(urb);
        retval = status;
        goto exit;
	}
	
	wait_for_completion_timeout(&done,USB_BULK_MES_TIMEOUT_READ);
	status = urb->status;
	
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "wait_for_completion_timeout status=%d!!\n",status);
		usb_free_urb(urb);
        retval = status;
        goto exit;
	}
	
	bytes_read = urb->actual_length;
	retval = bytes_read;
	
	usb_free_urb(urb);
	
#else
    /* do a blocking bulk read to get data from the device */
    retval = usb_bulk_msg(dev->udev,
                  usb_rcvbulkpipe(dev->udev, dev->bulk_in_endpointAddr_read),
                  buffer_addr,
                  read_count,
                  &bytes_read, USB_BULK_MES_TIMEOUT_READ);
#endif
#ifdef DEBUGON_USB_ACCESS
    {
        if ( fmb_usb_debugmsg_check( dev, FMB_USB_DEBUGMSG_BULKIN ) == 0 )
        {
            if ( retval == 0 )
            {
                int minor = -1;
                FMB_GET_HW_MINOR( minor, priv_p );
                sprintf( dbuf, "EA:%02x DS:%05d RDS:%05d", dev->bulk_in_endpointAddr_read, read_count, bytes_read );
                strcat( dbuf, " data(RCV):" );
                debug_data_dump( dbuf, buffer_addr, bytes_read, 10 );
                info( "%03d:usbBLK:%s", minor, dbuf );
            }
        }
    }
#endif /* DEBUGON_USB_ACCESS */

    /* if the read was successful, copy the data to userspace */
    if ( bytes_read > 0 ) {
        if ( buffer_addr == dev->bulk_in_buffer_read )
        {
            if (copy_to_user(buffer, dev->bulk_in_buffer_read, bytes_read))
                retval = -EFAULT;
            else
                retval = bytes_read;
        }else{
            retval = bytes_read;
        }
    }

exit:
    mutex_unlock(&dev->i_mutex);
    return retval;
}

/**
*    @brief        write API
*    @param[in]    file   linux fops struct file
*    @param[out]   buffer write data buffer
*    @param[in]    count  write data buffer size
*    @param[in]    ppos   linux fops struct ppos
*    @return       (0<=):actual write size (0>):error
*    @note         None
*    @attention    None
*/
static ssize_t fmb_hw_usb_write(struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
    struct fmb_hw_usb *dev;
    int retval;
    int bytes_write;
    struct fmb_hard_private*    priv_p;
    char *buf = NULL;
    size_t write_count;
    unsigned char *buffer_addr;
#ifdef USB_URB_FUNCTION
    int          minor = -1;
	struct urb *urb;
	struct completion	done;
	int			status;
	unsigned int pipe;
#endif
    
    priv_p = (struct fmb_hard_private *)file->private_data;
    PRIV_P_CHECK( priv_p, -ENODEV );
    dev    = priv_p->dev_priv;
    PRIV_P_CHECK( dev, -ENODEV );
    
    PARAM_CHECK( buffer, -EINVAL );

    if ( dev->bulk_out_endpointAddr == 0 ) return -EPERM;

    buffer_addr = fmb_hw_usb_trans_get_buf_by_useraddr(dev, (unsigned char *)buffer);
    if(buffer_addr == NULL)
    {
        buf = dev->bulk_out_buffer;
    }

    mutex_lock(&dev->o_mutex);
    if (!dev->interface)
    {        /* disconnect() was called */
        retval = -ENODEV;
        goto error;
    }

    if ( buffer_addr == NULL )
    {
        write_count = min(dev->bulk_out_size, count);
        if (copy_from_user(buf, buffer, write_count))
        {
            retval = -EFAULT;
            goto error;
        }
        buffer_addr = buf;
    }
    else
    {
        write_count = count;
    }
#ifdef USB_URB_FUNCTION
    pipe = usb_sndbulkpipe(dev->udev, dev->bulk_out_endpointAddr);
    
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb){
        MSG(KERN_ERR_LVL, minor, "urb allocate error!!\n");
        retval = -ENOMEM;
        goto error;
	}
	
	usb_fill_bulk_urb(urb, dev->udev, pipe, 
				(char *) buffer_addr, write_count, 
			    fmb_hw_usb_cmd_completion, NULL);
			     
	init_completion(&done); 	
	urb->context = &done;
	urb->actual_length = 0;
	status = usb_submit_urb(urb, GFP_KERNEL);
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "usb_submit_urb error!! status=%d\n",status);
		usb_free_urb(urb);
        retval = status;
        goto error;
	}
	
	wait_for_completion_timeout(&done,10*HZ);
	status = urb->status;
	
	if ( status != 0 ){
		MSG(KERN_ERR_LVL, minor, "wait_for_completion_timeout status=%d!!\n",status);
		usb_free_urb(urb);
        retval = status;
        goto error;
	}
	
	bytes_write = urb->actual_length;
	retval = bytes_write;
	
	usb_free_urb(urb);

#else
    /* do a blocking bulk write to set data from the device */
    retval = usb_bulk_msg(dev->udev,
                  usb_sndbulkpipe(dev->udev, dev->bulk_out_endpointAddr),
                  buffer_addr,
                  write_count,
                  &bytes_write, 10*HZ );
#endif

#ifdef DEBUGON_USB_ACCESS
    {
        if ( fmb_usb_debugmsg_check( dev, FMB_USB_DEBUGMSG_BULKOUT ) == 0 )
        {
            if ( retval == 0 )
            {
                int minor = -1;
                FMB_GET_HW_MINOR( minor, priv_p );
                sprintf( dbuf, "EA:%02x DS:%05d WDS:%05d", dev->bulk_out_endpointAddr, write_count, bytes_write );
                strcat( dbuf, " data(SND):" );
                debug_data_dump( dbuf, buffer_addr, bytes_write, 10 );
                info( "%03d:usbBLK:%s", minor, dbuf );
            }
        }
    }
#endif /* DEBUGON_USB_ACCESS */
    if (!retval)
    {
            retval = bytes_write;
    }

error:
    mutex_unlock(&dev->o_mutex);
    return retval;
}

#ifndef PAGE_SHIFT
#error not define PAGE_SHIFT
#endif
/**
*    @brief        mmap API
*    @param[in]    fp_p   linux fops struct file
*    @param[io]    vma    linux fops struct vm_area_struct
*    @return       
*    @note         None
*    @attention    None
*/
static int fmb_hw_usb_mmap(struct file* fp_p, struct vm_area_struct *vma)
{
    struct fmb_hard_private* priv;
    struct fmb_hw_usb* usb_priv_p;
  
    int retval = 0, bufidx, work;
    unsigned long size;
    
    int minor = 0;
    
    priv = (struct fmb_hard_private*)fp_p->private_data;
    PRIV_P_CHECK( priv, -ENODEV );
    
    usb_priv_p = priv->dev_priv;
    PRIV_P_CHECK( usb_priv_p, -ENODEV );

    MSG(DEBUG_LVL, minor,
            " ** Enter :%s()[%d] mmap request is recevied.s:%08x,e:%08x,pgoff:%d,PS:%d,SIZE:%d,PG<<PS:%d\n",
        __FUNCTION__,__LINE__,(unsigned int)vma->vm_start, (unsigned int)vma->vm_end, (int)vma->vm_pgoff, (int)PAGE_SHIFT,(int)FMB_USB_TRANS_BUFF_SIZE,(int)vma->vm_pgoff<<PAGE_SHIFT);

    // check parameters
    size = vma->vm_end - vma->vm_start;
//  bufidx = (vma->vm_pgoff << PAGE_SHIFT) / FMB_USB_TRANS_BUFF_SIZE;
//  bufidx = (vma->vm_pgoff) / ( FMB_USB_TRANS_BUFF_SIZE >> PAGE_SHIFT);

    //ToDo: for division by zero  OK???
    work   = ( FMB_USB_TRANS_BUFF_SIZE >> PAGE_SHIFT);
    if ( work != 0 )
    {
        bufidx = (vma->vm_pgoff) / work;
    }
    else
    {
        bufidx = 0;
    }

    if( size > FMB_USB_TRANS_BUFF_SIZE )
    {
            MSG(ERR_LVL, minor,
                "%s()[%d] : [USB TRANS] mmap request size is too big.[size:%ld]\n",
                __FUNCTION__,__LINE__,size);
            return -EINVAL;
    }
    if( bufidx >= FMB_USB_TRANS_BUFF_NUM)
    {
            MSG(ERR_LVL, minor,
                "%s()[%d] : [USB TRANS] mmap request offset is too big.[bufidx:%d, vm_pgoff:%ld]\n",
                __FUNCTION__,__LINE__,bufidx,vma->vm_pgoff);
            return -EINVAL;
    }

    // remap selected kernel buffer to the user buffer
    vma->vm_flags |= (VM_IO | VM_RESERVED);
#ifdef TV_LINUX_FUNCTION
    vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
#endif /* TV_LINUX_FUNCTION */
    retval = remap_pfn_range(vma, vma->vm_start, virt_to_phys(usb_priv_p->buffers[bufidx]) >> PAGE_SHIFT, size, vma->vm_page_prot);
    if( retval ) {
            MSG(ERR_LVL, minor,
                "%s()[%d] : [USB TRANS] Error!!!! remap_page_range (%d).\n",
                __FUNCTION__,__LINE__,retval);
            return -EAGAIN;
    }
    usb_priv_p->buffers_useraddr[bufidx] = (unsigned char *) vma->vm_start;

    MSG(DEBUG_LVL, minor,
            "[USB TRANS] buffer %d (Kern 0x%08x) is mapped to (User 0x%08x) in %s()[%d]\n",
                bufidx, (unsigned int) usb_priv_p->buffers[bufidx], (unsigned int) vma->vm_start
                ,__FUNCTION__,__LINE__);

    MSG(DEBUG_LVL, minor,
            " ** Exit  :%s()[%d]\n",__FUNCTION__,__LINE__);

    return 0;
}


/**
*    @brief        initialize of mmap area
*    @param[in]    dev    private data pointer
*    @return       0
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_mmaparea_init( struct fmb_hw_usb *dev )
{
    int i  = 0;

    for(i = 0; i < FMB_USB_TRANS_BUFF_NUM; i++)
    {
        dev->buffers[i] = NULL;
    }
    
    return 0;
}

/**
*    @brief        the memory for mmap is allocated. 
*    @param[in]    dev    private data pointer
*    @return       0
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_mmaparea_alloc( struct fmb_hw_usb *dev )
{
    int i  = 0;
    int rc = 0;
    int minor = -1;

    FMB_GET_HW_MINOR( minor, dev->hw_priv_p );

    /** mmap area alloc **/
    // added 090912
    /* allocate buffers for transfer user data */
    for(i = 0; i < FMB_USB_TRANS_BUFF_NUM; i++)
    {
        dev->buffers[i] = (unsigned char *)kmalloc(FMB_USB_TRANS_BUFF_SIZE, GFP_KERNEL);
        if(dev->buffers[i] == NULL)
        {
            err(
                "%03d:[USB TRANS] Error: Out of memory", minor );
            rc = -ENOMEM;
            break;
        }

        MSG(DEBUG_LVL, minor,
            "[USB TRANS] Buffer #%d: addr = 0x%08x (phys 0x%08x) size = 0x%08x", 
            i, (unsigned int) dev->buffers[i], (unsigned int) virt_to_phys(dev->buffers[i]), FMB_USB_TRANS_BUFF_SIZE);

        if( (((unsigned int) dev->buffers[i]) & ((1 << PAGE_SHIFT) - 1)) != 0)
        {
            err(
                "%03d:[USB TRANS] Error: Memory buffer is not aligned to memory page.[0x%08x]", 
                minor, (unsigned int) dev->buffers[i]);
            rc = -ENOMEM;
            break;
        }
    }
    
    if ( rc < 0 )
    {
        info( "%03d:mmap area  : Allocate Error", minor );
    }
    else
    {
        info( "%03d:mmap area  : page size[%d] page number[%d]", minor, FMB_USB_TRANS_BUFF_SIZE, FMB_USB_TRANS_BUFF_NUM );
    }
    
    return rc;
}

/**
*    @brief        free mmap area 
*    @param[in]    dev    private data pointer
*    @return       0
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_mmaparea_free( struct fmb_hw_usb *dev )
{
    int i;
    /** mmap area free **/
    for(i = FMB_USB_TRANS_BUFF_NUM - 1; i >= 0; i--)
    {
        if ( dev->buffers[i] != NULL )
        {
            kfree(dev->buffers[i]);
            dev->buffers[i] = NULL;
        }
    }
    
    return 0;
}

int
fmb_hw_usb_is_exist_intr_endpoint( struct fmb_hw_usb *dev )
{
    if ( dev->interrupt_in_endpointAddr == 0 )
    {
        return 0;
    }
    else
    {
        return 1;
    }
}

/**
*    @brief        initialize endpoint info
*    @param[in]    dev    private data pointer
*    @return       0
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_probe_endpoint_init( struct fmb_hw_usb * dev )
{
    dev->bulk_in_buffer = NULL;
    dev->bulk_in_buffer2 = NULL;
    dev->bulk_out_buffer = NULL;
    
    dev->bulk_out_endpointAddr     = 0;
    dev->bulk_in_endpointAddr      = 0;
    dev->bulk_in_endpointAddr2     = 0;
    dev->interrupt_in_endpointAddr = 0;

#ifdef USB_CTRLMSG_ONLY
    dev->control_msg_only_flg = 1;
#else
    dev->control_msg_only_flg = 0;
#endif

    return 0;
}

/**
*    @brief        probe endpoint info
*    @param[in]    dev    private data pointer
*    @return       0
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_probe_endpoint( struct fmb_hw_usb *dev )
{
    int i = 0;
    struct usb_host_interface *iface_desc;
    struct usb_endpoint_descriptor *endpoint;
    int bulk_in_cnt = 0;
    size_t buffer_size;
    int ret = 0;
    int minor =-1;
    
    dev->control_in_endpointAddr  = 0x80;
    dev->control_out_endpointAddr = 0x00;
    
    if ( dev->control_msg_only_flg == 1 ) return 0;

    /* set up the endpoint information */
    /* use only the first bulk-in and bulk-out endpoints */
    iface_desc = dev->interface->cur_altsetting;
    for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i) {
        endpoint = &iface_desc->endpoint[i].desc;
        if ( bulk_in_cnt == 0 )
        {
            if (!dev->bulk_in_endpointAddr &&
                usb_endpoint_is_bulk_in(endpoint))
            {
                /* we found a bulk in endpoint */
                buffer_size = le16_to_cpu(endpoint->wMaxPacketSize);
                dev->bulk_in_size = buffer_size;
                dev->bulk_in_endpointAddr = endpoint->bEndpointAddress;
                dev->bulk_in_buffer = (unsigned char *)kmalloc(buffer_size, GFP_KERNEL);
                if (!dev->bulk_in_buffer) {
                    err("Could not allocate bulk_in_buffer");
                    ret = -ENOMEM;
                    break;
                }
                MSG(DEBUG_LVL, minor,"endpoint[%x] BULK-IN      addr = 0x%02x size = 0x%08x", dev->bulk_in_endpointAddr & 0x7f, dev->bulk_in_endpointAddr, dev->bulk_in_size );
                bulk_in_cnt = 1;
            }
        }
        else if ( bulk_in_cnt == 1 )
        {
            if (!dev->bulk_in_endpointAddr2 &&
                usb_endpoint_is_bulk_in(endpoint))
            {
                /* we found a bulk in endpoint */
                buffer_size = le16_to_cpu(endpoint->wMaxPacketSize);
                dev->bulk_in_size2 = buffer_size;
                dev->bulk_in_endpointAddr2 = endpoint->bEndpointAddress;
                dev->bulk_in_buffer2 = (unsigned char *)kmalloc(buffer_size, GFP_KERNEL);
                if (!dev->bulk_in_buffer2) {
                    err("Could not allocate bulk_in_buffer2");
                    ret = -ENOMEM;
                    break;
                }
                MSG(DEBUG_LVL, minor,"endpoint[%x] BULK-IN      addr = 0x%02x size = 0x%08x",dev->bulk_in_endpointAddr2 & 0x7, dev->bulk_in_endpointAddr2, dev->bulk_in_size2 );
                bulk_in_cnt = 2;
            }
        }
        if (!dev->bulk_out_endpointAddr &&
            usb_endpoint_is_bulk_out(endpoint))
        {
            /* we found a bulk out endpoint */
            dev->bulk_out_endpointAddr = endpoint->bEndpointAddress;
            dev->bulk_out_size = le16_to_cpu(endpoint->wMaxPacketSize);
            dev->bulk_out_buffer = (unsigned char *)kmalloc(dev->bulk_out_size, GFP_KERNEL);
            if (!dev->bulk_out_buffer) {
                    err("Could not allocate bulk_out_buffer");
                    ret = -ENOMEM;
                    break;
            }
            MSG(DEBUG_LVL, minor,"endpoint[%x] BULK-OUT     addr = 0x%02x size = 0x%08x",dev->bulk_out_endpointAddr, dev->bulk_out_endpointAddr, dev->bulk_out_size );
        }
        
        if (!dev->interrupt_in_endpointAddr &&
            usb_endpoint_is_int_in(endpoint))
        {
                /* we found a interrupt in endpoint */
                dev->interrupt_in_endpointAddr = endpoint->bEndpointAddress;
                MSG(DEBUG_LVL, minor,"endpoint[%x] INTERRUPT-IN addr = 0x%02x size = 0x%08x",dev->interrupt_in_endpointAddr & 0x7f, dev->interrupt_in_endpointAddr, endpoint->wMaxPacketSize );
        }
    }

    dev->bulk_in_size_read         = dev->bulk_in_size;
    dev->bulk_in_endpointAddr_read = dev->bulk_in_endpointAddr;
    dev->bulk_in_buffer_read       = dev->bulk_in_buffer;

    if (!(dev->bulk_out_endpointAddr)) {
        info("Could not find bulk-out endpoints");
    }
    if (!(dev->bulk_in_endpointAddr)) {
        info("Could not find bulk-in endpoints");
    }
    if(dev->udev->descriptor.idProduct == USB_PRODUCT_ID_FUJITSU_MB86H57H58){
        if (!(dev->bulk_in_endpointAddr2)) {
            info("Could not find bulk-in2 endpoints");
        }
    }
    if (!(dev->interrupt_in_endpointAddr)) {
        info("Could not find interrupt in endpoints");
    }

    return ret;
}

/**
*    @brief        free endpoint info
*    @param[in]    dev    private data pointer
*    @return       0
*    @note         None
*    @attention    None
*/
int
fmb_hw_usb_probe_endpoint_free( struct fmb_hw_usb * dev )
{
    if ( dev->bulk_in_buffer != NULL )
    {
        kfree( dev->bulk_in_buffer );
        dev->bulk_in_buffer = NULL;
    }
    if ( dev->bulk_in_buffer2 != NULL )
    {
        kfree( dev->bulk_in_buffer2 );
        dev->bulk_in_buffer2 = NULL;
    }
    if ( dev->bulk_out_buffer != NULL )
    {
        kfree( dev->bulk_out_buffer );
        dev->bulk_out_buffer = NULL;
    }

    return 0;
}

extern struct fmb_hw_obj fmb_hw_obj_link;
static struct file_operations fmb_hw_usb_fops;

/*
 * usb class driver info in order to get a minor number from the usb core,
 * and to have the device registered with the driver core
 */
static struct usb_class_driver fmb_hw_usb_class;

/**
*    @brief        fops and class of usb driver are registered. 
*    @return       None
*    @note         None
*    @attention    None
*/
static void
fmb_hw_usb_fops_and_class_reg( void )
{
    fmb_hw_usb_fops.owner     =  THIS_MODULE;
    fmb_hw_usb_fops.read      =  fmb_hw_usb_read;
    fmb_hw_usb_fops.write     =  fmb_hw_usb_write;
//  fmb_hw_usb_fops.poll      =  fmb_hw_poll;
    fmb_hw_usb_fops.mmap      =  fmb_hw_usb_mmap;

    fmb_hw_usb_fops.ioctl     =  fmb_hw_obj_link.fops.ioctl;
    fmb_hw_usb_fops.open      =  fmb_hw_obj_link.fops.open;
    fmb_hw_usb_fops.release   =  fmb_hw_obj_link.fops.release;
    fmb_hw_usb_class.name       =   "h57fmb%d";
    fmb_hw_usb_class.fops       =   &fmb_hw_usb_fops;
    fmb_hw_usb_class.minor_base =   USB_FMB_MINOR_BASE;

    return;
}

int
fmb_hw_is_exist_intr( struct fmb_hard_private *hw_priv_p )
{
    return fmb_hw_usb_is_exist_intr_endpoint( hw_priv_p->dev_priv );
}

static int fmb_hw_usb_probe_get_devstring_info(  char *title, unsigned flag, char *out_work, char *in_work )
{
    int work_len = 0;
    char title_work[]=" *** "; 
    
    if ( flag != 0 )
    {
        if ( in_work != NULL )
        {
            if ( title == NULL ) title = title_work;
            
            work_len = strlen(in_work);
            if ( work_len > FMB_DEVICE_INFO_STRING_SIZE - 1 ) work_len = FMB_DEVICE_INFO_STRING_SIZE - 1;
            
            memcpy( out_work, in_work, work_len );
            out_work[work_len] = '\0';
            info("%14s : %s", title, out_work );
        }
        else strcpy( out_work, "" );
    }
    return 0;
}

/**
*    @brief        probe usb device 
*    @param[in]    interface
*    @param[in]    id
*    @return       None
*    @note         None
*    @attention    None
*/
static int fmb_hw_usb_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
    struct fmb_hw_usb *dev;
    struct fmb_hard_private*    priv_p;
    int retval = -ENOMEM;
    int result = 0;
    int rc;

    int             minor = -1;
    
    
    priv_p = fmb_hw_private_alloc();
    if ( priv_p == NULL )
    {
        return -ENOMEM;
    }

    fmb_hw_private_data_init(priv_p);        
    if(result < 0){
        MSG(INTERNAL_ERR_LVL, minor, "memmap error");
        return -ENOMEM;
    }

    /* allocate memory for our device state and initialize it */
    dev = (struct fmb_hw_usb *)kzalloc(sizeof(*dev), GFP_KERNEL);
    if (!dev) {
        err("fmb_hw_usb Out of memory");
        goto error;
    }
    /* allocate memory for our device state and initialize it */
    dev->device_info = (struct fmb_device_info *)kzalloc(sizeof(*(dev->device_info)), GFP_KERNEL);
    if (!dev) {
        err("fmb_device_info Out of memory");
        goto error;
    }

#ifdef MB86E61_FUNCTION
    dev->cmd_ctl_info_apl = kmalloc(sizeof(struct fmb_cmd_ctl_info), GFP_KERNEL);
    if(dev->cmd_ctl_info_apl == NULL){
        err("Out of memory(cmd_ctl_info_apl)");
        goto error;
    }
    dev->cmd_ctl_info_ep = kmalloc(sizeof(struct fmb_cmd_ctl_info), GFP_KERNEL);
    if(dev->cmd_ctl_info_ep == NULL){
        err("Out of memory(cmd_ctl_info_ep)");
        goto error;
    }
#endif /* MB86E61_FUNCTION */

    priv_p->dev_priv = dev;
    kref_init(&priv_p->kref);
    sema_init(&dev->limit_sem, WRITES_IN_FLIGHT);
    mutex_init(&dev->i_mutex);
    mutex_init(&dev->i2_mutex);
    mutex_init(&dev->o_mutex);
    mutex_init(&dev->io_mutex);

    dev->udev = usb_get_dev(interface_to_usbdev(interface));
    dev->interface = interface;
    
    dev->hw_priv_p = (void *)priv_p;

    fmb_hw_usb_probe_endpoint_init( dev );
    fmb_hw_usb_mmaparea_init( dev );
    
#ifdef MULTIENDIAN_FUNCTION
    dev->device_info->idVendor  = le16_to_cpu(dev->udev->descriptor.idVendor);
    dev->device_info->idProduct = le16_to_cpu(dev->udev->descriptor.idProduct);
#else
    dev->device_info->idVendor  = dev->udev->descriptor.idVendor;
    dev->device_info->idProduct = dev->udev->descriptor.idProduct;
#endif
    dev->device_info->speed     = dev->udev->speed;

    info("--- fmb driver version : %s ---", VERSION);
    info(" vendor    : 0x%04x", dev->device_info->idVendor );
    info(" product   : 0x%04x", dev->device_info->idProduct);
    info(" speed     : 0x%x",   dev->device_info->speed );
    
    fmb_hw_usb_probe_get_devstring_info( " Manufacturer ", dev->udev->descriptor.iManufacturer, dev->device_info->manufacturer, dev->udev->manufacturer );
    fmb_hw_usb_probe_get_devstring_info( " Product      ", dev->udev->descriptor.iProduct,      dev->device_info->product,      dev->udev->product );
    fmb_hw_usb_probe_get_devstring_info( " SerialNumber ", dev->udev->descriptor.iSerialNumber, dev->device_info->serial,       dev->udev->serial );
    
    result = Fmb_core_probe(priv_p);
    if(result < 0){
        MSG(INTERNAL_ERR_LVL, minor, "core_probe error");
        return -ENOMEM;
    }        

    priv_p->core_priv_p->eject_event_flag = FMB_EJECT_EVENT_OFF;
    priv_p->core_priv_p->eject_event_type = FMB_EJECT_EVENT_NO_CANCEL;

    priv_p->core_priv_p->fact_irq_flag     = FMB_FACT_IRQ_OFF;
    priv_p->core_priv_p->fact_irq_act_flag = FMB_FACT_IRQ_OFF;
    priv_p->core_priv_p->fact_irq_type     = FMB_FACT_IRQ_NO_CANCEL;
    priv_p->core_priv_p->fact_irq_stop_flag = FMB_FACT_IRQ_NO_CANCEL;

    priv_p->core_priv_p->wait_bit_flag     = FMB_WAIT_BIT_OFF;
    priv_p->core_priv_p->wait_bit_act_flag = FMB_WAIT_BIT_OFF;
    priv_p->core_priv_p->wait_bit_type     = FMB_WAIT_BIT_NO_CANCEL;
#ifdef MB86E61_FUNCTION
    priv_p->core_priv_p->boot_event_flag   = FMB_BOOT_EVENT_OFF;
    priv_p->core_priv_p->boot_event_type = FMB_BOOT_EVENT_NO_CANCEL;
#endif /* MB86E61_FUNCTION */



    result = fmb_hw_usb_probe_endpoint( dev );
    if ( result < 0 )
    {
        fmb_hw_usb_probe_endpoint_free( dev );
        goto error;
    }
#ifdef DEBUGON_USB_ACCESS
    fmb_usb_debug_msg_init( &(dev->usb_debug_msg) );
#endif /* DEBUGON_USB_ACCESS */
    
    /* save our data pointer in this interface device */
    usb_set_intfdata(interface, (void*)priv_p);

    fmb_hw_usb_fops_and_class_reg();

    /* we can register the device now, as it is ready */
    retval = usb_register_dev(interface, &fmb_hw_usb_class);
    if (retval) {
        /* something prevented us from registering this driver */
        err("Not able to get a minor for this device.");
        usb_set_intfdata(interface, NULL);
        goto error;
    }
    minor = priv_p->minor = interface->minor;
    
#ifdef USB_CTRLMSG_ONLY
    info("%03d:Endpoint construction : Control Message Only Mode", minor );
#endif
    if ( fmb_hw_usb_is_exist_intr_endpoint( dev ) )
    {
    /* create kernel thread. */
    init_completion(&dev->kt_comp);
    rc = kernel_thread(fmb_hw_usb_polling_thread, priv_p, 0);
    if(rc < 0){
        /* Release polling related resources. */
        fmb_hw_usb_polling_cleanup(dev);
        err("%03d:error kernel_thread creation", minor);
        return -EBUSY;
    }

    /* wait for kernel thread creation is completed. */
    wait_for_completion(&dev->kt_comp);

	}

    rc = fmb_hw_usb_mmaparea_alloc( dev );
    if ( rc < 0 )
    {
        goto error;
    }
    /* let the user know what node this device is now attached to */
    info("%03d:USB fmb device now attached to %s", minor, FMB_DRIVER_NAME );
    info("%03d: major     : %d     minor     : %d", minor, USB_MAJOR, priv_p->minor );
    
#ifdef DEBUGON_USB_ACCESS
    info("%03d:DEBUG MODE : USB ACCESS TRACE ON", minor);
#endif /* DEBUGON_USB_ACCESS */
#ifdef DEBUGON_I2C_ACCESS
    info("%03d:DEBUG MODE : I2C ACCESS TRACE ON", minor);
#endif /* DEBUGON_I2C_ACCESS */
#ifdef FMB_OPT_XERROR_WATCH
    info("%03d:OPTION     : XERROR Watch Mode ON", minor);
#endif /* FMB_OPT_XERROR_WATCH */

    fmb_usb_spirom_get_version( priv_p );
    
    return 0;

    
error:
    if (dev)
        kref_put(&priv_p->kref, fmb_hw_usb_delete );
    return retval;
}

/**
*   @brief      FMB system call remove()
*   @param[in]  interface     usb interface Structure pointer
*   @return     0
*   @note       None
*   @attention  None
*/
static void __devexit fmb_hw_usb_remove(struct usb_interface* interface)
{
    int             minor = -1;
    struct fmb_hard_private* priv_p;
    struct fmb_hw_usb *dev;

    priv_p = (struct fmb_hard_private*)usb_get_intfdata(interface);
    
    if ( priv_p == NULL ) MSG(INTERNAL_ERR_LVL, minor, "hard private data cannot be gotten.(abnormal)");
    
    minor = priv_p->minor;

    dev = priv_p->dev_priv;
    
    MSG(DRIVER_FUNC_LVL, minor, "START");
    
    if ( dev == NULL )     MSG(INTERNAL_ERR_LVL, minor, "usb private data cannot be gotten.(abnormal)");

    if ( fmb_hw_usb_is_exist_intr_endpoint( dev ) )
    {
    fmb_hw_usb_polling_cleanup(dev);
    }
    
    if ( priv_p->core_priv_p == NULL )     MSG(INTERNAL_ERR_LVL, minor, "core private data cannot be gotten.(abnormal)");
    
    priv_p->core_priv_p->eject_event_type = FMB_EJECT_EVENT_REMOVE_CANCEL;
    priv_p->core_priv_p->eject_event_flag = FMB_EJECT_EVENT_ON;
    FMB_SYS_ABS_wake_up( &priv_p->core_priv_p->eject_event_wait );

    priv_p->core_priv_p->fact_irq_type    = FMB_FACT_IRQ_REMOVE_CANCEL;
    priv_p->core_priv_p->fact_irq_stop_flag = FMB_FACT_IRQ_REMOVE_CANCEL;
    priv_p->core_priv_p->fact_irq_flag    = FMB_FACT_IRQ_ON;
    FMB_SYS_ABS_wake_up( &priv_p->core_priv_p->fact_irq_wait );

    priv_p->core_priv_p->wait_bit_type    = FMB_WAIT_BIT_REMOVE_CANCEL;
    priv_p->core_priv_p->wait_bit_flag    = FMB_WAIT_BIT_ON;
    FMB_SYS_ABS_wake_up( &priv_p->core_priv_p->wait_bit_wait );

#ifdef MB86E61_FUNCTION
    priv_p->core_priv_p->boot_event_type = FMB_BOOT_EVENT_REMOVE_CANCEL;
    priv_p->core_priv_p->boot_event_flag = FMB_BOOT_EVENT_ON;
    FMB_SYS_ABS_wake_up( &priv_p->core_priv_p->boot_event_wait );
#endif /* MB86E61_FUNCTION */

    lock_kernel();
    
    usb_set_intfdata(interface, NULL);
    
    usb_deregister_dev(interface, &fmb_hw_usb_class);
    
    mutex_lock(&dev->i_mutex);
    mutex_lock(&dev->i2_mutex);
    mutex_lock(&dev->o_mutex);
    mutex_lock(&dev->io_mutex);
    dev->interface = NULL;
    mutex_unlock(&dev->io_mutex);
    mutex_unlock(&dev->o_mutex);
    mutex_unlock(&dev->i2_mutex);
    mutex_unlock(&dev->i_mutex);

    unlock_kernel();
    
    kref_put(&priv_p->kref, fmb_hw_usb_delete);

    MSG(DRIVER_FUNC_LVL, minor, "END");
    
    info("%03d:USB fmb device now detached to %s(%d,%d)", minor, FMB_DRIVER_NAME, USB_MAJOR, minor );
}

static void fmb_hw_usb_polling_cleanup(struct fmb_hw_usb *dev)
{
    /* Parameter check*/
    if(dev==NULL){
        /* Noting to do. */
        return;
    }

    /* if kernel thread exists, kill it. */
    if(dev->thread_info.thread_running==1){
        
        init_completion(&dev->kt_comp);
        
        /* Clear running flag and wake up kernel thread. */
        dev->thread_info.thread_running=0;
        
        /* wait for kernel thread dies. */
        wait_for_completion(&dev->kt_comp);
    }
    

    return;
}

/**
*    @brief        usb polling(interrupt in messages) 
*    @param[in]    arg
*    @return       None
*    @note         None
*    @attention    None
*/
static int fmb_hw_usb_polling_thread( void  * arg)
{
    struct fmb_hard_private*    priv_p;
    struct fmb_hw_usb *hw_usb;
    struct fmb_core_private* core_priv_p;
    unsigned char *pData;
    int actual_length = 0;
    unsigned char continuity_counter;
    unsigned char cack_ID;
    unsigned char sub_cack_ID;
    int ret = 0;
    int minor;
    char    kthread_name[64];
#ifdef USB_URB_FUNCTION
    int isWait=0;
#endif
    
    /* Parameter check*/
    if(arg==NULL){
        return -1;
    }
    priv_p   = (struct fmb_hard_private *)arg;
    hw_usb   = priv_p->dev_priv;
    core_priv_p = priv_p->core_priv_p;

    /* Data area for receiving interrupt transfer data */
    pData = (unsigned char *)kmalloc(USB_INTR_MSG_MAXSIZE, GFP_KERNEL);
    if(pData == NULL){
        return -1;
    }

    minor = priv_p->minor;
    
    sprintf( kthread_name, "h57poll_%03d", minor );
    
    core_priv_p->rcv_get_msg_cnt = 0;

    /* Go to back-ground. */
    daemonize( kthread_name );
    current->flags |= PF_NOFREEZE;
    
    /* Set running flag. */
    hw_usb->thread_info.thread_running=1;
    
    /* Notify kernel thread creation completed. */
    complete(&hw_usb->kt_comp);

    /* 
     * kernel thread loop. 
     */
    while(1){
        /* Check thread running flag. */
        if(hw_usb->thread_info.thread_running==0){
            break;
        }

#ifdef USB_URB_FUNCTION
        if(isWait==1){
            ret = fmb_hw_usb_ep_interrupt_wait_for_completion(hw_usb, pData, USB_INTR_MSG_MAXSIZE, &actual_length, 1*HZ );
        }else{
            memset(pData, 0, USB_INTR_MSG_MAXSIZE);
            ret = fmb_hw_usb_ep_interrupt_read(hw_usb, pData, USB_INTR_MSG_MAXSIZE, &actual_length, 1*HZ );
        }
#else
        memset(pData, 0, USB_INTR_MSG_MAXSIZE);

        /* check interruput transfer */
        ret = fmb_hw_usb_ep_interrupt_read(hw_usb, pData, USB_INTR_MSG_MAXSIZE, &actual_length, 1*HZ );
#endif        
        if(ret < 0){
#ifdef USB_URB_FUNCTION
            if ( ret == -EINPROGRESS ){
                isWait = 1;
            }else{
                isWait = 0;
#endif        
            set_current_state(TASK_INTERRUPTIBLE);
            schedule_timeout(1);
#ifdef USB_URB_FUNCTION
            }
#endif        
        }
        else{
#ifdef USB_URB_FUNCTION
            isWait = 0;
#endif
            switch(*(pData)){
                case GET_CMD_ACK:
                    if ( actual_length != USB_CMDACK_SIZE )
                    {
                         MSG(KERN_ERR_LVL, minor, "The size of CMD_ACK format and actual is different.  [INTR]CD:0x%02x,CLEN:0x%02x,CCNT:0x%02x,ID:0x%02x,SUBID:0x%02x,AC:%d,DS:%d,RET:%d"
            ,*(pData), *(pData+2), *(pData+3), *(pData+4), *(pData+5), actual_length, USB_CMDACK_SIZE, ret  );
                    }
                    /* Show that there was command response corresponding to SET_API_CMD which issued it for MB86H57. */
                    continuity_counter = *(pData+3); /* ToDo: ???? */
                    cack_ID =  *(pData+4);
                    sub_cack_ID =  *(pData+5); 
                    fmb_core_tack_func( (unsigned long)priv_p, (unsigned short)cack_ID, (unsigned short)sub_cack_ID, continuity_counter );
                break;
                case GET_MSG:
                    if ( actual_length != USB_GETMSG_SIZE )
                    {
                         MSG(KERN_ERR_LVL, minor, "The size of GET_MSG format and actual is different.  [INTR]CD:0x%02x,CLEN:0x%02x,AC:%d,DS:%d,RET:%d"
            ,*(pData), *(pData+2), actual_length, USB_GETMSG_SIZE, ret  );
                    }
                    /* Show that there was message notification from MB86H57/H58. */
                    core_priv_p->rcv_get_msg_cnt++;

                    if(core_priv_p->get_msg_act_flag == 1){
                        core_priv_p->sys_cmd_msg_rcv_flag  = FMB_SYS_CMD_MSG_RECEIVE;
                        FMB_SYS_ABS_wake_up_interruptible( &core_priv_p->sys_msg_wait );
                    }else{
                    }
                    if(core_priv_p->fact_irq_act_flag == 1){
                        core_priv_p->fact_irq_type         = FMB_FACT_IRQ_EVENT;
                        core_priv_p->fact_irq_flag         = FMB_FACT_IRQ_ON;
                        FMB_SYS_ABS_wake_up( &core_priv_p->fact_irq_wait );
                    }else{
                    }
#ifdef FMB_OPT_XERROR_WATCH
                {
                    unsigned long val_status;
                    int ret;
                    ret = Fmb_core_lsi_get_irq_status( priv_p, &val_status );
                    if (  ret >= 0 )
                    {
                        if ( ( val_status & FMB_FACT_XERROR ) != 0 )
                        {
#define FMB_MW_MSG_XERROR_ADDR  0x80080
#define FMB_MW_MSG_XERROR_SIZE  (16)
                            int li = 0;
                            int lret;
                            int lwp = 0;
                            unsigned short work;
                            unsigned char lbuf[FMB_MW_MSG_XERROR_SIZE*3+1];
                            for(li=0;li<FMB_MW_MSG_XERROR_SIZE;li=li+2)
                            {
                                work = Fmb_hw_reg_read_lsi( priv_p, FMB_MW_MSG_XERROR_ADDR+li,  &lret );
                                if ( lret < 0 ) 
                                {
                                    lbuf[lwp] = '\0';
                                    break;
                                }
                                sprintf( &lbuf[lwp],"%04x ", work );
                                lwp = lwp + 5;
                            }
                            MSG(KERN_ERR_LVL, minor, "%03d: XError Info(16byte) 0x%x: %s", minor, FMB_MW_MSG_XERROR_ADDR, &lbuf[0] );
                            if ( lret < 0 ) MSG(KERN_ERR_LVL, minor, "%03d: XError dump info error %d", minor, lret );
                        }
                    }
                }
#endif /* FMB_OPT_XERROR_WATCH */
                break;
#ifdef MB86E61_FUNCTION
                case NOTIFY_FAILURE_MSG:
                    if ( actual_length != USB_NOTIFY_FAILURE_MSG_SIZE )
                    {
                         MSG(KERN_ERR_LVL, minor, "The size of NOTIFY_FAILURE_MSG format and actual is different.  [INTR]CD:0x%02x,AC:%d,DS:%d,RET:%d"
                             ,*(pData), actual_length, USB_NOTIFY_FAILURE_MSG_SIZE, ret  );
                    }
                    core_priv_p->boot_event_flag = FMB_BOOT_EVENT_ON;
                    FMB_SYS_ABS_wake_up( &core_priv_p->boot_event_wait );
                break;
#endif /* MB86E61_FUNCTION */
                default:/* Reserve */
                    /*--- No data ---*/
                break;
            }

#ifdef TV_LINUX_DEBUG_SUPPORT
            mdelay(200);
#else
            udelay(125);
#endif
        }
    }
    
    /* Notify end-up of kernel thread has completed. */
    complete(&hw_usb->kt_comp);
    kfree(pData);

    return 0;
}

/* specific function */
#ifdef SPECIFIC_FUNCTION_ON
/**
*  @brief           ioctl() FMB_API_SPEC_USB_CONTROL Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in]       argp        command parameter pointer(struct fmb_spec_usb_control *)
*                               Refer to define of struct fmb_spec_usb_control for the explanation of the argument. 
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @note            None
*  @attention       None
*/
int
fmb_hw_spec_usb_control( struct fmb_hard_private* hw_priv_p, unsigned long argp )
{
    int minor = -1;
    int ret = 0;
    struct fmb_spec_usb_control spec_usb_control;
    struct fmb_hw_usb_setup_packet setup_packet;
    struct fmb_hw_usb* usb_priv;
    unsigned char *data_work   = NULL;
    int            iomode_work;
    
    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );

    MSG(INTERNAL_LOG, minor, "START");
    

    if (copy_from_user(&spec_usb_control, (void*)argp, sizeof(struct fmb_spec_usb_control))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_spec_usb_control)");
        return -EFAULT;
    }

    setup_packet.bmRequestType = spec_usb_control.spacket.bmRequestType;
    setup_packet.bRequest      = spec_usb_control.spacket.bRequest;
    setup_packet.wValueLH      = spec_usb_control.spacket.wValueLH;
    setup_packet.wIndexLH      = spec_usb_control.spacket.wIndexLH;
    setup_packet.wLengthLH     = spec_usb_control.spacket.wLengthLH;
    
    spec_usb_control.result = -EPERM;
    
    if ( spec_usb_control.data != NULL && spec_usb_control.data_size > 0 )
    {
        data_work = (unsigned char *)kmalloc( spec_usb_control.data_size, GFP_KERNEL );
        if ( data_work == NULL )
        {
            if ( ret == 0 ) ret = -ENOMEM;
            goto endProc;
        }
    }
    switch( spec_usb_control.io_mode )
    {
    case FMB_SPEC_IOMODE_IN:
        iomode_work = USB_CMDTYPE_IN;
        break;
    case FMB_SPEC_IOMODE_OUT:
        iomode_work = USB_CMDTYPE_OUT;
        break;
    default:
        MSG(KERN_ERR_LVL, minor, "io mode error (%d)\n", spec_usb_control.io_mode );
        if ( ret == 0 ) ret = -EINVAL;
        goto endProc;
    }

    if ( iomode_work == USB_CMDTYPE_OUT )
    {
        if (copy_from_user(data_work, (void*)spec_usb_control.data, spec_usb_control.data_size )) {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_spec_usb_control.data)");
            if ( ret == 0 ) ret = -EFAULT;
            goto endProc;
        }
    }
    spec_usb_control.result = fmb_hw_usb_cmd( usb_priv, &setup_packet, spec_usb_control.endpoint, data_work, spec_usb_control.data_size, spec_usb_control.timeout, iomode_work );
    
    
    if ( iomode_work == USB_CMDTYPE_IN  && spec_usb_control.result >= 0 )
    {
        if (copy_to_user((void *)spec_usb_control.data, data_work, spec_usb_control.data_size ))
        {
            MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_spec_usb_control.data)");
            if ( ret == 0 ) ret = -EFAULT;
            goto endProc;
        }
    }

endProc:
    
    if ( data_work != NULL ) kfree( data_work ) ;
    
    if (copy_to_user((void *)argp, &spec_usb_control, sizeof(struct fmb_spec_usb_control)))
    {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_spec_usb_control)");
        if ( ret == 0 ) ret = -EFAULT;
    }
    
    if ( ret == 0 && spec_usb_control.result < 0 )  ret = spec_usb_control.result;
    
    MSG(INTERNAL_LOG, minor, "END");

    return ret;
    
}
/**
*  @brief           ioctl() FMB_API_SPEC_WAIT_BIT Command Function
*  @param[in]       hw_priv_p   private data pointer
*  @param[in]       argp        command parameter pointer(struct fmb_spec_wait_bit *)
*                               Refer to define of struct fmb_spec_wait_bit for the explanation of the argument. 
*  @return          0           normal end
*  @return          -EFAULT     user buffer(argp) access error
*  @return          -EINVAL     invalid content of parameter
*  @return          -ETIMEDOUT  time out. The value has not changed.
*  @return          -ECANCELED  wait cancel or remove(becomes -ENODEV occasionally. ). Refer to .cret member for the reason.  
*  @note            None
*  @attention       None
*/
int
fmb_hw_spec_wait_bit( struct fmb_hard_private* hw_priv_p, unsigned long argp )
{
    int             minor           = -1;
    int             result          =  0;
    int             result_internal =  0;
    struct fmb_spec_wait_bit    wait_bit;
    struct fmb_hw_usb*          usb_priv;
    struct fmb_core_private*    priv_p;
    unsigned short  mdata;
    unsigned short  val             =  0;
    unsigned long   limit_num       =  0;
    unsigned long   MinimumTime     =  0;
    unsigned long   pTimeout        =  0;
    
    PRIV_P_CHECK( hw_priv_p, -ENODEV );
    
    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    usb_priv = (struct fmb_hw_usb*)hw_priv_p->dev_priv;
    PRIV_P_CHECK( usb_priv, -ENODEV );
    
    MSG(INTERNAL_LOG, minor, "START");
    

    if (copy_from_user(&wait_bit, (void*)argp, sizeof(struct fmb_spec_wait_bit))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy from user's area.(fmb_spec_wait_bit)");
        return -EFAULT;
    }
    wait_bit.cret   = 0;
    wait_bit.status = 0;
    
    mdata = ~(wait_bit.en_mask);
    
    if ( wait_bit.interval_time == 0 )
    {
        return -EINVAL;
    }
    else
    {
        MinimumTime = 1000/HZ;
        if ( wait_bit.interval_time < MinimumTime )
        {
            MSG(INTERNAL_ERR_LVL, minor, "force set interval time.(%ld<---%ld)", MinimumTime, wait_bit.interval_time );
            pTimeout = MinimumTime;
        }
        else
        {
            pTimeout = wait_bit.interval_time;
        }
    }
    priv_p->wait_bit_act_flag = 1;
    
    for(;;)
    {
        /* get irq status */
        val = Fmb_hw_reg_read_lsi( hw_priv_p, wait_bit.offset,  &result_internal );
		if ( result_internal < 0 )
		{
			wait_bit.cret  = FMB_WAIT_BIT_REMOVE_CANCEL;
			 break;
		}

        if ( ( val & mdata ) != 0 )
        {
            wait_bit.cret   = FMB_WAIT_BIT_EVENT;
            wait_bit.status = val;
            result           = 0;
            break;

        }
        
        if ( priv_p->wait_bit_type == FMB_WAIT_BIT_REMOVE_CANCEL )
        {
            result          = -ECANCELED;
            wait_bit.cret  = FMB_WAIT_BIT_REMOVE_CANCEL;
            break;
        }

        if ( wait_bit.num != 0 )
        {
            if ( wait_bit.num <= limit_num )
            {
                result = -ETIMEDOUT;
                break;
            }
            else
            {
                limit_num++;
            }
        }
        result = FMB_SYS_ABS_wait_event_timeout(priv_p->wait_bit_wait,
                          priv_p->wait_bit_flag == FMB_WAIT_BIT_ON,
                          pTimeout );
        priv_p->wait_bit_flag = FMB_WAIT_BIT_OFF;

        if ( result < 0 )
        {
            MSG(INTERNAL_ERR_LVL, minor, "wait_event_timeout() error.(%d)", result );
            break;
        }

        switch( priv_p->wait_bit_type )
        {
        case FMB_WAIT_BIT_IOCTL_CANCEL:
                result = -ECANCELED;
                wait_bit.cret        = FMB_WAIT_BIT_IOCTL_CANCEL;
                priv_p->wait_bit_type = FMB_WAIT_BIT_NO_CANCEL;
                break;
        case FMB_WAIT_BIT_REMOVE_CANCEL:
                result = -ECANCELED;
                wait_bit.cret        = FMB_WAIT_BIT_REMOVE_CANCEL;  /* or -ENODEV[return Fmb_hw_reg_read_lsi()] */
                break;
        default:
                break;
        }
        if ( result < 0 ) break;
    }

    priv_p->wait_bit_act_flag = 0;

    if ( result_internal < 0 ) return result_internal;

    if (copy_to_user( (void*)argp, &wait_bit, sizeof(struct fmb_spec_wait_bit))) {
        MSG(KERN_ERR_LVL, minor, "Failed in a copy to user's area.(fmb_spec_wait_bit)");
        return -EFAULT;
    }

    MSG(INTERNAL_LOG, minor, "END");

    return result;
    
}
/**
*  @brief       ioctl() FMB_API_CANCEL_WAIT_BIT Command Function
*  @param[in]   hw_priv_p   private data pointer
*  @return      0           normal end
*  @note        None
*  @attention   None
*/
static int fmb_hw_cancel_wait_bit(struct fmb_hard_private*  hw_priv_p)
{
    int minor;
    struct fmb_core_private*    priv_p;

    PRIV_P_CHECK( hw_priv_p, -ENODEV );

    minor = hw_priv_p->minor;
    priv_p = hw_priv_p->core_priv_p;
    PRIV_P_CHECK( priv_p, -ENODEV );
    MSG(INTERNAL_LOG, minor, "START");

    priv_p->wait_bit_type = FMB_WAIT_BIT_IOCTL_CANCEL;
    priv_p->wait_bit_flag = FMB_WAIT_BIT_ON;
    FMB_SYS_ABS_wake_up( &priv_p->wait_bit_wait );

    MSG(INTERNAL_LOG, minor, "END");
    return 0;
}

#endif /* SPECIFIC_FUNCTION_ON */

#ifdef TV_LINUX_FUNCTION
#ifndef USE_UPPER_OS
#define USB_DEVICE_AND_INTERFACE_INFO(vend, prod, cl, sc, pr) \
        .match_flags = USB_DEVICE_ID_MATCH_INT_INFO \
                | USB_DEVICE_ID_MATCH_DEVICE, \
        .idVendor = (vend), \
        .idProduct = (prod), \
        .bInterfaceClass = (cl), \
        .bInterfaceSubClass = (sc), \
        .bInterfaceProtocol = (pr)
#endif
#endif

/* ------------------------------------------------------------------ */
/* USB ids + subsystem IDs                                            */
/* ------------------------------------------------------------------ */
static struct usb_device_id s_Fmb_hw_usb_tbl[] = {
    {
        USB_DEVICE(USB_VENDOR_ID_FUJITSU,USB_PRODUCT_ID_FUJITSU_MB86H57H58)
    },
    {
        USB_DEVICE(USB_VENDOR_ID_INTERNALROM,USB_PRODUCT_ID_FUJITSU_MB86H57H58)
    },
#ifdef MB86E61_FUNCTION
    {
        USB_DEVICE_AND_INTERFACE_INFO(USB_VENDOR_ID_FUJITSU,USB_PRODUCT_ID_FUJITSU_MB86E61, 0xFF, 0x00, 0x00)
    },
#endif
    { }             /* Terminating entry */
};
MODULE_DEVICE_TABLE (usb, s_Fmb_hw_usb_tbl);

struct usb_driver s_Fmb_hw_usb_driver = {
    .name       =   FMB_DRIVER_NAME,
    .id_table   =   s_Fmb_hw_usb_tbl,
    .probe      =   fmb_hw_usb_probe,
    .disconnect =   fmb_hw_usb_remove,
};

/**
*    @brief        get usb driver info
*    @return       usb driver info
*    @note         None
*    @attention    None
*/
struct usb_driver *
fmb_hw_usb_get_driver()
{
    return &s_Fmb_hw_usb_driver;
}


/**
*    @brief        hw driver abstruct device open
*    @param[in]    inode   linux fops struct inode pointer
*    @param[in]    file    linux fops struct file  pointer
*    @return       
*    @note         None
*    @attention    None
*/
int fmb_hw_device_open(  struct inode* ip_p, struct file* fp_p  )
{
    int ret;

    ret = fmb_hw_usb_open( ip_p, fp_p );
    
    return ret;
}

/**
*    @brief        hw driver abstruct device release(close)
*    @param[in]    inode   linux fops struct inode pointer
*    @param[in]    file    linux fops struct file  pointer
*    @return       
*    @note         None
*    @attention    None
*/
int fmb_hw_device_release( struct inode* ip_p, struct file* fp_p )
{
    int ret;
    
    ret = fmb_hw_usb_release( ip_p, fp_p );
    
    return ret;
}

/**
*    @brief        hw driver abstruct device ioctl
*    @param[in]    priv_hw_p  private data pointer
*    @param[in]    cmd        command id
*    @param[io]    arg        argument parameter
*    @return       
*    @note         None
*    @attention    None
*/
int fmb_hw_device_ioctl( struct fmb_hard_private* priv_hw_p, unsigned int cmd, unsigned long arg )
{
    int rc = 0;
    
    switch( cmd )
    {
    case FMB_SET_READ_EPADDR:
        rc = fmb_hw_usb_set_read_epaddr( priv_hw_p, (int)arg );
        break;
    case FMB_API_GET_WRITE_SPI_STATE:
        rc = fmb_hw_usb_get_write_spi_state( priv_hw_p, (int)arg );
        break;
    case FMB_API_LOAD_FIRM:
        rc = fmb_hw_usb_load_firm( priv_hw_p, (int)arg );
        break;
    case FMB_API_GET_DEVICE_INFO:
        rc = fmb_hw_usb_get_device_info( priv_hw_p, (int)arg );
        break;
#ifdef SPECIFIC_FUNCTION_ON
    case FMB_API_SPEC_USB_CONTROL:
        rc = fmb_hw_spec_usb_control( priv_hw_p, (int)arg );
        break;
    case FMB_API_SPEC_WAIT_BIT:
        rc = fmb_hw_spec_wait_bit( priv_hw_p, (int)arg );
        break;
    case FMB_API_SPEC_CANCEL_WAIT_BIT:
        rc = fmb_hw_cancel_wait_bit( priv_hw_p );
        break;
#endif /* SPECIFIC_FUNCTION_ON */
    case FMB_USB_SET_DEBUG_MSG:
#ifdef DEBUGON_USB_ACCESS
        rc = fmb_hw_usb_set_debug_msg( priv_hw_p, (int)arg );
#endif /* DEBUGON_USB_ACCESS */
        break;
    default:
        rc = -EINVAL;
        break;
    }
    return rc;
}


/**
*    @brief        hw driver abstruct device register
*    @return       
*    @note         None
*    @attention    None
*/
int fmb_hw_device_register( void )
{
    int rc;
    
    rc = fmb_hw_usb_register((long*)(&s_Fmb_hw_usb_driver));
    
    return rc;
}

/**
*    @brief        hw driver abstruct device unregister
*    @return       
*    @note         None
*    @attention    None
*/
void    fmb_hw_device_unregister( void )
{
    fmb_hw_usb_unregister((long*)(&s_Fmb_hw_usb_driver));
    return;
}


