/** @file aip.c
 *  
 * Auto IP control function
 * 
 * Copyright (c) 2007 SHARP CORPORATION
 * 
 * @author NWPF Makoto Adachi
 *
 * @change 2007/12/13 adachi@nwpf new 
 */
/* ---- standard header inclusion ---- */
#include <unistd.h> /* pipe() write() */
#include <stdlib.h> 
#include <errno.h>  /* errno */
#include <pthread.h>/* pthread_create(), pthread_mutex_init(), pthread_mutex_lock(), pthread_mutex_unlock() pthread_cond_init() pthread_cond_signal() pthread_cond_wiat() */
#include <string.h>

/* ---- original header inclusion ---- */
#include "sysapi.h"
#include "aip.h"
#include "AVH-IPv4LL.h"
#ifndef AIP_UNIT_TEST
#ifndef LINKED_LNX_AIP
//#include "lnxipfc.h"
#endif /* ^d LINKED_LNX_AIP */
#endif /* ^d AIP_UNIT_TEST */

/* ---- private macro definitions ---- */
#define AIP_PIPE_FD_NUM (2)

/* ---- public variable definitions ---- */
/* none */

/* ---- private type definitions ---- */
typedef struct
{
	boolean isEnable;
	uint8 aucAsignedAddr[IPV4_ADDR_LEN];
	boolean isReady;
	int32 iArpFd;
	boolean fHaveInitAddr;
}dAipState_t;

typedef struct
{
	pthread_t tAutoIpTaskHandle;
	pthread_mutex_t tAutoIpMutex;
	pthread_cond_t tAutoIpCond;
	int32 aiPipeFd[AIP_PIPE_FD_NUM];
	uint8 aucMessage[AIP_MSG_DATA_LEN];
}dAipInfo_t;

/* ---- private variable definitions ---- */
static dAipState_t tAipState;
static dAipInfo_t tAipInfo;

/* ---- private function declarations ---- */
static dResult_e eAipReset( uint8 *pucInterFace );
static void vStateInit(void);
static dResult_e eAipSetDisable(void);
static dResult_e eAipSetEnable(boolean fProbe,uint8 *pucAddr);
static boolean fIsAddressAsigned(uint8 *pucAddr);

/* ---- public function implementations ---- */
/***** XIMδؿ *****/
/**
 * Auto IP⥸塼ν
 * pipeκȡεư
 *
 * @return S_SUCCESS,ԤS_FAILURE
 */
extern dResult_e eAipInit( uint8 *pucInterFace )
{
	int32 iRet;
	static flag fIsAlreadyInit = FALSE;
	
	eDbgPrintf( DBG_LOG, "AIP : init %s\n", pucInterFace );
	if( fIsAlreadyInit )
	{
		eDbgPrintf( DBG_LOG, "AIP : Already Init\n" );
		if( strcmp( pcAvhGetIfName(), pucInterFace ) != 0 )
		{
			eDbgPrintf( DBG_LOG, "AIP : need update\n" );
			eAipReset( pucInterFace );
		}
		else
		{
			eDbgPrintf( DBG_LOG, "AIP : need not update\n" );
		}
		return S_SUCCESS;
	}

	pthread_mutex_init(&(tAipInfo.tAutoIpMutex),NULL);
	pthread_cond_init(&(tAipInfo.tAutoIpCond),NULL);

	vStateInit();
	iAvhInitParams( pucInterFace );

	iRet = pipe(tAipInfo.aiPipeFd);
	if(0 != iRet)
	{
		eDbgPrintf(DBG_FATAL_ERROR,"AIP: pipe create err(%d)\n",errno);
		return S_FAILURE;
	}

	iRet = pthread_create(&(tAipInfo.tAutoIpTaskHandle), NULL, vAutoIpTask, &(tAipInfo.aiPipeFd[0]) );

	if(0 != iRet)
	{
		eDbgPrintf(DBG_FATAL_ERROR,"AIP: auto ip thread create err(%d)\n",iRet);
		return S_FAILURE;
	}

	fIsAlreadyInit = TRUE;
	return S_SUCCESS;
}

/**
 * Auto IPξ֤enableˤ
 * ֤ѹAIP Taskؤ
 *
 * @return S_SUCCESS,ԤS_FAILURE
 */
extern dResult_e eAipEnableAutoIp(uint8 *pcAddr,uint32 uiAddrLen,boolean fProbe)
{
	dResult_e eRet;
	
	if( NULL == pcAddr )
	{
		return S_FAILURE;
	}

	if(IPV4_ADDR_LEN != uiAddrLen)
	{
		return S_FAILURE;
	}

	eRet = eAipSetEnable(fProbe,pcAddr); /* ֤ѹȥå */

	return eRet;
}

/**
 * Auto IPξ֤disableˤ
 * ֤ѹAIP Taskؤ
 * AIP Taskˤäarp socketcloseޤԤ
 *
 * @return S_SUCCESS,ԤS_FAILURE
 */
extern dResult_e eAipDisableAutoIp(void)
{
	return eAipSetDisable();
}

/**
 * RESET I/F Auto IP
 *
 * @return S_SUCCESS,ԤS_FAILURE
 */
static dResult_e eAipReset( uint8 *pucInterFace )
{
	int32 iRet;

	eDbgPrintf(DBG_LOG,"LNX : IN eAipReset IFNAME:%s;\n",pucInterFace);
	eDbgPrintf(DBG_LOG,"LNX : CALL eAipSetDisable\n");
	(void)eAipSetDisable();
	eDbgPrintf(DBG_LOG,"LNX : CALL vStateInit\n");
	vStateInit();
	eDbgPrintf(DBG_LOG,"LNX : CALL iAvhInitParams2\n");
	iAvhInitParams2( pucInterFace );
	eDbgPrintf(DBG_LOG,"LNX : END eAipReset\n");
	return S_SUCCESS;
}

/***** AIP task  *****/
extern boolean fAipStateCheck(boolean *isReady,boolean *fAssigned,uint8 *pucAddr)
{
	boolean fRet = TRUE;

	pthread_mutex_lock(&(tAipInfo.tAutoIpMutex));
	
	*isReady = tAipState.isReady;

	if(!(tAipState.isEnable))
	{
		vCloseArpSocket();

		pthread_cond_signal(&(tAipInfo.tAutoIpCond));
		fRet = FALSE;
		tAipState.fHaveInitAddr = FALSE;
		*fAssigned = FALSE;
	}
	else
	{
		if(tAipState.fHaveInitAddr)
		{ /* eAipSetEnableƤФ줿ľθƤӽФΤ */
			memcpy(pucAddr,tAipState.aucAsignedAddr,IPV4_ADDR_LEN);
			tAipState.fHaveInitAddr = FALSE;
			*fAssigned              = TRUE;
		}
		else
		{
			*fAssigned = FALSE;
		}

		/* ׸Ƥ ɬפ*/
		pthread_cond_signal(&(tAipInfo.tAutoIpCond));
	}

	pthread_mutex_unlock(&(tAipInfo.tAutoIpMutex));

	return fRet;
}

extern void vAipForceDisable(void)
{
	pthread_mutex_lock(&(tAipInfo.tAutoIpMutex));
	
	tAipState.isEnable      = FALSE;
	tAipState.fHaveInitAddr = FALSE;
	vCloseArpSocket();

	pthread_mutex_unlock(&(tAipInfo.tAutoIpMutex));

	return;
}

extern boolean fGetAssignedAddr(uint8 *pucBuf)
{
	boolean fAsigned;
	
	pthread_mutex_lock(&(tAipInfo.tAutoIpMutex));

	fAsigned = fIsAddressAsigned(tAipState.aucAsignedAddr);

	if(fAsigned)
	{
		//vpFastMemcpy(pucBuf,tAipState.aucAsignedAddr,IPV4_ADDR_LEN);
		memcpy(pucBuf,tAipState.aucAsignedAddr,IPV4_ADDR_LEN);
	}
	
	pthread_mutex_unlock(&(tAipInfo.tAutoIpMutex));

	return fAsigned;
}

extern void vAipNotifyAddress(uint8 *pucAddr,uint32 uiAddrLen)
{
	dResult_e eRet;
	
	eRet = eAipNotifyAddress(pucAddr,uiAddrLen);

	if(S_SUCCESS != eRet)
	{
		eDbgPrintf(DBG_ERROR,"AIP: Address Notify is fail\n");
	}

	return;
}

/* ---- private function implementations ---- */
static void vStateInit(void)
{
	int i;
	pthread_mutex_lock(&(tAipInfo.tAutoIpMutex));
	
	tAipState.isEnable = FALSE;
	tAipState.isReady  = FALSE;

	for(i = 0; i < IPV4_ADDR_LEN; i++)
	{
		tAipState.aucAsignedAddr[i] = 0;
	}

	for(i = 0; i < AIP_MSG_DATA_LEN; i++)
	{
		tAipInfo.aucMessage[i] = 0;
	}

	pthread_mutex_unlock(&(tAipInfo.tAutoIpMutex));
}

static dResult_e eAipSetEnable(boolean fProbe,uint8 *pucAddr)
{
	dResult_e eRet = S_SUCCESS;
	boolean fAsigned;
	size_t 	iWriteLen;

	pthread_mutex_lock(&(tAipInfo.tAutoIpMutex));
	
	//vpFastMemcpy(tAipState.aucAsignedAddr,pucAddr,IPV4_ADDR_LEN);

	fAsigned = fIsAddressAsigned(pucAddr);
	if(fAsigned)
	{
		tAipState.fHaveInitAddr = TRUE;
		memcpy(tAipState.aucAsignedAddr,pucAddr,IPV4_ADDR_LEN);
	}

	tAipState.isEnable = TRUE;
	if(fProbe)
	{
		vAvhInitState();
		tAipState.isReady = FALSE;
	}
	else
	{
		tAipState.isReady = TRUE;
	}
	
	tAipState.iArpFd = iOpenArpSocket();

	if(	-1 == tAipState.iArpFd)
	{
		eRet = S_FAILURE;
	}

	/* message for AIP task */
	
	vAvhPrintfMessageWithTime("write enable message\n");

	iWriteLen =	write(tAipInfo.aiPipeFd[1],tAipInfo.aucMessage,AIP_MSG_DATA_LEN);

	if(iWriteLen < 0)
	{
		eDbgPrintf(DBG_ERROR,"Aip Enable write err %d\n",errno);
		eRet = S_FAILURE;
	}
	else
	{
		pthread_cond_wait(&(tAipInfo.tAutoIpCond),&(tAipInfo.tAutoIpMutex));/*ɬפ׸Ƥ */
	}
	pthread_mutex_unlock(&(tAipInfo.tAutoIpMutex));
	
	return eRet;
}

static dResult_e eAipSetDisable(void)
{
	dResult_e eRet = S_SUCCESS;
	size_t 	iWriteLen;
	
	pthread_mutex_lock(&(tAipInfo.tAutoIpMutex));

	tAipState.isEnable = FALSE;

	/* message for AIP task */
	vAvhPrintfMessageWithTime("write disable message\n");
	iWriteLen =	write(tAipInfo.aiPipeFd[1],tAipInfo.aucMessage,AIP_MSG_DATA_LEN);

	if(iWriteLen < 0)
	{
		eDbgPrintf(DBG_ERROR,"Aip Disable write err %d\n",errno);
		eRet = S_FAILURE;
	}
	else
	{/* arp socket λԤ*/
		pthread_cond_wait(&(tAipInfo.tAutoIpCond),&(tAipInfo.tAutoIpMutex));
	}

	pthread_mutex_unlock(&(tAipInfo.tAutoIpMutex));
	
	return eRet;
}

static boolean fIsAddressAsigned(uint8 *pucAddr)
{
	if( NULL == pucAddr )
	{
		return FALSE;
	}
	if( (0 == pucAddr[0]) &&
		(0 == pucAddr[1]) &&
		(0 == pucAddr[2]) &&
		(0 == pucAddr[3]) )
	{
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}

#ifdef LINKED_LNX_AIP

extern void vEtmNotifyAimAddress(uint8 *pucAddr,uint32 uiAddrLen);
extern dResult_e eAipNotifyAddress(uint8 *pucAddr,uint32 uiAddrLen)
{
	eDbgPrintf(DBG_TEST,
			   "eAipNotifyAddress addr=%u.%u.%u.%u\n",
			   pucAddr[0],
			   pucAddr[1],
			   pucAddr[2],
			   pucAddr[3]);

	vEtmNotifyAimAddress(pucAddr,uiAddrLen);

	return S_SUCCESS;
}
#endif /* d LINKED_LNX_AIP */

#ifdef AIP_UNIT_TEST

#include <stdarg.h>
static boolean fGetAddr = FALSE;
static uint8 aucGetAddr[4];
static void v_eAipEnableTest(int iTestSubNum);
static void v_eAipInitTest(void);
static void v_eAipEnableTest(int iTestSubNum);
static void v_eAipDisableTest(int iTestSubNum);
static void v_eAipConflictTest(void);
static void v_eAipTestPrintResult(uint8 *pucFuncName,dResult_e fFuncRet,boolean fResult);
static void v_eAipEnableTestSub1(uint8 *pucAddr,uint32 uiLen,boolean fProbe);
static void v_eAipEnableTestSub2(uint8 *pucAddr,uint32 uiLen,boolean fProbe);

extern dResult_e eAipNotifyAddress(uint8 *pucAddr,uint32 uiAddrLen)
{


	if(IPV4_ADDR_LEN != uiAddrLen)
	{
		eDbgPrintf(DBG_TEST,"BAD addr len %d\n",uiAddrLen);
	}

	eDbgPrintf(DBG_TEST,
			   "eAipNotifyAddress addr=%u.%u.%u.%u\n",
			   pucAddr[0],
			   pucAddr[1],
			   pucAddr[2],
			   pucAddr[3]);

	memcpy(aucGetAddr,pucAddr,4);
	fGetAddr = TRUE;

	return S_SUCCESS;

}
extern dResult_e eDbgPrintf( int32 iLevel, char *fmt, ... )
{
	va_list list;
		
	va_start(list, fmt); 
	vfprintf(stdout, fmt, list); 

	va_end(list);

	return S_SUCCESS;
}

void *vpFastMemset( void *vpBuf, int cPat, size_t tSize )
{
	return memset(vpBuf,cPat,tSize);
}

void *vpFastMemcpy(void *dst,const void *src, size_t bytes)
{
	return memcpy(dst,src,bytes);
}


int main(int argc, char *argv[])
{
	int iTestNum = 1;
	int iTestSubNum = 1;	
	if( argc > 1)
	{
		iTestNum = atoi(argv[1]);
	}
	switch(iTestNum)
	{
	case 1:
		v_eAipInitTest();
		break;
	case 2:
		if( argc > 2 )
		{
			iTestSubNum = atoi(argv[2]);
		}
		v_eAipEnableTest(iTestSubNum);
		break;
	case 3:
		if( argc > 2 )
		{
			iTestSubNum = atoi(argv[2]);
		}
		v_eAipDisableTest(iTestSubNum);
		break;
	case 4:
		v_eAipConflictTest();
		break;
	default:
		eDbgPrintf(DBG_TEST,"unknown test num\n");
		break;
	}
	return 0;
}

static void v_eAipInitTest(void)
{
	dResult_e eRet;

	eRet = eAipInit();
	if(S_SUCCESS == eRet)
	{
		eDbgPrintf(DBG_TEST,"eAipInit OK\n");
	}
	else
	{
		eDbgPrintf(DBG_TEST,"eAipInit NG\n");
	}
}

static void v_eAipEnableTest(int iTestSubNum)
{
	dResult_e eRet;
	uint8 aucZeroAddr[4] = {0,0,0,0};
	uint8 aucTestAddr[4] = {169,254,1,234};
	
	eRet = eAipInit();
	if(S_SUCCESS == eRet)
	{
		eDbgPrintf(DBG_TEST,"eAipInit OK\n");
	}
	else
	{
		eDbgPrintf(DBG_TEST,"eAipInit NG\n");
		return;
	}
	sleep(2);
	
	switch (iTestSubNum)
	{
	case 1:
		v_eAipEnableTestSub1(aucZeroAddr,4,TRUE);
		break;
	case 2:
		v_eAipEnableTestSub1(aucTestAddr,4,TRUE);
		break;
	case 3:
		v_eAipEnableTestSub1(aucTestAddr,4,FALSE);
		break;
	case 4:
		v_eAipEnableTestSub2(aucZeroAddr,4,TRUE);
		break;
	case 5:
		v_eAipEnableTestSub2(aucTestAddr,4,TRUE);
		break;
	case 6:
		v_eAipEnableTestSub2(aucTestAddr,4,FALSE);
		break;
	case 7:
		eRet = eAipEnableAutoIp(NULL,4,TRUE);
		v_eAipTestPrintResult("eAipEnableAutoIp",eRet,FALSE);
		break;
	case 8:
		eRet = eAipEnableAutoIp(aucTestAddr,0,TRUE);
		v_eAipTestPrintResult("eAipEnableAutoIp",eRet,FALSE);
		break;
	case 9:
#if 0	/* AutoIP ܺٻԶNo.2б */
		eRet = eAipEnableAutoIp(aucZeroAddr,4,FALSE);
#else   /* AutoIP ܺٻԶNo.2б */

		eRet = eAipEnableAutoIp(aucZeroAddr,0,FALSE);
#endif   /* AutoIP ܺٻԶNo.2б */
		v_eAipTestPrintResult("eAipEnableAutoIp",eRet,TRUE);
		break;
	case 10:
		eRet = eAipEnableAutoIp(aucZeroAddr,4,TRUE);
		if(S_SUCCESS == eRet)
		{
			eRet = eAipEnableAutoIp(aucZeroAddr,4,TRUE);

			v_eAipTestPrintResult("second eAipEnableAutoIp",eRet,FALSE);
		}
		else
		{
			eDbgPrintf(DBG_TEST,"cannot try test\n");
		}
		break;
	default :
		eDbgPrintf(DBG_TEST,"unknown test num\n");
		break;
	}
}

static void v_eAipEnableTestSub1(uint8 *pucAddr,uint32 uiLen,boolean fProbe)
{
	dResult_e eRet;
	int32 iWait = 5;
	uint8 aucZeroAddr[4] = {0,0,0,0};
	int32 iCount = 0;

	eRet = eAipEnableAutoIp(pucAddr,uiLen,fProbe);

	while(TRUE)
	{
		eDbgPrintf(DBG_TEST,"wait %d sec for address assigned\n",iWait);
		sleep(iWait);
		if( fGetAddr )
		{
			break;
		}
		iCount++;

		if(iCount > 6)
		{
			eRet = eAipDisableAutoIp();
			if(S_SUCCESS != eRet)
			{
				eDbgPrintf(DBG_TEST,"eAipDisableAutoIp fail\n");
				break;
			}
			eRet = eAipEnableAutoIp(aucZeroAddr,4,TRUE);
			if(S_SUCCESS != eRet)
			{
				eDbgPrintf(DBG_TEST,"eAipEnableAutoIp retry fail\n");
				break;
			}
			iCount = 0;
		}
	}
	v_eAipTestPrintResult("eAipEnableAutoIp",eRet,TRUE);
	return;
}

static void v_eAipEnableTestSub2(uint8 *pucAddr,uint32 uiLen,boolean fProbe)
{
	dResult_e eRet;
	uint8 aucZeroAddr[4] = {0,0,0,0};
	int32 iWait = 5;
	int32 iCount = 0;
	
	eRet = eAipEnableAutoIp(aucZeroAddr,4,TRUE);
	if(S_SUCCESS != eRet)
	{
		eDbgPrintf(DBG_TEST,"cannot try test\n");
		return;
	}

	while(TRUE)
	{
		eDbgPrintf(DBG_TEST,"wait %d sec for address assigned\n",iWait);
		sleep(iWait);
		if( fGetAddr )
		{
			fGetAddr = FALSE;
			break;
		}
	}
	
	eRet = eAipDisableAutoIp();
	
	if(S_SUCCESS != eRet)
	{
		eDbgPrintf(DBG_TEST,"cannot try test\n");
	}

	eRet = eAipEnableAutoIp(pucAddr,uiLen,fProbe);

	while(TRUE)
	{
		eDbgPrintf(DBG_TEST,"wait %d sec for address assigned\n",iWait);
		sleep(iWait);
		if( fGetAddr )
		{
			break;
		}

		iCount++;

		if(iCount > 6)
		{
			eRet = eAipDisableAutoIp();
			if(S_SUCCESS != eRet)
			{
				eDbgPrintf(DBG_TEST,"eAipDisableAutoIp fail\n");
				break;
			}
			eRet = eAipEnableAutoIp(aucZeroAddr,4,TRUE);
			if(S_SUCCESS != eRet)
			{
				eDbgPrintf(DBG_TEST,"eAipEnableAutoIp retry fail\n");
				break;
			}
			iCount = 0;
		}
	}
	v_eAipTestPrintResult("eAipEnableAutoIp",eRet,TRUE);


	return;
}

static void v_eAipDisableTest(int iTestSubNum)
{
	dResult_e eRet;
	uint8 aucZeroAddr[4] = {0,0,0,0};
	
	eRet = eAipInit();
	if(S_SUCCESS == eRet)
	{
		eDbgPrintf(DBG_TEST,"eAipInit OK\n");
	}
	else
	{
		eDbgPrintf(DBG_TEST,"eAipInit NG\n");
		return;
	}
	sleep(2);

	switch (iTestSubNum)
	{
	case 1:
		eRet = eAipEnableAutoIp(aucZeroAddr,4,TRUE);
		if(S_SUCCESS != eRet)
		{
			eDbgPrintf(DBG_TEST,"cannot try test\n");
			return;
		}
		eRet = eAipDisableAutoIp();
		eDbgPrintf(DBG_TEST,"wait 60 sec for address assigned\n");
		sleep(60);

		if( fGetAddr )
		{
			eRet = S_FAILURE;
		}
		else
		{
			eRet = S_SUCCESS;
		}
		v_eAipTestPrintResult("quick eAipDisableAutoIp",eRet,TRUE);
		
		break;
	case 2:
		eRet = eAipDisableAutoIp();
		v_eAipTestPrintResult("eAipDisableAutoIp",eRet,TRUE);
		break;
	case 3:
		eRet = eAipEnableAutoIp(aucZeroAddr,4,TRUE);
		if(S_SUCCESS != eRet)
		{
			eDbgPrintf(DBG_TEST,"cannot try test\n");
			return;
		}
		eRet = eAipDisableAutoIp();
		if(S_SUCCESS != eRet)
		{
			eDbgPrintf(DBG_TEST,"cannot try test\n");
			return;
		}
		eRet = eAipDisableAutoIp();
		v_eAipTestPrintResult("eAipDisableAutoIp",eRet,TRUE);
		break;
	default:
		eDbgPrintf(DBG_TEST,"unknown test num\n");
		break;
	}
}

static void v_eAipConflictTest(void)
{
	int iWait = 5;
	dResult_e eRet;
	uint8 aucAddr[4] = {0};

	eRet = eAipInit();
	if(S_SUCCESS == eRet)
	{
		eDbgPrintf(DBG_TEST,"eAipInit OK\n");
	}
	else
	{
		eDbgPrintf(DBG_TEST,"eAipInit NG\n");
		exit(0);
	}

	iWait = 2;
	eDbgPrintf(DBG_TEST,"%d sec waiting...\n",iWait);
	sleep(iWait);

	eRet = eAipEnableAutoIp(aucAddr,4,TRUE);
	if(S_SUCCESS == eRet)
	{
		eDbgPrintf(DBG_TEST,"eAipEnableAutoIp OK\n");
	}
	else
	{
		eDbgPrintf(DBG_TEST,"eAipEnableAutoIp NG\n");
	}

	iWait = 3600;
	eDbgPrintf(DBG_TEST,"%d sec waiting...\n",iWait);
	sleep(iWait);
}

static void v_eAipTestPrintResult(uint8 *pucFuncName,dResult_e fFuncRet,boolean fResult)
{
	eDbgPrintf(DBG_TEST,"%s ",pucFuncName);
	if(S_SUCCESS == fFuncRet)
	{
		eDbgPrintf(DBG_TEST,"OK. ");
		if(fResult)
		{
			eDbgPrintf(DBG_TEST,"Test is OK.\n");
		}
		else
		{
			eDbgPrintf(DBG_TEST,"Test is NG.\n");
		}
	}
	else
	{
		eDbgPrintf(DBG_TEST,"NG. ");
		if(fResult)
		{
			eDbgPrintf(DBG_TEST,"Test is NG.\n");
		}
		else
		{
			eDbgPrintf(DBG_TEST,"Test is OK.\n");
		}
	}


	return;
}
#endif /* AIP_UNIT_TEST */

/* EOF */



